/** @file wlan_fw.c
  * @brief This file contains the initialization for FW and HW 
  * 
  * (c) Copyright  2003-2006, Marvell International Ltd. 
  * All Rights Reserved
  *
  * This software file (the "File") is distributed by Marvell International 
  * Ltd. under the terms of the GNU General Public License Version 2, June 1991 
  * (the "License").  You may use, redistribute and/or modify this File in 
  * accordance with the terms and conditions of the License, a copy of which 
  * is available along with the File in the license.txt file or by writing to 
  * the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 
  * 02111-1307 or on the worldwide web at http://www.gnu.org/licenses/gpl.txt.
  *
  * THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE 
  * IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE 
  * ARE EXPRESSLY DISCLAIMED.  The License provides additional details about 
  * this warranty disclaimer.
  *
  */
/********************************************************
Change log:
	09/28/05: Add Doxygen format comments
	01/05/06: Add kernel 2.6.x support	
	01/11/06: Conditionalize new scan/join functions.
	          Cleanup association response handler initialization.
	01/06/05: Add FW file read

********************************************************/

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/config.h>

#include <linux/vmalloc.h>
#include <linux/firmware.h>
#include <linux/version.h>

#include "include.h"

/********************************************************
		Local Variables
********************************************************/

char *helper_name=NULL;
char *fw_name=NULL;

module_param(helper_name, charp, 0644);
module_param(fw_name, charp, 0644);

/********************************************************
		Global Variables
********************************************************/

/********************************************************
		Local Functions
********************************************************/

/** 
 *  @brief This function downloads firmware image, gets
 *  HW spec from firmware and set basic parameters to
 *  firmware.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_setup_station_hw(wlan_private *priv)
{
	int 		ret = WLAN_STATUS_SUCCESS;
	wlan_adapter   *adapter = priv->adapter;
	u16		dbm = MRVDRV_MAX_TX_POWER;
 	u8           TempAddr[MRVDRV_ETH_ADDR_LEN];
	

    	ENTER();
    
	sbi_disable_host_int(priv);


	if ((ret=request_firmware(&priv->firmware, fw_name, priv->hotplug_device)) < 0) {
		PRINTM(FATAL, "request_firmware() failed, error code = %#x\n", ret);
		goto done;
	}

	ret = sbi_prog_firmware(priv);

	release_firmware(priv->firmware);

	if (ret) {
		PRINTM(INFO, "Bootloader in invalid state!\n");
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}
	

	/* check if the fimware is downloaded successfully or not */
	if (sbi_verify_fw_download(priv)) {
		PRINTM(INFO, "Firmware failed to be active in time!\n");
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

#define RF_REG_OFFSET 0x07
#define RF_REG_VALUE  0xc8

    	sbi_enable_host_int(priv);


	/*
	 * Read MAC address from HW
	 * permanent address is not read first before it gets set if
	 * NetworkAddress entry exists in the registry, so need to read it
	 * first
	 * 
	 * use permanentaddr as temp variable to store currentaddr if
	 * NetworkAddress entry exists
	 */
	memmove(TempAddr, adapter->CurrentAddr, MRVDRV_ETH_ADDR_LEN);
	memset(adapter->CurrentAddr, 0xff, MRVDRV_ETH_ADDR_LEN);
	memset(adapter->PermanentAddr, 0xff, MRVDRV_ETH_ADDR_LEN);

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_GET_HW_SPEC, 
			0, HostCmd_OPTION_WAITFORRSP,
			0, NULL);

	if (ret) {
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

	if (TempAddr[0] != 0xff) {
		/*
		 * an address was read, so need to send GET_HW_SPEC again to
		 * set the soft mac
		 */

		memmove(adapter->CurrentAddr, TempAddr, MRVDRV_ETH_ADDR_LEN);

		ret = PrepareAndSendCommand(priv, HostCmd_CMD_GET_HW_SPEC,
			0, HostCmd_OPTION_WAITFORRSP,
			0, NULL);

		if (ret) {
			ret = WLAN_STATUS_FAILURE;
			goto done;
		}
	}

	SetMacPacketFilter(priv);


	/* set the max tx power */
	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_RF_TX_POWER,
		HostCmd_ACT_TX_POWER_OPT_SET_LOW,
		HostCmd_OPTION_WAITFORRSP, 0,
		&dbm);

	if (ret) {
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}


	/* Get the supported Data Rates */
	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_DATA_RATE,
			HostCmd_ACT_GET_TX_RATE,
			HostCmd_OPTION_WAITFORRSP,
			0, NULL);

	if (ret) {
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

	ret = WLAN_STATUS_SUCCESS;
done:	
	LEAVE();

	return (ret);
}

/** 
 *  @brief This function initializes timers.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   n/a
 */
static void init_sync_objects(wlan_private *priv)
{
	wlan_adapter   *Adapter = priv->adapter;

	InitializeTimer(&Adapter->MrvDrvCommandTimer,
			MrvDrvCommandTimerFunction, priv);
	Adapter->CommandTimerIsSet = FALSE;

#ifdef REASSOCIATION
	/* Initialize the timer for the reassociation */
	InitializeTimer(&Adapter->MrvDrvTimer,
		       	MrvDrvTimerFunction, priv);
 	Adapter->TimerIsSet = FALSE;
#endif /* REASSOCIATION */

	return;
}

/** 
 *  @brief This function allocates buffer for the member of adapter
 *  structure like command buffer and BSSID list.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int wlan_allocate_adapter(wlan_private *priv)
{
	u32           ulBufSize;
	wlan_adapter   *Adapter = priv->adapter;

	BSSDescriptor_t* pTempScanTable;

	/* Allocate buffer to store the BSSID list */
	ulBufSize = sizeof(BSSDescriptor_t) * MRVDRV_MAX_BSSID_LIST;
	if (!(pTempScanTable = kmalloc(ulBufSize, GFP_KERNEL))) {
		wlan_free_adapter(priv);
		return WLAN_STATUS_FAILURE;
	}

	Adapter->ScanTable = pTempScanTable;
	memset(Adapter->ScanTable, 0, ulBufSize);

	spin_lock_init(&Adapter->QueueSpinLock);

	/* Allocate the command buffers */
	AllocateCmdBuffer(priv);

	memset(&Adapter->PSConfirmSleep, 0, sizeof(struct PS_CMD_ConfirmSleep));
	Adapter->PSConfirmSleep.SeqNum = wlan_cpu_to_le16(++Adapter->SeqNum);
	Adapter->PSConfirmSleep.Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_PS_MODE);
	Adapter->PSConfirmSleep.Size = wlan_cpu_to_le16(sizeof(struct PS_CMD_ConfirmSleep));
	Adapter->PSConfirmSleep.Result = 0;
	Adapter->PSConfirmSleep.Action = wlan_cpu_to_le16(HostCmd_SubCmd_Sleep_Confirmed);

	return WLAN_STATUS_SUCCESS;
}

/**
 *  @brief This function initializes the adapter structure
 *  and set default value to the member of adapter.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   n/a
 */
static void wlan_init_adapter(wlan_private *priv)
{
	wlan_adapter	*Adapter = priv->adapter;
	int		i;

	Adapter->ScanProbes = 0;

	Adapter->bcn_avg_factor  = DEFAULT_BCN_AVG_FACTOR;
	Adapter->data_avg_factor = DEFAULT_DATA_AVG_FACTOR;

	/* ATIM params */
  	Adapter->AtimWindow = 0;
	Adapter->ATIMEnabled = FALSE;

	Adapter->MediaConnectStatus = WlanMediaStateDisconnected;
	Adapter->LinkSpeed = MRVDRV_LINK_SPEED_1mbps;
	memset(Adapter->CurrentAddr, 0xff, MRVDRV_ETH_ADDR_LEN);

	/* Status variables */
	Adapter->HardwareStatus = WlanHardwareStatusInitializing;

	/* scan type */
	Adapter->ScanType = HostCmd_SCAN_TYPE_ACTIVE;

	/* scan mode */
	Adapter->ScanMode = HostCmd_BSS_TYPE_ANY;

	/* 802.11 specific */
	Adapter->SecInfo.WEPStatus = Wlan802_11WEPDisabled;
	for(i = 0; i < sizeof(Adapter->WepKey)/sizeof(Adapter->WepKey[0]); i++)
		memset(&Adapter->WepKey[i], 0, sizeof(struct MRVL_WEP_KEY));
	Adapter->CurrentWepKeyIndex = 0;
	Adapter->SecInfo.AuthenticationMode = Wlan802_11AuthModeOpen;
	Adapter->SecInfo.Auth1XAlg = WLAN_1X_AUTH_ALG_NONE;
	Adapter->SecInfo.EncryptionMode = CIPHER_NONE;
	Adapter->InfrastructureMode = Wlan802_11Infrastructure;

	Adapter->NumInScanTable = 0;
	Adapter->pAttemptedBSSDesc = NULL;
#ifdef REASSOCIATION
	init_MUTEX(&Adapter->ReassocSem);
#endif

	Adapter->Prescan = CMD_ENABLED;
	Adapter->HisRegCpy |= HIS_TxDnLdRdy;

	memset(&Adapter->CurBssParams, 0, sizeof(Adapter->CurBssParams));

	/* PnP and power profile */
	Adapter->SurpriseRemoved = FALSE;

	Adapter->CurrentPacketFilter =
		HostCmd_ACT_MAC_RX_ON | HostCmd_ACT_MAC_TX_ON;

	Adapter->RadioOn = RADIO_ON;
#ifdef REASSOCIATION
	Adapter->Reassoc_on = TRUE;
#endif /* REASSOCIATION */
	Adapter->TxAntenna = RF_ANTENNA_2;
	Adapter->RxAntenna = RF_ANTENNA_AUTO;

	Adapter->Is_DataRate_Auto = TRUE;
	Adapter->BeaconPeriod = MRVDRV_BEACON_INTERVAL;

	// set default value of capInfo.
#define SHORT_PREAMBLE_ALLOWED		1
	memset(&Adapter->capInfo, 0, sizeof(Adapter->capInfo));
	Adapter->capInfo.ShortPreamble = SHORT_PREAMBLE_ALLOWED;

	Adapter->AdhocChannel = DEFAULT_AD_HOC_CHANNEL;

	Adapter->PSMode = Wlan802_11PowerModeCAM;
	Adapter->MultipleDtim = MRVDRV_DEFAULT_MULTIPLE_DTIM;

	Adapter->ListenInterval = MRVDRV_DEFAULT_LISTEN_INTERVAL;
	
	Adapter->PSState = PS_STATE_FULL_POWER;
	Adapter->NeedToWakeup = FALSE;
	Adapter->LocalListenInterval = 0;	/* default value in firmware will be used */



	Adapter->DataRate = 0; // Initially indicate the rate as auto 

	Adapter->adhoc_grate_enabled=FALSE;

	Adapter->IntCounter = Adapter->IntCounterSaved = 0;
	INIT_LIST_HEAD((struct list_head *)&Adapter->RxSkbQ);

	INIT_LIST_HEAD((struct list_head *)&Adapter->TxSkbQ);
	Adapter->TxSkbNum = 0;


	Adapter->EncryptionStatus = Wlan802_11WEPDisabled;

	priv->adapter->linkmode = WLAN_LINKMODE_802_11;
	priv->adapter->radiomode = WLAN_RADIOMODE_RADIOTAP;
	
	spin_lock_init(&Adapter->CurrentTxLock);

	Adapter->CurrentTxSkb = NULL;
	Adapter->PktTxCtrl = 0;
	
	return;
}

/********************************************************
		Global Functions
********************************************************/

/** 
 *  @brief This function initializes firmware
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_init_fw(wlan_private *priv)
{
	int  ret = WLAN_STATUS_SUCCESS;

	ENTER();
	
	/* Allocate adapter structure */
	if ((ret = wlan_allocate_adapter(priv)) != WLAN_STATUS_SUCCESS) {
		goto done;
	}
	
	/* init adapter structure */
	wlan_init_adapter(priv);

	/* init timer etc. */
	init_sync_objects(priv);

	/* download fimrware etc. */
	if ((ret = wlan_setup_station_hw(priv)) != WLAN_STATUS_SUCCESS) {
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

	/* init 802.11d */
	wlan_init_11d( priv );

	ret = WLAN_STATUS_SUCCESS;
done:
	LEAVE();
	return ret;
}

/** 
 *  @brief This function frees the structure of adapter
 *    
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   n/a
 */
void wlan_free_adapter(wlan_private *priv)
{
	wlan_adapter   *Adapter = priv->adapter;
	
	ENTER();

	if (!Adapter) {
		PRINTM(INFO, "Why double free adapter?:)\n");
		return;
	}
	
	PRINTM(INFO, "Free Command buffer\n");
	FreeCmdBuffer(priv);

	PRINTM(INFO, "Free CommandTimer\n");
	if (Adapter->CommandTimerIsSet) {
		CancelTimer(&Adapter->MrvDrvCommandTimer);
		Adapter->CommandTimerIsSet = FALSE;
	}
	FreeTimer(&Adapter->MrvDrvCommandTimer);
#ifdef REASSOCIATION
	PRINTM(INFO, "Free MrvDrvTimer\n");
	if (Adapter->TimerIsSet) {
		CancelTimer(&Adapter->MrvDrvTimer);
		Adapter->TimerIsSet = FALSE;
	}
	FreeTimer(&Adapter->MrvDrvTimer);
#endif /* REASSOCIATION */

	PRINTM(INFO, "Free ScanTable\n");
	if (Adapter->ScanTable) {
		kfree(Adapter->ScanTable);
		Adapter->ScanTable = NULL;
	}


	PRINTM(INFO, "Free Adapter\n");

	/* Free the adapter object itself */
	kfree(Adapter);
	priv->adapter = NULL;
	LEAVE();
}

/** 
 *  @brief This function handles the timeout of command sending.
 *  It will re-send the same command again.
 *  
 *  @param FunctionContext    A pointer to FunctionContext
 *  @return 	   n/a
 */
void MrvDrvCommandTimerFunction(void *FunctionContext)
{
	wlan_private	*priv = (wlan_private *)FunctionContext;
	wlan_adapter	*Adapter = priv->adapter;
	struct CmdCtrlNode *pTempNode;
	ulong		flags;

	ENTER();

	PRINTM(FATAL, "MrvDrvCommandTimer fired.\n");

	Adapter->CommandTimerIsSet = FALSE;

	pTempNode = Adapter->CurCmd;

	if(pTempNode == NULL){
		PRINTM(INFO, "PTempnode Empty\n");
		return ;
	}

	spin_lock_irqsave(&Adapter->QueueSpinLock, flags);
	Adapter->CurCmd = NULL;
	spin_unlock_irqrestore(&Adapter->QueueSpinLock, flags);

	PRINTM(INFO, "Re-sending same command as it timeout...!\n");
	QueueCmd(Adapter,pTempNode, FALSE);

	wake_up_interruptible(&priv->MainThread.waitQ);

	LEAVE();
	return;
}

#ifdef REASSOCIATION
/** 
 *  @brief This function triggers re-association by waking up
 *  re-assoc thread.
 *  
 *  @param FunctionContext    A pointer to FunctionContext
 *  @return 	   n/a
 */
void MrvDrvTimerFunction(void *FunctionContext)
{
	wlan_private   *priv = (wlan_private *) FunctionContext;
	wlan_adapter   *Adapter = priv->adapter;

	ENTER();

	PRINTM(INFO, "MrvDrvTimer fired.\n");
	Adapter->TimerIsSet = FALSE;
	if (Adapter->PSState != PS_STATE_FULL_POWER) {
		/* wait until Exit_PS command returns */
		Adapter->TimerIsSet = TRUE;
		ModTimer(&Adapter->MrvDrvTimer, MRVDRV_TIMER_1S);
		PRINTM(INFO, "MrvDrvTimerFunction(PSState=%d) waiting" 
				"for Exit_PS done\n",Adapter->PSState);
		LEAVE();
		return;
	}

	PRINTM(INFO, "Waking Up the Event Thread\n");

	wake_up_interruptible(&priv->ReassocThread.waitQ);

	LEAVE();
	return;
}
#endif /* REASSOCIATION */


