/** @file  wlan_wext.c 
  * @brief This file contains ioctl functions
  * 
  * (c) Copyright  2003-2006, Marvell International Ltd. 
  * All Rights Reserved
  *
  * This software file (the "File") is distributed by Marvell International 
  * Ltd. under the terms of the GNU General Public License Version 2, June 1991 
  * (the "License").  You may use, redistribute and/or modify this File in 
  * accordance with the terms and conditions of the License, a copy of which 
  * is available along with the File in the license.txt file or by writing to 
  * the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 
  * 02111-1307 or on the worldwide web at http://www.gnu.org/licenses/gpl.txt.
  *
  * THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE 
  * IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE 
  * ARE EXPRESSLY DISCLAIMED.  The License provides additional details about 
  * this warranty disclaimer.
  *
  */
/********************************************************
Change log:
	10/10/05: Add Doxygen format comments
	12/23/05: Modify FindBSSIDInList to search entire table for
	          duplicate BSSIDs when earlier matches are not compatible
	12/26/05: Remove errant memcpy in wlanidle_off; overwriting stack space
	01/05/06: Add kernel 2.6.x support	
	01/11/06: Change compile flag BULVERDE_SDIO to SD to support Monahans/Zylonite
	01/11/06: Conditionalize new scan/join functions.
	          Update statics/externs.  Move forward decl. from wlan_decl.h
	04/06/06: Add TSPEC, queue metrics, and MSDU expiry support
	04/10/06: Add hostcmd generic API
	04/18/06: Remove old Subscrive Event and add new Subscribe Event
		  implementation through generic hostcmd API
********************************************************/


#include	"include.h"

#include	"wlan_version.h"

#define GETLOG_BUFSIZE  300

#define MAX_SCAN_CELL_SIZE      (IW_EV_ADDR_LEN + \
				MRVDRV_MAX_SSID_LENGTH + \
				IW_EV_UINT_LEN + IW_EV_FREQ_LEN + \
				IW_EV_QUAL_LEN + MRVDRV_MAX_SSID_LENGTH + \
				IW_EV_PARAM_LEN + 40) /* 40 for WPAIE */

#define WAIT_FOR_SCAN_RRESULT_MAX_TIME (10 * HZ)
/********************************************************
		Local Variables
********************************************************/

/********************************************************
		Global Variables
********************************************************/


/********************************************************
		Local Functions
********************************************************/
static int wlan_set_rate(struct net_device *dev, struct iw_request_info *info, 
                         struct iw_param *vwrq, char *extra);
static int wlan_get_rate(struct net_device *dev, struct iw_request_info *info, 
                         struct iw_param *vwrq, char *extra);

static int wlan_get_essid(struct net_device *dev, struct iw_request_info *info,
                          struct iw_point *dwrq, char *extra);

static int wlan_set_freq(struct net_device *dev, struct iw_request_info *info, 
                         struct iw_freq *fwrq, char *extra);
static int wlan_get_freq(struct net_device *dev, struct iw_request_info *info, 
                         struct iw_freq *fwrq, char *extra);

static int wlan_set_mode(struct net_device *dev, struct iw_request_info *info, 
                         u32 * uwrq, char *extra);
static int wlan_get_mode(struct net_device *dev, struct iw_request_info *info,
                         u32 * uwrq, char *extra);

static int wlan_set_encode(struct net_device *dev,struct iw_request_info *info,
                           struct iw_point *dwrq, char *extra);
static int wlan_get_encode(struct net_device *dev,struct iw_request_info *info,
                           struct iw_point *dwrq, u8 * extra);

static int wlan_set_txpow(struct net_device *dev, struct iw_request_info *info,
                          struct iw_param *vwrq, char *extra);
static int wlan_get_txpow(struct net_device *dev, struct iw_request_info *info,
                          struct iw_param *vwrq, char *extra);



/** 
 *  @brief Find a character in a string.
 *   
 *  @param s	   A pointer to string
 *  @param c	   Character to be located 
 *  @param dlen    the length of string
 *  @return 	   A pointer to the first occurrence of c in string, or NULL if c is not found.
 */
static void *wlan_memchr(void *s, int c, int n)
{
	const u8	*p = s;

	while(n-- != 0) {
		if((u8)c == *p++) {
			return (void *) (p - 1);
		}
	}
	return NULL;
}


#if WIRELESS_EXT > 14
/** 
 *  @brief Convert mw value to dbm value
 *   
 *  @param mw	   the value of mw
 *  @return 	   the value of dbm
 */
static int mw_to_dbm(int mw)
{
	if (mw < 2)
		return 0;
	else if (mw < 3)
		return 3;
	else if (mw < 4)
		return 5;
	else if (mw < 6)
		return 7;
	else if (mw < 7)
		return 8;
	else if (mw < 8)
		return 9;
	else if (mw < 10)
		return 10;
	else if (mw < 13)
		return 11;
	else if (mw < 16)
		return 12;
	else if (mw < 20)
		return 13;
	else if (mw < 25)
		return 14;
	else if (mw < 32)
		return 15;
	else if (mw < 40)
		return 16;
	else if (mw < 50)
		return 17;
	else if (mw < 63)
		return 18;
	else if (mw < 79)
		return 19;
	else if (mw < 100)
		return 20;
	else
		return 21;
}
#endif

/** 
 *  @brief Find the channel frequency power info with specific channel
 *   
 *  @param adapter 	A pointer to wlan_adapter structure
 *  @param band		it can be BAND_A, BAND_G or BAND_B
 *  @param channel      the channel for looking	
 *  @return 	   	A pointer to CHANNEL_FREQ_POWER structure or NULL if not find.
 */
CHANNEL_FREQ_POWER *find_cfp_by_band_and_channel(wlan_adapter *adapter, 
                                                 u8 band, u16 channel)
{
	CHANNEL_FREQ_POWER	*cfp = NULL;
	REGION_CHANNEL		*rc;
	int	count = sizeof(adapter->region_channel) / 
				sizeof(adapter->region_channel[0]);
	int i, j;

	for (j = 0; !cfp && (j < count); j++) {
		rc = &adapter->region_channel[j];

		if( adapter->State11D.Enable11D == ENABLE_11D ) {
			rc = &adapter->universal_channel[j];
		}
		if(!rc->Valid || !rc->CFP)
			continue;
		if (rc->Band != band)
			continue;
		for (i = 0; i < rc->NrCFP; i++) {
			if (rc->CFP[i].Channel == channel) {
				cfp = &rc->CFP[i];
				break;
			}
		}
	}

	if (!cfp && channel)
		PRINTM(INFO, "find_cfp_by_band_and_channel(): cannot find "
			"cfp by band %d & channel %d\n",band,channel);

	return cfp;
}
			
/** 
 *  @brief Find the channel frequency power info with specific frequency
 *   
 *  @param adapter 	A pointer to wlan_adapter structure
 *  @param band		it can be BAND_A, BAND_G or BAND_B
 *  @param freq	        the frequency for looking	
 *  @return 	   	A pointer to CHANNEL_FREQ_POWER structure or NULL if not find.
 */
static CHANNEL_FREQ_POWER *find_cfp_by_band_and_freq(wlan_adapter *adapter, 
							u8 band, u32 freq)
{
	CHANNEL_FREQ_POWER *cfp = NULL;
	REGION_CHANNEL *rc;
	int count = sizeof(adapter->region_channel) / 
				sizeof(adapter->region_channel[0]);
	int i, j;

	for (j = 0; !cfp && (j < count); j++) {
		rc = &adapter->region_channel[j];

		if( adapter->State11D.Enable11D == ENABLE_11D ) {
			rc = &adapter->universal_channel[j];
		}

		if(!rc->Valid || !rc->CFP)
			continue;
		if (rc->Band != band)
			continue;
		for (i = 0; i < rc->NrCFP; i++) {
			if (rc->CFP[i].Freq == freq) {
				cfp = &rc->CFP[i];
				break;
			}
		}
	}

	if(!cfp && freq)
		PRINTM(INFO, "find_cfp_by_band_and_freql(): cannot find cfp by "
				"band %d & freq %d\n",band,freq);

	return cfp;
}


/** 
 *  @brief Update Current Channel 
 *   
 *  @param priv 		A pointer to wlan_private structure
 *  @return 	   		WLAN_STATUS_SUCCESS--success, WLAN_STATUS_FAILURE--fail
 */
static int UpdateCurrentChannel(wlan_private *priv)
{
    int ret;
    
    /*
    ** the channel in f/w could be out of sync, get the current channel
    */
    ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_RF_CHANNEL,
                                HostCmd_OPT_802_11_RF_CHANNEL_GET,
                                HostCmd_OPTION_WAITFORRSP, 0,
                                NULL);
    
    PRINTM(INFO, "Current Channel = %d\n", priv->adapter->CurBssParams.channel);

    return ret;
}

/** 
 *  @brief Set Current Channel 
 *   
 *  @param priv 		A pointer to wlan_private structure
 *  @param channel		The channel to be set. 
 *  @return 	   		WLAN_STATUS_SUCCESS--success, WLAN_STATUS_FAILURE--fail
 */
static int SetCurrentChannel(wlan_private *priv, int channel)
{
    PRINTM(INFO, "Set Channel = %d\n", channel);

    /* 
    **  Current channel is not set to AdhocChannel requested, set channel
    */
    return (PrepareAndSendCommand(priv, HostCmd_CMD_802_11_RF_CHANNEL,
                                  HostCmd_OPT_802_11_RF_CHANNEL_SET,
                                  HostCmd_OPTION_WAITFORRSP, 0,
                                  &channel));
}

/** 
 *  @brief Change Adhoc Channel
 *   
 *  @param priv 		A pointer to wlan_private structure
 *  @param channel		The channel to be set. 
 *  @return 	   		WLAN_STATUS_SUCCESS--success, WLAN_STATUS_FAILURE--fail
 */
static int ChangeAdhocChannel(wlan_private *priv, int channel)
{
    	int             ret      = WLAN_STATUS_SUCCESS;
	wlan_adapter    *Adapter = priv->adapter;

    	Adapter->AdhocChannel = channel;

    	UpdateCurrentChannel(priv);

    	if (Adapter->CurBssParams.channel == Adapter->AdhocChannel){
        	/* AdhocChannel is set to the current Channel already */
        	LEAVE();
        	return WLAN_STATUS_SUCCESS;
    	}

    	PRINTM(INFO, "Updating Channel from %d to %d\n", 
	Adapter->CurBssParams.channel, Adapter->AdhocChannel);

    	SetCurrentChannel(priv, Adapter->AdhocChannel);

    	UpdateCurrentChannel(priv);

    	if (Adapter->CurBssParams.channel != Adapter->AdhocChannel){
        	PRINTM(INFO, "Failed to updated Channel to %d, channel = %d\n", 
                Adapter->AdhocChannel, Adapter->CurBssParams.channel);
        	LEAVE();
        	return WLAN_STATUS_FAILURE;
    	}

	if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
		int			i;
		struct WLAN_802_11_SSID	curAdhocSsid;
		
		PRINTM(INFO, "Channel Changed while in an IBSS\n");

		/* Copy the current ssid */
		memcpy(&curAdhocSsid, &Adapter->CurBssParams.ssid, 
				sizeof(struct WLAN_802_11_SSID));
		
		/* Exit Adhoc mode */
		PRINTM(INFO, "In ChangeAdhocChannel(): Sending Adhoc Stop\n");
		ret = StopAdhocNetwork(priv);

		if (ret) {
			LEAVE();
			return ret;
		}
        	/* Scan for the network, do not save previous results.  Stale
         	 *   scan data will cause us to join a non-existant adhoc network
         	 */
        	SendSpecificSSIDScan(priv, &curAdhocSsid, FALSE);


		// find out the BSSID that matches the current SSID 
		i = FindSSIDInList(Adapter, &curAdhocSsid, NULL, 
				Wlan802_11IBSS);

		if (i >= 0) {
			PRINTM(INFO, "SSID found at %d in List," 
					"so join\n", i);
			JoinAdhocNetwork(priv, &Adapter->ScanTable[i]);
		} else {
			// else send START command
			PRINTM(INFO, "SSID not found in list, "
					"so creating adhoc with ssid = %s\n",
					curAdhocSsid.Ssid);
			StartAdhocNetwork(priv, &curAdhocSsid);
		}	// end of else (START command)
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set WPA key
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_encode_wpa(struct net_device *dev, 
			struct iw_request_info *info,
				struct iw_point *dwrq, char *extra)
{
	int			ret = WLAN_STATUS_SUCCESS;
	wlan_private		*priv = dev->priv;
	struct WLAN_802_11_KEY	*pKey;

	ENTER();


	pKey = (struct WLAN_802_11_KEY *) extra;

	HEXDUMP("Key buffer: ", extra, dwrq->length);

	HEXDUMP("KeyMaterial: ", (u8 *) pKey->KeyMaterial,
			pKey->KeyLength);

	// current driver only supports key length of up to 32 bytes
	if (pKey->KeyLength > MRVL_MAX_WPA_KEY_LENGTH) {
		PRINTM(INFO, " Error in key length \n");
		return WLAN_STATUS_FAILURE;
	}

	ret = PrepareAndSendCommand(priv,
		HostCmd_CMD_802_11_KEY_MATERIAL,
		HostCmd_ACT_SET,
		HostCmd_OPTION_WAITFORRSP,
		KEY_INFO_ENABLED, pKey);
	if (ret) {
		LEAVE();
		return ret;
	}

	LEAVE();
	return ret;
}

/*
 *  iwconfig ethX key on:	WEPEnabled;
 *  iwconfig ethX key off:	WEPDisabled;
 *  iwconfig ethX key [x]:	CurrentWepKeyIndex = x; WEPEnabled;
 *  iwconfig ethX key [x] kstr:	WepKey[x] = kstr;
 *  iwconfig ethX key kstr:	WepKey[CurrentWepKeyIndex] = kstr;
 *
 *  all:			Send command SET_WEP;
 				SetMacPacketFilter;
 */

/** 
 *  @brief Set WEP key
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_encode_nonwpa(struct net_device *dev,
			       struct iw_request_info *info,
				       struct iw_point *dwrq, char *extra)
{
	int			ret = WLAN_STATUS_SUCCESS;
	wlan_private		*priv = dev->priv;
	wlan_adapter		*Adapter = priv->adapter;
	struct MRVL_WEP_KEY	*pWep;
	struct WLAN_802_11_SSID	ssid;
	int			index, PrevAuthMode;

	ENTER();

	pWep = &Adapter->WepKey[Adapter->CurrentWepKeyIndex];
	PrevAuthMode = Adapter->SecInfo.AuthenticationMode;

	index = (dwrq->flags & IW_ENCODE_INDEX) - 1;

	if(index >= 4) {
		PRINTM(INFO, "Key index #%d out of range.\n", index + 1);
		return -EINVAL;
	}


	PRINTM(INFO, "Flags=0x%x, Length=%d Index=%d CurrentWepKeyIndex=%d\n", 
			dwrq->flags, dwrq->length, index, Adapter->CurrentWepKeyIndex);

	if (dwrq->length > 0) {
		/* iwconfig ethX key [n] xxxxxxxxxxx 
		 * Key has been provided by the user 
		 */

		/*
		 * Check the size of the key 
		 */

		if (dwrq->length > MAX_KEY_SIZE) {
			return -EINVAL;
		}

		/*
		 * Check the index (none -> use current) 
		 */

		if (index < 0 || index > 3)		//invalid index or no index
			index = Adapter->CurrentWepKeyIndex;
		else					//index is given & valid
			pWep = &Adapter->WepKey[index];

		/*
		 * Check if the key is not marked as invalid 
		 */
		if (!(dwrq->flags & IW_ENCODE_NOKEY)) {
			/* Cleanup */
			memset(pWep, 0, sizeof(struct MRVL_WEP_KEY));

			/* Copy the key in the driver */
			memcpy(pWep->KeyMaterial, extra, dwrq->length);

			/* Set the length */
			if (dwrq->length > MIN_KEY_SIZE) {
				pWep->KeyLength = MAX_KEY_SIZE;
			} else {
				if (dwrq->length > 0) {
					pWep->KeyLength = MIN_KEY_SIZE;
				} else {
					/* Disable the key */
					pWep->KeyLength = 0;
				}
			}
			pWep->KeyIndex = index;

			if(Adapter->SecInfo.WEPStatus != Wlan802_11WEPEnabled) {
				/*
				 * The status is set as Key Absent 
				 * so as to make sure we display the 
				 * keys when iwlist ethX key is 
				 * used - MPS 
				 */

				Adapter->SecInfo.WEPStatus = 
					Wlan802_11WEPKeyAbsent;
			}

			PRINTM(INFO, "KeyIndex=%u KeyLength=%u\n", 
					pWep->KeyIndex,
					pWep->KeyLength);
			HEXDUMP("WepKey", (u8 *) pWep->KeyMaterial, 
					pWep->KeyLength);
		}
	} else {
		/*
		 * No key provided so it is either enable key, 
		 * on or off */
		if (dwrq->flags & IW_ENCODE_DISABLED) {
			PRINTM(INFO, "******** iwconfig ethX key off **********\n");

			Adapter->SecInfo.WEPStatus = Wlan802_11WEPDisabled;
			if (Adapter->SecInfo.AuthenticationMode == Wlan802_11AuthModeShared)
				Adapter->SecInfo.AuthenticationMode = Wlan802_11AuthModeOpen;
		} else {
			/* iwconfig ethX key [n]
			 * iwconfig ethX key on 
			 * Do we want to just set the transmit key index ? 
			 */

			if (index < 0 || index > 3) {
				PRINTM(INFO, "******** iwconfig ethX key on **********\n");
				index = Adapter->CurrentWepKeyIndex;
			}
			else {
				PRINTM(INFO, "******** iwconfig ethX key [x=%d] **********\n",index);
				Adapter->CurrentWepKeyIndex = index;
			}

			/* Copy the required key as the current key */
			pWep = &Adapter->WepKey[index];

			if (!pWep->KeyLength) {
				PRINTM(INFO, "Key not set,so cannot enable it\n");
				return -EPERM;
			}
	    
			Adapter->SecInfo.WEPStatus = Wlan802_11WEPEnabled;

			HEXDUMP("KeyMaterial", (u8 *) pWep->KeyMaterial,
					pWep->KeyLength);
		}
	}

	if(pWep->KeyLength) {
		ret = PrepareAndSendCommand(priv,
			HostCmd_CMD_802_11_SET_WEP,
			0, HostCmd_OPTION_WAITFORRSP,
			OID_802_11_ADD_WEP, NULL);

		if (ret) {
			LEAVE();
			return ret;
		}
	}

	if (Adapter->SecInfo.WEPStatus == Wlan802_11WEPEnabled) {
		Adapter->CurrentPacketFilter |= HostCmd_ACT_MAC_WEP_ENABLE;
	} else {
		Adapter->CurrentPacketFilter &= ~HostCmd_ACT_MAC_WEP_ENABLE;
	}

	SetMacPacketFilter(priv);

	if (dwrq->flags & IW_ENCODE_RESTRICTED) {
		/* iwconfig ethX restricted key [1] */
		Adapter->SecInfo.AuthenticationMode = Wlan802_11AuthModeShared;
		PRINTM(INFO, "Authentication mode restricted!\n");
	} else if (dwrq->flags & IW_ENCODE_OPEN) {
		/* iwconfig ethX key [2] open */
		Adapter->SecInfo.AuthenticationMode = Wlan802_11AuthModeOpen;
		PRINTM(INFO, "Authentication mode open!\n");
	}

	/*
	 * If authentication mode changed - de-authenticate, set authentication
	 * method and re-associate if we were previously associated.
	 */
	if (Adapter->SecInfo.AuthenticationMode != PrevAuthMode) {
		if (Adapter->MediaConnectStatus == WlanMediaStateConnected &&
				Adapter->InfrastructureMode == 
				Wlan802_11Infrastructure) {

			/* keep a copy of the ssid associated with */
			memcpy(&ssid, &Adapter->CurBssParams.ssid, sizeof(ssid));

			/*
			 * De-authenticate from AP 
			 */
				
			ret = SendDeauthentication(priv);

			if (ret) {
				LEAVE();
				return ret;
			}

		} else {
			/* reset ssid */
			memset(&ssid, 0, sizeof(ssid));
		}
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief Set RX Antenna
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param Mode			RF antenna mode
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int SetRxAntenna(wlan_private *priv, int Mode)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_adapter	*Adapter = priv->adapter;

	if (Mode != RF_ANTENNA_1 && Mode != RF_ANTENNA_2
					&& Mode != RF_ANTENNA_AUTO) {
		return -EINVAL;
	}

	Adapter->RxAntennaMode = Mode;

	PRINTM(INFO, "SET RX Antenna Mode to 0x%04x\n", Adapter->RxAntennaMode);
			
	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_RF_ANTENNA,
			HostCmd_ACT_SET_RX, HostCmd_OPTION_WAITFORRSP,
			0, &Adapter->RxAntennaMode);
	return ret;
}

/** 
 *  @brief Set TX Antenna
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param Mode			RF antenna mode
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int SetTxAntenna(wlan_private *priv, int Mode)
{
	int		ret = 0;
	wlan_adapter	*Adapter = priv->adapter;

	if ((Mode != RF_ANTENNA_1) && (Mode != RF_ANTENNA_2)
			&& (Mode != RF_ANTENNA_AUTO)) {
		return -EINVAL;
	}
		
	Adapter->TxAntennaMode = Mode; 

	PRINTM(INFO, "SET TX Antenna Mode to 0x%04x\n", Adapter->TxAntennaMode);

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_RF_ANTENNA,
			HostCmd_ACT_SET_TX, HostCmd_OPTION_WAITFORRSP,
			0, &Adapter->TxAntennaMode);

	return ret;
}

/** 
 *  @brief Get RX Antenna
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param buf			A pointer to recieve antenna mode
 *  @return 	   		length of buf 
 */
static int GetRxAntenna(wlan_private *priv, char *buf)
{
	int		ret = 0;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	// clear it, so we will know if the value 
	// returned below is correct or not.
	Adapter->RxAntennaMode = 0;	

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_RF_ANTENNA,
			HostCmd_ACT_GET_RX, HostCmd_OPTION_WAITFORRSP,
			0, NULL);

	if (ret) {
		LEAVE();
		return ret;
	}

	PRINTM(INFO, "Get Rx Antenna Mode:0x%04x\n", Adapter->RxAntennaMode);

	LEAVE();
			
	return sprintf(buf, "0x%04x", Adapter->RxAntennaMode) + 1;
}

/** 
 *  @brief Get TX Antenna
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param buf			A pointer to recieve antenna mode
 *  @return 	   		length of buf 
 */
static int GetTxAntenna(wlan_private *priv, char *buf)
{
	int		ret = 0;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();
	
	// clear it, so we will know if the value 
	// returned below is correct or not.
	Adapter->TxAntennaMode = 0;	

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_RF_ANTENNA,
			HostCmd_ACT_GET_TX, HostCmd_OPTION_WAITFORRSP,
			0, NULL);		

	if (ret) {
		LEAVE();
		return ret;
	}

	PRINTM(INFO, "Get Tx Antenna Mode:0x%04x\n", Adapter->TxAntennaMode);

	LEAVE();

	return sprintf(buf, "0x%04x", Adapter->TxAntennaMode) + 1;
}

/** 
 *  @brief Set Radio On/OFF
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @option 			Radio Option
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail 
 */
static int wlan_radio_ioctl(wlan_private *priv, u8 option)
{
	int		ret = 0;
	wlan_adapter	*Adapter	= priv->adapter;

	ENTER();

	if (Adapter->RadioOn != option) {
		PRINTM(INFO, "Switching %s the Radio\n", option ? "On" : "Off");
		Adapter->RadioOn = option;
		
		ret = PrepareAndSendCommand(priv,
				HostCmd_CMD_802_11_RADIO_CONTROL,
				HostCmd_ACT_GEN_SET, HostCmd_OPTION_WAITFORRSP,
			      	0, NULL);
	}

	LEAVE();
	return ret;
}

#ifdef REASSOCIATION
/** 
 *  @brief Set Auto Reassociation On
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail 
 */
static int reassociation_on(wlan_private *priv)
{
	wlan_adapter	*Adapter	= priv->adapter;

	ENTER();


	Adapter->Reassoc_on = TRUE;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set Auto Reassociation Off
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail 
 */
static int reassociation_off(wlan_private *priv)
{
	wlan_adapter		*Adapter = priv->adapter;

	ENTER();

	
	if (Adapter->TimerIsSet == TRUE) {
		CancelTimer(&Adapter->MrvDrvTimer);
		Adapter->TimerIsSet = FALSE;
	}	

	Adapter->Reassoc_on = FALSE;
	
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}
#endif /* REASSOCIATION */


/** 
 *  @brief Set Region
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param region_code		region code
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail 
 */
static int wlan_set_region(wlan_private *priv, u16 region_code)
{
	int i;

	ENTER();
	
	for (i = 0; i < MRVDRV_MAX_REGION_CODE; i++) {
		// use the region code to search for the index
		if (region_code == RegionCodeToIndex[i]) {
			priv->adapter->RegionTableIndex = (u16) i;
			priv->adapter->RegionCode = region_code;
			break;
		}
	}

	// if it's unidentified region code
	if (i >= MRVDRV_MAX_REGION_CODE) {
		PRINTM(INFO, "Region Code not identified\n");
		LEAVE();
		return WLAN_STATUS_FAILURE;
	}

	if (wlan_set_regiontable(priv, priv->adapter->RegionCode, 0)) {
		LEAVE();
		return -EINVAL;
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief Copy Rates
 *   
 *  @param dest                 A pointer to Dest Buf
 *  @param src		        A pointer to Src Buf
 *  @param len                  The len of Src Buf
 *  @return 	   	        Number of Rates copyed 
 */
static inline int CopyRates(u8 *dest, int pos, u8 *src, int len)
{
	int	i;

	for (i = 0; i < len && src[i]; i++, pos++) {
		if (pos >= sizeof(WLAN_802_11_RATES))
			break;
		dest[pos] = src[i];
	}

	return pos;
}

/** 
 *  @brief Get active data rates
 *   
 *  @param Adapter              A pointer to wlan_adapter structure
 *  @param rate		        The buf to return the active rates
 *  @return 	   	        The number of Rates
 */
static int get_active_data_rates(wlan_adapter *Adapter, WLAN_802_11_RATES rates)
{
	int	k = 0;

	ENTER();
	
	if (Adapter->MediaConnectStatus != WlanMediaStateConnected) {
		if (Adapter->InfrastructureMode == Wlan802_11Infrastructure) {	
			//Infra. mode
			PRINTM(INFO, "Infra\n");
			k = CopyRates(rates, k, SupportedRates, 
						sizeof(SupportedRates));
		} else {
			//ad-hoc mode
			PRINTM(INFO, "Adhoc G\n");
			k = CopyRates(rates, k, AdhocRates_G, 
							sizeof(AdhocRates_G));
		}
	} else {
		k = CopyRates(rates, 0, Adapter->CurBssParams.DataRates, 
					Adapter->CurBssParams.NumOfRates);
	}

	LEAVE();

	return k;
}

/** 
 *  @brief Get/Set Firmware wakeup method
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param wrq	   	A pointer to user data
 *  @return 	   	WLAN_STATUS_SUCCESS--success, otherwise fail
 */
static int wlan_txcontrol(wlan_private *priv, struct iwreq *wrq)
{
	wlan_adapter	*Adapter = priv->adapter;
	int data;
	ENTER();

	if((int) wrq->u.data.length == 0) {
	    if(copy_to_user(wrq->u.data.pointer, &Adapter->PktTxCtrl, sizeof(u32))) {
		PRINTM(MSG, "copy_to_user failed!\n");
		return -EFAULT;
	    }	    
	} else{
	     if((int) wrq->u.data.length > 1) {
	  	PRINTM(MSG, "ioctl too many args!\n");
	   	return -EFAULT;
	     }
	     if (copy_from_user(&data,wrq->u.data.pointer,sizeof(int))) {
	         PRINTM(INFO, "Copy from user failed\n");
	         return -EFAULT;
	     }

		 Adapter->PktTxCtrl = (u32)data;
	}

	wrq->u.data.length = 1;
			
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}



/** 
 *  @brief Get/Set NULL Package generation interval
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param wrq	   	A pointer to user data
 *  @return 	   	WLAN_STATUS_SUCCESS--success, otherwise fail
 */
static int wlan_null_pkt_interval(wlan_private *priv, struct iwreq *wrq)
{
	wlan_adapter	*Adapter = priv->adapter;
	int data;	
	ENTER();

	if((int) wrq->u.data.length == 0) {
	    data = Adapter->NullPktInterval;
	    
	    if(copy_to_user(wrq->u.data.pointer, &data, sizeof(int))) {
		PRINTM(MSG, "copy_to_user failed!\n");
		return -EFAULT;
	    }
	} else{
	     if((int) wrq->u.data.length > 1) {
	  	PRINTM(MSG, "ioctl too many args!\n");
	   	return -EFAULT;
	     }
	     if (copy_from_user(&data,wrq->u.data.pointer,sizeof(int))) {
	         PRINTM(INFO, "Copy from user failed\n");
	         return -EFAULT;
	     }

	     Adapter->NullPktInterval = data;
	}

	wrq->u.data.length = 1;
			
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Commit handler: called after a bunch of SET operations
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_config_commit(struct net_device *dev, 
		struct iw_request_info *info, 
		char *cwrq, char *extra)
{
	ENTER();

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get protocol name 
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_name(struct net_device *dev, struct iw_request_info *info, 
		char *cwrq, char *extra)
{
	const char *cp;
	char comm[6] = {"COMM-"};
	char mrvl[6] = {"MRVL-"};
	int cnt;

	ENTER();

	strcpy(cwrq, mrvl);

	cp = strstr(driver_version, comm);
	if (cp == driver_version)	//skip leading "COMM-"
		cp = driver_version + strlen(comm);
	else
		cp = driver_version;

	cnt = strlen(mrvl);
	cwrq += cnt;
	while (cnt < 16 && (*cp != '-')) {
		*cwrq++ = toupper(*cp++);
		cnt++;
	}
	*cwrq = '\0';

	LEAVE();

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get frequency
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param fwrq 		A pointer to iw_freq structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_freq(struct net_device *dev, struct iw_request_info *info, 
		struct iw_freq *fwrq, char *extra)
{
	wlan_private		*priv = dev->priv;
	wlan_adapter		*Adapter = priv->adapter;
	CHANNEL_FREQ_POWER	*cfp;

	ENTER();


	cfp = find_cfp_by_band_and_channel(Adapter, 0, 
						Adapter->CurBssParams.channel);
	
	if (!cfp) {
		if (Adapter->CurBssParams.channel)
			PRINTM(INFO, "Invalid channel=%d\n", 
					Adapter->CurBssParams.channel);
		return -EINVAL;
	} 
	
	fwrq->m = (long) cfp->Freq * 100000;
	fwrq->e = 1;

	PRINTM(INFO, "freq=%u\n",fwrq->m);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get current BSSID
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param awrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_wap(struct net_device *dev, struct iw_request_info *info, 
		struct sockaddr *awrq, char *extra)
{
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
		memcpy(awrq->sa_data, Adapter->CurBssParams.bssid, ETH_ALEN);
	} else {
		memset(awrq->sa_data, 0, ETH_ALEN);
	}
	awrq->sa_family = ARPHRD_ETHER;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set Adapter Node Name
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_nick(struct net_device *dev, struct iw_request_info *info, 
		struct iw_point *dwrq, char *extra)
{
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	/*
	 * Check the size of the string 
	 */
	
	if (dwrq->length > 16) {
		return -E2BIG;
	}

	memset(Adapter->nodeName, 0, sizeof(Adapter->nodeName));
	memcpy(Adapter->nodeName, extra, dwrq->length);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get Adapter Node Name
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_nick(struct net_device *dev, struct iw_request_info *info, 
		struct iw_point *dwrq, char *extra)
{
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	/*
	 * Get the Nick Name saved 
	 */

	strncpy(extra, Adapter->nodeName, 16);

	extra[16] = '\0';

	/*
	 * If none, we may want to get the one that was set 
	 */

	/*
	 * Push it out ! 
	 */
	dwrq->length = strlen(extra) + 1;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set RTS threshold
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_rts(struct net_device *dev, struct iw_request_info *info, 
		struct iw_param *vwrq, char *extra)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;
	int		rthr = vwrq->value;

	ENTER();


	if (vwrq->disabled) {
		Adapter->RTSThsd = rthr = MRVDRV_RTS_MAX_VALUE;
	} else {
		if (rthr < MRVDRV_RTS_MIN_VALUE || rthr > MRVDRV_RTS_MAX_VALUE)
			return -EINVAL;
		Adapter->RTSThsd = rthr;
	}

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_SNMP_MIB,
			HostCmd_ACT_SET, HostCmd_OPTION_WAITFORRSP,
			OID_802_11_RTS_THRESHOLD, &rthr);

	LEAVE();
	return ret;
}

/** 
 *  @brief Get RTS threshold
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_rts(struct net_device *dev, struct iw_request_info *info, 
		struct iw_param *vwrq, char *extra)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();


	Adapter->RTSThsd = 0;
	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_SNMP_MIB,
			HostCmd_ACT_GET, HostCmd_OPTION_WAITFORRSP,
			OID_802_11_RTS_THRESHOLD, NULL);
	if (ret) {
		LEAVE();
		return ret;
	}
	

	vwrq->value = Adapter->RTSThsd;
	vwrq->disabled = ((vwrq->value < MRVDRV_RTS_MIN_VALUE)
			 || (vwrq->value > MRVDRV_RTS_MAX_VALUE));
	vwrq->fixed = 1;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set Fragment threshold
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_frag(struct net_device *dev, struct iw_request_info *info, 
		struct iw_param *vwrq, char *extra)
{
	int		ret = WLAN_STATUS_SUCCESS;
	int		fthr = vwrq->value;
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();


	if (vwrq->disabled) {
		Adapter->FragThsd = fthr = MRVDRV_FRAG_MAX_VALUE;
	} else {
		if (fthr < MRVDRV_FRAG_MIN_VALUE || fthr > MRVDRV_FRAG_MAX_VALUE)
			return -EINVAL;
		Adapter->FragThsd = fthr;
	}

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_SNMP_MIB,
			HostCmd_ACT_SET, HostCmd_OPTION_WAITFORRSP,
			OID_802_11_FRAGMENTATION_THRESHOLD,
			&fthr);
	LEAVE();
	return ret;
}

/** 
 *  @brief Get Fragment threshold
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_frag(struct net_device *dev, struct iw_request_info *info, 
		struct iw_param *vwrq, char *extra)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	Adapter->FragThsd = 0;
	ret = PrepareAndSendCommand(priv,
				HostCmd_CMD_802_11_SNMP_MIB, HostCmd_ACT_GET,
				HostCmd_OPTION_WAITFORRSP,
				OID_802_11_FRAGMENTATION_THRESHOLD,
				NULL);
	if (ret) {
		LEAVE();
		return ret;
	}

	vwrq->value = Adapter->FragThsd;
	vwrq->disabled = ((vwrq->value < MRVDRV_FRAG_MIN_VALUE)
			 || (vwrq->value > MRVDRV_FRAG_MAX_VALUE));
	vwrq->fixed = 1;

	LEAVE();
	return ret;
}

/** 
 *  @brief Get Wlan Mode
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_mode(struct net_device *dev,
	      struct iw_request_info *info, u32 * uwrq, char *extra)
{
	wlan_private	*priv = dev->priv;
	wlan_adapter	*adapter = priv->adapter;

	ENTER();

	switch (adapter->InfrastructureMode) {
		case Wlan802_11IBSS:
			*uwrq = IW_MODE_ADHOC;
			break;

		case Wlan802_11Infrastructure:
			*uwrq = IW_MODE_INFRA;
			break;

		default:
		case Wlan802_11AutoUnknown:
			*uwrq = IW_MODE_AUTO;
			break;
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get Encryption key
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_encode(struct net_device *dev,
			struct iw_request_info *info,
				struct iw_point *dwrq, u8 * extra)
{

	wlan_private	*priv = dev->priv;
	wlan_adapter	*adapter = priv->adapter;
	int		index = (dwrq->flags & IW_ENCODE_INDEX);

	ENTER();

	PRINTM(INFO, "flags=0x%x index=%d length=%d CurrentWepKeyIndex=%d\n", 
			dwrq->flags, index, dwrq->length, 
			adapter->CurrentWepKeyIndex);

	dwrq->flags = 0;

	/*
	 * Check encryption mode 
	 */

	switch (adapter->SecInfo.AuthenticationMode) {
		case Wlan802_11AuthModeOpen:
			dwrq->flags = IW_ENCODE_OPEN;
			break;

		case Wlan802_11AuthModeShared:
		case Wlan802_11AuthModeNetworkEAP:
			dwrq->flags = IW_ENCODE_RESTRICTED;
			break;
		default:
			dwrq->flags = IW_ENCODE_DISABLED | IW_ENCODE_OPEN;
			break;
	}

	if ((adapter->SecInfo.WEPStatus == Wlan802_11WEPEnabled)
		|| (adapter->SecInfo.WEPStatus == Wlan802_11WEPKeyAbsent)
		|| adapter->SecInfo.WPAEnabled
		|| adapter->SecInfo.WPA2Enabled
		) {
		dwrq->flags &= ~IW_ENCODE_DISABLED;
	} else {
		dwrq->flags |= IW_ENCODE_DISABLED;
	}

	memset(extra, 0, 16);

	if (!index) {
		/* Handle current key request	*/
		if ((adapter->WepKey[adapter->CurrentWepKeyIndex].KeyLength) &&
				(adapter->SecInfo.WEPStatus == Wlan802_11WEPEnabled)) {
			index = adapter->WepKey[adapter->CurrentWepKeyIndex].KeyIndex;
			memcpy(extra, adapter->WepKey[index].KeyMaterial,
					adapter->WepKey[index].KeyLength);
			dwrq->length = adapter->WepKey[index].KeyLength;
			/* return current key */
			dwrq->flags |= (index + 1);
			/* return WEP enabled */
			dwrq->flags &= ~IW_ENCODE_DISABLED;
		} else if ((adapter->SecInfo.WPAEnabled)
		|| (adapter->SecInfo.WPA2Enabled)
		) {
			/* return WPA enabled */
			dwrq->flags &= ~IW_ENCODE_DISABLED;
		} else {
			dwrq->flags |= IW_ENCODE_DISABLED;
		}
	} else {
		/* Handle specific key requests */
		index--;
		if (adapter->WepKey[index].KeyLength) {
			memcpy(extra, adapter->WepKey[index].KeyMaterial,
					adapter->WepKey[index].KeyLength);
			dwrq->length = adapter->WepKey[index].KeyLength;
			/* return current key */
			dwrq->flags |= (index + 1);
			/* return WEP enabled */
			dwrq->flags &= ~IW_ENCODE_DISABLED;
		} else if ((adapter->SecInfo.WPAEnabled)
		|| (adapter->SecInfo.WPA2Enabled)
		) {
			/* return WPA enabled */
			dwrq->flags &= ~IW_ENCODE_DISABLED;
		} else {
			dwrq->flags |= IW_ENCODE_DISABLED;
		}
	}

	dwrq->flags |= IW_ENCODE_NOKEY;

	PRINTM(INFO, "Key:%02x:%02x:%02x:%02x:%02x:%02x KeyLen=%d\n",
			extra[0], extra[1], extra[2],
			extra[3], extra[4], extra[5], dwrq->length);

	if (adapter->EncryptionStatus == Wlan802_11Encryption2Enabled
			&& !dwrq->length) {
		dwrq->length = MAX_KEY_SIZE;
	}

	PRINTM(INFO, "Return flags=0x%x\n", dwrq->flags);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get TX Power
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_txpow(struct net_device *dev,
			   struct iw_request_info *info,
			   struct iw_param *vwrq, char *extra)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();


	ret = PrepareAndSendCommand(priv, 
			HostCmd_CMD_802_11_RF_TX_POWER,
			HostCmd_ACT_TX_POWER_OPT_GET,
			HostCmd_OPTION_WAITFORRSP,
		      	0, NULL);

	if (ret) {
		LEAVE();
		return ret;
	}

  	PRINTM(INFO, "TXPOWER GET %d dbm.\n", Adapter->TxPowerLevel);
	vwrq->value = Adapter->TxPowerLevel;
	vwrq->fixed = 1;
	if (Adapter->RadioOn) {
		vwrq->disabled = 0;
		vwrq->flags = IW_TXPOW_DBM;
	} else {
		vwrq->disabled = 1;
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set TX Retry Count
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_retry(struct net_device *dev, struct iw_request_info *info,
	       struct iw_param *vwrq, char *extra)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_private   *priv = dev->priv;
	wlan_adapter   *adapter = priv->adapter;

	ENTER();


	if (vwrq->flags == IW_RETRY_LIMIT) {
		/* The MAC has a 4-bit Total_Tx_Count register
		   Total_Tx_Count = 1 + Tx_Retry_Count */
#define TX_RETRY_MIN 0
#define TX_RETRY_MAX 14
		if (vwrq->value < TX_RETRY_MIN || vwrq->value > TX_RETRY_MAX)
			return -EINVAL;

		/* Adding 1 to convert retry count to try count */
		adapter->TxRetryCount = vwrq->value + 1;

		ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_SNMP_MIB, 
				HostCmd_ACT_SET, HostCmd_OPTION_WAITFORRSP,
				OID_802_11_TX_RETRYCOUNT,
				NULL);

		if (ret) {
			LEAVE();
			return ret;
		}
	} else {
		return -EOPNOTSUPP;
	}

	LEAVE();
	return  WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get TX Retry Count
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_retry(struct net_device *dev, struct iw_request_info *info,
		struct iw_param *vwrq, char *extra)
{
	wlan_private   *priv = dev->priv;
	wlan_adapter   *Adapter = priv->adapter;
	int		ret = WLAN_STATUS_SUCCESS;

	ENTER();
	Adapter->TxRetryCount = 0;
	ret = PrepareAndSendCommand(priv,
				HostCmd_CMD_802_11_SNMP_MIB, HostCmd_ACT_GET,
				HostCmd_OPTION_WAITFORRSP,
				OID_802_11_TX_RETRYCOUNT,
				NULL);
	if (ret) {
		LEAVE();
		return ret;
	}
	vwrq->disabled = 0;
	if (!vwrq->flags) {
		vwrq->flags = IW_RETRY_LIMIT;
		/* Subtract 1 to convert try count to retry count */
		vwrq->value = Adapter->TxRetryCount - 1;
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Sort Channels
 *   
 *  @param freq 		A pointer to iw_freq structure
 *  @param num		        number of Channels
 *  @return 	   		NA
 */
static inline void sort_channels(struct iw_freq *freq, int num)
{
	int i, j;
	struct iw_freq temp;

	for (i = 0; i < num; i++)
		for (j = i + 1; j < num; j++)
			if (freq[i].i > freq[j].i) {
				temp.i = freq[i].i;
				temp.m = freq[i].m;
				
				freq[i].i = freq[j].i;
				freq[i].m = freq[j].m;
				
				freq[j].i = temp.i;	
				freq[j].m = temp.m;	
			}
}

/* data rate listing
	MULTI_BANDS:
		abg		a	b	b/g
   Infra 	G(12)		A(8)	B(4)	G(12)
   Adhoc 	A+B(12)		A(8)	B(4)	B(4)

	non-MULTI_BANDS:
		   		 	b	b/g
   Infra 	     		    	B(4)	G(12)
   Adhoc 	      		    	B(4)	B(4)
 */
/** 
 *  @brief Get Range Info
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_range(struct net_device *dev, struct iw_request_info *info,
		struct iw_point *dwrq, char *extra)
{
	int			i, j;
	wlan_private		*priv = dev->priv;
	wlan_adapter		*Adapter = priv->adapter;
	struct iw_range		*range = (struct iw_range *) extra;
	CHANNEL_FREQ_POWER	*cfp;
	WLAN_802_11_RATES	rates;

	u8 flag = FALSE;

	ENTER();

	dwrq->length = sizeof(struct iw_range);
	memset(range, 0, sizeof(struct iw_range));

	range->min_nwid = 0;
	range->max_nwid = 0;

	memset(rates, 0, sizeof(rates));
	range->num_bitrates = get_active_data_rates(Adapter, rates);

	for (i=0; i < MIN(range->num_bitrates,IW_MAX_BITRATES) && rates[i]; i++) {
		range->bitrate[i] = (rates[i] & 0x7f) * 500000;
	}
	range->num_bitrates = i;
	PRINTM(INFO, "IW_MAX_BITRATES=%d num_bitrates=%d\n",IW_MAX_BITRATES,range->num_bitrates);

	range->num_frequency = 0;
	if( wlan_get_state_11d( priv) == ENABLE_11D &&
			Adapter->MediaConnectStatus == WlanMediaStateConnected) {
		u8 chan_no;
		u8 band;
		
		parsed_region_chan_11d_t *parsed_region_chan = &Adapter->parsed_region_chan;

		if( parsed_region_chan == NULL ) {
			PRINTM(INFO, "11D:parsed_region_chan is NULL\n");
			LEAVE();
			return 0;
		}
		band = parsed_region_chan->band;
		PRINTM(INFO, "band=%d NoOfChan=%d\n", band, parsed_region_chan->NoOfChan);

		for (i = 0; (range->num_frequency < IW_MAX_FREQUENCIES) 
				&& (i < parsed_region_chan->NoOfChan); i++) {
			chan_no = parsed_region_chan->chanPwr[i].chan;
			PRINTM(INFO, "chan_no=%d\n", chan_no );
			range->freq[range->num_frequency].i = (long) chan_no;
			range->freq[range->num_frequency].m = 
					(long) chan_2_freq(chan_no, band) * 100000;
			range->freq[range->num_frequency].e = 1;
			range->num_frequency++;
		}
		flag = TRUE;
	}
	if ( !flag )
	{
		for (j = 0; (range->num_frequency < IW_MAX_FREQUENCIES) 
			&& (j < sizeof(Adapter->region_channel) 
				/ sizeof(Adapter->region_channel[0]));
				j++) {
			cfp = Adapter->region_channel[j].CFP;
			for (i = 0; (range->num_frequency < IW_MAX_FREQUENCIES) 
				&& Adapter->region_channel[j].Valid 
				&& cfp 
				&& (i < Adapter->region_channel[j].NrCFP);
					i++) {
				range->freq[range->num_frequency].i = (long) cfp->Channel;
				range->freq[range->num_frequency].m = (long) cfp->Freq * 100000;
				range->freq[range->num_frequency].e = 1;
				cfp++;
				range->num_frequency++;
			}
		}
	}

	PRINTM(INFO, "IW_MAX_FREQUENCIES=%d num_frequency=%d\n",IW_MAX_FREQUENCIES,
							range->num_frequency);

	range->num_channels = range->num_frequency;
	
	sort_channels(&range->freq[0], range->num_frequency);
	
	/*
	 * Set an indication of the max TCP throughput in bit/s that we can
	 * expect using this interface 
	 */
	if (i > 2)
		range->throughput = 5000 * 1000;
	else
		range->throughput = 1500 * 1000;

	range->min_rts = MRVDRV_RTS_MIN_VALUE;
	range->max_rts = MRVDRV_RTS_MAX_VALUE;
	range->min_frag = MRVDRV_FRAG_MIN_VALUE;
	range->max_frag = MRVDRV_FRAG_MAX_VALUE;

	range->encoding_size[0] = 5;
	range->encoding_size[1] = 13;
	range->num_encoding_sizes = 2;
	range->max_encoding_tokens = 4;

	range->min_pmp = 1000000;
	range->max_pmp = 120000000;	
	range->min_pmt = 1000;		
	range->max_pmt = 1000000;	
	range->pmp_flags = IW_POWER_PERIOD;
	range->pmt_flags = IW_POWER_TIMEOUT;
	range->pm_capa = IW_POWER_PERIOD | IW_POWER_TIMEOUT | IW_POWER_ALL_R;

	/*
	 * Minimum version we recommend 
	 */
	range->we_version_source = 15;

	/*
	 * Version we are compiled with 
	 */
	range->we_version_compiled = WIRELESS_EXT;

	range->retry_capa = IW_RETRY_LIMIT;
	range->retry_flags = IW_RETRY_LIMIT | IW_RETRY_MAX;

	range->min_retry = TX_RETRY_MIN;
	range->max_retry = TX_RETRY_MAX;

	/*
	 * Set the qual, level and noise range values 
	 */
	/*
	 * need to put the right values here 
	 */
	range->max_qual.qual = 10;
	range->max_qual.level = 0;
	range->max_qual.noise = 0;
	range->sensitivity = 0;

	/*
	 * Setup the supported power level ranges 
	 */
	memset(range->txpower, 0, sizeof(range->txpower));
	range->txpower[0] = 5;
	range->txpower[1] = 7;
	range->txpower[2] = 9;
	range->txpower[3] = 11;
	range->txpower[4] = 13;
	range->txpower[5] = 15;
	range->txpower[6] = 17;
	range->txpower[7] = 19;

	range->num_txpower = 8;
	range->txpower_capa = IW_TXPOW_DBM;
	range->txpower_capa |= IW_TXPOW_RANGE;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief  Set power management 
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_power(struct net_device *dev, struct iw_request_info *info,
		struct iw_param *vwrq, char *extra)
{
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();


	/* PS is currently supported only in Infrastructure Mode 
	 * Remove this check if it is to be supported in IBSS mode also 
	 */

	if (vwrq->disabled) {
		Adapter->PSMode = Wlan802_11PowerModeCAM;
		if (Adapter->PSState != PS_STATE_FULL_POWER) {
			PSWakeup(priv, HostCmd_OPTION_WAITFORRSP);
		}

		return 0;
	}

	if ((vwrq->flags & IW_POWER_TYPE) == IW_POWER_TIMEOUT) {
		PRINTM(INFO, "Setting power timeout command is not supported\n");
		return -EINVAL;
	} else if ((vwrq->flags & IW_POWER_TYPE) == IW_POWER_PERIOD) {
		PRINTM(INFO, "Setting power period command is not supported\n");
		return -EINVAL;
	}

	if (Adapter->PSMode != Wlan802_11PowerModeCAM) {
		return WLAN_STATUS_SUCCESS;
	}
     
	Adapter->PSMode = Wlan802_11PowerModeMAX_PSP;
     
	if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
		PSSleep(priv, HostCmd_OPTION_WAITFORRSP);
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief  Get power management 
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_power(struct net_device *dev, struct iw_request_info *info,
		struct iw_param *vwrq, char *extra)
{
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;
	int		mode;

	ENTER();

	mode = Adapter->PSMode;

	if ((vwrq->disabled = (mode == Wlan802_11PowerModeCAM))
			|| Adapter->MediaConnectStatus == 
					WlanMediaStateDisconnected) {
		LEAVE();
		return WLAN_STATUS_SUCCESS;
	}

	vwrq->value = 0;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief  Set sensitivity threshold
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_sens(struct net_device *dev, struct iw_request_info *info, 
		struct iw_param *vwrq, char *extra)
{
	ENTER();
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief  Get sensitivity threshold
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_FAILURE
 */
static int wlan_get_sens(struct net_device *dev,
	      struct iw_request_info *info, struct iw_param *vwrq,
	      char *extra)
{
	ENTER();
	LEAVE();
	return WLAN_STATUS_FAILURE;
}



/*
 * iwconfig settable callbacks 
 */
static const iw_handler wlan_handler[] = {
    (iw_handler) wlan_config_commit,        /* SIOCSIWCOMMIT */
    (iw_handler) wlan_get_name,             /* SIOCGIWNAME */
    (iw_handler) NULL,                      /* SIOCSIWNWID */
    (iw_handler) NULL,                      /* SIOCGIWNWID */
    (iw_handler) wlan_set_freq,             /* SIOCSIWFREQ */
    (iw_handler) wlan_get_freq,             /* SIOCGIWFREQ */
    (iw_handler) wlan_set_mode,             /* SIOCSIWMODE */
    (iw_handler) wlan_get_mode,             /* SIOCGIWMODE */
    (iw_handler) wlan_set_sens,             /* SIOCSIWSENS */
    (iw_handler) wlan_get_sens,             /* SIOCGIWSENS */
    (iw_handler) NULL,                      /* SIOCSIWRANGE */
    (iw_handler) wlan_get_range,            /* SIOCGIWRANGE */
    (iw_handler) NULL,                      /* SIOCSIWPRIV */
    (iw_handler) NULL,                      /* SIOCGIWPRIV */
    (iw_handler) NULL,                      /* SIOCSIWSTATS */
    (iw_handler) NULL,                      /* SIOCGIWSTATS */
#if WIRELESS_EXT > 15
    iw_handler_set_spy,                     /* SIOCSIWSPY */
    iw_handler_get_spy,                     /* SIOCGIWSPY */
    iw_handler_set_thrspy,                  /* SIOCSIWTHRSPY */
    iw_handler_get_thrspy,                  /* SIOCGIWTHRSPY */
#else  /* WIRELESS_EXT > 15 */
#ifdef WIRELESS_SPY
    (iw_handler) wlan_set_spy,              /* SIOCSIWSPY */
    (iw_handler) wlan_get_spy,              /* SIOCGIWSPY */
#else  /* WIRELESS_SPY */
    (iw_handler) NULL,                      /* SIOCSIWSPY */
    (iw_handler) NULL,                      /* SIOCGIWSPY */
#endif /* WIRELESS_SPY */
    (iw_handler) NULL,                      /* -- hole -- */
    (iw_handler) NULL,                      /* -- hole -- */
#endif /* WIRELESS_EXT > 15 */
    (iw_handler) wlan_set_wap,              /* SIOCSIWAP */
    (iw_handler) wlan_get_wap,              /* SIOCGIWAP */
    (iw_handler) NULL,                      /* -- hole -- */
    //(iw_handler) wlan_get_aplist,         /* SIOCGIWAPLIST */
    NULL,                                   /* SIOCGIWAPLIST */
#if WIRELESS_EXT > 13
    (iw_handler) wlan_set_scan,             /* SIOCSIWSCAN */
    (iw_handler) wlan_get_scan,             /* SIOCGIWSCAN */
#else  /* WIRELESS_EXT > 13 */
    (iw_handler) NULL,                      /* SIOCSIWSCAN */
    (iw_handler) NULL,                      /* SIOCGIWSCAN */
#endif /* WIRELESS_EXT > 13 */
    (iw_handler) wlan_set_essid,            /* SIOCSIWESSID */
    (iw_handler) wlan_get_essid,            /* SIOCGIWESSID */
    (iw_handler) wlan_set_nick,             /* SIOCSIWNICKN */
    (iw_handler) wlan_get_nick,             /* SIOCGIWNICKN */
    (iw_handler) NULL,                      /* -- hole -- */
    (iw_handler) NULL,                      /* -- hole -- */
    (iw_handler) wlan_set_rate,             /* SIOCSIWRATE */
    (iw_handler) wlan_get_rate,             /* SIOCGIWRATE */
    (iw_handler) wlan_set_rts,              /* SIOCSIWRTS */
    (iw_handler) wlan_get_rts,              /* SIOCGIWRTS */
    (iw_handler) wlan_set_frag,             /* SIOCSIWFRAG */
    (iw_handler) wlan_get_frag,             /* SIOCGIWFRAG */
    (iw_handler) wlan_set_txpow,            /* SIOCSIWTXPOW */
    (iw_handler) wlan_get_txpow,            /* SIOCGIWTXPOW */
    (iw_handler) wlan_set_retry,            /* SIOCSIWRETRY */
    (iw_handler) wlan_get_retry,            /* SIOCGIWRETRY */
    (iw_handler) NULL,                      /* -- hole -- */
    (iw_handler) wlan_get_encode,           /* SIOCGIWENCODE */
    (iw_handler) wlan_set_power,            /* SIOCSIWPOWER */
    (iw_handler) wlan_get_power,            /* SIOCGIWPOWER */
#if (WIRELESS_EXT >= 18)
    (iw_handler) NULL,                      /* -- hole -- */
    (iw_handler) NULL,                      /* -- hole -- */
    (iw_handler) NULL,                      /* -- hole -- */
    (iw_handler) NULL,                      /* -- hole -- */
#endif /* WIRELESSS_EXT >= 18 */
};

/*
 * iwpriv settable callbacks 
 */

static const iw_handler wlan_private_handler[] = {
	NULL,			/* SIOCIWFIRSTPRIV */
};

static const struct iw_priv_args wlan_private_args[] = {
	/*
	 * { cmd, set_args, get_args, name } 
	 */
	{
	WLANEXTSCAN,
	IW_PRIV_TYPE_INT,
	IW_PRIV_TYPE_CHAR | 2,
	"extscan"
	},

	{
	WLANCISDUMP,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_BYTE | 512,
	"getcis"
	},

     	{
	WLANSCAN_TYPE,
	IW_PRIV_TYPE_CHAR | 8,
	IW_PRIV_TYPE_CHAR | 8,
	"scantype"
	},

	{ 
	WLAN_SETINT_GETINT,
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	""
	},
	{ 
	WLANNF, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	"getNF"
	},
	{ 
	WLANRSSI, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	"getRSSI"
	},
	{
	WLANENABLE11D,		
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,  
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 
	"enable11d"
	},
	{ 
	WLANADHOCGRATE,
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,  
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 
	"adhocgrate"
	},
		
	{ 
	WLAN_SUBCMD_SET_PRESCAN,
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 
	"prescan"
	},	
	{ 
	WLAN_SETONEINT_GETONEINT,
	IW_PRIV_TYPE_INT | 1, 
	IW_PRIV_TYPE_INT | 1, 
	""
	},
	{ 
	WLAN_BEACON_INTERVAL,
	IW_PRIV_TYPE_INT | 1, 
	IW_PRIV_TYPE_INT | 1, 
	"bcninterval"
	},
	{
	WLAN_LISTENINTRVL,
	IW_PRIV_TYPE_INT | 1,
	IW_PRIV_TYPE_INT | 1,
	"lolisteninter"
	},
	{
	WLAN_TXCONTROL,
	IW_PRIV_TYPE_INT | 1,
	IW_PRIV_TYPE_INT | 1,
	"txcontrol"
	},
	{
	WLAN_NULLPKTINTERVAL,
	IW_PRIV_TYPE_INT | 1,
	IW_PRIV_TYPE_INT | 1,
	"psnullinterval"
	},
	/* Using iwpriv sub-command feature */
	{ 
	WLAN_SETONEINT_GETNONE,  /* IOCTL: 24 */
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE, 
	""
	},

	{ 
	WLAN_SUBCMD_SETRXANTENNA, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE,
	"setrxant"
	},
	{ 
	WLAN_SUBCMD_SETTXANTENNA, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE, 
	"settxant"
	},
	{
	WLANSETAUTHALG,
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE,
	"authalgs",
	},
	{
	WLANSET8021XAUTHALG,
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE,
	"8021xauthalgs",
	},
	{
	WLANSETENCRYPTIONMODE,
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE,
	"encryptionmode",
	},
	{ 
	WLANSETREGION,
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE,
	"setregioncode"
	},
	{ 
	WLAN_SET_LISTEN_INTERVAL,
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE,
	"setlisteninter"
	},
	{ 
	WLAN_SET_MULTIPLE_DTIM,
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE,
	"setmultipledtim"
	},
	{ 
	WLAN_SET_ATIM_WINDOW,
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE,
	"atimwindow"
	},
	{ 
	WLANSETBCNAVG, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE,
	"setbcnavg"
	},
	{ 
	WLANSETDATAAVG, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	IW_PRIV_TYPE_NONE,
	"setdataavg"
	},
	{
		WLAN_SET_LINKMODE,
		IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 
		IW_PRIV_TYPE_NONE,
		"linkmode"
	},
	{
		WLAN_SET_RADIOMODE,
		IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 
		IW_PRIV_TYPE_NONE,
		"radiomode"
	},
	{
		WLAN_SET_DEBUGMODE,
		IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 
		IW_PRIV_TYPE_NONE,
		"debugmode"
	},
	{ 
	WLAN_SETNONE_GETONEINT, 
	IW_PRIV_TYPE_NONE, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	""
	},
	{ 
	WLANGETREGION,
	IW_PRIV_TYPE_NONE, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	"getregioncode"
	},
	{ 
	WLAN_GET_LISTEN_INTERVAL,
	IW_PRIV_TYPE_NONE, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	"getlisteninter"
	},
	{ 
	WLAN_GET_MULTIPLE_DTIM,
	IW_PRIV_TYPE_NONE, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	"getmultipledtim"
	},
	{ 
	WLAN_GET_TX_RATE,
	IW_PRIV_TYPE_NONE, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	"gettxrate"
	},
	{ 
	WLANGETBCNAVG,
	IW_PRIV_TYPE_NONE, 
	IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1,
	"getbcnavg"
	},
	{
		WLAN_GET_LINKMODE,
		IW_PRIV_TYPE_NONE,
		IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 
		"get_linkmode"
	},
	{
		WLAN_GET_RADIOMODE,
		IW_PRIV_TYPE_NONE,
		IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 
		"get_radiomode"
	},
	{
		WLAN_GET_DEBUGMODE,
		IW_PRIV_TYPE_NONE,
		IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 
		"get_debugmode"
	},
	{ 
	WLAN_SETNONE_GETTWELVE_CHAR,
	IW_PRIV_TYPE_NONE, 
	IW_PRIV_TYPE_CHAR | 12,
	""
	},
	{ 
	WLAN_SUBCMD_GETRXANTENNA, 
	IW_PRIV_TYPE_NONE, 
	IW_PRIV_TYPE_CHAR | 12, 
	"getrxant"
	},
	{ 
	WLAN_SUBCMD_GETTXANTENNA, 
	IW_PRIV_TYPE_NONE, 
	IW_PRIV_TYPE_CHAR | 12, 
	"gettxant"
	},
	{ 
	WLAN_GET_TSF, 
	IW_PRIV_TYPE_NONE, 
	IW_PRIV_TYPE_CHAR | 12, 
	"gettsf"
	},
	{
	WLAN_SETNONE_GETNONE,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_NONE,
	""
	},
	{
	WLANDEAUTH,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_NONE,
	"deauth"
	},
	{
	WLANADHOCSTOP,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_NONE,
	"adhocstop"
	},
	{
	WLANRADIOON,		
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_NONE,
	"radioon"
	},
	{
	WLANRADIOOFF,		
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_NONE,
	"radiooff"
	},
#ifdef REASSOCIATION
	{
	WLANREASSOCIATIONAUTO,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_NONE,
	"reasso-on"
	},
	{
	WLANREASSOCIATIONUSER,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_NONE,
	"reasso-off"
	},
#endif /* REASSOCIATION */
	{
	WLANWLANIDLEON,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_NONE,
	"wlanidle-on"
	},
	{
	WLANWLANIDLEOFF,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_NONE,
	"wlanidle-off"
	},
	{
	WLAN_SET64CHAR_GET64CHAR,
	IW_PRIV_TYPE_CHAR | 64,
	IW_PRIV_TYPE_CHAR | 64,
	""
	},
	{
	WLANSLEEPPARAMS,
	IW_PRIV_TYPE_CHAR | 64,
	IW_PRIV_TYPE_CHAR | 64,
	"sleepparams"
	},

	{
	WLANSCAN_MODE,
	IW_PRIV_TYPE_CHAR | 64,
	IW_PRIV_TYPE_CHAR | 64,
	"scanmode"
	},
	{ 
	WLAN_GET_ADHOC_STATUS,
	IW_PRIV_TYPE_CHAR | 64,
	IW_PRIV_TYPE_CHAR | 64,
	"getadhocstatus"
	},
	{
	WLAN_SETWORDCHAR_GETNONE,
	IW_PRIV_TYPE_CHAR | 32,
	IW_PRIV_TYPE_NONE,
	""
	},
	{
	WLAN_SETNONE_GETWORDCHAR,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_CHAR | 128,
	""
	},
	{
	WLANVERSION,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_CHAR | 128,
	"version"
	},
	{
	WLANSETWPAIE,
	IW_PRIV_TYPE_CHAR | IW_PRIV_SIZE_FIXED | 24,
	IW_PRIV_TYPE_NONE,
	"setwpaie"
	},
	{ 
	WLAN_SETTENCHAR_GETNONE,
	IW_PRIV_TYPE_CHAR | 10, 
	IW_PRIV_TYPE_NONE,
	""
	},
	{ 
	WLAN_SETNONE_GETTENCHAR,
	IW_PRIV_TYPE_NONE, 
	IW_PRIV_TYPE_CHAR | 10, 
	""
	},
	{
	WLANGETLOG,
	IW_PRIV_TYPE_NONE,
	IW_PRIV_TYPE_CHAR | GETLOG_BUFSIZE,
	"getlog"
	},
        {
        WLAN_SET_GET_SIXTEEN_INT,
        IW_PRIV_TYPE_INT | 16,
        IW_PRIV_TYPE_INT | 16,
        ""
        },
        {
        WLAN_TPCCFG,
        IW_PRIV_TYPE_INT | 16,
        IW_PRIV_TYPE_INT | 16,
        "tpccfg"
        },
        {
        WLAN_POWERCFG,
        IW_PRIV_TYPE_INT | 16,
        IW_PRIV_TYPE_INT | 16,
        "powercfg"
        },
	{
	WLAN_AUTO_FREQ_SET,
	IW_PRIV_TYPE_INT | 16,
	IW_PRIV_TYPE_INT | 16,
	"setafc"
	},
	{
	WLAN_AUTO_FREQ_GET,
	IW_PRIV_TYPE_INT | 16,
	IW_PRIV_TYPE_INT | 16,
	"getafc"
	},
	{
	WLAN_SCANPROBES,
	IW_PRIV_TYPE_INT | 16,
	IW_PRIV_TYPE_INT | 16,
	"scanprobes"
	},
        {
        WLAN_LED_GPIO_CTRL,
        IW_PRIV_TYPE_INT | 16,
        IW_PRIV_TYPE_INT | 16,
        "ledgpio"
        },
	{ 
	WLAN_SLEEP_PERIOD,
        IW_PRIV_TYPE_INT | 16,
        IW_PRIV_TYPE_INT | 16,
	"sleeppd"
	},
	{ 
	WLAN_ADAPT_RATESET,
        IW_PRIV_TYPE_INT | 16,
        IW_PRIV_TYPE_INT | 16,
	"rateadapt"
	},

	{ 
	WLAN_INACTIVITY_TIMEOUT,
        IW_PRIV_TYPE_INT | 16,
        IW_PRIV_TYPE_INT | 16,
	"inactivityto"
	},
	{ 
	WLANSNR, 
	IW_PRIV_TYPE_INT | 16, 
	IW_PRIV_TYPE_INT | 16, 
	"getSNR"
	},
	{ 
	WLAN_GET_RATE,
	IW_PRIV_TYPE_INT | 16, 
	IW_PRIV_TYPE_INT | 16, 
	"getrate"
	},
	{ 
	WLAN_GET_RXINFO,
	IW_PRIV_TYPE_INT | 16, 
	IW_PRIV_TYPE_INT | 16, 
	"getrxinfo"
	},

	{ 
        WLAN_SET_GET_2K, /* IOCTL : 14 */
		IW_PRIV_TYPE_BYTE | 2000,
		IW_PRIV_TYPE_BYTE | 2000,
        ""
	},

	{
		WLAN_SET_USER_SCAN,
		IW_PRIV_TYPE_BYTE | 2000,
		IW_PRIV_TYPE_BYTE | 2000,
		"setuserscan"
	},
	{
		WLAN_GET_SCAN_TABLE,
		IW_PRIV_TYPE_BYTE | 2000,
		IW_PRIV_TYPE_BYTE | 2000,
		"getscantable"
	},


};

struct iw_handler_def wlan_handler_def = {
	num_standard:sizeof(wlan_handler) / sizeof(iw_handler),
	num_private:sizeof(wlan_private_handler) / sizeof(iw_handler),
	num_private_args:sizeof(wlan_private_args) /
	sizeof(struct iw_priv_args),
	standard:(iw_handler *) wlan_handler,
	private:(iw_handler *) wlan_private_handler,
	private_args:(struct iw_priv_args *) wlan_private_args,
};

/** 
 *  @brief wlan hostcmd ioctl handler
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param req		        A pointer to ifreq structure
 *  @param cmd			command 
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_hostcmd_ioctl(struct net_device *dev, struct ifreq *req, int cmd)
{
	u8		*tempResponseBuffer;
	struct CmdCtrlNode *pCmdNode;
	struct HostCmd_DS_GEN	*gencmd, *pCmdPtr;
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;
	u16		wait_option = 0;

	ENTER();

	/*
	 * Get a free command control node 
	 */
	if (!(pCmdNode = GetFreeCmdCtrlNode(priv))) {
		PRINTM(INFO, "Failed GetFreeCmdCtrlNode\n");
		return -ENOMEM;
	}

	if (!(tempResponseBuffer = kmalloc(3000, GFP_KERNEL))) {
		PRINTM(INFO, "ERROR: Failed to allocate response buffer!\n");
		return -ENOMEM;
	}

	wait_option |= HostCmd_OPTION_WAITFORRSP;
	
	SetCmdCtrlNode(priv, pCmdNode, 0, wait_option, NULL);
	init_waitqueue_head(&pCmdNode->cmdwait_q);

	pCmdPtr = (struct HostCmd_DS_GEN *) pCmdNode->BufVirtualAddr;
	if (copy_from_user(&gencmd, req->ifr_data, sizeof(gencmd))) {
		PRINTM(INFO, "Copy from user failed\n");
		return -EFAULT;
	}

	/*
	 * Copy the whole command into the command buffer 
	 */
	if (copy_from_user(pCmdPtr, req->ifr_data, gencmd->Size)) {
		PRINTM(INFO, "Copy from user failed\n");
		return -EFAULT;
	}

	pCmdNode->pdata_buf = tempResponseBuffer;
	pCmdNode->CmdFlags |= CMD_F_HOSTCMD;

	pCmdPtr->SeqNum = ++priv->adapter->SeqNum;
	pCmdPtr->Result = 0;

	PRINTM(INFO, "HOSTCMD Command: 0x%04x Size: %d SeqNum: %d\n",
			pCmdPtr->Command, pCmdPtr->Size, pCmdPtr->SeqNum);
	HEXDUMP("Command Data", (u8 *) (pCmdPtr), MIN(32, pCmdPtr->Size));
	PRINTM(INFO, "Copying data from : (user)0x%p -> 0x%p(driver)\n",
			req->ifr_data, pCmdPtr);

	pCmdNode->CmdWaitQWoken = FALSE;
	QueueCmd(Adapter, pCmdNode, TRUE);
	wake_up_interruptible(&priv->MainThread.waitQ);

	if (wait_option & HostCmd_OPTION_WAITFORRSP) {
		/* Sleep until response is generated by FW */
		wait_event_interruptible(pCmdNode->cmdwait_q, 
						pCmdNode->CmdWaitQWoken);
	}
		
	/* Copy the response back to user space */
	pCmdPtr = (struct HostCmd_DS_GEN *) tempResponseBuffer;

	if (copy_to_user(req->ifr_data, tempResponseBuffer, pCmdPtr->Size))
		PRINTM(INFO, "ERROR: copy_to_user failed!\n");

	kfree(tempResponseBuffer);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get Rx Info 
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param wreq		        A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success
 */
static int wlan_get_rxinfo(wlan_private *priv, struct iwreq *wrq)
{
  	wlan_adapter *Adapter = priv->adapter;
  	int data[2];
	ENTER();
	data[0] = Adapter->SNR[TYPE_RXPD][TYPE_NOAVG];
	data[1] = Adapter->RxPDRate;
	if (copy_to_user(wrq->u.data.pointer,data,sizeof(int) * 2)) {
		PRINTM(INFO, "Copy to user failed\n");
		return -EFAULT;
	}
	wrq->u.data.length = 2;
	LEAVE();
  	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get SNR 
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param wreq		        A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_snr(wlan_private *priv, struct iwreq *wrq)
{
  	int ret = WLAN_STATUS_SUCCESS;
  	wlan_adapter *Adapter = priv->adapter;
  	int data[4];

  	ENTER();
  	memset(data,0,sizeof(data));
  	if(wrq->u.data.length) {
		if (copy_from_user(data,wrq->u.data.pointer,MIN(wrq->u.data.length,4) * sizeof(int))) {
			PRINTM(INFO, "Copy from user failed\n");
			return -EFAULT;
		}
	}
  	if ((wrq->u.data.length == 0) || (data[0] == 0) || (data[0] == 1))
  	{ 
		if (Adapter->MediaConnectStatus == WlanMediaStateConnected){
    			ret = PrepareAndSendCommand(priv, 
          			HostCmd_CMD_802_11_RSSI,
          			0, HostCmd_OPTION_WAITFORRSP,
          			0, NULL);

    			if (ret) 
    			{
      				LEAVE();
      				return ret;
    			}
		}
	}

  	if (wrq->u.data.length == 0)
  	{
    		data[0] = Adapter->SNR[TYPE_BEACON][TYPE_NOAVG];
    		data[1] = Adapter->SNR[TYPE_BEACON][TYPE_AVG];
      		data[2] = Adapter->SNR[TYPE_RXPD][TYPE_NOAVG];
    		data[3] = Adapter->SNR[TYPE_RXPD][TYPE_AVG] / AVG_SCALE;
    		if (copy_to_user(wrq->u.data.pointer,data,sizeof(int) * 4)) {
				PRINTM(INFO, "Copy to user failed\n");
				return -EFAULT;
			}
    		wrq->u.data.length = 4;
  	}
  	else if (data[0] == 0)
  	{
    		data[0] = Adapter->SNR[TYPE_BEACON][TYPE_NOAVG];
    		if (copy_to_user(wrq->u.data.pointer,data,sizeof(int))) {
				PRINTM(INFO, "Copy to user failed\n");
				return -EFAULT;
			}
    		wrq->u.data.length = 1;
  	}
  	else if (data[0] == 1)
  	{
    		data[0] = Adapter->SNR[TYPE_BEACON][TYPE_AVG];
    		if (copy_to_user(wrq->u.data.pointer,data,sizeof(int))) {
				PRINTM(INFO, "Copy to user failed\n");
				return -EFAULT;
			}
    		wrq->u.data.length = 1;
  	}
  	else if (data[0] == 2)
  	{
      		data[0] = Adapter->SNR[TYPE_RXPD][TYPE_NOAVG];
    		if (copy_to_user(wrq->u.data.pointer,data,sizeof(int))) {
				PRINTM(INFO, "Copy to user failed\n");
				return -EFAULT;
			}
    		wrq->u.data.length = 1;
  	}
  	else if (data[0] == 3)
  	{
    		data[0] = Adapter->SNR[TYPE_RXPD][TYPE_AVG] / AVG_SCALE;
    		if (copy_to_user(wrq->u.data.pointer,data,sizeof(int))) {
				PRINTM(INFO, "Copy to user failed\n");
				return -EFAULT;
			}
    		wrq->u.data.length = 1;
  	}
  	else
  	{
    		return -ENOTSUPP;
  	}
    
  	LEAVE();
  	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief Get/Set Adhoc beacon Interval 
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param wreq		        A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_beacon_interval(wlan_private *priv, struct iwreq *wrq)
{
	int data;
	wlan_adapter   *Adapter = priv->adapter;

	ENTER();

	if ( wrq->u.data.length > 0 ) {
		if (copy_from_user(&data,wrq->u.data.pointer,sizeof(int))) {
			PRINTM(INFO, "Copy from user failed\n");
			return -EFAULT;
		}

		PRINTM(INFO, "WLAN SET BEACON INTERVAL: %d\n", data);
		if((data > MRVDRV_MAX_BEACON_INTERVAL) || (data < MRVDRV_MIN_BEACON_INTERVAL))
			return -ENOTSUPP;
		Adapter->BeaconPeriod = data;
	}
	data = Adapter->BeaconPeriod;
	if (copy_to_user(wrq->u.data.pointer, &data, sizeof(int))) {
		PRINTM(INFO, "Copy to user failed\n");
		return -EFAULT;
	}

	wrq->u.data.length = 1;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get RSSI 
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param wreq		        A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_rssi(wlan_private *priv, struct iwreq *wrq)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_adapter	*Adapter = priv->adapter;
	int		temp;
	int data = 0;
	int 		*val;
	
	ENTER();
	data = *( (int *) (wrq->u.name + SUBCMD_OFFSET) );			
	if ((data == 0) || (data == 1)) { 
		ret = PrepareAndSendCommand(priv, 
				HostCmd_CMD_802_11_RSSI,
				0,HostCmd_OPTION_WAITFORRSP,
				0, NULL);
		if (ret) {
			LEAVE();
			return ret;
		}
	}

	switch (data) {
		case 0:
		
		temp = CAL_RSSI(Adapter->SNR[TYPE_BEACON][TYPE_NOAVG],
					Adapter->NF[TYPE_BEACON][TYPE_NOAVG]);
				break;
		case 1:
		temp = CAL_RSSI(Adapter->SNR[TYPE_BEACON][TYPE_AVG],
				Adapter->NF[TYPE_BEACON][TYPE_AVG]);
				break;
		case 2:
		temp = CAL_RSSI(Adapter->SNR[TYPE_RXPD][TYPE_NOAVG],
					Adapter->NF[TYPE_RXPD][TYPE_NOAVG]);
				break;
		case 3:
		temp = CAL_RSSI(Adapter->SNR[TYPE_RXPD][TYPE_AVG] / AVG_SCALE,
				Adapter->NF[TYPE_RXPD][TYPE_AVG] / AVG_SCALE);
				break;
		default:
			return -ENOTSUPP;
	}
	val = (int *)wrq->u.name;
	*val = 	temp;
 
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get NF
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param wreq		        A pointer to iwreq structure 
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_nf(wlan_private *priv, struct iwreq *wrq)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_adapter	*Adapter = priv->adapter;
	int		temp;
	int data = 0;
	int 		*val;

	ENTER();
	data = *( (int *) (wrq->u.name + SUBCMD_OFFSET) );
	if ((data == 0) || (data == 1) ) { 
		ret = PrepareAndSendCommand(priv, 
				HostCmd_CMD_802_11_RSSI,
				0,HostCmd_OPTION_WAITFORRSP,
				0, NULL);

		if (ret) {
			LEAVE();
			return ret;
		}
	}

	switch (data) {
		case 0:
			temp = Adapter->NF[TYPE_BEACON][TYPE_NOAVG];
			break;
		case 1:
			temp = Adapter->NF[TYPE_BEACON][TYPE_AVG];
			break;
		case 2:
			temp = Adapter->NF[TYPE_RXPD][TYPE_NOAVG];
			break;
		case 3:
			temp = Adapter->NF[TYPE_RXPD][TYPE_AVG] / AVG_SCALE;
			break;
		default:
			return -ENOTSUPP;
	}

	temp = CAL_NF(temp);
		
	PRINTM(INFO, "***temp = %d\n", temp);		
	val = (int *)wrq->u.name;
	*val = 	temp;
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}



/** 
 *  @brief Get Support Rates
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param wrq			A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_getrate_ioctl(wlan_private *priv, struct iwreq *wrq)
{
	wlan_adapter		*Adapter = priv->adapter;
	WLAN_802_11_RATES	rates;
	int rate[16];
	int			i;

	ENTER();

	memset(rates, 0, sizeof(rates));
	memset(rate,0,sizeof(rate));
	wrq->u.data.length = get_active_data_rates(Adapter, rates);
	if(wrq->u.data.length > 16)
		wrq->u.data.length = 16;

	for (i = 0; i < wrq->u.data.length; i++)
	{
		rates[i] &= ~0x80;
		rate[i] = rates[i];
	}

	if (copy_to_user(wrq->u.data.pointer, rate, wrq->u.data.length * sizeof(int))) {
		PRINTM(INFO, "Copy to user failed\n");
		return -EFAULT;
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get TxRate
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_txrate_ioctl(wlan_private *priv, struct ifreq *req)
{
	wlan_adapter		*Adapter = priv->adapter;
	int * pdata;
	struct iwreq	*wrq = (struct iwreq *) req;
	int 	ret = WLAN_STATUS_SUCCESS;
	ENTER();
	Adapter->TxRate = 0;
	PRINTM(INFO,"wlan_get_txrate_ioctl\n");
	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_TX_RATE_QUERY,
			HostCmd_ACT_GET, HostCmd_OPTION_WAITFORRSP,
			0, NULL);
	if (ret) {
		LEAVE();
		return ret;
	}
	pdata = (int *)wrq->u.name;
	*pdata = (int) Adapter->TxRate;	
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get Adhoc Status
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param wrq			A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_get_adhoc_status_ioctl(wlan_private *priv, struct iwreq *wrq)
{
	char 		status[64];
	wlan_adapter	*Adapter = priv->adapter;
	
	memset(status, 0, sizeof(status));

	switch (Adapter->InfrastructureMode) {
	case Wlan802_11IBSS:
		if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
			if (Adapter->AdhocCreate == TRUE)
				memcpy(&status, "AdhocStarted", sizeof(status));
			else
				memcpy(&status, "AdhocJoined", sizeof(status));
		} else {
			memcpy(&status, "AdhocIdle", sizeof(status));
		}
		break;
	case Wlan802_11Infrastructure:	
		memcpy(&status, "InfraMode", sizeof(status));
		break;
	default:	
		memcpy(&status, "AutoUnknownMode", sizeof(status));
		break;
	}

	PRINTM(INFO, "Status = %s\n", status);
	wrq->u.data.length = strlen(status) + 1;

	if (wrq->u.data.pointer) {
		if (copy_to_user(wrq->u.data.pointer, 
					&status, wrq->u.data.length))
			return -EFAULT;
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get Driver Version
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_version_ioctl(wlan_private *priv, struct ifreq *req)
{
	int	len;
	char	buf[128];
	struct iwreq	*wrq = (struct iwreq *) req;

	ENTER();

	get_version(priv->adapter, buf, sizeof(buf) - 1);

	len = strlen(buf);
	if (wrq->u.data.pointer) {
		if(copy_to_user(wrq->u.data.pointer, buf, len)){
			PRINTM(INFO, "CopyToUser failed\n");
			return -EFAULT;
		}
		wrq->u.data.length = len;
	}

	PRINTM(INFO, "wlan version: %s\n", buf);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Read/Write adapter registers
 *   
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_regrdwr_ioctl(wlan_private *priv, struct ifreq *req)
{
	wlan_ioctl_regrdwr	regrdwr;
	struct wlan_offset_value offval;
	wlan_adapter		*Adapter = priv->adapter;
	int			ret = WLAN_STATUS_SUCCESS;

	ENTER();

	if(copy_from_user(&regrdwr, req->ifr_data,	sizeof(regrdwr))){
		PRINTM(INFO, "copy of regrdwr for wlan_regrdwr_ioctl from user failed \n");
		LEAVE();
 		return -EFAULT;	
	}

	if(regrdwr.WhichReg == REG_EEPROM) {
		PRINTM(MSG, "Inside RDEEPROM\n");
		Adapter->pRdeeprom = (char *) kmalloc((regrdwr.NOB + sizeof(regrdwr)), GFP_KERNEL);
		if(!Adapter -> pRdeeprom)
			return -ENOMEM;
		memcpy(Adapter->pRdeeprom, &regrdwr,sizeof(regrdwr));
		/* +14 is for Action, Offset, and NOB in 
		 * response */
		PRINTM(INFO, "Action:%d Offset: %x NOB: %02x\n",
					regrdwr.Action, 
					regrdwr.Offset, regrdwr.NOB);
		
		ret = PrepareAndSendCommand(priv,
				HostCmd_CMD_802_11_EEPROM_ACCESS,
				regrdwr.Action,HostCmd_OPTION_WAITFORRSP,
				0, &regrdwr);
	
		if (ret) {
			if(Adapter -> pRdeeprom)
				kfree(Adapter -> pRdeeprom);
			LEAVE();
			return ret;
		}

		mdelay(10);

		/*
		 * Return the result back to the user 
		 */

		if (regrdwr.Action == HostCmd_ACT_GEN_READ) {
			if(copy_to_user(req->ifr_data, Adapter -> pRdeeprom, sizeof(regrdwr) + regrdwr.NOB))
			{
				if(Adapter -> pRdeeprom)
					kfree(Adapter -> pRdeeprom);
				PRINTM(INFO, "copy of regrdwr for wlan_regrdwr_ioctl to user failed \n");
				LEAVE();
				return -EFAULT;	
			}
		}
			
		if(Adapter -> pRdeeprom)
			kfree(Adapter -> pRdeeprom);

		return WLAN_STATUS_SUCCESS;
	}

	offval.offset = regrdwr.Offset;
	offval.value = (regrdwr.Action) ? regrdwr.Value : 0x00;

	PRINTM(INFO, "RegAccess: %02x Action:%d "
			"Offset: %04x Value: %04x\n",
			regrdwr.WhichReg, regrdwr.Action, 
			offval.offset, offval.value);

	/*
	 * regrdwr.WhichReg should contain the command that
	 * corresponds to which register access is to be 
	 * performed HostCmd_CMD_MAC_REG_ACCESS 0x0019
	 * HostCmd_CMD_BBP_REG_ACCESS 0x001a 
	 * HostCmd_CMD_RF_REG_ACCESS 0x001b 
	 */
	ret = PrepareAndSendCommand(priv, regrdwr.WhichReg, 
			regrdwr.Action,HostCmd_OPTION_WAITFORRSP,
			0, &offval);

	if (ret) {
		LEAVE();
		return ret;
	}

	mdelay(10);

	/*
	 * Return the result back to the user 
	 */
	regrdwr.Value = Adapter->OffsetValue.value;
	if (regrdwr.Action == HostCmd_ACT_GEN_READ) {
		if(copy_to_user(req->ifr_data, &regrdwr, sizeof(regrdwr))){
			PRINTM(INFO, "copy of regrdwr for wlan_regrdwr_ioctl to user failed \n");
			LEAVE();
 			return -EFAULT;	
		}
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}
		



/** 
 *  @brief Set/Get WPA IE   
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_setwpaie_ioctl(wlan_private *priv, struct ifreq *req)
{
	struct iwreq	*wrq = (struct iwreq *) req;
	wlan_adapter	*Adapter = priv->adapter;
	int		ret = WLAN_STATUS_SUCCESS;

	ENTER();

	if (wrq->u.data.length) {
		if (wrq->u.data.length > sizeof(Adapter->Wpa_ie)){
			PRINTM(INFO, "failed to copy WPA IE, too big \n");
			return -EFAULT;
		}
		if (copy_from_user(Adapter->Wpa_ie, wrq->u.data.pointer,
							wrq->u.data.length)) {
			PRINTM(INFO, "failed to copy WPA IE \n");
			return -EFAULT;
		}
		Adapter->Wpa_ie_len = wrq->u.data.length;
		PRINTM(INFO, "Set Wpa_ie_len=%d IE=%#x\n", Adapter->Wpa_ie_len, 
							Adapter->Wpa_ie[0]);
		HEXDUMP("Wpa_ie", Adapter->Wpa_ie, Adapter->Wpa_ie_len);
		if (Adapter->Wpa_ie[0] == WPA_IE)
			Adapter->SecInfo.WPAEnabled = TRUE;
		else if (Adapter->Wpa_ie[0] == WPA2_IE)
			Adapter->SecInfo.WPA2Enabled = TRUE;
		else {
			Adapter->SecInfo.WPAEnabled = FALSE;
			Adapter->SecInfo.WPA2Enabled = FALSE;
		}
	} else {
		memset(Adapter->Wpa_ie, 0, sizeof(Adapter->Wpa_ie));
		Adapter->Wpa_ie_len = wrq->u.data.length;
		PRINTM(INFO, "Reset Wpa_ie_len=%d IE=%#x\n", Adapter->Wpa_ie_len,
							Adapter->Wpa_ie[0]);
		Adapter->SecInfo.WPAEnabled = FALSE;
		Adapter->SecInfo.WPA2Enabled = FALSE;
	}

	// enable/disable RSN in firmware if WPA is enabled/disabled
	// depending on variable Adapter->SecInfo.WPAEnabled is set or not
	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_ENABLE_RSN,
			HostCmd_ACT_SET, HostCmd_OPTION_WAITFORRSP,
			0, NULL);

	LEAVE();
	return ret;
}

/** 
 *  @brief Set Auto Prescan
 *  @param priv                 A pointer to wlan_private structure
 *  @param wrq			A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_subcmd_setprescan_ioctl(wlan_private *priv,  struct iwreq *wrq)
{
	int data; 
	wlan_adapter   *Adapter = priv->adapter;
	int 		*val;
	
	ENTER();

	data = *( (int *) (wrq->u.name + SUBCMD_OFFSET) );
	PRINTM(INFO, "WLAN_SUBCMD_SET_PRESCAN %d\n", data);
	switch ( data ) {
		case CMD_ENABLED:
			Adapter->Prescan = TRUE;
			break;
		case CMD_DISABLED:
			Adapter->Prescan = FALSE;
			break;
		default:			
			break;
	}

	data = 	Adapter->Prescan;
	val = (int *)wrq->u.name;
	*val = 	data;	
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set multiple dtim
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_multiple_dtim_ioctl(wlan_private *priv, struct ifreq *req)
{
	struct iwreq	*wrq = (struct iwreq *) req;
	u32		mdtim;
	int 		idata;
	int		ret = -EINVAL;

	ENTER();
	
	idata = *( (int *) (wrq->u.name + SUBCMD_OFFSET) );	
	mdtim = (u32)idata;
	if (((mdtim >= MRVDRV_MIN_MULTIPLE_DTIM) && (mdtim <= MRVDRV_MAX_MULTIPLE_DTIM))
		|| (mdtim == MRVDRV_IGNORE_MULTIPLE_DTIM)) {
		priv->adapter->MultipleDtim = mdtim;
		ret = WLAN_STATUS_SUCCESS;
	}
	if (ret)
		PRINTM(INFO, "Invalid parameter, MultipleDtim not changed.\n");

	LEAVE();
	return ret;
}

/** 
 *  @brief Set authentication mode
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_setauthalg_ioctl(wlan_private *priv, struct ifreq *req)
{
	int 	alg;
	struct iwreq	*wrq = (struct iwreq *) req;
	wlan_adapter	*Adapter = priv->adapter;
	
	ENTER();

	if (wrq->u.data.flags == 0) {
		//from iwpriv subcmd
		alg = *( (int *) (wrq->u.name + SUBCMD_OFFSET) );
	} else {	
		//from wpa_supplicant subcmd
		if (copy_from_user(&alg,wrq->u.data.pointer,sizeof(alg))) {
			PRINTM(INFO, "Copy from user failed\n");
			return -EFAULT;
		}
	}
	
	PRINTM(INFO, "auth alg is %#x\n",alg);

	switch (alg) {
		case AUTH_ALG_SHARED_KEY:
			Adapter->SecInfo.AuthenticationMode = Wlan802_11AuthModeShared;
			break;
		case AUTH_ALG_NETWORK_EAP:
			Adapter->SecInfo.AuthenticationMode = 
						Wlan802_11AuthModeNetworkEAP;
			break;
		case AUTH_ALG_OPEN_SYSTEM:
		default:
			Adapter->SecInfo.AuthenticationMode = Wlan802_11AuthModeOpen;
			break;
	}
	
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set 802.1x authentication mode
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set8021xauthalg_ioctl(wlan_private *priv, struct ifreq *req)
{
	int 		alg;
	struct iwreq	*wrq = (struct iwreq *) req;
	
	ENTER();
	
	if (wrq->u.data.flags == 0) {
		//from iwpriv subcmd
		alg =  *( (int *) (wrq->u.name + SUBCMD_OFFSET) );
	} else {	
		//from wpa_supplicant subcmd
		if (copy_from_user(&alg,wrq->u.data.pointer,sizeof(int))) {
			PRINTM(INFO, "Copy from user failed\n");
			return -EFAULT;
		}
	}
	PRINTM(INFO, "802.1x auth alg is %#x\n",alg);
	priv->adapter->SecInfo.Auth1XAlg = alg;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set Encryption mode
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_setencryptionmode_ioctl(wlan_private *priv, struct ifreq *req)
{
	int		mode;
	struct iwreq	*wrq = (struct iwreq *) req;

	ENTER();
	
	if (wrq->u.data.flags == 0) {
		//from iwpriv subcmd
		mode = *( (int *) (wrq->u.name + SUBCMD_OFFSET) );
	}
	else {	
		//from wpa_supplicant subcmd
		if (copy_from_user(&mode,wrq->u.data.pointer,sizeof(int))) {
			PRINTM(INFO, "Copy from user failed\n");
			return -EFAULT;
		}
	}
	PRINTM(INFO, "encryption mode is %#x\n",mode);
	priv->adapter->SecInfo.EncryptionMode = mode;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set Link-Layer Layer mode
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_linkmode_ioctl(wlan_private *priv, struct ifreq *req)
{
	int		mode;

	ENTER();
	
	mode = (int) ((struct ifreq *)((u8 *)req + 4))->ifr_data;

	switch (mode) {
		case 0:
			// mode: normal (802.3)
			priv->adapter->linkmode = mode;
			break;
		case 2:
			// mode: monitor (802.11)
			priv->adapter->linkmode = mode;
			break;
		default:
			printk(KERN_ALERT "usb8388-5: invalid link-layer mode (%#x)\n", mode);
			return WLAN_STATUS_NOT_ACCEPTED;
			break;
	}
	printk(KERN_ALERT "usb8388-5: link-layer mode is %#x\n", mode);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set Radio Header mode
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_radiomode_ioctl(wlan_private *priv, struct ifreq *req)
{
	int		mode;

	ENTER();
	
	mode = (int) ((struct ifreq *)((u8 *)req + 4))->ifr_data;

	switch (mode) {
		case 0:
			// mode: no radioheader
			priv->adapter->radiomode = mode;
			break;
		case 1:
			// mode: prism
			printk(KERN_ALERT "usb8388-5: prism radio header still unsupported\n");
			break;
		case 2:
			// mode: radiotap
			priv->adapter->radiomode = mode;
			break;
		default:
			printk(KERN_ALERT "usb8388-5: invalid radio header mode (%#x)\n", mode);
			return WLAN_STATUS_NOT_ACCEPTED;
			break;
	}
	printk(KERN_ALERT "usb8388-5: radio-header mode is %#x\n", mode);

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set Debug Header mode
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_set_debugmode_ioctl(wlan_private *priv, struct ifreq *req)
{
	ENTER();
	priv->adapter->debugmode = (int) ((struct ifreq*)
					  ((u8 *)req + 4))->ifr_data;
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get Rx antenna
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_subcmd_getrxantenna_ioctl(wlan_private *priv, struct ifreq *req)
{
	int		len;
	char		buf[8];
	struct iwreq	*wrq = (struct iwreq *) req;

	ENTER();

	PRINTM(INFO, "WLAN_SUBCMD_GETRXANTENNA\n");
	len = GetRxAntenna(priv, buf);

	wrq->u.data.length = len;
	if (wrq->u.data.pointer) {
		if (copy_to_user(wrq->u.data.pointer, &buf, len)) {
			PRINTM(INFO, "CopyToUser failed\n");
			return -EFAULT;
		}
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get Tx antenna
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_subcmd_gettxantenna_ioctl(wlan_private *priv, struct ifreq *req)
{
	int		len;
	char		buf[8];
	struct iwreq	*wrq = (struct iwreq *) req;

	ENTER();

	PRINTM(INFO, "WLAN_SUBCMD_GETTXANTENNA\n");
	len = GetTxAntenna(priv, buf);

	wrq->u.data.length = len;
	if (wrq->u.data.pointer) {
		if (copy_to_user(wrq->u.data.pointer, &buf, len)) {
			PRINTM(INFO, "CopyToUser failed\n");
			return -EFAULT;
		}
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get the MAC TSF value from the firmware
 *
 *  @param priv         A pointer to wlan_private structure
 *  @param wrq          A pointer to iwreq structure containing buffer
 *                      space to store a TSF value retrieved from the firmware
 *
 *  @return             0 if successful; IOCTL error code otherwise
 */
static int wlan_get_tsf_ioctl(wlan_private *priv, struct iwreq *wrq)
{
    u64 tsfVal;
    int ret;
    
    ENTER();

    ret = PrepareAndSendCommand(priv, 
                                HostCmd_CMD_GET_TSF,
                                0, HostCmd_OPTION_WAITFORRSP, 0,
                                &tsfVal);

    PRINTM(INFO, "IOCTL: Get TSF = 0x%016llx\n", tsfVal);

    if (ret != WLAN_STATUS_SUCCESS)
    {
        PRINTM(INFO, "IOCTL: Get TSF; Command exec failed\n");        
        ret = -EFAULT;
    }
    else
    {
        if (copy_to_user(wrq->u.data.pointer,
                         &tsfVal,
                         MIN(wrq->u.data.length, sizeof(tsfVal))) != 0) {
            
            PRINTM(INFO, "IOCTL: Get TSF; Copy to user failed\n");
            ret = -EFAULT;
        }
        else
        {
            ret = 0;
        }
    }

    LEAVE();

    return ret;
}





/** 
 *  @brief Get/Set sleep period 
 *  @param priv                 A pointer to wlan_private structure
 *  @param wrq			A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_sleep_period(wlan_private *priv, struct iwreq *wrq)
{
	int ret;		
	int data;
	wlan_adapter *Adapter = priv->adapter;
	struct HostCmd_DS_802_11_SLEEP_PERIOD	sleeppd;
       
	ENTER();

	if (wrq->u.data.length > 1)
		return -ENOTSUPP;
	
	memset(&sleeppd, 0, sizeof(sleeppd));
	memset(&Adapter->sleep_period, 0, sizeof(SleepPeriod));

	if (wrq->u.data.length == 0) {
		sleeppd.Action = wlan_cpu_to_le16(HostCmd_ACT_GEN_GET);
	} else {
		if (copy_from_user(&data,wrq->u.data.pointer,sizeof(int))) {
			PRINTM(INFO, "Copy from user failed\n");
			return -EFAULT;
		}

		/* sleep period is between 0 or 10 ... 60 */
		if ((data <= 60 && data >= 10) || (data == 0)){
			sleeppd.Action = 
				wlan_cpu_to_le16(HostCmd_ACT_GEN_SET);
			sleeppd.Period = data;
		} else
			return -EINVAL;
        }

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_SLEEP_PERIOD,
			0, HostCmd_OPTION_WAITFORRSP,
			0, (void *) &sleeppd);
	
	data = (int) Adapter->sleep_period.period;
	if (copy_to_user(wrq->u.data.pointer,&data,sizeof(int))) {
		PRINTM(INFO, "Copy to user failed\n");
		return -EFAULT;
	}
	wrq->u.data.length = 1;

	LEAVE();
	return ret;
}

/** 
 *  @brief Get/Set adapt rate 
 *  @param priv                 A pointer to wlan_private structure
 *  @param wrq			A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_adapt_rateset(wlan_private *priv, struct iwreq *wrq)
{
	int			ret;
	wlan_adapter		*Adapter = priv->adapter;
	int data[2];

	ENTER();
	memset(data,0,sizeof(data));
	if (!wrq->u.data.length) {
		PRINTM(INFO, "Get ADAPT RATE SET\n");
		ret = PrepareAndSendCommand(priv,
				HostCmd_CMD_802_11_RATE_ADAPT_RATESET,
				HostCmd_ACT_GEN_GET, HostCmd_OPTION_WAITFORRSP,
			      	0, NULL);
		data[0] = Adapter->EnableHwAuto;
		data[1] = Adapter->RateBitmap;
		if (copy_to_user(wrq->u.data.pointer,data,sizeof(int) * 2)) {
			PRINTM(INFO, "Copy to user failed\n");
			return -EFAULT;
		}

#define GET_TWO_INT	2
		wrq->u.data.length = GET_TWO_INT;
	} else {
		PRINTM(INFO, "Set ADAPT RATE SET\n");
		if(wrq->u.data.length > 2)
			return -EINVAL;
		if (copy_from_user(data,wrq->u.data.pointer,sizeof(int) * wrq->u.data.length)) {
			PRINTM(INFO, "Copy from user failed\n");
			return -EFAULT;
		}

		Adapter->EnableHwAuto = data[0];
		Adapter->RateBitmap = data[1];
		ret = PrepareAndSendCommand(priv,
				HostCmd_CMD_802_11_RATE_ADAPT_RATESET,
				HostCmd_ACT_GEN_SET,HostCmd_OPTION_WAITFORRSP,
			      	0, NULL);
	}

	LEAVE();
	return ret;	
}

/** 
 *  @brief Get/Set inactivity timeout
 *  @param priv                 A pointer to wlan_private structure
 *  @param wrq			A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_inactivity_timeout(wlan_private *priv, struct iwreq *wrq)
{
	int	ret;		
	int	data = 0;
	u16	timeout = 0;

	ENTER();
	if (wrq->u.data.length > 1)	
		return -ENOTSUPP;
	
	if (wrq->u.data.length == 0) {
		/* Get*/
		ret = PrepareAndSendCommand(priv,
				HostCmd_CMD_802_11_INACTIVITY_TIMEOUT,
				HostCmd_ACT_GET, HostCmd_OPTION_WAITFORRSP,
				0, &timeout);
		data = timeout;
		if (copy_to_user(wrq->u.data.pointer,&data,sizeof(int))) {
			PRINTM(INFO, "Copy to user failed\n");
			return -EFAULT;
		}
	} else {
		/* Set */
		if (copy_from_user(&data,wrq->u.data.pointer,sizeof(int))) {
			PRINTM(INFO, "Copy from user failed\n");
			return -EFAULT;
		}

		timeout =data;
		ret = PrepareAndSendCommand(priv, 
				HostCmd_CMD_802_11_INACTIVITY_TIMEOUT,
				HostCmd_ACT_SET, HostCmd_OPTION_WAITFORRSP,
				0, &timeout);
	}

	wrq->u.data.length = 1;

	LEAVE();
	return ret;
}

/** 
 *  @brief Get LOG
 *  @param priv                 A pointer to wlan_private structure
 *  @param wrq			A pointer to iwreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_do_getlog_ioctl(wlan_private *priv, struct iwreq *wrq)
{
	int		ret;
	char		buf[GETLOG_BUFSIZE-1];
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	PRINTM(INFO, " GET STATS\n"); 

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_GET_LOG,
			0, HostCmd_OPTION_WAITFORRSP,
			0, NULL);

	if (ret) {
		LEAVE();
		return ret;
	}

	if (wrq->u.data.pointer) {
		sprintf(buf, "\n  mcasttxframe %u failed %u retry %u " 
				"multiretry %u framedup %u "
				"rtssuccess %u rtsfailure %u ackfailure %u\n"
				"rxfrag %u mcastrxframe %u fcserror %u "
				"txframe %u wepundecryptable %u ",
				Adapter->LogMsg.mcasttxframe,
				Adapter->LogMsg.failed,
				Adapter->LogMsg.retry,
				Adapter->LogMsg.multiretry,
				Adapter->LogMsg.framedup,
				Adapter->LogMsg.rtssuccess,
				Adapter->LogMsg.rtsfailure,
				Adapter->LogMsg.ackfailure,
				Adapter->LogMsg.rxfrag,
				Adapter->LogMsg.mcastrxframe,
				Adapter->LogMsg.fcserror,
				Adapter->LogMsg.txframe,
				Adapter->LogMsg.wepundecryptable);
		wrq->u.data.length = strlen(buf) + 1;
		if (copy_to_user(wrq->u.data.pointer, buf, wrq->u.data.length)) {
			PRINTM(INFO, "Copy to user failed\n");
			return -EFAULT;
		}
	}

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief config sleep parameters
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_sleep_params_ioctl(wlan_private *priv, struct iwreq *wrq)
{
	int ret;
	wlan_adapter 	*Adapter = priv->adapter;
	wlan_ioctl_sleep_params_config  sp;

	ENTER();

	memset(&sp, 0, sizeof(sp));
	
	if (copy_from_user(&sp, wrq->u.data.pointer,
				MIN(sizeof(sp), wrq->u.data.length))) {
		PRINTM(INFO, "Copy from user failed\n");
		return -EFAULT;
	}
	
	memcpy(&Adapter->sp, &sp.Error, sizeof(SleepParams));
	
	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_SLEEP_PARAMS,
			sp.Action, HostCmd_OPTION_WAITFORRSP,
			0, NULL);

	if (!ret && !sp.Action && wrq->u.data.pointer) {
		memcpy(&sp.Error, &Adapter->sp, sizeof(SleepParams));
		if (copy_to_user(wrq->u.data.pointer, &sp, sizeof(sp))) {
			PRINTM(INFO, "Copy to user failed\n");
			return -EFAULT;
		}
		wrq->u.data.length = sizeof(sp);
	}

	LEAVE();
	return ret;
}

/** 
 *  @brief Read the CIS Table
 *  @param priv                 A pointer to wlan_private structure
 *  @param req			A pointer to ifreq structure
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
static int wlan_do_getcis_ioctl(wlan_private *priv, struct ifreq *req)
{
	int 		ret = WLAN_STATUS_SUCCESS;
	struct iwreq	*wrq = (struct iwreq *) req;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	if (wrq->u.data.pointer) {
		if (copy_to_user(wrq->u.data.pointer, Adapter->CisInfoBuf, 
							Adapter->CisInfoLen)) {
			PRINTM(INFO, "Copy to user failed\n");
			return -EFAULT;
		}
		wrq->u.data.length = Adapter->CisInfoLen;
	}

	LEAVE();
	return ret;
}


/** 
 *  @brief Set scan type
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param wrq	   	A pointer to user data
 *  @return 	   	WLAN_STATUS_SUCCESS--success, otherwise fail
 */
static int wlan_scan_type_ioctl(wlan_private *priv, struct iwreq *wrq)
{
	u8	buf[12];
	u8	*option[] = { "active", "passive", "get", };
	int	i, max_options = (sizeof(option)/sizeof(option[0]));
	int	ret = WLAN_STATUS_SUCCESS;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	if( wlan_get_state_11d( priv) == ENABLE_11D ) {
		PRINTM(INFO, "11D: Cannot set scantype when 11D enabled\n");
		return -EFAULT;
	}

	memset(buf, 0, sizeof(buf));

	if (copy_from_user(buf, wrq->u.data.pointer, MIN(sizeof(buf), 
					wrq->u.data.length))) {
		PRINTM(INFO, "Copy from user failed\n");
		return -EFAULT;
	}

	PRINTM(INFO, "Scan Type Option = %s\n", buf);

	buf[sizeof(buf)-1] = '\0';

	for (i = 0; i < max_options; i++) {
		if (!strcmp(buf, option[i]))
			break;
	}

	switch (i) {
		case 0:
			Adapter->ScanType = HostCmd_SCAN_TYPE_ACTIVE;
			break;
		case 1:
			Adapter->ScanType = HostCmd_SCAN_TYPE_PASSIVE;
			break;
		case 2:
			wrq->u.data.length = 
				strlen(option[Adapter->ScanType]) + 1;

			if (copy_to_user(wrq->u.data.pointer, 
						option[Adapter->ScanType],
						wrq->u.data.length)) {
				PRINTM(INFO, "Copy to user failed\n");
				ret = -EFAULT;
			}

			break;
		default:
			PRINTM(INFO, "Invalid Scan Type Ioctl Option\n");
			ret = -EINVAL;
			break;
	}

	LEAVE();
	return ret;
}

/** 
 *  @brief Set scan mode
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param wrq	   	A pointer to user data
 *  @return 	   	WLAN_STATUS_SUCCESS--success, otherwise fail
 */
static int wlan_scan_mode_ioctl(wlan_private *priv, struct iwreq *wrq)
{
	wlan_adapter *Adapter = priv->adapter;
	u8	buf[12];
	u8	*option[] = { "bss", "ibss", "any","get"};
	int	i, max_options = (sizeof(option)/sizeof(option[0]));
	int	ret = WLAN_STATUS_SUCCESS;

	ENTER();

	memset(buf, 0, sizeof(buf));

	if (copy_from_user(buf, wrq->u.data.pointer, MIN(sizeof(buf), 
					wrq->u.data.length))) {
		PRINTM(INFO, "Copy from user failed\n");
		return -EFAULT;
	}

	PRINTM(INFO, "Scan Mode Option = %s\n", buf);

	buf[sizeof(buf)-1] = '\0';

	for (i = 0; i < max_options; i++) {
		if (!strcmp(buf, option[i]))
			break;
	}

	switch (i) {

		case 0:
			Adapter->ScanMode = HostCmd_BSS_TYPE_BSS;
			break;
		case 1:
			Adapter->ScanMode = HostCmd_BSS_TYPE_IBSS;
			break;
		case 2:
			Adapter->ScanMode = HostCmd_BSS_TYPE_ANY;
			break;
		case 3:

			wrq->u.data.length = 
				strlen(option[Adapter->ScanMode-1]) + 1;

			PRINTM(INFO, "Get Scan Mode Option = %s\n", 
					option[Adapter->ScanMode-1]);

			PRINTM(INFO, "Scan Mode Length %d\n", wrq->u.data.length);

			if (copy_to_user(wrq->u.data.pointer, 
						option[Adapter->ScanMode-1],
							wrq->u.data.length)) {
				PRINTM(INFO, "Copy to user failed\n");
				ret = -EFAULT;
			}
			PRINTM(INFO, "GET Scan Type Option after copy = %s\n", 
					(char *)wrq->u.data.pointer);

			break;

		default:
			PRINTM(INFO, "Invalid Scan Mode Ioctl Option\n");
			ret = -EINVAL;
			break;
	}

	LEAVE();
	return ret;
}

/** 
 *  @brief Get/Set Adhoc G Rate
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param wrq	   	A pointer to user data
 *  @return 	   	WLAN_STATUS_SUCCESS--success, otherwise fail
 */
static int wlan_do_set_grate_ioctl(wlan_private *priv, struct iwreq *wrq)
{
	wlan_adapter	*Adapter = priv->adapter;
	int 		data,data1;
	int 		*val;

	ENTER();
    
	data1 =  *( (int *) (wrq->u.name + SUBCMD_OFFSET) );	
	switch (data1) {
		case 0:
			Adapter->adhoc_grate_enabled = FALSE;
			break;
		case 1:
			Adapter->adhoc_grate_enabled = TRUE;
			break;
		case 2:
			break;
		default:
			return -EINVAL;
	}
	data = Adapter->adhoc_grate_enabled;
	val = (int *)wrq->u.name;
	*val = 	data;		
	LEAVE();
	return WLAN_STATUS_SUCCESS;
}


/********************************************************
		Global Functions
********************************************************/
/** 
 *  @brief ioctl function - entry point
 *  
 *  @param dev		A pointer to net_device structure
 *  @param req	   	A pointer to ifreq structure
 *  @param cmd 		command
 *  @return 	   	WLAN_STATUS_SUCCESS--success, otherwise fail
 */
int wlan_do_ioctl(struct net_device *dev, struct ifreq *req, int cmd)
{
	int		subcmd = 0;
	int 		idata = 0;
	int 		*pdata;
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;
	struct iwreq	*wrq = (struct iwreq *) req;

	ENTER();


	PRINTM(INFO, "wlan_do_ioctl: ioctl cmd = 0x%x\n", cmd);
	switch (cmd) {
	case WLANEXTSCAN: 
		ret = wlan_extscan_ioctl(priv, req);
		break;
	case WLANHOSTCMD:
		ret = wlan_hostcmd_ioctl(dev, req, cmd);
		break;

	case WLANCISDUMP:	/* Read CIS Table  */
		ret = wlan_do_getcis_ioctl(priv, req);
		break;
		
	case WLANSCAN_TYPE :
		PRINTM(INFO, "Scan Type Ioctl\n");
		ret = wlan_scan_type_ioctl(priv, wrq);
		break;


	case WLANREGRDWR:	/* Register read write command */
		ret = wlan_regrdwr_ioctl(priv, req);
		break;


	case SIOCSIWENCODE:	/* set encoding token & mode for WPA */
		ret = wlan_set_encode(dev, NULL, &(wrq->u.data),
							wrq->u.data.pointer);
		break;
	case WLAN_SETNONE_GETNONE : /* set WPA mode on/off ioctl #20 */
		switch(wrq->u.data.flags) {
		case WLANDEAUTH :
			PRINTM(INFO, "Deauth\n");
			wlan_send_deauth(priv);
			break;
			
		case WLANADHOCSTOP:
			PRINTM(INFO, "Adhoc stop\n");
			ret = wlan_do_adhocstop_ioctl(priv);
			break;

		case WLANRADIOON:
			wlan_radio_ioctl(priv, RADIO_ON);
			break;
			
		case WLANRADIOOFF:
			wlan_radio_ioctl(priv, RADIO_OFF);
			break;
#ifdef REASSOCIATION
		case WLANREASSOCIATIONAUTO:
			reassociation_on(priv);
			break;
		case WLANREASSOCIATIONUSER:
			reassociation_off(priv);
			break;
#endif /* REASSOCIATION */
		case WLANWLANIDLEON:
			wlanidle_on(priv);
			break;
		case WLANWLANIDLEOFF:
			wlanidle_off(priv);
			break;
		}	/* End of switch */
		break;

	case WLAN_SETWORDCHAR_GETNONE :
		switch(wrq->u.data.flags) {
		}
		break;

	case WLAN_SETNONE_GETWORDCHAR:
		switch(wrq->u.data.flags) {
		case WLANVERSION:	/* Get driver version */
			ret = wlan_version_ioctl(priv, req);
			break;
		}
		break;
	case WLANSETWPAIE:
		ret = wlan_setwpaie_ioctl(priv, req);
		break;
	case WLAN_SETINT_GETINT:
		/* The first 4 bytes of req->ifr_data is sub-ioctl number
		 * after 4 bytes sits the payload.
		 */
		subcmd = (int)req->ifr_data;	//from iwpriv subcmd
		switch(subcmd){
		case WLANNF :		   
			ret  = wlan_get_nf(priv, wrq);
			break;
		case WLANRSSI :		
			ret = wlan_get_rssi(priv, wrq);
			break;			
		case WLANENABLE11D:
			ret = wlan_cmd_enable_11d( priv, wrq );
			break;
		case WLANADHOCGRATE:
			ret = wlan_do_set_grate_ioctl(priv, wrq);
			break;
		case WLAN_SUBCMD_SET_PRESCAN:
			ret = wlan_subcmd_setprescan_ioctl(priv,wrq);
			break;			
		}
		break;
	
	case WLAN_SETONEINT_GETONEINT:
		switch(wrq->u.data.flags) {
		case WLAN_BEACON_INTERVAL:
			ret = wlan_beacon_interval(priv,wrq);
			break;			

		case WLAN_LISTENINTRVL:
			if(!wrq->u.data.length) {				
				int data;
				PRINTM(INFO, "Get LocalListenInterval Value\n");
#define GET_ONE_INT	1				
				data = Adapter->LocalListenInterval;
				if (copy_to_user(wrq->u.data.pointer,
						       &data,sizeof(int))) {
					PRINTM(INFO, "Copy to user failed\n");
					return -EFAULT;
				}

				wrq->u.data.length = GET_ONE_INT;
			} else {
				int data;
				if (copy_from_user(&data,wrq->u.data.pointer,sizeof(int))) {
					PRINTM(INFO, "Copy from user failed\n");
					return -EFAULT;
				}

				PRINTM(INFO, "Set LocalListenInterval = %d\n",data);
#define MAX_U16_VAL	65535
				if(data > MAX_U16_VAL) {
					PRINTM(INFO, "Exceeds U16 value\n");
					return -EINVAL;
				}
				Adapter->LocalListenInterval = data;
			}
			break;
		case WLAN_TXCONTROL:
			ret = wlan_txcontrol(priv, wrq ); //adds for txcontrol ioctl
			break;

		case WLAN_NULLPKTINTERVAL:
			ret = wlan_null_pkt_interval(priv, wrq);
			break;
	
		default:
			ret = -EOPNOTSUPP;
			break;
		}
		break;	
	
	case WLAN_SETONEINT_GETNONE:
		/* The first 4 bytes of req->ifr_data is sub-ioctl number
		 * after 4 bytes sits the payload.
		 */
		subcmd = wrq->u.data.flags;	//from wpa_supplicant subcmd
		
		if(!subcmd)
			subcmd = (int)req->ifr_data;	//from iwpriv subcmd
		
		switch (subcmd) {
		case WLAN_SUBCMD_SETRXANTENNA : /* SETRXANTENNA */
			idata = *( (int *) (wrq->u.name + SUBCMD_OFFSET) );	
			ret = SetRxAntenna(priv, idata);
			break;
		case WLAN_SUBCMD_SETTXANTENNA : /* SETTXANTENNA */
			idata = *( (int *) (wrq->u.name + SUBCMD_OFFSET) );
			ret = SetTxAntenna(priv, idata);
			break;
		case WLAN_SET_ATIM_WINDOW:
			Adapter->AtimWindow =  *( (int *) (wrq->u.name + SUBCMD_OFFSET) );
			Adapter->AtimWindow = MIN(Adapter->AtimWindow, 50);
			break;
		case WLANSETBCNAVG:
			Adapter->bcn_avg_factor = *( (int *) (wrq->u.name + SUBCMD_OFFSET) );
			if(Adapter->bcn_avg_factor == 0)
				Adapter->bcn_avg_factor = DEFAULT_BCN_AVG_FACTOR;
			if(Adapter->bcn_avg_factor > DEFAULT_BCN_AVG_FACTOR)
				Adapter->bcn_avg_factor = DEFAULT_BCN_AVG_FACTOR;
			break;
		case WLANSETDATAAVG:
			Adapter->data_avg_factor =  *( (int *) (wrq->u.name + SUBCMD_OFFSET) );
			if(Adapter->data_avg_factor == 0)
				Adapter->data_avg_factor = DEFAULT_DATA_AVG_FACTOR;
			if(Adapter->data_avg_factor > DEFAULT_DATA_AVG_FACTOR)
				Adapter->data_avg_factor = DEFAULT_DATA_AVG_FACTOR;
			break;
		case WLANSETREGION:
			idata = *( (int *) (wrq->u.name + SUBCMD_OFFSET));
			ret = wlan_set_region(priv, (u16)idata);
			break;
				
		case WLAN_SET_LISTEN_INTERVAL:
			idata = *( (int *) (wrq->u.name + SUBCMD_OFFSET));
			Adapter->ListenInterval = (u16)idata;
			break;
				
		case WLAN_SET_MULTIPLE_DTIM:
			ret = wlan_set_multiple_dtim_ioctl(priv, req);
			break;

		case WLANSETAUTHALG:
			ret = wlan_setauthalg_ioctl(priv, req);
			break;
		
		case WLANSET8021XAUTHALG:
			ret = wlan_set8021xauthalg_ioctl(priv, req);
			break;

		case WLANSETENCRYPTIONMODE:
			ret = wlan_setencryptionmode_ioctl(priv, req);
			break;

		case WLAN_SET_LINKMODE:
			ret = wlan_set_linkmode_ioctl(priv, req);
			break;

		case WLAN_SET_RADIOMODE:
			ret = wlan_set_radiomode_ioctl(priv, req);
			break;

		case WLAN_SET_DEBUGMODE:
			ret = wlan_set_debugmode_ioctl(priv, req);
			break;

		default:
			ret = -EOPNOTSUPP;
			break;
		}
		
		break;


	case WLAN_SETNONE_GETTWELVE_CHAR: /* Get Antenna settings */
		/* 
		 * We've not used IW_PRIV_TYPE_FIXED so sub-ioctl number is
		 * in flags of iwreq structure, otherwise it will be in
		 * mode member of iwreq structure.
		 */
		switch ((int)wrq->u.data.flags) {
		case WLAN_SUBCMD_GETRXANTENNA: /* Get Rx Antenna */
			ret = wlan_subcmd_getrxantenna_ioctl(priv, req);
			break;

		case WLAN_SUBCMD_GETTXANTENNA:  /* Get Tx Antenna */
			ret = wlan_subcmd_gettxantenna_ioctl(priv, req);
			break;

		case WLAN_GET_TSF:
			ret = wlan_get_tsf_ioctl(priv, wrq);
			break;
		}
		break;
		
		

			
	case WLAN_SET64CHAR_GET64CHAR:
		switch ((int)wrq->u.data.flags) {

		case WLANSLEEPPARAMS:
			ret = wlan_sleep_params_ioctl(priv, wrq);
			break;

		case WLANSCAN_MODE :
			PRINTM(INFO, "Scan Mode Ioctl\n");
			ret = wlan_scan_mode_ioctl(priv, wrq);
			break;
		
		case WLAN_GET_ADHOC_STATUS:
			ret = wlan_get_adhoc_status_ioctl(priv, wrq);
			break;
		}
		break;

	case WLAN_SETCONF_GETCONF:
		PRINTM(INFO, "The WLAN_SETCONF_GETCONF=0x%x is %d\n",WLAN_SETCONF_GETCONF, *(u8 *)req->ifr_data);
		switch (*(u8 *)req->ifr_data) {

		}
		break;

	case WLAN_SETNONE_GETONEINT:
		switch((int)req->ifr_data) {
		case WLANGETBCNAVG:
			pdata = (int *)wrq->u.name;
			*pdata =(int) Adapter->bcn_avg_factor;
			break;

		case WLANGETREGION:
			pdata = (int *)wrq->u.name;
			*pdata =(int) Adapter->RegionCode;
			break;

		case WLAN_GET_LISTEN_INTERVAL:
			pdata = (int *)wrq->u.name;
			*pdata =(int) Adapter->ListenInterval;
			break;

		case WLAN_GET_LINKMODE:
			req->ifr_data = (char *)((u32)Adapter->linkmode);
			break;

		case WLAN_GET_RADIOMODE:
			req->ifr_data = (char *)((u32)Adapter->radiomode);
			break;

		case WLAN_GET_DEBUGMODE:
			req->ifr_data = (char *)((u32)Adapter->debugmode);
			break;

		case WLAN_GET_MULTIPLE_DTIM:
			pdata = (int *)wrq->u.name;
			*pdata =(int) Adapter->MultipleDtim;
			break;
		case WLAN_GET_TX_RATE:
			ret = wlan_get_txrate_ioctl(priv, req);
			break;			
		default:
			ret = -EOPNOTSUPP;

		}

		break;
		
	case WLAN_SETTENCHAR_GETNONE:
		switch ((int)wrq->u.data.flags) {

		}
		break;

	case WLAN_SETNONE_GETTENCHAR:
		switch ((int)wrq->u.data.flags) {
		}
		break;

    	case WLANGETLOG:
        	ret = wlan_do_getlog_ioctl(priv, wrq);
        	break;


  	case WLAN_SET_GET_SIXTEEN_INT:
    		switch ((int)wrq->u.data.flags) {
      		case WLAN_TPCCFG:
     	 	{
        		int data[5];
			struct HostCmd_DS_802_11_TPC_CFG cfg;
        		memset(&cfg, 0, sizeof(cfg));
        		if ((wrq->u.data.length > 1) && (wrq->u.data.length != 5))
          			return WLAN_STATUS_FAILURE;

        		if (wrq->u.data.length == 0){
          			cfg.Action  = wlan_cpu_to_le16(HostCmd_ACT_GEN_GET);
        		}
        		else
        		{
			        if (copy_from_user(data,wrq->u.data.pointer, sizeof(int) * 5)) {
			        	PRINTM(INFO, "Copy from user failed\n");
			        	return -EFAULT;
			        }

			        cfg.Action  = wlan_cpu_to_le16(HostCmd_ACT_GEN_SET);
          			cfg.Enable  = data[0];
          			cfg.UseSNR  = data[1];
          			cfg.P0      = data[2];
          			cfg.P1      = data[3];
          			cfg.P2      = data[4];
        		}

        		ret = PrepareAndSendCommand(priv,HostCmd_CMD_802_11_TPC_CFG,0, 
          			HostCmd_OPTION_WAITFORRSP,0,(void *) &cfg);

        		data[0] = cfg.Enable;
        		data[1] = cfg.UseSNR;
        		data[2] = cfg.P0;
        		data[3] = cfg.P1;
        		data[4] = cfg.P2;
        		if (copy_to_user(wrq->u.data.pointer,data,sizeof(int) * 5)) {
        			PRINTM(INFO, "Copy to user failed\n");
        			return -EFAULT;
        		}

        		wrq->u.data.length = 5;
      		}
      		break;
  
      		case WLAN_POWERCFG:
      		{
      			int data[4];
        		struct HostCmd_DS_802_11_PWR_CFG cfg;
        		memset(&cfg, 0, sizeof(cfg));
        		if ((wrq->u.data.length > 1) && (wrq->u.data.length != 4))
          			return WLAN_STATUS_FAILURE;
        		if (wrq->u.data.length == 0){
				cfg.Action  = wlan_cpu_to_le16(HostCmd_ACT_GEN_GET);
        		}
        		else
        		{
			        if (copy_from_user(data,wrq->u.data.pointer, sizeof(int) * 4)) {
			        	PRINTM(INFO, "Copy from user failed\n");
			        	return -EFAULT;
			        }

          			cfg.Action  = wlan_cpu_to_le16(HostCmd_ACT_GEN_SET);
          			cfg.Enable  = data[0];
          			cfg.PA_P0   = data[1];
          			cfg.PA_P1   = data[2];
          			cfg.PA_P2   = data[3];
        		}
        		ret = PrepareAndSendCommand(priv,HostCmd_CMD_802_11_PWR_CFG,0, 
          			HostCmd_OPTION_WAITFORRSP,0,(void *) &cfg);
        		data[0] = cfg.Enable;
        		data[1] = cfg.PA_P0;
        		data[2] = cfg.PA_P1;
        		data[3] = cfg.PA_P2; 
        		if (copy_to_user(wrq->u.data.pointer,data,sizeof(int) * 4)) {
        			PRINTM(INFO, "Copy to user failed\n");
        			return -EFAULT;
        		}

        		wrq->u.data.length = 4;
      		}
      		break;
  		case WLAN_AUTO_FREQ_SET: 
      		{
			int data[3];
        		struct HostCmd_DS_802_11_AFC afc;
        		memset(&afc, 0, sizeof(afc));
			if (wrq->u.data.length != 3)
		 		return WLAN_STATUS_FAILURE;
        		if (copy_from_user(data,wrq->u.data.pointer,sizeof(int)*3)) {
        			PRINTM(INFO, "Copy from user failed\n");
        			return -EFAULT;
        		}
        		afc.afc_auto    = data[0];

        		if (afc.afc_auto != 0) {
          			afc.afc_thre    = data[1];
          			afc.afc_period  = data[2];
        		} else {
          			afc.afc_toff    = data[1];
          			afc.afc_foff    = data[2];
        		}
        		ret = PrepareAndSendCommand(priv,HostCmd_CMD_802_11_SET_AFC,0, 
				HostCmd_OPTION_WAITFORRSP,0,(void *) &afc);
      		}
      		break;
		case WLAN_AUTO_FREQ_GET: 
      		{
        		int data[3];
        		struct HostCmd_DS_802_11_AFC afc;
        		memset(&afc, 0, sizeof(afc));
        		ret = PrepareAndSendCommand(priv,HostCmd_CMD_802_11_GET_AFC,0, 
          			HostCmd_OPTION_WAITFORRSP,0,(void *) &afc);
        		data[0] = afc.afc_auto;
        		data[1] = afc.afc_toff;
        		data[2] = afc.afc_foff;
        		if (copy_to_user(wrq->u.data.pointer,data,sizeof(int) * 3)) {
        			PRINTM(INFO, "Copy to user failed\n");
        			return -EFAULT;
        		}

        		wrq->u.data.length = 3;
      		}
      		break;
		case WLAN_SCANPROBES: 
      		{
			int data;      
			if (wrq->u.data.length > 0){
				if (copy_from_user(&data,wrq->u.data.pointer,sizeof(int))) {
					PRINTM(INFO, "Copy from user failed\n");
					return -EFAULT;
				}

				Adapter->ScanProbes = data;
			}
			else
			{
				data = Adapter->ScanProbes;
	  			if (copy_to_user(wrq->u.data.pointer,&data,sizeof(int))) {
					PRINTM(INFO, "Copy to user failed\n");
					return -EFAULT;
				}
			}
			wrq->u.data.length = 1;
			}
			break;
    		case WLAN_LED_GPIO_CTRL:
    		{
      			int i;
      			int data[16];
      
      			struct HostCmd_DS_802_11_LED_CTRL ctrl;
      			MrvlIEtypes_LedGpio_t *gpio = (MrvlIEtypes_LedGpio_t *) ctrl.data;

      			memset(&ctrl, 0, sizeof(ctrl));
        		if (wrq->u.data.length > MAX_LEDS*2)
        			return -ENOTSUPP;
      			if ((wrq->u.data.length % 2) != 0)
        			return -ENOTSUPP;
      			if (wrq->u.data.length == 0) {
        			ctrl.Action = wlan_cpu_to_le16(HostCmd_ACT_GEN_GET);
      			} else {
	        		if (copy_from_user(data,wrq->u.data.pointer,sizeof(int)*wrq->u.data.length)) {
	        			PRINTM(INFO, "Copy from user failed\n");
	        			return -EFAULT;
	        		}

        			ctrl.Action = wlan_cpu_to_le16(HostCmd_ACT_GEN_SET);
        			ctrl.NumLed = wlan_cpu_to_le16(0);
        			gpio->Header.Type = wlan_cpu_to_le16(TLV_TYPE_LED_GPIO);
        			gpio->Header.Len  = wrq->u.data.length;
        			for (i=0; i < wrq->u.data.length; i+=2) {
          				gpio->LedPin[i/2].Led = data[i];
         	 			gpio->LedPin[i/2].Pin = data[i+1];
        			}
      			}
      			ret = PrepareAndSendCommand( priv, HostCmd_CMD_802_11_LED_GPIO_CTRL,0, 
					HostCmd_OPTION_WAITFORRSP,0, (void *) &ctrl);
        		for (i=0; i < gpio->Header.Len; i+=2) {
        			data[i] = gpio->LedPin[i/2].Led;
        			data[i+1] = gpio->LedPin[i/2].Pin;
      			}
        		if (copy_to_user(wrq->u.data.pointer,data,
					       sizeof(int)* gpio->Header.Len)) {
	       			PRINTM(INFO, "Copy to user failed\n");
					return -EFAULT;
        		}

      			wrq->u.data.length = gpio->Header.Len;
    		}
    		break;
		case WLAN_SLEEP_PERIOD:
			ret = wlan_sleep_period(priv, wrq);
			break;
		case WLAN_ADAPT_RATESET:
			ret = wlan_adapt_rateset(priv, wrq);
			break;
		case WLAN_INACTIVITY_TIMEOUT:
			ret = wlan_inactivity_timeout(priv, wrq); 
			break;
		case WLANSNR :
			ret = wlan_get_snr(priv, wrq);
			break;
		case WLAN_GET_RATE:
			ret = wlan_getrate_ioctl(priv, wrq);
			break;
		case WLAN_GET_RXINFO:
			ret = wlan_get_rxinfo(priv,wrq);			
  		}
  		break;

    case WLAN_SET_GET_2K:
        switch ((int)wrq->u.data.flags) {
  	case WLAN_SET_USER_SCAN:
            ret = wlan_set_user_scan_ioctl(priv, wrq);
            break;
        case WLAN_GET_SCAN_TABLE:
            ret = wlan_get_scan_table_ioctl(priv, wrq);
            break;

        default:
            ret = -EOPNOTSUPP;
        }
        break;
	default:
 		ret = -EINVAL;
		break;
	}
	LEAVE();
	return ret;
}

/** 
 *  @brief Get wireless statistics
 *  
 *  @param dev		A pointer to net_device structure
 *  @return 	   	A pointer to iw_statistics buf
 */
struct iw_statistics *wlan_get_wireless_stats(struct net_device *dev)
{
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;
	int		ret = WLAN_STATUS_SUCCESS;

	ENTER();

	priv->wstats.status = Adapter->InfrastructureMode;
	priv->wstats.discard.retries = priv->stats.tx_errors;

	/* send RSSI command to get beacon RSSI/NF, valid only if associated */
	ret = PrepareAndSendCommand(priv, 
			HostCmd_CMD_802_11_RSSI,
			0, HostCmd_OPTION_WAITFORRSP,
			0, NULL);

	priv->wstats.qual.level =
		CAL_RSSI(Adapter->SNR[TYPE_BEACON][TYPE_NOAVG],
		Adapter->NF[TYPE_BEACON][TYPE_NOAVG]);
	priv->wstats.qual.noise = CAL_NF(Adapter->NF[TYPE_BEACON][TYPE_NOAVG]);
	if (Adapter->NF[TYPE_BEACON][TYPE_NOAVG] == 0
		&& Adapter->MediaConnectStatus == WlanMediaStateConnected)
		priv->wstats.qual.noise = MRVDRV_NF_DEFAULT_SCAN_VALUE;
	else
		priv->wstats.qual.noise = 
				CAL_NF(Adapter->NF[TYPE_BEACON][TYPE_NOAVG]);
	priv->wstats.qual.qual = 0;

	PRINTM(INFO, "Signal Level = %#x\n", priv->wstats.qual.level);
	PRINTM(INFO, "Noise = %#x\n", priv->wstats.qual.noise);

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_GET_LOG,
			0, HostCmd_OPTION_WAITFORRSP,
			0, NULL);

	if (!ret) {
		priv->wstats.discard.code = Adapter->LogMsg.wepundecryptable;
		priv->wstats.discard.fragment = Adapter->LogMsg.fcserror;
		priv->wstats.discard.retries = Adapter->LogMsg.retry;
		priv->wstats.discard.misc = Adapter->LogMsg.ackfailure;
	}

	return &priv->wstats;
}

/** 
 *  @brief Set frequency
 *   
 *  @param priv 		A pointer to wlan_private structure
 *  @param info			A pointer to iw_request_info structure 
 *  @param fwrq			A pointer to iw_freq structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
int wlan_set_freq(struct net_device *dev, struct iw_request_info *info, 
		struct iw_freq *fwrq, char *extra)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;
	int             rc = -EINPROGRESS;	/* Call commit handler */
	CHANNEL_FREQ_POWER	*cfp;

	ENTER();



	/*
	 * If setting by frequency, convert to a channel 
	 */
	if (fwrq->e == 1) {

		long	f = fwrq->m / 100000;
		int	c = 0;

		cfp = find_cfp_by_band_and_freq(Adapter, 0, f);
		if (!cfp) {
			PRINTM(INFO, "Invalid freq=%ld\n", f);
			return -EINVAL;
		} 
		
		c = (int) cfp->Channel;

		if (c < 0) 
			return -EINVAL;
	
		fwrq->e = 0;
		fwrq->m = c;
	}

	/*
	 * Setting by channel number 
	 */
	if (fwrq->m > 1000 || fwrq->e > 0) {
		rc = -EOPNOTSUPP;
	} else {
		int	channel = fwrq->m;

		cfp = find_cfp_by_band_and_channel(Adapter, 0, channel);
		if (!cfp) {
			rc = -EINVAL;
		} else {
			if (Adapter->InfrastructureMode == Wlan802_11IBSS) {
				rc = ChangeAdhocChannel(priv, channel);
				/*  If station is WEP enabled, send the 
				 *  command to set WEP in firmware
				 */
				if (Adapter->SecInfo.WEPStatus == 
							Wlan802_11WEPEnabled) {
					PRINTM(INFO, "set_freq: WEP Enabled\n");
					ret = PrepareAndSendCommand(priv,
						HostCmd_CMD_802_11_SET_WEP, 
						0,HostCmd_OPTION_WAITFORRSP
						, OID_802_11_ADD_WEP, 
						NULL);

					if (ret) {
						LEAVE();
						return ret;
					}
						
					Adapter->CurrentPacketFilter |=	
						HostCmd_ACT_MAC_WEP_ENABLE;

					SetMacPacketFilter(priv);
				}
			} else {
				rc = -EOPNOTSUPP;
			}
		}
	}

	LEAVE();
	return rc;
}


/** 
 *  @brief use index to get the data rate
 *   
 *  @param index                The index of data rate
 *  @return 	   		data rate or 0 
 */
u32 index_to_data_rate(u8 index)
{
	if (index >= sizeof(WlanDataRates))
		index = 0;

	return WlanDataRates[index];
}

/** 
 *  @brief use rate to get the index
 *   
 *  @param rate                 data rate
 *  @return 	   		index or 0 
 */
u8 data_rate_to_index(u32 rate)
{
	u8	*ptr;

	if (rate)
		if ((ptr = wlan_memchr(WlanDataRates, (u8)rate, 
						sizeof(WlanDataRates))))
			return (ptr - WlanDataRates);

	return 0;
}

/** 
 *  @brief set data rate
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
int wlan_set_rate(struct net_device *dev, struct iw_request_info *info, 
		struct iw_param *vwrq, char *extra)
{
	wlan_private		*priv = dev->priv;
	wlan_adapter		*Adapter = priv->adapter;
	u32			data_rate;
	u16			action;
	int 			ret = WLAN_STATUS_SUCCESS;
	WLAN_802_11_RATES	rates;
	u8			*rate;
    
	ENTER();


	PRINTM(INFO, "Vwrq->value = %d\n", vwrq->value);

	if (vwrq->value == -1) {
		action = HostCmd_ACT_SET_TX_AUTO;	// Auto
		Adapter->Is_DataRate_Auto = TRUE;
		Adapter->DataRate = 0;
	} else {
		if (vwrq->value % 100000) {
			return -EINVAL;
		}

		data_rate = vwrq->value / 500000;

		memset(rates, 0, sizeof(rates));
		get_active_data_rates(Adapter, rates);
		rate = rates;
		while (*rate) {
			PRINTM(INFO, "Rate=0x%X  Wanted=0x%X\n",*rate,data_rate);
			if ((*rate & 0x7f) == (data_rate & 0x7f))
				break;
			rate++;
		}
		if (!*rate) {
			PRINTM(MSG, "The fixed data rate 0x%X is out "
					"of range.\n",data_rate);
			return -EINVAL;
		}

		Adapter->DataRate = data_rate;
		action = HostCmd_ACT_SET_TX_FIX_RATE;
		Adapter->Is_DataRate_Auto = FALSE;
	}

	ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_DATA_RATE, 
			action,	HostCmd_OPTION_WAITFORRSP,
			0, NULL);

	LEAVE();
	return ret;
}

/** 
 *  @brief get data rate
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
int wlan_get_rate(struct net_device *dev, struct iw_request_info *info, 
		struct iw_param *vwrq, char *extra)
{
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();

	if (Adapter->Is_DataRate_Auto) {
		vwrq->fixed = 0;
	} else {
		vwrq->fixed = 1;
	}
	
	vwrq->value = Adapter->DataRate * 500000;

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief set wireless mode 
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
int wlan_set_mode(struct net_device *dev,
	      struct iw_request_info *info, u32 * uwrq, char *extra)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;
	
	WLAN_802_11_NETWORK_INFRASTRUCTURE	WantedMode;

	ENTER();


	switch (*uwrq) {
		case IW_MODE_ADHOC:
			PRINTM(INFO, "Wanted Mode is ad-hoc: current DataRate=%#x\n",Adapter->DataRate);
			WantedMode = Wlan802_11IBSS;
				Adapter->AdhocChannel = DEFAULT_AD_HOC_CHANNEL;
			break;

		case IW_MODE_INFRA:
			PRINTM(INFO, "Wanted Mode is Infrastructure\n");
			WantedMode = Wlan802_11Infrastructure;
			break;

		case IW_MODE_AUTO:
			PRINTM(INFO, "Wanted Mode is Auto\n");
			WantedMode = Wlan802_11AutoUnknown;
			break;

		default:
			PRINTM(INFO, "Wanted Mode is Unknown: 0x%x\n", *uwrq);
			return -EINVAL;
	}

	if (Adapter->InfrastructureMode == WantedMode ||
			WantedMode == Wlan802_11AutoUnknown) {
		PRINTM(INFO, "Already set to required mode! No change!\n");

		Adapter->InfrastructureMode = WantedMode;

		LEAVE();
		return WLAN_STATUS_SUCCESS;
	}

	if (Adapter->InfrastructureMode == Wlan802_11Infrastructure) {
		if (Adapter->PSState != PS_STATE_FULL_POWER) {
			PSWakeup(priv, HostCmd_OPTION_WAITFORRSP);
		}
		Adapter->PSMode = Wlan802_11PowerModeCAM;
	}
	
	if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
		if (Adapter->InfrastructureMode == Wlan802_11Infrastructure) {
			ret = SendDeauthentication(priv);

			if (ret) {
				LEAVE();
				return ret;
			}
		} else if (Adapter->InfrastructureMode == Wlan802_11IBSS) {
			/* If current mode is Adhoc, clean stale information */
			ret = StopAdhocNetwork(priv);

			if (ret) {
				LEAVE();
				return ret;
			}
		}
	} 

	if (Adapter->SecInfo.WEPStatus == Wlan802_11WEPEnabled) {
		/* If there is a key with the specified SSID, 
		 * send REMOVE WEP command, to make sure we clean up
		 * the WEP keys in firmware
		 */
		ret = PrepareAndSendCommand(priv,
			HostCmd_CMD_802_11_SET_WEP,
			0, HostCmd_OPTION_WAITFORRSP
			, OID_802_11_REMOVE_WEP,
			NULL);

		if (ret) {
			LEAVE();
			return ret;
		}

		Adapter->CurrentPacketFilter &= ~HostCmd_ACT_MAC_WEP_ENABLE;

		SetMacPacketFilter(priv);
	}

	Adapter->SecInfo.WEPStatus = Wlan802_11WEPDisabled;
	Adapter->SecInfo.AuthenticationMode = Wlan802_11AuthModeOpen;

	Adapter->InfrastructureMode = WantedMode;

	ret = PrepareAndSendCommand(priv,
			HostCmd_CMD_802_11_SNMP_MIB,
			0, HostCmd_OPTION_WAITFORRSP,
			OID_802_11_INFRASTRUCTURE_MODE,
			NULL);

	if (ret) {
		LEAVE();
		return ret;
	}

	LEAVE();
	return  WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Set Encryption key
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
int wlan_set_encode(struct net_device *dev,
		struct iw_request_info *info,
		struct iw_point *dwrq, char *extra)
{

	struct WLAN_802_11_KEY *pKey = NULL;

	ENTER();
	
	if (dwrq->length > MAX_KEY_SIZE) {
		pKey = (struct WLAN_802_11_KEY *) extra;

		if(pKey->KeyLength <= MAX_KEY_SIZE) {
			//dynamic WEP
			dwrq->length = pKey->KeyLength;
			dwrq->flags = pKey->KeyIndex + 1;
			return wlan_set_encode_nonwpa(dev, info, dwrq, 
					pKey->KeyMaterial);
		}
		else {
			//WPA
			return wlan_set_encode_wpa(dev, info, dwrq, extra);
		}
	}
	else {
		//static WEP
		PRINTM(INFO, "Setting WEP\n");
		return wlan_set_encode_nonwpa(dev, info, dwrq, extra);
	}

	return -EINVAL;
}

/** 
 *  @brief set tx power 
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
int wlan_set_txpow(struct net_device *dev, struct iw_request_info *info,
			   		struct iw_param *vwrq, char *extra)
{
	int		ret = WLAN_STATUS_SUCCESS;
	wlan_private *priv= dev->priv;
	wlan_adapter *Adapter= priv->adapter;

	int wlan_radio_ioctl(wlan_private *priv, u8 option);

	u16	dbm;
	
	ENTER();

	
	if(vwrq->disabled) {
		wlan_radio_ioctl(priv, RADIO_OFF);
		return WLAN_STATUS_SUCCESS;
	}

	Adapter->Preamble = HostCmd_TYPE_AUTO_PREAMBLE;

	wlan_radio_ioctl(priv, RADIO_ON);

#if WIRELESS_EXT > 14
	if ((vwrq->flags & IW_TXPOW_TYPE) == IW_TXPOW_MWATT) {
		dbm = (u16) mw_to_dbm(vwrq->value);
	}
	else
#endif
		dbm = (u16) vwrq->value;


	/* auto tx power control */

	if (vwrq->fixed == 0)
		dbm = 0xffff;

  	PRINTM(INFO, "<1>TXPOWER SET %d dbm.\n", dbm);

	ret = PrepareAndSendCommand(priv, 
		HostCmd_CMD_802_11_RF_TX_POWER,
		HostCmd_ACT_TX_POWER_OPT_SET_LOW,
		HostCmd_OPTION_WAITFORRSP,
		0, (void *) &dbm);

	LEAVE();
	return ret;
}

/** 
 *  @brief Get current essid 
 *   
 *  @param dev                  A pointer to net_device structure
 *  @param info			A pointer to iw_request_info structure
 *  @param vwrq 		A pointer to iw_param structure
 *  @param extra		A pointer to extra data buf
 *  @return 	   		WLAN_STATUS_SUCCESS --success, otherwise fail
 */
int wlan_get_essid(struct net_device *dev, struct iw_request_info *info,
		struct iw_point *dwrq, char *extra)
{
	wlan_private	*priv = dev->priv;
	wlan_adapter	*Adapter = priv->adapter;

	ENTER();
	/*
	 * Note : if dwrq->flags != 0, we should get the relevant SSID from
	 * the SSID list... 
	 */

	/*
	 * Get the current SSID 
	 */
	if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
		memcpy(extra, Adapter->CurBssParams.ssid.Ssid,
				Adapter->CurBssParams.ssid.SsidLength);
		extra[Adapter->CurBssParams.ssid.SsidLength] = '\0';
	} else {
		memset(extra, 0, 32);
		extra[Adapter->CurBssParams.ssid.SsidLength] = '\0';
	}
	/*
	 * If none, we may want to get the one that was set 
	 */

	/* To make the driver backward compatible with WPA supplicant v0.2.4 */
	if (dwrq->length == 32)  /* check with WPA supplicant buffer size */
		dwrq->length = MIN(Adapter->CurBssParams.ssid.SsidLength, 
							IW_ESSID_MAX_SIZE);
	else 		
		dwrq->length = Adapter->CurBssParams.ssid.SsidLength + 1;

	dwrq->flags = 1;		/* active */

	LEAVE();
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get version 
 *   
 *  @param adapter              A pointer to wlan_adapter structure
 *  @param version		A pointer to version buffer
 *  @param maxlen		max length of version buffer
 *  @return 	   		NA
 */
void get_version(wlan_adapter *adapter, char *version, int maxlen)
{
	union {
		u32	l;
		u8	c[4];
	} ver;
	char	fwver[32];

	ver.l = adapter->FWReleaseNumber;
	if(ver.c[3] == 0)
		sprintf(fwver, "%u.%u.%u",
			ver.c[2], ver.c[1], ver.c[0]);
	else
		sprintf(fwver, "%u.%u.%u.p%u",
			ver.c[2], ver.c[1], ver.c[0],ver.c[3]);

	snprintf(version, maxlen, driver_version, fwver);
}


