/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil -*- */
/* vi: set expandtab shiftwidth=4 tabstop=4 textwidth=78: */

/** @file wlan_join.c
  *
  *  @brief Functions implementing wlan infrastructure and adhoc join routines
  *
  *  IOCTL handlers as well as command preperation and response routines
  *   for sending adhoc start, adhoc join, and association commands
  *   to the firmware.
  *
  *  @sa wlan_join.h
  *
  * (c) Copyright  2003-2006, Marvell International Ltd. 
  * All Rights Reserved
  *
  * This software file (the "File") is distributed by Marvell International 
  * Ltd. under the terms of the GNU General Public License Version 2, June 1991 
  * (the "License").  You may use, redistribute and/or modify this File in 
  * accordance with the terms and conditions of the License, a copy of which 
  * is available along with the File in the license.txt file or by writing to 
  * the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 
  * 02111-1307 or on the worldwide web at http://www.gnu.org/licenses/gpl.txt.
  *
  * THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE 
  * IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE 
  * ARE EXPRESSLY DISCLAIMED.  The License provides additional details about 
  * this warranty disclaimer.
  *
  */
/*************************************************************
Change Log:
    01/11/06: Initial revision. Match new scan code, relocate related functions
    01/19/06: Fix failure to save adhoc ssid as current after adhoc start
    03/16/06: Add a semaphore to protect reassociation thread

************************************************************/

#include    "include.h"


/**
 *  @brief This function finds out the common rates between rate1 and rate2.
 *
 * It will fill common rates in rate1 as output if found.
 *
 * NOTE: Setting the MSB of the basic rates need to be taken
 *   care, either before or after calling this function
 *
 *  @param Adapter     A pointer to wlan_adapter structure
 *  @param rate1       the buffer which keeps input and output
 *  @param rate1_size  the size of rate1 buffer
 *  @param rate2       the buffer which keeps rate2
 *  @param rate2_size  the size of rate2 buffer.
 *
 *  @return            WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int get_common_rates(wlan_adapter* Adapter, u8* rate1,
                            int rate1_size, u8* rate2, int rate2_size)
{
    u8* ptr = rate1;
    int ret = WLAN_STATUS_SUCCESS;
    u8  tmp[30];
    int i;

    memset(&tmp, 0, sizeof(tmp));
    memcpy(&tmp, rate1, MIN(rate1_size,sizeof(tmp)));
    memset(rate1, 0, rate1_size);

    /* Mask the top bit of the original values */
    for (i = 0; tmp[i] && i < sizeof(tmp); i++)
        tmp[i] &= 0x7F;

    for (i = 0; rate2[i] && i < rate2_size; i++) {
        /* Check for Card Rate in tmp, excluding the top bit */
        if (strchr(tmp, rate2[i] & 0x7F)) {
            /* Values match, so copy the Card Rate to rate1 */
            *rate1++ = rate2[i];
        }
    }

    HEXDUMP("rate1 (AP) Rates:", tmp, sizeof(tmp));
    HEXDUMP("rate2 (Card) Rates:", rate2, rate2_size);
    HEXDUMP("Common Rates:", ptr, rate1_size);
    PRINTM(INFO, "Tx DataRate is set to 0x%X\n", Adapter->DataRate);

    if (!Adapter->Is_DataRate_Auto) {
        while (*ptr) {
            if ((*ptr & 0x7f) == Adapter->DataRate) {
                ret = WLAN_STATUS_SUCCESS;
                goto done;
            }
            ptr++;
        }
        PRINTM(MSG, "Previously set fixed data rate %#x isn't "
               "compatible with the network.\n",Adapter->DataRate);

        ret = WLAN_STATUS_FAILURE;
        goto done;
    }

    ret = WLAN_STATUS_SUCCESS;
done:
    return ret;
}


/**
 *  @brief Send Deauth Request
 *
 *  @param priv      A pointer to wlan_private structure
 *  @return          WLAN_STATUS_SUCCESS--success, otherwise fail
 */
int wlan_send_deauth(wlan_private* priv)
{
    wlan_adapter* Adapter = priv->adapter;
    int           ret     = WLAN_STATUS_SUCCESS;

    ENTER();

    if (Adapter->InfrastructureMode == Wlan802_11Infrastructure &&
        Adapter->MediaConnectStatus == WlanMediaStateConnected) {

        ret = SendDeauthentication(priv);

    } else {
        LEAVE();
        return -ENOTSUPP;
    }

    LEAVE();
    return ret;
}


/**
 *  @brief Stop Adhoc Network
 *
 *  @param priv         A pointer to wlan_private structure
 *  @return             WLAN_STATUS_SUCCESS --success, otherwise fail
 */
int wlan_do_adhocstop_ioctl(wlan_private* priv)
{
    wlan_adapter* Adapter = priv->adapter;
    int           ret     = WLAN_STATUS_SUCCESS;

    ENTER();

    if (Adapter->InfrastructureMode == Wlan802_11IBSS &&
        Adapter->MediaConnectStatus == WlanMediaStateConnected) {

        ret = StopAdhocNetwork(priv);

    } else {
        LEAVE();
        return -ENOTSUPP;
    }

    LEAVE();

    return WLAN_STATUS_SUCCESS;
}


/**
 *  @brief Set essid
 *
 *  @param dev          A pointer to net_device structure
 *  @param info         A pointer to iw_request_info structure
 *  @param dwrq         A pointer to iw_point structure
 *  @param extra        A pointer to extra data buf
 *  @return             WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
int wlan_set_essid(struct net_device* dev, struct iw_request_info* info,
                   struct iw_point* dwrq, char* extra)
{
    wlan_private*       priv    = dev->priv;
    wlan_adapter*       Adapter = priv->adapter;
    int                 ret     = WLAN_STATUS_SUCCESS;
    struct WLAN_802_11_SSID    reqSSID;
    int                 i;

    ENTER();



#ifdef REASSOCIATION
    // cancel re-association timer if there's one
    if (Adapter->TimerIsSet == TRUE) {
        CancelTimer(&Adapter->MrvDrvTimer);
        Adapter->TimerIsSet = FALSE;
    }

    if (down_interruptible(&Adapter->ReassocSem)) {
        PRINTM(FATAL, "Acquire semaphore error, wlan_set_essid\n");
        return -EBUSY;
    }
#endif /* REASSOCIATION */

    /* Check the size of the string */
    if (dwrq->length > IW_ESSID_MAX_SIZE + 1) {
        ret = -E2BIG ;
        goto setessid_ret;
    }

    memset(&reqSSID, 0, sizeof(struct WLAN_802_11_SSID));

    /*
     * Check if we asked for `any' or 'particular'
     */
    if (!dwrq->flags) {
        if (FindBestNetworkSsid(priv, &reqSSID)) {
            PRINTM(INFO, "Could not find best network\n");
            ret = WLAN_STATUS_SUCCESS;
            goto setessid_ret;
        }
    } else {
        /* Set the SSID */
        memcpy(reqSSID.Ssid, extra, dwrq->length);
        reqSSID.SsidLength = dwrq->length;
    }

    PRINTM(INFO, "Requested new SSID = %s\n",
           (reqSSID.SsidLength > 0) ? (char *)reqSSID.Ssid : "NULL");

    if (!reqSSID.SsidLength || reqSSID.Ssid[0] < 0x20) {
        PRINTM(INFO, "Invalid SSID - aborting set_essid\n");
        ret = -EINVAL;
        goto setessid_ret;
    }

    /* If the requested SSID is not a NULL string, join */

    if (Adapter->InfrastructureMode == Wlan802_11Infrastructure) {
        /* infrastructure mode */
        PRINTM(INFO, "SSID requested = %s\n", reqSSID.Ssid);

        if (Adapter->MediaConnectStatus == WlanMediaStateConnected){
            PRINTM(INFO, "Already Connected ..\n");
            ret = SendDeauthentication(priv);

            if (ret){
                goto setessid_ret;
            }
        }
        if(Adapter->Prescan)
            SendSpecificSSIDScan(priv, &reqSSID, TRUE);
        i = FindSSIDInList(Adapter, &reqSSID, NULL,
                           Wlan802_11Infrastructure);
        if (i >= 0) {
            PRINTM(INFO, "SSID found in scan list ... associating...\n");

           
            ret = wlan_associate(priv, &Adapter->ScanTable[i]);

            if (ret) {
                goto setessid_ret;
            }
        } else { /* i >= 0 */
            ret = i; /* return -ENETUNREACH, passed from FindSSIDInList */
            goto setessid_ret;
        }
    } else {
        /* ad hoc mode */
        /* If the requested SSID matches current SSID return */
        if (!SSIDcmp(&Adapter->CurBssParams.ssid, &reqSSID)) {
            ret = WLAN_STATUS_SUCCESS;
            goto setessid_ret;
        }

        if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
            /*
             * Exit Adhoc mode
             */
            PRINTM(INFO, "Sending Adhoc Stop\n");
            ret = StopAdhocNetwork(priv);

            if (ret) {
                goto setessid_ret;
            }

        }

        /* Scan for the network, do not save previous results.  Stale
         *   scan data will cause us to join a non-existant adhoc network
         */
        SendSpecificSSIDScan(priv, &reqSSID, FALSE);
        
        /* Search for the requested SSID in the scan table */
        i = FindSSIDInList(Adapter, &reqSSID, NULL, Wlan802_11IBSS);

        if (i >= 0) {
            PRINTM(INFO, "SSID found at %d in List, so join\n", i);
            JoinAdhocNetwork(priv, &Adapter->ScanTable[i]);
        } else {
            /* else send START command */
            PRINTM(INFO, "SSID not found in list, "
                   "so creating adhoc with ssid = %s\n",
                   reqSSID.Ssid);

            StartAdhocNetwork(priv, &reqSSID);
        }   /* end of else (START command) */
    }       /* end of else (Ad hoc mode) */


    /*
     * The MediaConnectStatus change can be removed later when
     *   the ret code is being properly returned.
     */
    /* Check to see if we successfully connected */
    if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
        ret = WLAN_STATUS_SUCCESS;
    } else {
        ret = -ENETDOWN;
    }

setessid_ret:
#ifdef REASSOCIATION
    up(&Adapter->ReassocSem);
#endif

    LEAVE();
    return ret;
}

/**
 *  @brief Connect to the AP or Ad-hoc Network with specific bssid
 *
 *  @param dev          A pointer to net_device structure
 *  @param info         A pointer to iw_request_info structure
 *  @param awrq         A pointer to iw_param structure
 *  @param extra        A pointer to extra data buf
 *  @return             WLAN_STATUS_SUCCESS --success, otherwise fail
 */
int wlan_set_wap(struct net_device* dev, struct iw_request_info* info,
                 struct sockaddr* awrq, char* extra)
{
    wlan_private*  priv            = dev->priv;
    wlan_adapter*  Adapter         = priv->adapter;
    int            ret             = WLAN_STATUS_SUCCESS;
    const u8       bcast[ETH_ALEN] = { 255, 255, 255, 255, 255, 255 };
    u8             reqBSSID[ETH_ALEN];
    int            i;

    ENTER();


//Application should call scan before call this function.    

    if (awrq->sa_family != ARPHRD_ETHER)
        return -EINVAL;

    PRINTM(INFO, "ASSOC: WAP: sa_data: %02x:%02x:%02x:%02x:%02x:%02x\n",
           (u8)awrq->sa_data[0], (u8)awrq->sa_data[1],
           (u8)awrq->sa_data[2], (u8)awrq->sa_data[3],
           (u8)awrq->sa_data[4], (u8)awrq->sa_data[5]);

#ifdef REASSOCIATION
    // cancel re-association timer if there's one
    if (Adapter->TimerIsSet == TRUE) {
        CancelTimer(&Adapter->MrvDrvTimer);
        Adapter->TimerIsSet = FALSE;
    }
#endif /* REASSOCIATION */

    if (!memcmp(bcast, awrq->sa_data, ETH_ALEN)) {
        i = FindBestSSIDInList(Adapter);
    } else {
        memcpy(reqBSSID, awrq->sa_data, ETH_ALEN);

        PRINTM(INFO, "ASSOC: WAP: Bssid = %02x:%02x:%02x:%02x:%02x:%02x\n",
               reqBSSID[0], reqBSSID[1], reqBSSID[2],
               reqBSSID[3], reqBSSID[4], reqBSSID[5]);

        /* Search for index position in list for requested MAC */
        i = FindBSSIDInList(Adapter, reqBSSID, Adapter->InfrastructureMode);
    }

    if (i < 0) {
        PRINTM(INFO, "ASSOC: WAP: MAC address not found in BSSID List\n");
        return -ENETUNREACH;
    }
    
    if (Adapter->InfrastructureMode == Wlan802_11Infrastructure) {
        if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
            ret = SendDeauthentication(priv);

            if (ret) {
                LEAVE();
                return ret;
            }
        }
        ret = wlan_associate(priv, &Adapter->ScanTable[i]);

        if (ret) {
            LEAVE();
            return ret;
        }
    } else {
        if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
            /* Exit Adhoc mode */
            ret = StopAdhocNetwork(priv);

            if (ret) {
                LEAVE();
                return ret;
            }
        }

        JoinAdhocNetwork(priv, &Adapter->ScanTable[i]);
    }

    /* Check to see if we successfully connected */
    if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
        ret = WLAN_STATUS_SUCCESS;
    } else {
        ret = -ENETDOWN;
    }

    LEAVE();
    return ret;
}


/**
 *  @brief Associated to a specific BSS discovered in a scan
 *
 *  @param priv      A pointer to wlan_private structure
 *  @param pBSSDesc  Pointer to the BSS descriptor to associate with.
 *
 *  @return          WLAN_STATUS_SUCCESS-success, otherwise fail
 */
int wlan_associate(wlan_private* priv, BSSDescriptor_t* pBSSDesc)
{
    int ret;


    ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_AUTHENTICATE,
                                0, HostCmd_OPTION_WAITFORRSP,
                                0, pBSSDesc->MacAddress);

    if (ret) {
        LEAVE();
        return ret;
    }

    ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_ASSOCIATE,
                                0, HostCmd_OPTION_WAITFORRSP,
                                0, pBSSDesc);

    LEAVE();
    return ret;
}


/**
 *  @brief Start an Adhoc Network
 *
 *  @param priv         A pointer to wlan_private structure
 *  @param AdhocSSID    The ssid of the Adhoc Network
 *  @return             WLAN_STATUS_SUCCESS--success, WLAN_STATUS_FAILURE--fail
 */
int StartAdhocNetwork(wlan_private* priv, struct WLAN_802_11_SSID* AdhocSSID)
{
    wlan_adapter* Adapter = priv->adapter;
    int           ret     = WLAN_STATUS_SUCCESS;

    ENTER();

    Adapter->AdhocCreate = TRUE;

    if (!Adapter->capInfo.ShortPreamble) {
        PRINTM(INFO, "AdhocStart: Long Preamble\n");
        Adapter->Preamble = HostCmd_TYPE_LONG_PREAMBLE;
    } else {
        PRINTM(INFO, "AdhocStart: Short Preamble\n");
        Adapter->Preamble = HostCmd_TYPE_SHORT_PREAMBLE;
    }

    SetRadioControl(priv);


    PRINTM(INFO, "Adhoc Channel = %d\n", Adapter->AdhocChannel);
    PRINTM(INFO, "CurBssParams.channel = %d\n", Adapter->CurBssParams.channel);
    PRINTM(INFO, "CurBssParams.band = %d\n", Adapter->CurBssParams.band);

    ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_AD_HOC_START,
                                0, HostCmd_OPTION_WAITFORRSP,
                                0, AdhocSSID);

    LEAVE();
    return ret;
}

/**
 *  @brief Join an adhoc network found in a previous scan
 *
 *  @param priv         A pointer to wlan_private structure
 *  @param pBSSDesc     Pointer to a BSS descriptor found in a previous scan
 *                      to attempt to join
 *
 *  @return             WLAN_STATUS_SUCCESS--success, WLAN_STATUS_FAILURE--fail
 */
int JoinAdhocNetwork(wlan_private* priv, BSSDescriptor_t* pBSSDesc)
{
    wlan_adapter* Adapter = priv->adapter;
    int           ret     = WLAN_STATUS_SUCCESS;

    ENTER();

    PRINTM(INFO, "JoinAdhocNetwork: CurBss.ssid =%s\n",
           Adapter->CurBssParams.ssid.Ssid);
    PRINTM(INFO, "JoinAdhocNetwork: CurBss.ssid_len =%u\n",
           Adapter->CurBssParams.ssid.SsidLength);
    PRINTM(INFO, "JoinAdhocNetwork: ssid =%s\n",
           pBSSDesc->Ssid.Ssid);
    PRINTM(INFO, "JoinAdhocNetwork: ssid len =%u\n",
           pBSSDesc->Ssid.SsidLength);

    /* check if the requested SSID is already joined */
    if (Adapter->CurBssParams.ssid.SsidLength
        && !SSIDcmp(&pBSSDesc->Ssid, &Adapter->CurBssParams.ssid)
        && (Adapter->CurBssParams.BSSDescriptor.InfrastructureMode ==
            Wlan802_11IBSS)) {

        PRINTM(INFO,
               "ADHOC_J_CMD: New ad-hoc SSID is the same as current, "
               "not attempting to re-join");

        return WLAN_STATUS_FAILURE;
    }


    /*Use ShortPreamble only when both creator and card supports
      short preamble*/
    if (!pBSSDesc->Cap.ShortPreamble
        || !Adapter->capInfo.ShortPreamble) {
        PRINTM(INFO, "AdhocJoin: Long Preamble\n");
        Adapter->Preamble = HostCmd_TYPE_LONG_PREAMBLE;
    }
    else {
        PRINTM(INFO, "AdhocJoin: Short Preamble\n");
        Adapter->Preamble = HostCmd_TYPE_SHORT_PREAMBLE;
    }

    SetRadioControl(priv);

    PRINTM(INFO, "CurBssParams.channel = %d\n", Adapter->CurBssParams.channel);
    PRINTM(INFO, "CurBssParams.band = %c\n", Adapter->CurBssParams.band);

    Adapter->AdhocCreate = FALSE;

    // store the SSID info temporarily
    memset(&Adapter->AttemptedSSIDBeforeScan, 0, sizeof(struct WLAN_802_11_SSID));
    memcpy(&Adapter->AttemptedSSIDBeforeScan,
           &pBSSDesc->Ssid,
           sizeof(struct WLAN_802_11_SSID));

    ret = PrepareAndSendCommand(priv, HostCmd_CMD_802_11_AD_HOC_JOIN,
                                0, HostCmd_OPTION_WAITFORRSP,
                                OID_802_11_SSID, pBSSDesc);

    LEAVE();
    return ret;
}

/**
 *  @brief Stop the Adhoc Network
 *
 *  @param priv      A pointer to wlan_private structure
 *  @return          WLAN_STATUS_SUCCESS--success, WLAN_STATUS_FAILURE--fail
 */
int StopAdhocNetwork(wlan_private* priv)
{
    return PrepareAndSendCommand(priv, HostCmd_CMD_802_11_AD_HOC_STOP,
                                 0, HostCmd_OPTION_WAITFORRSP,
                                 0, NULL);
}

/**
 *  @brief Send Deauthentication Request
 *
 *  @param priv      A pointer to wlan_private structure
 *  @return          WLAN_STATUS_SUCCESS--success, WLAN_STATUS_FAILURE--fail
 */
int SendDeauthentication(wlan_private* priv)
{
    return PrepareAndSendCommand(priv, HostCmd_CMD_802_11_DEAUTHENTICATE,
                                 0, HostCmd_OPTION_WAITFORRSP,
                                 0, NULL);
}

/**
 *  @brief Set Idle Off
 *
 *  @param priv         A pointer to wlan_private structure
 *  @return             WLAN_STATUS_SUCCESS --success, otherwise fail
 */
int wlanidle_off(wlan_private* priv)
{
    wlan_adapter* Adapter   = priv->adapter;
    int           ret       = WLAN_STATUS_SUCCESS;
    const u8      zeroMac[] = {0, 0, 0, 0, 0, 0};
    int           i;

    ENTER();

    if (Adapter->MediaConnectStatus == WlanMediaStateDisconnected) {
        if (Adapter->InfrastructureMode == Wlan802_11Infrastructure) {
            if (memcmp(Adapter->PreviousBSSID, zeroMac,
                       sizeof(zeroMac)) != 0) {

                PRINTM(INFO, "Previous SSID = %s\n",
                       Adapter->PreviousSSID.Ssid);
                PRINTM(INFO, "Previous BSSID = "
                       "%02x:%02x:%02x:%02x:%02x:%02x:\n",
                       Adapter->PreviousBSSID[0], Adapter->PreviousBSSID[1],
                       Adapter->PreviousBSSID[2], Adapter->PreviousBSSID[3],
                       Adapter->PreviousBSSID[4], Adapter->PreviousBSSID[5]);

                i = FindSSIDInList(Adapter,
                                   &Adapter->PreviousSSID,
                                   Adapter->PreviousBSSID,
                                   Adapter->InfrastructureMode);

                if (i < 0) {
                    SendSpecificBSSIDScan(priv, Adapter->PreviousBSSID, TRUE);
                    i = FindSSIDInList(Adapter,
                                       &Adapter->PreviousSSID,
                                       Adapter->PreviousBSSID,
                                       Adapter->InfrastructureMode);
                }

                if (i < 0) {
                    /* If the BSSID could not be found, try just the SSID */
                    i = FindSSIDInList(Adapter,
                                       &Adapter->PreviousSSID,
                                       NULL,
                                       Adapter->InfrastructureMode);
                }

                if (i < 0) {
                    SendSpecificSSIDScan(priv, &Adapter->PreviousSSID, TRUE);
                    i = FindSSIDInList(Adapter,
                                       &Adapter->PreviousSSID,
                                       NULL,
                                       Adapter->InfrastructureMode);
                }

                if (i >= 0) {
                    ret = wlan_associate(priv, &Adapter->ScanTable[i]);
                }
            }
        } else if (Adapter->InfrastructureMode == Wlan802_11IBSS) {
            ret = PrepareAndSendCommand(priv,
                                        HostCmd_CMD_802_11_AD_HOC_START,
                                        0, HostCmd_OPTION_WAITFORRSP,
                                        0, &Adapter->PreviousSSID);
        }
    }
    /* else it is connected */

    PRINTM(INFO, "\nwlanidle is off");
    LEAVE();
    return ret;
}

/**
 *  @brief Set Idle On
 *
 *  @param priv         A pointer to wlan_private structure
 *  @return             WLAN_STATUS_SUCCESS --success, otherwise fail
 */
int wlanidle_on(wlan_private* priv)
{
    wlan_adapter* Adapter = priv->adapter;
    int           ret     = WLAN_STATUS_SUCCESS;

    ENTER();

    if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
        if (Adapter->InfrastructureMode == Wlan802_11Infrastructure) {
            PRINTM(INFO, "Previous SSID = %s\n",
                   Adapter->PreviousSSID.Ssid);
            memmove(&Adapter->PreviousSSID,
                    &Adapter->CurBssParams.ssid,
                    sizeof(struct WLAN_802_11_SSID));
            wlan_send_deauth(priv);

        } else if (Adapter->InfrastructureMode == Wlan802_11IBSS) {
            ret = StopAdhocNetwork(priv);
        }

    }

#ifdef REASSOCIATION
    if (Adapter->TimerIsSet == TRUE) {
        CancelTimer(&Adapter->MrvDrvTimer);
        Adapter->TimerIsSet = FALSE;
    }
#endif /* REASSOCIATION */

    PRINTM(INFO, "\nwlanidle is on");

    LEAVE();
    return ret;
}




/**
 *  @brief This function prepares command of authenticate.
 *
 *  @param priv      A pointer to wlan_private structure
 *  @param cmd       A pointer to HostCmd_DS_COMMAND structure
 *  @param pdata_buf Void cast of pointer to a BSSID to authenticate with
 *
 *  @return         WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_cmd_802_11_authenticate(wlan_private* priv,
                                 struct HostCmd_DS_COMMAND* cmd,
                                 void* pdata_buf)
{
    wlan_adapter*                   Adapter       = priv->adapter;
    struct HostCmd_DS_802_11_AUTHENTICATE* pAuthenticate = &cmd->params.auth;
    u8*                             bssid         = (u8*)pdata_buf;

    cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_AUTHENTICATE);
    cmd->Size = wlan_cpu_to_le16(sizeof(struct HostCmd_DS_802_11_AUTHENTICATE)
                                 + S_DS_GEN);

    pAuthenticate->AuthType = Adapter->SecInfo.AuthenticationMode;
    memcpy(pAuthenticate->MacAddr, bssid, MRVDRV_ETH_ADDR_LEN);

    PRINTM(INFO, "AUTH_CMD: Bssid is : %x:%x:%x:%x:%x:%x\n",
           bssid[0], bssid[1], bssid[2], bssid[3], bssid[4], bssid[5]);

    return WLAN_STATUS_SUCCESS;
}


/**
 *  @brief This function prepares command of deauthenticat.
 *
 *  @param priv     A pointer to wlan_private structure
 *  @param cmd      A pointer to HostCmd_DS_COMMAND structure
 *  @return         WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_cmd_802_11_deauthenticate(wlan_private* priv,
                                   struct HostCmd_DS_COMMAND* cmd)
{
    wlan_adapter*                     Adapter = priv->adapter;
    struct HostCmd_DS_802_11_DEAUTHENTICATE* dauth   = &cmd->params.deauth;

    ENTER();

    cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_DEAUTHENTICATE);
    cmd->Size =
        wlan_cpu_to_le16(sizeof(struct HostCmd_DS_802_11_DEAUTHENTICATE) + S_DS_GEN);

    /* set AP MAC address */
    memmove(dauth->MacAddr, Adapter->CurBssParams.bssid, MRVDRV_ETH_ADDR_LEN);

    /* Reason code 3 = Station is leaving */
#define REASON_CODE_STA_LEAVING 3
    dauth->ReasonCode = wlan_cpu_to_le16(REASON_CODE_STA_LEAVING);

    LEAVE();
    return WLAN_STATUS_SUCCESS;
}


/**
 *  @brief This function prepares command of association.
 *
 *  @param priv      A pointer to wlan_private structure
 *  @param cmd       A pointer to HostCmd_DS_COMMAND structure
 *  @param pdata_buf Void cast of BSSDescriptor_t from the scan table to assoc
 *  @return          WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_cmd_802_11_associate(wlan_private* priv,
                              struct HostCmd_DS_COMMAND* cmd,
                              void* pdata_buf)
{
    wlan_adapter*                 Adapter  = priv->adapter;
    struct HostCmd_DS_802_11_ASSOCIATE*  pAsso    = &cmd->params.associate;
    int                           ret      = WLAN_STATUS_SUCCESS;
    BSSDescriptor_t*              pBSSDesc;
    u8*                           card_rates;
    u8*                           pos;
    int                           card_rates_size;
    u16                           TmpCap;
    MrvlIEtypes_SsIdParamSet_t*   ssid;
    MrvlIEtypes_PhyParamSet_t*    phy;
    MrvlIEtypes_SsParamSet_t*     ss;
    MrvlIEtypes_RatesParamSet_t*  rates;
    MrvlIEtypes_RsnParamSet_t*    rsn;

    ENTER();

    pBSSDesc = (BSSDescriptor_t*)pdata_buf;
    pos      = (u8*)pAsso;

    if (!Adapter) {
        ret = WLAN_STATUS_FAILURE;
        goto done;
    }

    cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_ASSOCIATE);

    /* Save so we know which BSS Desc to use in the response handler */
    Adapter->pAttemptedBSSDesc = pBSSDesc;

    memcpy(pAsso->PeerStaAddr,
           pBSSDesc->MacAddress,
           sizeof(pAsso->PeerStaAddr));
    pos += sizeof(pAsso->PeerStaAddr);

    /* set preamble to firmware */
    if (Adapter->capInfo.ShortPreamble && pBSSDesc->Cap.ShortPreamble) {
        Adapter->Preamble = HostCmd_TYPE_SHORT_PREAMBLE;
    } else {
        Adapter->Preamble = HostCmd_TYPE_LONG_PREAMBLE;
    }

    SetRadioControl(priv);

    /* set the listen interval */
    pAsso->ListenInterval = Adapter->ListenInterval;

    pos += sizeof(pAsso->CapInfo);
    pos += sizeof(pAsso->ListenInterval);
    pos += sizeof(pAsso->BcnPeriod);
    pos += sizeof(pAsso->DtimPeriod);

    ssid = (MrvlIEtypes_SsIdParamSet_t *)pos;
    ssid->Header.Type = wlan_cpu_to_le16(TLV_TYPE_SSID);
    ssid->Header.Len = pBSSDesc->Ssid.SsidLength;
    memcpy(ssid->SsId, pBSSDesc->Ssid.Ssid, ssid->Header.Len);
    pos += sizeof(ssid->Header) + ssid->Header.Len;
    ssid->Header.Len = wlan_cpu_to_le16(ssid->Header.Len);

    phy = (MrvlIEtypes_PhyParamSet_t *)pos;
    phy->Header.Type = wlan_cpu_to_le16(TLV_TYPE_PHY_DS);
    phy->Header.Len = sizeof(phy->fh_ds.DsParamSet);
    memcpy(&phy->fh_ds.DsParamSet,
           &pBSSDesc->PhyParamSet.DsParamSet.CurrentChan,
           sizeof(phy->fh_ds.DsParamSet));
    pos += sizeof(phy->Header) + phy->Header.Len;
    phy->Header.Len = wlan_cpu_to_le16(phy->Header.Len);

    ss = (MrvlIEtypes_SsParamSet_t *)pos;
    ss->Header.Type = wlan_cpu_to_le16(TLV_TYPE_CF);
    ss->Header.Len = sizeof(ss->cf_ibss.CfParamSet);
    pos += sizeof(ss->Header) + ss->Header.Len;
    ss->Header.Len = wlan_cpu_to_le16(ss->Header.Len);

    rates = (MrvlIEtypes_RatesParamSet_t *)pos;
    rates->Header.Type = wlan_cpu_to_le16(TLV_TYPE_RATES);

    memcpy(&rates->Rates,&pBSSDesc->SupportedRates, WLAN_SUPPORTED_RATES);

    card_rates = SupportedRates;
    card_rates_size = sizeof(SupportedRates);

    if (get_common_rates(Adapter, rates->Rates, WLAN_SUPPORTED_RATES,
                         card_rates, card_rates_size)) {
        ret = WLAN_STATUS_FAILURE;
        goto done;
    }

    rates->Header.Len = MIN(strlen(rates->Rates), WLAN_SUPPORTED_RATES);
    Adapter->CurBssParams.NumOfRates = rates->Header.Len;

    pos += sizeof(rates->Header) + rates->Header.Len;
    rates->Header.Len = wlan_cpu_to_le16(rates->Header.Len);


    if (Adapter->SecInfo.WPAEnabled
        || Adapter->SecInfo.WPA2Enabled
        )
    {
        rsn = (MrvlIEtypes_RsnParamSet_t *)pos;
        rsn->Header.Type = (u16)Adapter->Wpa_ie[0]; /* WPA_IE or WPA2_IE */
        rsn->Header.Type = wlan_cpu_to_le16(rsn->Header.Type);
        rsn->Header.Len = (u16)Adapter->Wpa_ie[1];
        memcpy(rsn->RsnIE, &Adapter->Wpa_ie[2], rsn->Header.Len);
        HEXDUMP("ASSOC_CMD: RSN IE", (u8*)rsn,
                sizeof(rsn->Header) + rsn->Header.Len);
        pos += sizeof(rsn->Header) + rsn->Header.Len;
        rsn->Header.Len = wlan_cpu_to_le16(rsn->Header.Len);
    }





    /* update CurBssParams */
    Adapter->CurBssParams.channel = (pBSSDesc
                                     ->PhyParamSet.DsParamSet.CurrentChan);

    /* Copy the infra. association rates into Current BSS state structure */
    memcpy(&Adapter->CurBssParams.DataRates, &rates->Rates,
           MIN(sizeof(Adapter->CurBssParams.DataRates), rates->Header.Len));

    PRINTM(INFO, "ASSOC_CMD: rates->Header.Len = %d\n", rates->Header.Len);

    /* set IBSS field */
    if (pBSSDesc->InfrastructureMode == Wlan802_11Infrastructure) {
#define CAPINFO_ESS_MODE 1
        pAsso->CapInfo.Ess = CAPINFO_ESS_MODE;
    }

    if (wlan_parse_dnld_countryinfo_11d( priv )) {
        ret = WLAN_STATUS_FAILURE;
        goto done;
    }



    cmd->Size = wlan_cpu_to_le16((u16)(pos - (u8 *)pAsso) + S_DS_GEN);

    /* set the Capability info at last */
    memcpy(&TmpCap, &pBSSDesc->Cap, sizeof(pAsso->CapInfo));
    TmpCap &= CAPINFO_MASK;
    PRINTM(INFO, "ASSOC_CMD: TmpCap=%4X CAPINFO_MASK=%4X\n",
           TmpCap, CAPINFO_MASK );
    TmpCap = wlan_cpu_to_le16(TmpCap);
    memcpy(&pAsso->CapInfo, &TmpCap, sizeof(pAsso->CapInfo));

done:
    LEAVE();
    return ret;
}


/**
 *  @brief This function prepares command of ad_hoc_start.
 *
 *  @param priv     A pointer to wlan_private structure
 *  @param cmd      A pointer to HostCmd_DS_COMMAND structure
 *  @param pssid    A pointer to WLAN_802_11_SSID structure
 *  @return         WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_cmd_802_11_ad_hoc_start(wlan_private* priv,
                                 struct HostCmd_DS_COMMAND * cmd,
                                 void *pssid)
{
    wlan_adapter*                    Adapter       = priv->adapter;
    struct HostCmd_DS_802_11_AD_HOC_START*  adhs          = &cmd->params.ads;
    int                              ret           = WLAN_STATUS_SUCCESS;
    int                              cmdAppendSize = 0;
    int                              i;
    u16                              TmpCap;
    BSSDescriptor_t*                 pBSSDesc;

    ENTER();

    if (!Adapter) {
        ret = WLAN_STATUS_FAILURE;
        goto done;
    }

    cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_AD_HOC_START);

    pBSSDesc = &Adapter->CurBssParams.BSSDescriptor;
    Adapter->pAttemptedBSSDesc = pBSSDesc;

    /*
     * Fill in the parameters for 2 data structures:
     *   1. HostCmd_DS_802_11_AD_HOC_START Command
     *   2. Adapter->ScanTable[i]
     *
     * Driver will fill up SSID, BSSType,IBSS param, Physical Param,
     *   probe delay, and Cap info.
     *
     * Firmware will fill up beacon period, DTIM, Basic rates
     *   and operational rates.
     */

    memset(adhs->SSID, 0, MRVDRV_MAX_SSID_LENGTH);

    memcpy(adhs->SSID,((struct WLAN_802_11_SSID *) pssid)->Ssid,
           ((struct WLAN_802_11_SSID *) pssid)->SsidLength);

    PRINTM(INFO, "ADHOC_S_CMD: SSID = %s\n", adhs->SSID);

    memset(pBSSDesc->Ssid.Ssid, 0, MRVDRV_MAX_SSID_LENGTH);
    memcpy(pBSSDesc->Ssid.Ssid,
           ((struct WLAN_802_11_SSID *) pssid)->Ssid,
           ((struct WLAN_802_11_SSID *) pssid)->SsidLength);

    pBSSDesc->Ssid.SsidLength = ((struct WLAN_802_11_SSID *) pssid)->SsidLength;

    /* set the BSS type */
    adhs->BSSType = HostCmd_BSS_TYPE_IBSS;
    pBSSDesc->InfrastructureMode = Wlan802_11IBSS;
    adhs->BeaconPeriod = Adapter->BeaconPeriod; 

    /* set Physical param set */
#define DS_PARA_IE_ID   3
#define DS_PARA_IE_LEN  1

    adhs->PhyParamSet.DsParamSet.ElementId = DS_PARA_IE_ID;
    adhs->PhyParamSet.DsParamSet.Len = DS_PARA_IE_LEN;

    ASSERT(Adapter->AdhocChannel);
 
    PRINTM(INFO, "ADHOC_S_CMD: Creating ADHOC on Channel %d\n",
           Adapter->AdhocChannel);

    Adapter->CurBssParams.channel = Adapter->AdhocChannel;

    pBSSDesc->Channel = Adapter->AdhocChannel;
    adhs->PhyParamSet.DsParamSet.CurrentChan = Adapter->AdhocChannel;

    memcpy(&pBSSDesc->PhyParamSet,
           &adhs->PhyParamSet, sizeof(IEEEtypes_PhyParamSet_t));

    pBSSDesc->NetworkTypeInUse = Wlan802_11DS;

    /* set IBSS param set */
#define IBSS_PARA_IE_ID   6
#define IBSS_PARA_IE_LEN  2

    adhs->SsParamSet.IbssParamSet.ElementId = IBSS_PARA_IE_ID;
    adhs->SsParamSet.IbssParamSet.Len = IBSS_PARA_IE_LEN;
    adhs->SsParamSet.IbssParamSet.AtimWindow = Adapter->AtimWindow;
    memcpy(&pBSSDesc->SsParamSet,
           &adhs->SsParamSet,
           sizeof(IEEEtypes_SsParamSet_t));

    /* set Capability info */
    adhs->Cap.Ess = 0;
    adhs->Cap.Ibss = 1;
    pBSSDesc->Cap.Ibss = 1;

    /* ProbeDelay */
    adhs->ProbeDelay = wlan_cpu_to_le16(HostCmd_SCAN_PROBE_DELAY_TIME);

    /* set up privacy in Adapter->ScanTable[i] */
    if (Adapter->SecInfo.WEPStatus == Wlan802_11WEPEnabled
        ) {

#define AD_HOC_CAP_PRIVACY_ON 1
        PRINTM(INFO, "ADHOC_S_CMD: WEPStatus set, Privacy to WEP\n");
        pBSSDesc->Privacy = Wlan802_11PrivFilter8021xWEP;
        adhs->Cap.Privacy = AD_HOC_CAP_PRIVACY_ON;
    } else {
        PRINTM(INFO, "ADHOC_S_CMD: WEPStatus NOT set, Setting "
               "Privacy to ACCEPT ALL\n");
        pBSSDesc->Privacy = Wlan802_11PrivFilterAcceptAll;
    }

    memset(adhs->DataRate, 0, sizeof(adhs->DataRate));

    if (Adapter->adhoc_grate_enabled == TRUE) {
        memcpy(adhs->DataRate, AdhocRates_G,
               MIN(sizeof(adhs->DataRate), sizeof(AdhocRates_G)));
    } else {
        memcpy(adhs->DataRate, AdhocRates_B,
               MIN(sizeof(adhs->DataRate), sizeof(AdhocRates_B)));
    }

    /* Find the last non zero */
    for (i = 0; i < sizeof(adhs->DataRate) && adhs->DataRate[i]; i++)
        ;

    Adapter->CurBssParams.NumOfRates = i;

    /* Copy the ad-hoc creating rates into Current BSS state structure */
    memcpy(&Adapter->CurBssParams.DataRates,
           &adhs->DataRate,
           Adapter->CurBssParams.NumOfRates);

    PRINTM(INFO, "ADHOC_S_CMD: Rates=%02x %02x %02x %02x \n",
           adhs->DataRate[0], adhs->DataRate[1],
           adhs->DataRate[2], adhs->DataRate[3]);

    PRINTM(INFO, "ADHOC_S_CMD: AD HOC Start command is ready\n");


    if (wlan_create_dnld_countryinfo_11d( priv )) {
        PRINTM(INFO, "ADHOC_S_CMD: dnld_countryinfo_11d failed\n");
        ret = WLAN_STATUS_FAILURE;
        goto done;
    }


    cmd->Size = wlan_cpu_to_le16(sizeof(struct HostCmd_DS_802_11_AD_HOC_START)
                                 + S_DS_GEN + cmdAppendSize);

    memcpy(&TmpCap, &adhs->Cap, sizeof(u16));
    TmpCap = wlan_cpu_to_le16(TmpCap);
    memcpy(&adhs->Cap, &TmpCap, sizeof(u16));

    ret = WLAN_STATUS_SUCCESS;
done:
    LEAVE();
    return ret;
}

/**
 *  @brief This function prepares command of ad_hoc_stop.
 *
 *  @param priv     A pointer to wlan_private structure
 *  @param cmd      A pointer to HostCmd_DS_COMMAND structure
 *  @return         WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_cmd_802_11_ad_hoc_stop(wlan_private* priv,
                                struct HostCmd_DS_COMMAND* cmd)
{
    cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_AD_HOC_STOP);
    cmd->Size = wlan_cpu_to_le16(sizeof(struct HostCmd_DS_802_11_AD_HOC_STOP)
                                 + S_DS_GEN);


    return WLAN_STATUS_SUCCESS;
}

/**
 *  @brief This function prepares command of ad_hoc_join.
 *
 *  @param priv      A pointer to wlan_private structure
 *  @param cmd       A pointer to HostCmd_DS_COMMAND structure
 *  @param pdata_buf Void cast of BSSDescriptor_t from the scan table to join
 *
 *  @return          WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_cmd_802_11_ad_hoc_join(wlan_private* priv,
                                struct HostCmd_DS_COMMAND* cmd,
                                void* pdata_buf)
{
    wlan_adapter*                  Adapter       = priv->adapter;
    struct HostCmd_DS_802_11_AD_HOC_JOIN* pAdHocJoin    = &cmd->params.adj;
    BSSDescriptor_t*               pBSSDesc      = (BSSDescriptor_t*)pdata_buf;
    int                            cmdAppendSize = 0;
    int                            ret           = WLAN_STATUS_SUCCESS;
    u8*                            card_rates;
    int                            card_rates_size;
    u16                            TmpCap;
    int                            i;


    ENTER();

    Adapter->pAttemptedBSSDesc = pBSSDesc;

    cmd->Command = wlan_cpu_to_le16(HostCmd_CMD_802_11_AD_HOC_JOIN);

    pAdHocJoin->BssDescriptor.BSSType = HostCmd_BSS_TYPE_IBSS;

    pAdHocJoin->BssDescriptor.BeaconPeriod = pBSSDesc->BeaconPeriod;

    memcpy(&pAdHocJoin->BssDescriptor.BSSID,
           &pBSSDesc->MacAddress,
           MRVDRV_ETH_ADDR_LEN);

    memcpy(&pAdHocJoin->BssDescriptor.SSID,
           &pBSSDesc->Ssid.Ssid,
           pBSSDesc->Ssid.SsidLength);

    memcpy(&pAdHocJoin->BssDescriptor.PhyParamSet,
           &pBSSDesc->PhyParamSet,
           sizeof(IEEEtypes_PhyParamSet_t));

    memcpy(&pAdHocJoin->BssDescriptor.SsParamSet,
           &pBSSDesc->SsParamSet,
           sizeof(IEEEtypes_SsParamSet_t));

    memcpy(&TmpCap, &pBSSDesc->Cap, sizeof(IEEEtypes_CapInfo_t));

    TmpCap &= CAPINFO_MASK;

    PRINTM(INFO, "ADHOC_J_CMD: TmpCap=%4X CAPINFO_MASK=%4X\n",
           TmpCap, CAPINFO_MASK);
    memcpy(&pAdHocJoin->BssDescriptor.Cap, &TmpCap,
           sizeof(IEEEtypes_CapInfo_t));

    /* information on BSSID descriptor passed to FW */
    PRINTM(INFO, "ADHOC_J_CMD: BSSID = %2x-%2x-%2x-%2x-%2x-%2x, SSID = %s\n",
           pAdHocJoin->BssDescriptor.BSSID[0],
           pAdHocJoin->BssDescriptor.BSSID[1],
           pAdHocJoin->BssDescriptor.BSSID[2],
           pAdHocJoin->BssDescriptor.BSSID[3],
           pAdHocJoin->BssDescriptor.BSSID[4],
           pAdHocJoin->BssDescriptor.BSSID[5],
           pAdHocJoin->BssDescriptor.SSID);

    PRINTM(INFO, "ADHOC_J_CMD: Data Rate = %x\n",
           (u32)pAdHocJoin->BssDescriptor.DataRates);

    /* FailTimeOut */
    pAdHocJoin->FailTimeOut = wlan_cpu_to_le16(MRVDRV_ASSOCIATION_TIME_OUT);

    /* ProbeDelay */
    pAdHocJoin->ProbeDelay = wlan_cpu_to_le16(HostCmd_SCAN_PROBE_DELAY_TIME);

    /* Copy Data Rates from the Rates recorded in scan response */
    memset(pAdHocJoin->BssDescriptor.DataRates, 0,
           sizeof(pAdHocJoin->BssDescriptor.DataRates));
    memcpy(pAdHocJoin->BssDescriptor.DataRates, pBSSDesc->DataRates,
           MIN(sizeof(pAdHocJoin->BssDescriptor.DataRates),
               sizeof(pBSSDesc->DataRates)));

    card_rates = SupportedRates;
    card_rates_size = sizeof(SupportedRates);

    Adapter->CurBssParams.channel = pBSSDesc->Channel;

    if (get_common_rates(Adapter, pAdHocJoin->BssDescriptor.DataRates,
                         sizeof(pAdHocJoin->BssDescriptor.DataRates),
                         card_rates,
                         card_rates_size)) {
        PRINTM(INFO, "ADHOC_J_CMD: get_common_rates returns error.\n");
        ret = WLAN_STATUS_FAILURE;
        goto done;
    }

    /* Find the last non zero */
    for (i=0; i < sizeof(pAdHocJoin->BssDescriptor.DataRates)
             && pAdHocJoin->BssDescriptor.DataRates[i]; i++)
        ;

    Adapter->CurBssParams.NumOfRates = i;

    /*
     * Copy the adhoc joining rates to Current BSS State structure
     */
    memcpy(Adapter->CurBssParams.DataRates,
           pAdHocJoin->BssDescriptor.DataRates,
           Adapter->CurBssParams.NumOfRates);

    pAdHocJoin->BssDescriptor.SsParamSet.IbssParamSet.AtimWindow =
        wlan_cpu_to_le16(pBSSDesc->ATIMWindow);

    if (Adapter->SecInfo.WEPStatus == Wlan802_11WEPEnabled
        ) {
        pAdHocJoin->BssDescriptor.Cap.Privacy = AD_HOC_CAP_PRIVACY_ON;
    }

    if (Adapter->PSMode == Wlan802_11PowerModeMAX_PSP) {
        /* wake up first */
        WLAN_802_11_POWER_MODE LocalPSMode;

        LocalPSMode = Wlan802_11PowerModeCAM;
        ret = PrepareAndSendCommand(priv,
                                    HostCmd_CMD_802_11_PS_MODE,
                                    HostCmd_ACT_GEN_SET,
                                    0, 0, &LocalPSMode);

        if (ret) {
            ret = WLAN_STATUS_FAILURE;
            goto done;
        }
    }


    if (wlan_parse_dnld_countryinfo_11d( priv )) {
        ret = WLAN_STATUS_FAILURE;
        goto done;
    }


    cmd->Size = wlan_cpu_to_le16(sizeof(struct HostCmd_DS_802_11_AD_HOC_JOIN)
                                 + S_DS_GEN + cmdAppendSize);

    memcpy(&TmpCap, &pAdHocJoin->BssDescriptor.Cap,
           sizeof(IEEEtypes_CapInfo_t));
    TmpCap = wlan_cpu_to_le16(TmpCap);

    memcpy(&pAdHocJoin->BssDescriptor.Cap,
           &TmpCap,
           sizeof(IEEEtypes_CapInfo_t));

done:
    LEAVE();
    return ret;
}


/**
 *  @brief This function handles the command response of authenticate
 *
 *  @param priv    A pointer to wlan_private structure
 *  @param resp    A pointer to HostCmd_DS_COMMAND
 *  @return        WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_ret_802_11_authenticate(wlan_private* priv,
                                 struct HostCmd_DS_COMMAND* resp)
{
    return WLAN_STATUS_SUCCESS;
}


/**
 *  @brief This function handles the command response of associate
 *
 *  @param priv    A pointer to wlan_private structure
 *  @param resp    A pointer to HostCmd_DS_COMMAND
 *  @return        WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_ret_802_11_associate(wlan_private* priv,
                              struct HostCmd_DS_COMMAND* resp)
{
    wlan_adapter*           Adapter = priv->adapter;
    int                     ret     = WLAN_STATUS_SUCCESS;
    union iwreq_data        wrqu;
    IEEEtypes_AssocRsp_t*   pAssocRsp;
    BSSDescriptor_t*        pBSSDesc;

    ENTER();

    pAssocRsp = (IEEEtypes_AssocRsp_t*)&resp->params;

    if (pAssocRsp->StatusCode) {

        if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
            MacEventDisconnected(priv);
        }

        PRINTM(INFO, "ASSOC_RESP: Association Failed, status code = %d\n",
               pAssocRsp->StatusCode);

        ret = WLAN_STATUS_FAILURE;
        goto done;
    }

    HEXDUMP("ASSOC_RESP:", (void*)&resp->params,
            wlan_le16_to_cpu(resp->Size) - S_DS_GEN);


    /* Send a Media Connected event, according to the Spec */
    Adapter->MediaConnectStatus = WlanMediaStateConnected;
    Adapter->LinkSpeed = MRVDRV_LINK_SPEED_11mbps;

    /* Set the attempted BSSID Index to current */
    pBSSDesc = Adapter->pAttemptedBSSDesc;

    PRINTM(INFO, "ASSOC_RESP: %s\n", pBSSDesc->Ssid.Ssid);

    /* Set the new SSID to current SSID */
    memcpy(&Adapter->CurBssParams.ssid,
           &pBSSDesc->Ssid,
           sizeof(struct WLAN_802_11_SSID));

    /* Set the new BSSID (AP's MAC address) to current BSSID */
    memcpy(Adapter->CurBssParams.bssid,
           pBSSDesc->MacAddress,
           MRVDRV_ETH_ADDR_LEN);

    /* Make a copy of current BSSID descriptor */
    memcpy(&Adapter->CurBssParams.BSSDescriptor,
           pBSSDesc,
           sizeof(BSSDescriptor_t));



    PRINTM(INFO, "ASSOC_RESP: CurrentPacketFilter is %x\n",
           Adapter->CurrentPacketFilter);

    Adapter->MediaConnectStatus = WlanMediaStateConnected;

    if (Adapter->SecInfo.WPAEnabled
        || Adapter->SecInfo.WPA2Enabled
        )
        Adapter->IsGTK_SET = FALSE;

    Adapter->SNR[TYPE_RXPD][TYPE_AVG] = 0;
    Adapter->NF[TYPE_RXPD][TYPE_AVG] = 0;

    memset(Adapter->rawSNR, 0x00, sizeof(Adapter->rawSNR));
    memset(Adapter->rawNF, 0x00, sizeof(Adapter->rawNF));
    Adapter->nextSNRNF = 0;
    Adapter->numSNRNF = 0;


    {
        os_carrier_on(priv);
        os_start_queue(priv);
    }

    PRINTM(INFO, "ASSOC_RESP: Associated \n");

    memcpy(wrqu.ap_addr.sa_data, Adapter->CurBssParams.bssid, ETH_ALEN);
    wrqu.ap_addr.sa_family = ARPHRD_ETHER;
    wireless_send_event(priv->wlan_dev.netdev, SIOCGIWAP, &wrqu, NULL);

done:
    LEAVE();
    return ret;
}

/**
 *  @brief This function handles the command response of disassociate
 *
 *  @param priv    A pointer to wlan_private structure
 *  @param resp    A pointer to HostCmd_DS_COMMAND
 *  @return        WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_ret_802_11_disassociate(wlan_private* priv,
                                 struct HostCmd_DS_COMMAND* resp)
{
    ENTER();

    MacEventDisconnected(priv);

    LEAVE();
    return WLAN_STATUS_SUCCESS;
}

/**
 *  @brief This function handles the command response of ad_hoc_start
 *
 *  @param priv    A pointer to wlan_private structure
 *  @param resp    A pointer to HostCmd_DS_COMMAND
 *  @return        WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_ret_802_11_ad_hoc_start(wlan_private* priv,
                                 struct HostCmd_DS_COMMAND* resp)
{
    wlan_adapter*                    Adapter = priv->adapter;
    int                              ret     = WLAN_STATUS_SUCCESS;
    u16                              Command = wlan_le16_to_cpu(resp->Command);
    u16                              Result  = wlan_le16_to_cpu(resp->Result);
    struct HostCmd_DS_802_11_AD_HOC_RESULT* pAdHocResult;
    union iwreq_data                 wrqu;
    BSSDescriptor_t*                 pBSSDesc;

    ENTER();

    pAdHocResult = &resp->params.result;

    PRINTM(INFO, "ADHOC_S_RESP: Size = %d\n", wlan_le16_to_cpu(resp->Size));
    PRINTM(INFO, "ADHOC_S_RESP: Command = %x\n", Command);
    PRINTM(INFO, "ADHOC_S_RESP: Result = %x\n", Result);

    pBSSDesc = Adapter->pAttemptedBSSDesc;

    /*
     * Join result code 0 --> SUCCESS
     */
    if (Result) {
        PRINTM(INFO, "ADHOC_RESP Failed\n");
        if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
            MacEventDisconnected(priv);
        }

        memset(&Adapter->CurBssParams.BSSDescriptor,
               0x00,
               sizeof(Adapter->CurBssParams.BSSDescriptor));

        Adapter->AdHocFailed = TRUE;
        LEAVE();
        return WLAN_STATUS_FAILURE;
    }

    /*
     * Now the join cmd should be successful
     * If BSSID has changed use SSID to compare instead of BSSID
     */
    PRINTM(INFO, "ADHOC_J_RESP  %s\n", pBSSDesc->Ssid.Ssid);

    /* Send a Media Connected event, according to the Spec */
    Adapter->MediaConnectStatus = WlanMediaStateConnected;
    Adapter->LinkSpeed = MRVDRV_LINK_SPEED_11mbps;

    if (Command == HostCmd_RET_802_11_AD_HOC_START) {
        Adapter->AdHocCreated = TRUE;

        /* Update the created network descriptor with the new BSSID */
        memcpy(pBSSDesc->MacAddress,
               pAdHocResult->BSSID,
               MRVDRV_ETH_ADDR_LEN);
    } else {

        /* Make a copy of current BSSID descriptor, only needed for join since
         *   the current descriptor is already being used for adhoc start
         */
        memmove(&Adapter->CurBssParams.BSSDescriptor,
                pBSSDesc,
                sizeof(BSSDescriptor_t));
    }

    /* Set the BSSID from the joined/started descriptor */
    memcpy(&Adapter->CurBssParams.bssid,
           pBSSDesc->MacAddress,
           MRVDRV_ETH_ADDR_LEN);

    /* Set the new SSID to current SSID */
    memcpy(&Adapter->CurBssParams.ssid,
           &pBSSDesc->Ssid,
           sizeof(struct WLAN_802_11_SSID));

    os_carrier_on(priv);
    os_start_queue(priv);

    memset(&wrqu, 0, sizeof(wrqu));
    memcpy(wrqu.ap_addr.sa_data, Adapter->CurBssParams.bssid, ETH_ALEN);
    wrqu.ap_addr.sa_family = ARPHRD_ETHER;
    wireless_send_event(priv->wlan_dev.netdev, SIOCGIWAP, &wrqu, NULL);


    PRINTM(INFO, "ADHOC_RESP: - Joined/Started Ad Hoc\n");
    PRINTM(INFO, "ADHOC_RESP: Channel = %d\n", Adapter->AdhocChannel);
    PRINTM(INFO, "ADHOC_RESP: BSSID = %02x:%02x:%02x:%02x:%02x:%02x\n",
           pAdHocResult->BSSID[0], pAdHocResult->BSSID[1],
           pAdHocResult->BSSID[2], pAdHocResult->BSSID[3],
           pAdHocResult->BSSID[4], pAdHocResult->BSSID[5]);

    LEAVE();
    return ret;
}


/**
 *  @brief This function handles the command response of ad_hoc_stop
 *
 *  @param priv    A pointer to wlan_private structure
 *  @param resp    A pointer to HostCmd_DS_COMMAND
 *  @return        WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_ret_802_11_ad_hoc_stop(wlan_private* priv,
                                struct HostCmd_DS_COMMAND* resp)
{
    ENTER();

    MacEventDisconnected(priv);

    LEAVE();
    return WLAN_STATUS_SUCCESS;
}


#ifdef REASSOCIATION
/**
 *  @brief This function handles re-association. it is triggered
 *  by re-assoc timer.
 *
 *  @param data    A pointer to wlan_thread structure
 *  @return        WLAN_STATUS_SUCCESS
 */
int wlan_reassociation_thread(void *data)
{
    wlan_thread*  thread  = data;
    wlan_private* priv    = thread->priv;
    wlan_adapter* Adapter = priv->adapter;
    wait_queue_t  wait;
    int           i;

    ENTER();

    wlan_activate_thread(thread);
    init_waitqueue_entry(&wait, current);

    for (;;) {
        add_wait_queue(&thread->waitQ, &wait);
        OS_SET_THREAD_STATE(TASK_INTERRUPTIBLE);

        PRINTM(INFO, "Reassoc: Thread sleeping...\n");

        schedule();

        OS_SET_THREAD_STATE(TASK_RUNNING);
        remove_wait_queue(&thread->waitQ, &wait);

        if (Adapter->SurpriseRemoved) {
            break;
        }

        if (kthread_should_stop()) {
            break;
        }

        PRINTM(INFO, "Reassoc: Thread waking up...\n");

        if (Adapter->InfrastructureMode != Wlan802_11Infrastructure) {
            PRINTM(MSG, "Reassoc: non infra mode is not supported\n");
            continue;
        }

        /* The semaphore is used to avoid reassociation thread and 
           wlan_set_scan/wlan_set_essid interrupting each other.
           Reassociation should be disabled completely by application if 
           wlan_set_user_scan_ioctl/wlan_set_wap is used.
        */
        if (down_interruptible(&Adapter->ReassocSem)) {
            PRINTM(FATAL, "Acquire semaphore error, reassociation thread\n");
            goto settimer;
        }

        if (Adapter->MediaConnectStatus != WlanMediaStateDisconnected) {
            up(&Adapter->ReassocSem);
            PRINTM(MSG, "Reassoc: Adapter->MediaConnectStatus is wrong\n");
            continue;
        }

        PRINTM(INFO, "Reassoc: Required ESSID: %s\n",
               Adapter->PreviousSSID.Ssid);

        PRINTM(INFO, "Reassoc: Performing Active Scan @ %lu\n", jiffies);

        SendSpecificSSIDScan(priv, &Adapter->PreviousSSID, TRUE);

        /* Try to find the specific BSSID we were associated to first */
        i = FindSSIDInList(Adapter,
                           &Adapter->PreviousSSID,
                           Adapter->PreviousBSSID,
                           Adapter->InfrastructureMode);

        if (i < 0) {
            /* If the BSSID could not be found, try just the SSID */
            i = FindSSIDInList(Adapter,
                               &Adapter->PreviousSSID,
                               NULL,
                               Adapter->InfrastructureMode);
        }

        if (i >= 0) {
            wlan_associate(priv, &Adapter->ScanTable[i]);
        }

        up(&Adapter->ReassocSem);

settimer:
        if (Adapter->MediaConnectStatus == WlanMediaStateDisconnected) {
            PRINTM(INFO, "Reassoc: No AP found or assoc failed."
                   "Restarting re-assoc Timer @ %lu\n", jiffies);

            Adapter->TimerIsSet = TRUE;
            ModTimer(&Adapter->MrvDrvTimer, 10 * 1000); /* 10s (in ms) */
        }
    }

    wlan_deactivate_thread(thread);

    LEAVE();
    return WLAN_STATUS_SUCCESS;
}
#endif /* REASSOCIATION */
