/** @file wlan_rx.c
  * @brief This file contains the handling of RX in wlan driver.
  * 
  * (c) Copyright  2003-2006, Marvell International Ltd. 
  * All Rights Reserved
  *
  * This software file (the "File") is distributed by Marvell International 
  * Ltd. under the terms of the GNU General Public License Version 2, June 1991 
  * (the "License").  You may use, redistribute and/or modify this File in 
  * accordance with the terms and conditions of the License, a copy of which 
  * is available along with the File in the license.txt file or by writing to 
  * the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 
  * 02111-1307 or on the worldwide web at http://www.gnu.org/licenses/gpl.txt.
  *
  * THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE 
  * IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE 
  * ARE EXPRESSLY DISCLAIMED.  The License provides additional details about 
  * this warranty disclaimer.
  *
  */
/********************************************************
Change log:
	09/28/05: Add Doxygen format comments
	12/09/05: ADD Sliding window SNR/NF Average Calculation support
	
********************************************************/

#include	"include.h"

/********************************************************
		Local Variables
********************************************************/

typedef struct
{
    u8  dest_addr[6];
    u8  src_addr[6];
    u16 h803_len;

} __attribute__ ((packed)) Eth803Hdr_t;

typedef struct
{
    u8  llc_dsap;
    u8  llc_ssap;
    u8  llc_ctrl;
    u8  snap_oui[3];
    u16 snap_type;

} __attribute__ ((packed)) Rfc1042Hdr_t;

typedef struct
{
    struct RxPD         rx_pd;
    Eth803Hdr_t  eth803_hdr;
    Rfc1042Hdr_t rfc1042_hdr;

} __attribute__ ((packed)) RxPacketHdr_t;

typedef struct
{
    u8  dest_addr[6];
    u8  src_addr[6];
    u16 ethertype;

} __attribute__ ((packed)) EthII_Hdr_t; 

typedef struct
{
    struct RxPD         rx_pd;
    void*        eth80211_hdr;

} __attribute__ ((packed)) Rx80211PacketHdr_t;

typedef struct
{
    struct ieee80211_radiotap_header hdr;
    u8           flags;
    u8           rate;
    u16          chan_freq;
    u16          chan_flags;
    u8           antenna;
    u8           antsignal;
    u16          rx_flags;
    //u8           pad[IEEE80211_RADIOTAP_HDRLEN - 18];
} __attribute__ ((packed)) RxRadiotapHdr_t;

#define RX_RADIOTAP_PRESENT (   \
		(1 << IEEE80211_RADIOTAP_FLAGS)   | \
		(1 << IEEE80211_RADIOTAP_RATE)    | \
		(1 << IEEE80211_RADIOTAP_CHANNEL) | \
		(1 << IEEE80211_RADIOTAP_ANTENNA) | \
		(1 << IEEE80211_RADIOTAP_DB_ANTSIGNAL)  | \
		(1 << IEEE80211_RADIOTAP_RX_FLAGS)  | \
		0)

/********************************************************
		Global Variables
********************************************************/

/********************************************************
		Local Functions
********************************************************/

/** 
 *  @brief This function computes the AvgSNR .
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   AvgSNR
 */
static u8 wlan_getAvgSNR(wlan_private * priv)
{
	u8 i;
	u16 temp = 0;
	wlan_adapter   *Adapter = priv->adapter;
	if(Adapter->numSNRNF == 0)
		return 0;
	for (i = 0; i <Adapter->numSNRNF; i++)
		temp += Adapter->rawSNR[i];
	return (u8)(temp/Adapter->numSNRNF);
	
}

/** 
 *  @brief This function computes the AvgNF
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   AvgNF
 */
static u8 wlan_getAvgNF(wlan_private * priv)
{
	u8 i;
	u16 temp = 0;
	wlan_adapter   *Adapter = priv->adapter;
	if(Adapter->numSNRNF == 0)
		return 0;
	for (i = 0; i <Adapter->numSNRNF; i++)
		temp += Adapter->rawNF[i];
	return (u8)(temp/Adapter->numSNRNF);
	
}

/** 
 *  @brief This function save the raw SNR/NF to our internel buffer
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param pRxPD   A pointer to RxPD structure of received packet
 *  @return 	   n/a
 */
static void wlan_save_rawSNRNF(wlan_private * priv, struct RxPD * pRxPD)
{
	wlan_adapter   *Adapter = priv->adapter;
	if(Adapter->numSNRNF < Adapter->data_avg_factor)
		Adapter->numSNRNF++;
	Adapter->rawSNR[Adapter->nextSNRNF] = pRxPD->SNR;
	Adapter->rawNF[Adapter->nextSNRNF] = pRxPD->NF;
	Adapter->nextSNRNF++;
	if(Adapter->nextSNRNF >= Adapter->data_avg_factor)
		Adapter->nextSNRNF = 0;
	return;
}


/** 
 *  @brief This function computes the RSSI in received packet.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param pRxPD   A pointer to RxPD structure of received packet
 *  @return 	   n/a
 */
static void wlan_compute_rssi(wlan_private *priv, struct RxPD *pRxPD)
{
	wlan_adapter   *Adapter = priv->adapter;

	ENTER();

	PRINTM(INFO, "RxPD: SNR = %d, NF = %d\n", pRxPD->SNR, pRxPD->NF);
	PRINTM(INFO, "Before computing SNR: SNR- avg = %d, NF-avg = %d\n",
		Adapter->SNR[TYPE_RXPD][TYPE_AVG] / AVG_SCALE, 
		Adapter->NF[TYPE_RXPD][TYPE_AVG] / AVG_SCALE);

	Adapter->SNR[TYPE_RXPD][TYPE_NOAVG] = pRxPD->SNR;
	Adapter->NF[TYPE_RXPD][TYPE_NOAVG] = pRxPD->NF;
	wlan_save_rawSNRNF(priv , pRxPD);

	Adapter->RxPDSNRAge = jiffies;
	Adapter->RxPDRate = pRxPD->RxRate;
	
	Adapter->SNR[TYPE_RXPD][TYPE_AVG] = wlan_getAvgSNR(priv) * AVG_SCALE;
	Adapter->NF[TYPE_RXPD][TYPE_AVG] = wlan_getAvgNF(priv ) * AVG_SCALE;
	PRINTM(INFO, "After computing SNR: SNR-avg = %d, NF-avg = %d\n",
		Adapter->SNR[TYPE_RXPD][TYPE_AVG] / AVG_SCALE, 
		Adapter->NF[TYPE_RXPD][TYPE_AVG] / AVG_SCALE);

	Adapter->RSSI[TYPE_RXPD][TYPE_NOAVG] = 
		CAL_RSSI(Adapter->SNR[TYPE_RXPD][TYPE_NOAVG],
			Adapter->NF[TYPE_RXPD][TYPE_NOAVG]);

	Adapter->RSSI[TYPE_RXPD][TYPE_AVG] = 
		CAL_RSSI(Adapter->SNR[TYPE_RXPD][TYPE_AVG] / AVG_SCALE,
			Adapter->NF[TYPE_RXPD][TYPE_AVG] / AVG_SCALE);
	
	LEAVE();
}

/********************************************************
		Global functions
********************************************************/

/**
 *  @brief This function processes received packet and forwards it
 *  to kernel/upper layer
 *  
 *  @param priv    A pointer to wlan_private
 *  @param skb     A pointer to skb which includes the received packet
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int ProcessRxedPacket(wlan_private *priv, struct sk_buff *skb)
{
	wlan_adapter *Adapter = priv->adapter;
	int          ret = WLAN_STATUS_SUCCESS;

	RxPacketHdr_t* pRxPkt;
	struct RxPD*   pRxPD;

	int            hdrChop;
	EthII_Hdr_t*   pEthHdr;

	
	const u8 rfc1042_eth_hdr[] = { 0xaa, 0xaa, 0x03, 0x00, 0x00, 0x00 };

	ENTER();

	if ( priv->adapter->debugmode & MRVDRV_DEBUG_RX_PATH )
		HEXDUMP("RX packet: ", skb->data, MIN(skb->len, 100));

	if ( priv->adapter->linkmode == WLAN_LINKMODE_802_11 )
		return ProcessRxed_802_11_Packet(priv, skb);

	pRxPkt = (RxPacketHdr_t*)skb->data;
	pRxPD  = &pRxPkt->rx_pd;

 	HEXDUMP("RX Data: Before chop RxPD", skb->data, MIN(skb->len, 100));


		if (skb->len < (ETH_HLEN + 8 + sizeof(struct RxPD))) {
			PRINTM(INFO, "RX Error: FRAME RECEIVED WITH BAD LENGTH\n");
			priv->stats.rx_length_errors++;
			ret = WLAN_STATUS_SUCCESS;
			goto done;
		}

	/* 
	 * Check RxPD status and update 802.3 stat,
	 */
	if (!(pRxPD->Status & MRVDRV_RXPD_STATUS_OK)) {
		PRINTM(INFO, "RX Error: frame received with bad status\n");
		printk("<1> RxPD Not OK\n");
		priv->stats.rx_errors++;
		ret = WLAN_STATUS_SUCCESS;
		goto done;
	}

	
	PRINTM(INFO, "RX Data: skb->len - sizeof(RxPd) = %d - %d = %d\n",
	       skb->len, sizeof(struct RxPD), skb->len - sizeof(struct RxPD));

	HEXDUMP("RX Data: Dest", pRxPkt->eth803_hdr.dest_addr,
            sizeof(pRxPkt->eth803_hdr.dest_addr));
	HEXDUMP("RX Data: Src", pRxPkt->eth803_hdr.src_addr,
            sizeof(pRxPkt->eth803_hdr.src_addr));

	if (memcmp(&pRxPkt->rfc1042_hdr, 
               rfc1042_eth_hdr, sizeof(rfc1042_eth_hdr)) == 0) {
        /* 
         *  Replace the 803 header and rfc1042 header (llc/snap) with an 
         *    EthernetII header, keep the src/dst and snap_type (ethertype)
         *
         *  The firmware only passes up SNAP frames converting
         *    all RX Data from 802.11 to 802.2/LLC/SNAP frames.
         *
         *  To create the Ethernet II, just move the src, dst address right
         *    before the snap_type.
         */
        pEthHdr = (EthII_Hdr_t*)
            ((u8*)&pRxPkt->eth803_hdr
             + sizeof(pRxPkt->eth803_hdr) + sizeof(pRxPkt->rfc1042_hdr)
             - sizeof(pRxPkt->eth803_hdr.dest_addr)
             - sizeof(pRxPkt->eth803_hdr.src_addr)
             - sizeof(pRxPkt->rfc1042_hdr.snap_type));
        
        memcpy(pEthHdr->src_addr, pRxPkt->eth803_hdr.src_addr,
               sizeof(pEthHdr->src_addr));
        memcpy(pEthHdr->dest_addr, pRxPkt->eth803_hdr.dest_addr,
               sizeof(pEthHdr->dest_addr));

        /* Chop off the RxPD + the excess memory from the 802.2/llc/snap header
         *   that was removed 
         */
        hdrChop = (u8*)pEthHdr - (u8*)pRxPkt;
	} else {
        HEXDUMP("RX Data: LLC/SNAP", 
                (u8*)&pRxPkt->rfc1042_hdr, sizeof(pRxPkt->rfc1042_hdr));

        /* Chop off the RxPD */
        hdrChop = (u8*)&pRxPkt->eth803_hdr - (u8*)pRxPkt;
    }
    
    /* Chop off the leading header bytes so the skb points to the start of 
     *   either the reconstructed EthII frame or the 802.2/llc/snap frame
     */
	skb_pull(skb, hdrChop);

	/* Take the data rate from the RxPD structure 
	 * only if the rate is auto
	 */
	if (Adapter->Is_DataRate_Auto) {
		Adapter->DataRate = index_to_data_rate(pRxPD->RxRate);
	}

	wlan_compute_rssi(priv, pRxPD);

		PRINTM(INFO, "RX Data: Size of actual packet = %d\n", skb->len);
		if (os_upload_rx_packet(priv, skb)) {
			PRINTM(INFO, "RX Error: os_upload_rx_packet"
							" returns failure\n");
			ret = WLAN_STATUS_FAILURE;
			goto done;
		}
 		priv->stats.rx_bytes += skb->len; 
		priv->stats.rx_packets++;
	
	ret = WLAN_STATUS_SUCCESS;
done:
	LEAVE();

	return (ret);
}

/**
 *  @brief This function converts Tx/Rx rates from the Marvell WLAN format
 *  (see Table 2 in Section 3.1) to IEEE80211_RADIOTAP_RATE units (500 Kb/s)
 *  
 *  @param rate    Input rate
 *  @return 	   Output Rate (0 if invalid)
 */
u8 convert_mv_rate_to_radiotap(u8 rate)
{
	switch (rate) {
		case  0: /*   1 Mbps */ return 2; 
		case  1: /*   2 Mbps */ return 4; 
		case  2: /* 5.5 Mbps */ return 11; 
		case  3: /*  11 Mbps */ return 22; 
		case  4: /*   6 Mbps */ return 12; 
		case  5: /*   9 Mbps */ return 18; 
		case  6: /*  12 Mbps */ return 24; 
		case  7: /*  18 Mbps */ return 36; 
		case  8: /*  24 Mbps */ return 48; 
		case  9: /*  36 Mbps */ return 72; 
		case 10: /*  48 Mbps */ return 96; 
		case 11: /*  54 Mbps */ return 108; 
	}
	PRINTM(MSG, "Invalid Marvell WLAN rate (%i)\n", rate);
	return 0;
}

/**
 *  @brief This function processes a received 802.11 packet and forwards it
 *  to kernel/upper layer
 *  
 *  @param priv    A pointer to wlan_private
 *  @param skb     A pointer to skb which includes the received packet
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int ProcessRxed_802_11_Packet(wlan_private *priv, struct sk_buff *skb)
{
	wlan_adapter *Adapter = priv->adapter;
	int          ret = WLAN_STATUS_SUCCESS;

	Rx80211PacketHdr_t* pRxPkt;
	struct RxPD*          pRxPD;
	RxRadiotapHdr_t radiotap_hdr;
	RxRadiotapHdr_t *pradiotap_hdr;

	ENTER();

	pRxPkt = (Rx80211PacketHdr_t*)skb->data;
	pRxPD  = &pRxPkt->rx_pd;

 	// HEXDUMP("RX Data: Before chop RxPD", skb->data, MIN(skb->len, 100));

	if (skb->len < (ETH_HLEN + 8 + sizeof(struct RxPD))) {
		PRINTM(INFO, "RX Error: FRAME RECEIVED WITH BAD LENGTH\n");
		priv->stats.rx_length_errors++;
		ret = WLAN_STATUS_SUCCESS;
		goto done;
	}

	/* 
	 * Check RxPD status and update 802.3 stat,
	 */
	if (!(pRxPD->Status & MRVDRV_RXPD_STATUS_OK)) {
		//PRINTM(INFO, "RX Error: frame received with bad status\n");
		priv->stats.rx_errors++;
	}

	PRINTM(INFO, "RX Data: skb->len - sizeof(RxPd) = %d - %d = %d\n",
	       skb->len, sizeof(struct RxPD), skb->len - sizeof(struct RxPD));

	/* create the exported radio header */
	switch (priv->adapter->radiomode) {
		case WLAN_RADIOMODE_NONE:
			/* no radio header */
			/* chop the RxPD */
			skb_pull(skb, sizeof(struct RxPD));
			break;

		case WLAN_RADIOMODE_RADIOTAP:
			/* radiotap header */
			radiotap_hdr.hdr.it_version = 0;
			/* XXX must check this value for pad */
			radiotap_hdr.hdr.it_pad = 0;
			radiotap_hdr.hdr.it_len = sizeof(RxRadiotapHdr_t);
			radiotap_hdr.hdr.it_present = RX_RADIOTAP_PRESENT;
			/* unknown values */
			radiotap_hdr.flags = 0;
			radiotap_hdr.chan_freq = 0;
			radiotap_hdr.chan_flags = 0;
			radiotap_hdr.antenna = 0;
			/* known values */
			radiotap_hdr.rate = convert_mv_rate_to_radiotap(pRxPD->RxRate);
			/* XXX must check no carryout */
			radiotap_hdr.antsignal = pRxPD->SNR + pRxPD->NF;
			radiotap_hdr.rx_flags = 0;
			if (!(pRxPD->Status & MRVDRV_RXPD_STATUS_OK))
				radiotap_hdr.rx_flags |= IEEE80211_RADIOTAP_F_RX_BADFCS;
			//memset(radiotap_hdr.pad, 0x11, IEEE80211_RADIOTAP_HDRLEN - 18);

			// HEXDUMP1("RX radiomode packet BEF: ", skb->data, MIN(skb->len, 100));

			/* chop the RxPD */
			skb_pull(skb, sizeof(struct RxPD));

			/* add space for the new radio header */
			if ( (skb_headroom(skb) < sizeof(RxRadiotapHdr_t)) &&
					pskb_expand_head(skb, sizeof(RxRadiotapHdr_t), 0, GFP_ATOMIC)) {
				PRINTM(MSG, "%s: couldn't pskb_expand_head\n", __func__);
			}


			pradiotap_hdr = (RxRadiotapHdr_t *) skb_push(skb, sizeof(RxRadiotapHdr_t));
			memcpy(pradiotap_hdr, &radiotap_hdr, sizeof(RxRadiotapHdr_t));
			//HEXDUMP1("RX radiomode packet AFT: ", skb->data, MIN(skb->len, 100));
			break;

		default:
			/* unknown header */
			PRINTM(MSG, "Unknown radiomode (%i)\n", priv->adapter->radiomode);
			/* don't export any header */
			/* chop the RxPD */
			skb_pull(skb, sizeof(struct RxPD));
			break;
	}

	/* Take the data rate from the RxPD structure 
	 * only if the rate is auto
	 */
	if (Adapter->Is_DataRate_Auto) {
		Adapter->DataRate = index_to_data_rate(pRxPD->RxRate);
	}

	wlan_compute_rssi(priv, pRxPD);

	PRINTM(INFO, "RX Data: Size of actual packet = %d\n", skb->len);

	if (os_upload_rx_packet(priv, skb)) {
		PRINTM(INFO, "RX Error: os_upload_rx_packet returns failure\n");
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

 	priv->stats.rx_bytes += skb->len; 
	priv->stats.rx_packets++;
	
	ret = WLAN_STATUS_SUCCESS;
done:
	LEAVE();

	skb->protocol = __constant_htons(0x0019);  /* ETH_P_80211_RAW */

	return (ret);
}
