#!/usr/local/bin/perl
#  recur.pl - Manage and update recurring transactions in a .cbb file.
#
#  Written by Curtis Olson.  Started January 16, 1996.
#
#  Copyright (C) 1996  Curtis L. Olson  - curt@sledge.mn.org
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# $Id: recur.pl,v 1.3 1996/02/11 20:25:23 curt Exp $
# (Log is kept at end of this file)


# specify the installed location of the necessary pieces.
$cbb_incl_dir = "..";
unshift(@INC, $cbb_incl_dir);

require "categories.pl";
require "common.pl";
require "engine.pl";
require "timelocal.pl";


($#ARGV >= 0) || 
    die "Usage: $0 account";

# how many days to plan ahead
$future_days = 366;		# approximately 1 year;
$secs_per_day = 86400;		# seconds per day;

$cur_date = &raw_date;

$account = shift(@ARGV);
(&load_trans($account) eq "ok") || die "Cannot open account:  $account";

#-----------------------------------------------------------------------
# Traverse all transactions and perform the following steps on entries with
# $cleared = "?":
# 
#   1.  If date has passed, change $cleared to "!"
#   2.  If date is present or future, delete entry.  These entries will be
#       reinserted later.
#-----------------------------------------------------------------------

print "Updating/deleting current recurring transactions:  ";

$result = &first_trans();
while ( $result ne "none" ) {
    ($key, $date, $check, $desc, $debit, $credit, $cat, $com, $cleared,
     $total) = split(/\t/, $result);

    if ( $cleared eq "?" ) {
	if ( $date < $cur_date ) {
	    # set $cleared to "!"
	    # print "updating - $result\n";
	    print ".";
	    $update = "$key\t$date\t$check\t$desc\t$debit\t$credit\t".
		"$cat\t$com\t!\t$total";
	    &update_trans($update);
	} else {
	    # delete
	    # print "deleting - $result\n";
	    print ".";
	    &delete_trans($key);
	}
    }
    
    $result = &next_trans();
}

print "\n";


#-----------------------------------------------------------------------
# Now add in all future recurring transactions
#-----------------------------------------------------------------------

print "Adding in future recurring transactions:  ";

# print "$account - " . &file_root($account) . "\n";
$recur = &file_root($account) . ".rcr";	# 
# print "$recur\n";		# 
open(RECUR, "<$recur") || die "Cannot open:  $recur";

while ( <RECUR> ) {
    # print length($_) . " - $_";
    if ( m/^#/ || ! m/\t/ ) {
	# ignore this line
    } else {
	# Ok, we found one!
	&add_all_recurs($_);
    }
}

close(RECUR);

# Finally, save the result

(&save_trans("$account") eq "ok") || die "Cannot save account:  $account";


#-----------------------------------------------------------------------
# Supporting Routines
#-----------------------------------------------------------------------

# Add all recuring transactions specified

sub add_all_recurs {
    local($line) = @_;

    chop($line);

    ($days, $months, $years, $desc, $debit, $credit, $com, $cat,
     $begindate, $cutoff) = split(/\t/, $line);

    if ( $begindate eq "" ) {
	$begindate = $cur_date;
    }

    if ( $cutoff eq "" ) {
	$cutoff = &calc_cutoff();
    }

    # print "cutoff = $cutoff\n";

    if ( ($days < 32) || ($days =~ m/,/) || ($days eq "*") ) {
	# type 1 recurring transaction
	$dates = &gen_dates_1($days, $months, $years, $begindate, $cutoff);
	# print "$dates\n";
    } else {
	# type 2 recurring transaction
	$dates = &gen_dates_2($days, $months, $years, $begindate, $cutoff);
	# print "$dates\n";
    }

    @DATES = split(/,/, $dates);

    foreach $date (@DATES) {
	print ".";
	$trans = "$date\t\t$desc\t$debit\t$credit\t$cat\t$com\t?\t";
	$key = &create_trans($trans);
    }
}

print "\n";


# Calculate cutoff date

sub calc_cutoff {
    local($csec,$cmin,$chour,$cmday,$cmon,$cyear,$cwday,$cyday,$cisdst) = 
        localtime(time);

    $today_secs = &timelocal(0, 0, 0, $cmday, $cmon, $cyear);
    $cutoff_secs = $today_secs + ($future_days * $secs_per_day);

    ($csec,$cmin,$chour,$cmday,$cmon,$cyear,$cwday,$cyday,$cisdst) = 
        localtime($cutoff_secs);

    return "19$cyear" . &pad($cmon + 1) . &pad($cmday);  
}


# Generate a list of type 1 dates

sub gen_dates_1 {
    local($days, $months, $years, $begindate, $cutoff) = @_;
    local($csec,$cmin,$chour,$cmday,$cmon,$cyear,$cwday,$cyday,$cisdst) = 
        localtime(time);
    local($ldates) = "";

    # print "$days - $months - $years\n";

    if ( $months eq "*" ) {
	$months = "1,2,3,4,5,6,7,8,9,10,11,12";
    }
    @MONTHS = split(/,/, $months);

    if ( $years eq "*" ) {
	$years = "$cyear," . ($cyear+1);
    }
    @YEARS = split(/,/, $years);

    foreach $year (@YEARS) {
	foreach $month (@MONTHS) {
	    # print "19$year" . &pad($month) . "\n";
	    if ( $month == 12 ) {
		$tyear = $year + 1; $tmonth = 1;
	    } else {
		$tyear = $year; $tmonth = $month + 1;
	    }
	    # note in perl the months start at 0 ... :(
	    $next_month = &timelocal(0, 0, 0, 1, ($tmonth - 1), $tyear);
	    # subtract the number of seconds in a day to get the last
	    # day of the previous month
	    $month_end = $next_month - $secs_per_day;
	    ($csec,$cmin,$chour,$cmday,$cmon,$cyear,$cwday,$cyday,$cisdst) = 
		localtime($month_end);

	    if ( $days eq "*" ) {
		$tdays = "1";
		$day = 2;
		while ( $day <= $cmday ) {
		    $tdays .= "," . $day;
		    $day++;
		}
	    } else {
		$tdays = $days;
		$tdays =~ s/last/$cmday/;
	    }

	    # print "$tdays\n";

	    @DAYS = split(/,/, $tdays);
	    foreach $day (@DAYS) {
		$this_date =  "19$year" . &pad($month) . &pad($day);
		if ( ($this_date >= $begindate) && ($this_date <= $cutoff) ) {
		    # print "$this_date\n";
		    if ( $ldates eq "" ) {
			$ldates = $this_date;
		    } else {
			$ldates .= "," . $this_date;
		    }
		}
	    }
	}
    }

    return $ldates;
}


# Generate a list of type 2 dates

sub gen_dates_2 {
    local($start, $incr, $junk, $begindate, $cutoff) = @_;
    local($ldates) = "";
    local($csec,$cmin,$chour,$cmday,$cmon,$cyear,$cwday,$cyday,$cisdst);
    local($junk, $syear, $smonth, $sday) = $start =~ /(\d\d)(\d\d)(\d\d)(\d\d)/;

    # print "$syear - $smonth - $sday\n";

    $start_secs = &timelocal(0, 0, 0, $sday, ($smonth - 1), $syear);
    $sincr = $incr * $secs_per_day;

    $secs = $start_secs;
    ($csec,$cmin,$chour,$cmday,$cmon,$cyear,$cwday,$cyday,$cisdst) = 
	localtime($secs);
    $this_date = "19$cyear" . &pad($cmon + 1) . &pad($cmday);
    while ( $this_date <= $cutoff ) {
	if ( $this_date >= $begindate ) {
	    if ( $ldates eq "" ) {
		$ldates = $this_date;
	    } else {
		$ldates .= "," . $this_date;
	    }
	    # print "$this_date\n";
	}
	$secs += $sincr;
	($csec,$cmin,$chour,$cmday,$cmon,$cyear,$cwday,$cyday,$cisdst) = 
	    localtime($secs);
	$this_date = "19$cyear" . &pad($cmon + 1) . &pad($cmday);
    }

    return $ldates;
}


#----------------------------------------------------------------------------
# $Log: recur.pl,v $
# Revision 1.3  1996/02/11  20:25:23  curt
# Added support for effective date range.
#
# Revision 1.2  1996/01/19  02:11:14  curt
# A couple of tweaks to make this more usable.
#
# Revision 1.1  1996/01/18  05:15:51  curt
# Initial Revision ... not done, but the hard stuff is working.
#

