;;; lj-editf.el --- friends list editing support for ljupdate

;; Copyright (C) 2002  Edward O'Connor <ted@oconnor.cx>

;; Author: Edward O'Connor <ted@oconnor.cx>
;; Keywords: convenience

;; This file is part of ljupdate.

;; ljupdate is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License as
;; published by the Free Software Foundation; either version 2, or
;; {at your option} any later version.

;; ljupdate is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public
;; License along with ljupdate, or with your Emacs. See the file
;; COPYING, or, if you're using GNU Emacs, try typing C-h C-c to
;; bring it up. If you're using XEmacs, C-h C-l does this. If you
;; do not have a copy, you can obtain one by writing to the Free
;; Software Foundation at this address:

;;                Free Software Foundation, Inc.
;;                59 Temple Place, Suite 330
;;                Boston, MA  02111-1307
;;                USA


;;; Commentary:

;; This is my first attempt to do anything non-trivial with the
;; widget library. So beware, all ye who enter here.

;;; Code:

(require 'widget)
(require 'wid-edit)

(require 'ljcompat)

(defcustom lj-edit-friends-mode-hook '()
  "*Hooks to be run when entering `lj-edit-friends-mode'."
  :type 'hook
  :group 'ljupdate)

(defun lj-edit-friends-mode ()
  "Major mode for editing someone's friends list."
  (kill-all-local-variables)
  (setq major-mode 'lj-edit-friends-mode)
  (setq mode-name "Edit Friends")
  (use-local-map widget-keymap)
  (run-hooks 'lj-edit-friends-mode-hook))

;; OK, what I really want to do is define a friend widget.

(defun lj--create-new-friend-widget (widget)
  "Create a new, empty friend widget.
Note that this is for the new friends list, and so there be no
remove button, and the username is editable."
  (let* ((username (widget-create-child-and-convert
                    widget 'editable-text
                    :value "Foo"))
         (fg (widget-create-child-and-convert
              widget 'color :value "Black"))
         (bg (widget-create-child-and-convert
              widget 'color :value "White")))
    widget))

(define-widget 'lj-friend-widget 'group
  "A widget for editing an ljupdate friend object."
  :create 'lj--create-new-friend-widget)

(defun lj--editf-remove-notify (widget &rest ignore)
  (let* ((remove-widget widget)
         (remove-widget-on (widget-value remove-widget))
         (remove-widget-friend (widget-get remove-widget :friend))
         (group-widget (widget-get remove-widget :parent))
         (username-widget (widget-get group-widget :editf-username)))
  (if remove-widget-on
      ;; add to removal list
      (progn
        (add-to-list 'lj-editf-friends-to-remove remove-widget-friend)
        (add-to-list 'lj-editf-widgets-to-remove group-widget))
    ;; remove from removal lists
    (setq lj-editf-friends-to-remove
          (remove remove-widget-friend lj-editf-friends-to-remove))
    (setq lj-editf-widgets-to-remove
          (delq group-widget lj-editf-widgets-to-remove))
    (widget-setup))))

(defun lj--create-read-friend-widget-from-friend (friend)
  "Create a friend widget from FRIEND."
  (let* ((group (widget-create 'group))
         (username (widget-create-child-and-convert
                    group 'item
                    :format "%v (%t)"
                    :tag ""
                    :value (lj-friend-username friend)))
         (remove (progn
                   (widget-insert "Remove? ")
                   (widget-create-child-and-convert
                    group 'checkbox
                    :friend (lj-friend-username friend)
                    :notify 'lj--editf-remove-notify)))
         (ignore (widget-insert "\n"))
         (fg (widget-create-child-and-convert
              group 'color :tag "Foreground"
              :value (lj-friend-fg friend)))
         (bg (widget-create-child-and-convert
              group 'color :tag "Background"
              :value (lj-friend-bg friend))))
    (widget-put group :editf-username username)
    (widget-put group :editf-remove remove)
    (widget-put group :editf-fg fg)
    (widget-put group :editf-bg bg)
    group))

(defun lj--insert-edit-friends-form ()
  (set (make-local-variable 'lj-editf-friends-to-remove) '())
  (set (make-local-variable 'lj-editf-widgets-to-remove) '())
  (set (make-local-variable 'lj-editf-friends-to-add) '())
  (let ((friends (lj-profile-friends lj-editf-profile)))
    (widget-insert "Editing "
                   (lj-profile-username lj-editf-profile)
                   "'s friends\n")
    (mapcar (lambda (friend-cons)
              (let ((friend (cdr friend-cons)))
                (widget-insert "\n")
                (lj--create-read-friend-widget-from-friend friend)))
            (sort friends
                  (lambda (cons1 cons2)
                    (string-lessp (car cons1) (car cons2)))))
    (widget-insert "\n")
    (widget-insert "Friends to add: ")
    (widget-create 'editable-list 'lj-friend-widget)
    (widget-create
     'push-button :tag "Commit Changes"
     :notify (lambda (widget &rest ignore)
               (lj--commit-edit-friends-changes)))
    (widget-setup)))

(defun lj--commit-edit-friends-changes ()
  "Commit the changes specified by the current edit-friends buffer
to the sever."
  (let* ((fields
          (mapcar (lambda (friend-to-remove)
                    (cons (concat "editfriend_delete_" friend-to-remove)
                          1))
                  lj-editf-friends-to-remove))
         (profile lj-editf-profile)
         (request (lj--make-request 'editfriends fields)))
    (setq lj-editf-friends-to-remove '())
    (when (car (lj--send-request profile request))
      (mapcar (lambda (widget)
                (widget-apply widget :deactivate))
              lj-editf-widgets-to-remove)
      (widget-setup)
      (setq lj-editf-widgets-to-remove '()))))

;;;###autoload
(defun lj-edit-friends (profile)
  "Edit PROFILE's friends."
  (interactive (list (lj--read-profile)))
  ;; Ensure that we have a current version of PROFILE's friends
  ;; list.
  (lj--initialize)
  (lj-profile-update-friends profile)
  (switch-to-buffer (get-buffer-create
                     (format "*%s's Friends*"
                             (lj-profile-username profile))))
  (let ((inhibit-read-only t))
    (erase-buffer))
  (or (eq major-mode 'lj-edit-friends-mode)
      (lj-edit-friends-mode))
  (set (make-local-variable 'lj-editf-profile) profile)
  (lj--insert-edit-friends-form))

(provide 'lj-editf)
;;; lj-editf.el ends here
