;;; lj-entry.el --- listing and editing of previous entries

;; Copyright (C) 2001, 2002 Edward O'Connor <ted@oconnor.cx>
;; Copyright (C) 2002 Alexander Koller  <koller@coli.uni-sb.de>

;; Maintainer: Edward O'Connor <ted@oconnor.cx>
;; Keywords: convenience

;; This file is part of ljupdate.

;; ljupdate is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License as
;; published by the Free Software Foundation; either version 2, or
;; {at your option} any later version.

;; ljupdate is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public
;; License along with ljupdate, or with your Emacs. See the file
;; COPYING, or, if you're using GNU Emacs, try typing C-h C-c to
;; bring it up. If you're using XEmacs, C-h C-l does this. If you
;; do not have a copy, you can obtain one by writing to the Free
;; Software Foundation at this address:

;;                Free Software Foundation, Inc.
;;                59 Temple Place, Suite 330
;;                Boston, MA  02111-1307
;;                USA

;;; Commentary:

;;; History:

;;; Code:

(require 'ljcompat)

(defcustom lj-list-entries-mode-hook nil
  "Hooks to be executed when `lj-list-entries-mode' is entered."
  :type 'hook
  :group 'ljupdate)

(defvar lj-list-entries-mode-map nil
  "Keyboard mapping used by `lj-list-entries-mode'.")

(unless lj-list-entries-mode-map
  (setq lj-list-entries-mode-map (make-sparse-keymap))
  (suppress-keymap lj-list-entries-mode-map)
  (define-key lj-list-entries-mode-map "\C-m" 'lj--edit-entry-at-point))

(defun lj--get-entries (profile selector selector-argument)
  "Get an overview list of entries from PROFILE which match the
SELECTOR. The selector is one of the following symbols, with the
SELECTOR-ARGUMENT listed.

     day (<month> <day> <year>)  entries for that day
     lastn <n>                   last n entries
     one <itemid>                entry with that itemid

The arguments are all numeric, and must be either integers or
strings representing integers. The `syncitems' select type allowed
by the LJ protocol is not supported by this function."

  ;; Make sure that we've logged in before posting.
  (let* ((system (lj-profile-system profile)))
    (unless (get profile 'lj-logged-in-p)
      (lj-profile-login profile))
    (let* ((fields '((noprops . t) ; don't return music, moods, etc.
                     (truncate . 30)))
           (request nil))

      (case selector
        (day (add-to-list 'fields '(selecttype . "day"))
             (add-to-list 'fields `(month . ,(nth 0 selector-argument)))
             (add-to-list 'fields `(day   . ,(nth 1 selector-argument)))
             (add-to-list 'fields `(year  . ,(nth 2 selector-argument))))
        (lastn (add-to-list 'fields '(selecttype . "lastn"))
               (add-to-list 'fields `(howmany . ,selector-argument)))
        (one (add-to-list 'fields '(selecttype . "one"))
             (add-to-list 'fields `(itemid . ,selector-argument))))

      (setq request (lj--make-request 'getevents fields))

      (lj--send-request profile request))))

(defun lj--get-day-counts (profile)
  "Get a list of day counts for PROFILE."

  ;; Make sure that we've logged in before posting.
  (let* ((system (lj-profile-system profile)))
    (unless (get profile 'lj-logged-in-p)
      (lj-profile-login profile))

    (lj--send-request profile (lj--make-request 'getdaycounts '()))))

(defun lj--get-entry (profile itemid)
  "Get a single entry from PROFILE with itemid ITEMID."

  ;; Make sure that we've logged in before posting.
  (let* ((system (lj-profile-system profile)))
    (unless (get profile 'lj-logged-in-p)
      (lj-profile-login profile))
    (let* ((fields `((selecttype . "one")
                     (itemid . ,itemid)))
           (request (lj--make-request 'getevents fields)))

      (lj--send-request profile request))))

(defun lj--entries-entry (entries which i)
  (cdr (assoc (concat "entries_" (number-to-string i) "_" which) entries)))

(defun lj-list-entries-mode ()
  "Major mode for displaying a list of LiveJournal entries.
   Press Return on a line to download and display an entry.
   You can then simply post an updated entry from the new buffer."
  (interactive)
  (lj--initialize)

  (kill-all-local-variables)
  (setq major-mode 'lj-list-entries-mode)
  (setq mode-name "LiveJournal List")

  ;; install keymap and local variable
  (use-local-map lj-list-entries-mode-map)
  (make-local-variable 'lj--lines-to-itemids)
  (setq lj--lines-to-itemids nil)

  (run-hooks 'lj-list-entries-mode-hook))

(defun lj--url-unescape (string ignore-newlines)
  "Unescape the STRING according to IGNORE-NEWLINES."
  (let ((result ())
	(decoded-char))

    (while (not (string-equal string ""))
      (case (elt string 0)
	    (?+
	     (add-to-list 'result " ")
	     (setq string (substring string 1)))

	    (?%
	     (setq decoded-char
		   (string (string-to-number (substring string 1 3) 16)))

	     (when (and
		    (not (string-equal decoded-char "\r"))
		    (not (and ignore-newlines
			      (string-equal decoded-char "\n"))))
	       (add-to-list 'result decoded-char))

	     (setq string (substring string 3)))

	    (t
             (when (elt string 0)
               (add-to-list 'result (string (elt string 0))))
             (if (> (length string) 1)
                 (setq string (substring string 1))
               (setq string "")))))

    (apply 'concat (nreverse result))))

(defun lj--fill-entries-list-buffer (entries-responses)
  "Fills the current buffer with the ENTRIES-RESPONSES. See also
   lj--create-entries-list-buffer. The function assumes that
   the current buffer is a properly initialized buffer in
   lj-list-entries-mode."

  (let ((line-counter 1))
    (setq lj--lines-to-itemids nil)


    (mapc (lambda (entry)
            (let ((i 1))
              (while (<= i (or (string-to-number
                                (or (cdr (assoc "entries_count" entry)) "0"))
                               0))
                (insert (format "%s %-28s %s\n"
                                (substring
                                 (or
                                  (lj--entries-entry
                                   entry "entrytime" i)
                                  "Something went horribly wrong.")
                                 0 16)
                                (substring
                                 (or
                                  (lj--entries-entry entry "subject" i)
                                  "Something else happened that's bad.")
                                 0
                                 (min 28
                                      (length
                                       (lj--entries-entry
                                        entry "subject" i))))
                                (lj--url-unescape
                                 (lj--entries-entry entry "entry" i)
                                 t) ))
                (push (cons line-counter
                            (string-to-number
                             (or
                              (lj--entries-entry entry "itemid" i)
                              "0")))
                      lj--lines-to-itemids)

                (setq line-counter (+ 1 line-counter))

                (setq i (+ i 1)))))

          entries-responses)

    (set-buffer-modified-p nil)
    (goto-char 1)))




(defun lj--create-entries-list-buffer (entries-responses)
  "Create a new buffer in LiveJournal List mode, and fill it with
   the ENTRIES-RESPONSES. ENTRIES-RESPONSES is a list of responses
   from `getentries' queries."
  (let ((entry-buffer (generate-new-buffer "*Livejournal Entries*")))

    (switch-to-buffer entry-buffer)
    (lj-list-entries-mode)
    (lj--fill-entries-list-buffer entries-responses)))



(defun lj--get-entry-property (entry property)
  (let* ((propkey (car (rassoc property entry))))

    (if propkey
	(cdr (assoc (concat "prop_"
			    (number-to-string
                             (nth 1 (split-string propkey "_")))
			    "_value")
		    entry))

      nil)))



;; this function assumes that ENTRY contains precisely 1 entry
(defun lj--decode-entry-properties (entry profile)
  (let* ((system (lj-profile-system profile))
	 (music (lj--get-entry-property entry "current_music"))
	 (moodid (lj--get-entry-property entry "current_moodid"))
	 (mood (lj--get-entry-property entry "current_mood"))
	 (picture (lj--get-entry-property entry "picture_keyword"))
	 (nocomments (lj--get-entry-property entry "opt_nocomments"))
	 (subject (cdr (assoc "entries_1_subject" entry)))
	 (entry-time (cdr (assoc "entries_1_entrytime" entry)))
	 (security (cdr (assoc "entries_1_security" entry))))

;    (unless (get profile 'lj-logged-in-p)
    ; BUG: We can't trust the value of LJ-LOGGED-IN-P, as it
    ; may remain T when everythinge else in the SYSTEM is
    ; reset to NIL periodically. This is very weird.
    (lj-profile-login profile)


  (append
   `((subject . ,subject))
   `((entry-time . ,entry-time))

   ;; if we don't get a value, it's because the user didn't
   ;; specify music -- we certainly shouldn't guess and add the
   ;; now-current music.
   `((music . ,(or music "")))

   (if moodid
       `((mood . ,(car (rassoc (string-to-number moodid)
                               (lj-system-moods system)))))
     (if mood
         `((mood . ,mood))
       nil))

   ; picture
   (if picture
       `((picture . ,picture))
     nil)

   (case nocomments
     (1 '((allow . no)))
     (0 '((allow . yes)))
     (t nil))

   (if security
       `((allow . ,security))
     nil))))


(defun lj--edit-entry-at-point ()
  (interactive)

  (let* ((line-number (count-lines (window-start) (point)))
;;			 (if (= (current-column) 0) 1 0)))
	 (itemid (cdr (assoc line-number lj--lines-to-itemids)))
	 (profile (or lj--current-profile lj-default-profile))
	 (entry (lj--get-entry profile itemid))
	 (entry-properties (lj--decode-entry-properties entry profile))
	 (subject (cdr (assoc 'subject entry-properties)))
	 (display-buffer (generate-new-buffer
			  (concat "*Livejournal: " subject "*"))))

    ;; first do just as we would for lj-compose ...
    (switch-to-buffer display-buffer)
    (lj-update-mode)
    (lj--insert-initial-buffer-contents entry-properties)


    ;; ... then add the item ID ...
    (save-excursion
      (goto-char header-end)
      (insert (format "X-LJ-ItemID: %d\n" itemid))
      (insert (format "X-LJ-Entry-Time: %s\n"
		      (cdr (assoc 'entry-time entry-properties)))))


    ;; ... and the entry text.
    (insert (lj--url-unescape (cdr (assoc "entries_1_entry" entry)) nil))

    (set-buffer-modified-p nil)))

(defun lj--extract-entry-day (current-daycount mon year)
  (let* ((split-day (split-string (car current-daycount) "-")))
    (if (and
         (= 3 (length split-day))
         (= (string-to-number (car split-day)) year)
         (= (string-to-number (nth 1 split-day)) mon))
        (string-to-number (nth 2 split-day))
      nil)))

(defun lj--extract-entry-days (daycounts mon year)
  "Computes the list of days in month MON and year YEAR on which
   entries take place from DAYCOUNTS, which is the return value
   of a `getdaycounts' talk-to-server call. The result returned
   by this function is a list of integers (the days)."

  (let ((mapret (mapcar (lambda (daycount)
                          (lj--extract-entry-day daycount mon year))
                        daycounts))
        (retval '()))
    (mapc (lambda (item)
            (when item
              (add-to-list 'retval item)))
          mapret)
    (nreverse retval)))

(defun lj--get-entries-for-month (month year)
  "Prompt the user for a month and a year and download the Livejournal
   entries from that month, to display in a Livejournal List buffer."
  (interactive "dGet entries for month: \ndGet entries for year: ")
  (lj--initialize)

  (let* ((daycounts (lj--get-day-counts (or lj--current-profile lj-default-profile)))
	 (entry-days (sort (lj--extract-entry-days (cdr daycounts) month year)
                           #'<)))

    (if (not (car daycounts))
	(lj--message 1 "Server error, errmsg is %s"
                     (cdr (assoc "errmsg" daycounts)))

      (lj--create-entries-list-buffer
       (mapcar #'(lambda (day)
		   (lj--get-entries (or lj--current-profile lj-default-profile)
                                    'day (list month day year)))
	       entry-days)))))

(provide 'lj-entry)
