;;; lj-hack.el --- tools for hacking on ljupdate

;; Copyright (C) 2002  Edward O'Connor <ted@oconnor.cx>
;; Copyright (C) 2001, 2002 John Wiegley <johnw@gnu.org>

;; Author: Edward O'Connor <ted@oconnor.cx>
;; Keywords: convenience

;; This file is part of ljupdate.

;; ljupdate is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License as
;; published by the Free Software Foundation; either version 2, or
;; {at your option} any later version.

;; ljupdate is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public
;; License along with ljupdate, or with your Emacs. See the file
;; COPYING, or, if you're using GNU Emacs, try typing C-h C-c to
;; bring it up. If you're using XEmacs, C-h C-l does this. If you
;; do not have a copy, you can obtain one by writing to the Free
;; Software Foundation at this address:

;;                Free Software Foundation, Inc.
;;                59 Temple Place, Suite 330
;;                Boston, MA  02111-1307
;;                USA

;;; Commentary:

;; No user-serviceable parts inside. Refer all maintenance to your
;; local union technician. Note that I'm not following decent
;; elisp conventions in this file. Also, this file is only tested
;; under GNU Emacs 21.2, and no attempts are made to have it work
;; with other Emacsen.

;;; Code:

(require 'pp)

(defvar ljupdate-dev-directory
  "/home/ted/elisp/cvs-ljupdate/")

(defvar ljupdate-lisp-directory
  (concat ljupdate-dev-directory "lisp/"))

(defvar ljupdate-texi-directory
  (concat ljupdate-dev-directory "texi/"))

(defvar ljupdate-info-directory
  (concat ljupdate-dev-directory "info/"))

(when noninteractive
  ;; It would be cooler if there were some other way to discover
  ;; where the BBDB lives.
  (add-to-list 'load-path "~/elisp/bbdb-2.34/lisp")
  (add-to-list 'load-path ljupdate-lisp-directory))

;; Make ljupdate code prettier. Yay fruit salad.
(and (fboundp 'font-lock-add-keywords)
     (font-lock-add-keywords
      'emacs-lisp-mode
      '(("(\\(lj--define-protocol-mode\\)\\>.\\([a-z]+\\).[']?\\(.*\\)"
         (1 font-lock-keyword-face)
         (2 font-lock-variable-name-face)
         (3 font-lock-function-name-face))
        ("(\\(lj--deferror\\)\\>.\\([-a-z_]+\\)"
         (1 font-lock-keyword-face)
         (2 font-lock-variable-name-face))
	("(\\(lj-defsystem\\|lj-defprofile\\)\\>"
         (1 font-lock-keyword-face))
	("(\\(lj--make-request\\)\\>"
         (1 font-lock-function-name-face))
	("(\\(lj--message\\)\\>"
         (1 font-lock-warning-face)))))

(defun ljupdate-files ()
  ;; Sorting is important; we want `lj-auto.el' to be first.
  (directory-files ljupdate-lisp-directory nil "^lj.*\\.el$" nil))

(defun lj-generate-autoloads-file ()
  "Generate a new lj-auto.el file."
  (interactive)
  (save-excursion
    (let* ((default-directory ljupdate-lisp-directory)
           (generated-autoload-file (concat default-directory "lj-auto.el")))
      (auto-insert-mode -1)
      (when (file-exists-p generated-autoload-file)
      (delete-file generated-autoload-file))
      (find-file generated-autoload-file)
      (set-buffer (get-file-buffer generated-autoload-file))
      (insert ";;; lj-auto.el --- autoloads for ljupdate\n\n"
              "(provide 'lj-auto)\n")
      (save-buffer)
      (mapc 'update-file-autoloads
            ;; We `cdr' because `lj-auto.el' is the first entry.
            (cdr (ljupdate-files)))
      (set-buffer (get-file-buffer generated-autoload-file))
      (save-buffer)
      (kill-buffer (current-buffer))
      (auto-insert-mode 1))))

;; Straight-up pilfered from chess/lispdoc.el. johnw rules.
(unless (fboundp 'update-lispdoc-tags)
  (defun update-lispdoc-tags ()
    (interactive)
    (save-excursion
      (goto-char (point-min))
      (while (re-search-forward "^@c lispfun \\(.+\\)" nil t)
        (let ((name (match-string 1)) begin end)
          (message "Update lispdoc for function '%s'" name)
          (if (re-search-forward (concat "^@defun " name) nil t)
              (setq begin (match-beginning 0)))
          (if (re-search-forward "^@end defun" nil t)
              (setq end (match-end 0)))
          (if (and begin end)
              (delete-region begin end))
          (let* ((sym (or (intern-soft name)
                          (signal 'wrong-type-argument
                                  (list 'functionp name))))
                 (data (let ((func (symbol-function sym)))
                         (while (symbolp func)
                           (setq func (symbol-function func)))
                         func))
                 (args (pp-to-string (if (listp data)
                                         (cadr data)
                                       (aref data 0))))
                 (doc (documentation sym)))
            (if (or (null doc) (= (length doc) 0))
                (message "warning: no documentation available for '%s'" name)
              (unless (and begin end)
                (insert ?\n ?\n))
              (insert (format "@defun %s %s\n" name
                              (substring args 1 (- (length args) 2))))
              (setq begin (point))
              (insert doc ?\n)
              (save-restriction
                (narrow-to-region begin (point))
                (goto-char (point-min))
                (let ((case-fold-search nil))
                  (while (re-search-forward "[A-Z][A-Z-]+" nil t)
                    (replace-match (format "@var{%s}"
                                           (downcase (match-string 0))) t t)))
                (goto-char (point-max)))
              (insert "@end defun"))))))))

;; Basically pilfered from chess-maint.el.
(defun lj-generate-texinfo-file ()
  "Generate a new ljupdate-final.texi from ljupdate.texi."
  (interactive)
  (require 'texinfo)
  (with-temp-buffer
    (let ((default-directory ljupdate-texi-directory))
      (insert-file-contents-literally "ljupdate.texi")
      (texinfo-mode)
      (lj-load-all)
      (texinfo-insert-node-lines (point-min) (point-max) t)
      (texinfo-every-node-update)
      (texinfo-all-menus-update t)
      (texinfo-every-node-update)
      (texinfo-all-menus-update t)
      (update-lispdoc-tags)
      (write-file "ljupdate-final.texi"))))

;; Note that this is now redundant with the Makefile.
(defun lj-generate-info-file ()
  "Generate a new ljupdate.info from ljupdate-final.texi."
  (let ((default-directory ljupdate-texi-directory))
    (shell-command (concat "makeinfo ./ljupdate-final.texi -o "
                   ljupdate-info-directory "ljupdate.info"))))

(defun lj-compile-files (&optional load)
  "Compile everything."
  (interactive "P")
  (let ((default-directory ljupdate-lisp-directory))
    (mapc (lambda (file)
            (byte-compile-file file load))
          (ljupdate-files))))

(defun lj-load-all ()
  "Load everything."
  (interactive)
  (let ((default-directory ljupdate-lisp-directory))
    (mapc 'load (ljupdate-files))))

(defun lj-test-checkf ()
  "Test checkfriends stuff."
  (setq debug-on-error t)
  (require 'ljcheckf)
  (lj-checkfriends-mode 1))

(defun lj-make-all ()
  "Build stuff."
  (interactive)
  (lj-generate-autoloads-file)
  (lj-generate-texinfo-file)
  (lj-generate-info-file))

(defun lj-generate-snapshot-tarball ()
  "Make a new snapshot tarball."
  (interactive)
  (lj-make-all)
  (let ((files-to-include (mapcar (lambda (file) (concat "lisp/" file))
                                  (ljupdate-files)))
        (archive-file-name (format-time-string "ljupdate-%Y%m%d.tar.gz"))
        (command nil))
    (setq files-to-include
          (append (list "CONTRIBUTORS" "COPYING" "ChangeLog"
                        "DEV-NOTES" "HISTORY" "INSTALL"
                        "README" "TODO" "info/ljupdate.info"
                        "texi/ljupdate.texi" "texi/ljupdate-final.texi")
                  files-to-include))
    (setq command
          (format "tar czf %s %s"
                  archive-file-name
                  (mapconcat (lambda (file-name)
                               (concat "ljupdate/" file-name))
                             files-to-include
                             " ")))
    (let ((default-directory
            "/home/ted/public_html/documents/ljupdate/development/"))
      (shell-command command))))

(provide 'lj-hack)

;;; lj-hack.el ends here
