;;; lj-objs.el --- Object definitions for ljupdate

;; Copyright (C) 2002  Edward O'Connor <ted@oconnor.cx>

;; Author: Edward O'Connor <ted@oconnor.cx>
;; Keywords: convenience

;; This file is part of ljupdate.

;; ljupdate is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License as
;; published by the Free Software Foundation; either version 2, or
;; {at your option} any later version.

;; ljupdate is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public
;; License along with ljupdate, or with your Emacs. See the file
;; COPYING, or, if you're using GNU Emacs, try typing C-h C-c to
;; bring it up. If you're using XEmacs, C-h C-l does this. If you
;; do not have a copy, you can obtain one by writing to the Free
;; Software Foundation at this address:

;;                Free Software Foundation, Inc.
;;                59 Temple Place, Suite 330
;;                Boston, MA  02111-1307
;;                USA

;;; Commentary:

;; 

;;; Code:

(require 'ljcompat)

;; Look Ma, some documentation!

;;          A primer on ljupdate objects and methods
;;          ----------------------------------------

;; There are four kinds of ljupdate objects: users, systems,
;; profiles, and friends.

;; Generally, you access a field of an object by using calls like
;; this:

;;      (lj-CLASS-FIELD OBJECT)
;;      (lj-CLASS-set-FIELD OBJECT NEW-FIELD-VALUE)

;; You shouldn't need to know anything about the inner
;; representation of these objects, or at least that's the hope.

;; Here's the rough explanation of what each kind of object is:

;; A SYSTEM is an object that holds information about some server
;; that has the LiveJournal software running on it. The two
;; systems defined by default are `livejournal' and `deadjournal'.

;; System objects are named by symbols; the data lives in the
;; symbol's property list.

;; System objects contain several fields:

;;      hostname - the hostname of the system, e.g.
;;                 "www.livejournal.com".
;;          port - what port the web server runs on (default is 80)
;;         moods - Each LiveJournal system maintains a table of
;;                 commonly-typed moods, such as "happy" and "sad".
;;                 These actually do differ between systems, though
;;                 not by much (for instance, `deadjournal' doesn't
;;                 define "nervous", but `livejournal' does).
;;         users - each system has many, many users on it. System
;;                 objects certainly don't know about all of them;
;;                 we just remember ones that for one reason or
;;                 another have come up (such as your friends).
;;                 This is an alist whose elements are of the form
;;                     (username . lj-user)

;; A USER is a journal on some SYSTEM. For instance, bradfitz is a
;; user on `livejournal'.

;; User objects contain these fields:

;;      username - the username of the user, e.g. "bradfitz".
;;          type - the type of user; valid values are the symbols
;;                 `user' and `community'.
;;     real-name - the real name of the user, e.g. "Brad Fitzpatrick".

;; A PROFILE is an account on some system that you can use to log
;; in. For instance, I can log in as "hober" on `livejournal' and
;; also as hober" on deajournal, but I can't log in as "bradfitz",
;; so I have two profiles defined:

;;      1. User "hober" on system `livejournal'
;;      2. User "hober" on system `deadjournal'

;; Profiles are named by symbols; the data lives in the symbol's
;; property list. The profile symbol is generated by
;; `lj-defprofile' by combining the profile's username and system
;; name with a dash, e.g. `hober-livejournal'.

;; Profile objects contain these fields:

;;        system - The system this profile lives on, e.g. `livejournal'.
;;      username - The username of this profile, e.g. "hober".
;;        fast-p - Whether or not we should use the lj fast cookie when
;;                 doing things as this user.
;;      pictures - A list of strings naming this user's pictures.
;;   communities - A list of strings naming the communities that this
;;                 user is authorized to post to.
;;     real-name - This user's real name, e.g. "Edward O'Connor".
;;       friends - A list of friend objects, representing the users on
;;                 this user's friends list.
;; friend-groups - An alist of string-number pairs, denoting the
;;                 name and number of each of the profile's
;;                 defined friend groups.

;; A FRIEND is an object that keeps track of how your friends
;; should be displayed. They have these fields:

;;      username - The username of this friend, e.g. "hober". This
;;                 can be used to look up additional user information
;;                 (type, real name) in the profile's system's user
;;                 list.
;;           num - each of your friends has a unique friends list id
;;                 number.
;;        fg, bg - these fields define the foreground and background
;;                 colors that you use on your friends page for this
;;                 friend.


;; lj-message methods

(defalias 'lj-message-p 'listp)

(defsubst lj-message-get-field (message field)
  ""
  (cdr (assoc field message)))

(defsubst lj-message-set-field (message field)
  ""
)

;;; lj-user methods

(defalias 'lj-user-p 'vectorp)

(defsubst lj-user-username (user)
  "Fetch USER's username."
  (aref user 0))

(defsubst lj-user-set-username (user username)
  "Set USER's username to USERNAME."
  (aset user 0 username))

(defsubst lj-user-type (user)
  "Fetch USER's type."
  (aref user 1))

(defsubst lj-user-set-type (user type)
  "Set USER's type to TYPE."
  (aset user 1 type))

(defsubst lj-user-real-name (user)
  "Fetch USER's real name."
  (aref user 2))

(defsubst lj-user-set-real-name (user real-name)
  "Set USER's real name to REAL-NAME."
  (aset user 2 real-name))

(defun lj-user-create (username &optional type real-name)
  "Create a user named USERNAME optionally with TYPE and REAL-NAME."
  (let ((user (make-vector 10 nil)))
    (lj-user-set-username user username)
    (lj-user-set-type user (if (string-equal type "community")
                               'community
                             'user))
    (lj-user-set-real-name user real-name)
    user))

;;; lj-friend methods

(defalias 'lj-friend-p 'vectorp)

(defun lj-friend-create (username num fg bg)
  "Create a new friend with USERNAME, NUM, FG, and BG."
  (let ((friend (make-vector 5 nil)))
    (lj-friend-set-username friend username)
    (lj-friend-set-num friend num)
    (lj-friend-set-fg friend fg)
    (lj-friend-set-bg friend bg)
    friend))

(defsubst lj-friend-username (friend)
  "Fetch the username of FRIEND."
  (aref friend 0))

(defsubst lj-friend-set-username (friend username)
  "Set FRIEND's username to USERNAME."
  (aset friend 0 username))

(defsubst lj-friend-num (friend)
  "Fetch FRIEND's id."
  (aref friend 1))

(defsubst lj-friend-set-num (friend num)
  "Set FRIEND's id to NUM."
  (aset friend 1 num))

(defsubst lj-friend-fg (friend)
  "Fetch FRIEND's foreground color."
  (aref friend 2))

(defsubst lj-friend-set-fg (friend fg)
  "Set FRIEND's foreground color to FG."
  (aset friend 2 fg))

(defsubst lj-friend-bg (friend)
  "Fetch FRIEND's background color."
  (aref friend 3))

(defsubst lj-friend-set-bg (friend bg)
  "Set FRIEND's background color to BG."
  (aset friend 3 bg))

(defvar lj-use-friend-faces-flag
  (or (and (fboundp 'display-color-p)
           (display-color-p))
      window-system)
  "*Non-nil means to use your friends list colors.")

;;; lj-system variables and methods

(defun lj-system-p (object)
  "Is OBJECT an lj-system?"
  (and (symbolp object)
       (let ((sysv (get object 'lj-system)))
         (and sysv
              (vectorp sysv)))))

(defsubst lj-system-hostname (system)
  "Fetch the hostname of SYSTEM."
  (aref (get system 'lj-system) 0))

(defsubst lj-system-set-hostname (system hostname)
  "Set the hostname of SYSTEM to HOSTNAME."
  (aset (get system 'lj-system) 0 hostname))

(defsubst lj-system-port (system)
  "Fetch the port of SYSTEM."
  (aref (get system 'lj-system) 1))

(defsubst lj-system-set-port (system port)
  "Set the port of SYSTEM to PORT."
  (aset (get system 'lj-system) 1 port))

(defsubst lj-system-moods (system)
  "Fetch the moods alist of SYSTEM."
  (aref (get system 'lj-system) 2))

(defsubst lj-system-set-moods (system moods)
  "Set the moods alist of SYSTEM to MOODS."
  (aset (get system 'lj-system) 2 moods))

(defsubst lj-system-users (system)
  "Fetch an alist of (username . lj-user object) pairs for SYSTEM."
  (aref (get system 'lj-system) 3))

(defsubst lj-system-set-users (system users)
  "Set SYSTEM's users alist to USERS."
  (aset (get system 'lj-system) 3 users))

(defun lj-system-get-user (system username)
  "Fetch the lj-user object at SYSTEM for USERNAME."
  (let ((entry (assoc username (lj-system-users system))))
    (if entry
        (cdr entry)
      nil)))

(defun lj-system-add-user (system user)
  "Create a user entry at SYSTEM for USER."
  (let ((users (lj-system-users system)))
    (add-to-list 'users (cons (lj-user-username user) user))
    (lj-system-set-users system users))
  (setq lj--startup-modified-p t))

(defun lj-system-get-user-create (system username)
  "Fetch the lj-user object at SYSTEM for USERNAME.

If SYSTEM doesn't have an entry for USERNAME, create a new lj-user
for USERNAME, and add it to SYSTEM."
  (let ((user (lj-system-get-user system username)))
    (unless user
      (setq user (lj-user-create username))
      (lj-system-add-user system user))
    user))

(defvar lj--systems nil
  "A list of LiveJournal-based systems.")

;;;###autoload
(defun lj-defsystem (system hostname &optional port)
  "Define a LiveJournal system SYSTEM (a symbol) at host HOSTNAME.

Optionally specify a non-default PORT."
  (put system 'lj-system (make-vector 5 nil))

  (lj-system-set-hostname system hostname)
  (lj-system-set-port     system (or port 80))

  (add-to-list 'lj--systems system)
  system)

(lj-defsystem 'livejournal "www.livejournal.com")
(lj-defsystem 'deadjournal "www.deadjournal.com")

(defcustom lj-default-system 'livejournal
  "*The LiveJournal system to post updates to by default."
  :type '(choice (restricted-sexp :match-alternatives (lj-system-p))
                 (set (restricted-sexp :match-alternatives (lj-system-p))))
  :group 'ljupdate)

;;; lj-profile variables and methods

(defvar lj--profiles nil
  "A list of different profiles that you use.

A profile is a symbol with an lj-profile property. Use
`lj-defprofile' to create a profile.")

(defun lj--profile-alist ()
  "Fetch an alist of profiles you have set up.

For use in various completion mechanisms."
  (mapcar (lambda (sym)
            (cons (symbol-name sym)
                  (symbol-name sym)))
          lj--profiles))

(defcustom lj-default-profile nil
  "*The profile (or profiles) to use by default."
  :type '(choice (restricted-sexp :match-alternatives (lj-profile-p))
                 (set (restricted-sexp :match-alternatives (lj-profile-p))))
  :group 'ljupdate)

(defvar lj--current-profile nil
  "The profile (or profiles) we used last time.")

(defsubst lj-profile-system (profile)
  "Fetch the system of PROFILE."
  (aref (get profile 'lj-profile) 0))

(defsubst lj-profile-set-system (profile system)
  "Set PROFILE's system to SYSTEM."
  (aset (get profile 'lj-profile) 0 system))

(defsubst lj-profile-username (profile)
  "Fetch the username of PROFILE."
  (aref (get profile 'lj-profile) 1))

(defsubst lj-profile-set-username (profile username)
  "Set PROFILE's username to USERNAME."
  (aset (get profile 'lj-profile) 1 username))

(defsubst lj-profile-fast-p (profile)
  "Should PROFILE use an LJ fast cookie?"
  (aref (get profile 'lj-profile) 2))

(defsubst lj-profile-set-fast-p (profile fast-p)
  "Set PROFILE's fast-p flag to FAST-P."
  (aset (get profile 'lj-profile) 2 fast-p))

(defsubst lj-profile-pictures (profile)
  "Fetch the pictures alist of PROFILE."
  (aref (get profile 'lj-profile) 3))

(defsubst lj-profile-set-pictues (profile pictures)
  "Set PROFILE's pictures alist to PICTURES."
  (aset (get profile 'lj-profile) 3 pictures))

(defsubst lj-profile-communities (profile)
  "Fetch the communities that PROFILE can post to."
  (aref (get profile 'lj-profile) 4))

(defsubst lj-profile-set-communities (profile communities)
  "Set PROFILE's communities to COMMUNITIES."
  (aset (get profile 'lj-profile) 4 communities))

(defsubst lj-profile-real-name (profile)
  "Fetch the real name of PROFILE."
  (aref (get profile 'lj-profile) 5))

(defsubst lj-profile-set-real-name (profile real-name)
  "Set PROFILE's real name to REAL-NAME."
  (aset (get profile 'lj-profile) 5 real-name))

(defsubst lj-profile-friend-groups (profile)
  "Fetch the friend groups of PROFILE."
  (aref (get profile 'lj-profile) 7))

(defsubst lj-profile-set-friend-groups (profile friend-groups)
  "Set PROFILE's friend groups to FRIEND-GROUPS."
  (aset (get profile 'lj-profile) 7 friend-groups))

(defsubst lj-profile-friends (profile)
  "Fetch the friends of PROFILE."
  (aref (get profile 'lj-profile) 6))

(defsubst lj-profile-set-friends (profile friends)
  "Set the friends list of PROFILE to FRIENDS."
  (aset (get profile 'lj-profile) 6 friends))

(defun lj-profile-add-friend (profile friend)
  "Ensure that PROFILE's friends list contains FRIEND."
  (aset (get profile 'lj-profile) 6
        (cons (cons (lj-friend-username friend) friend)
              (lj-profile-friends profile))))

(defun lj--profile-get-friend-1 (friends username)
  (cond ((null friends)
         nil)
        ((string-equal (caar friends) username)
         (cdar friends))
        (t
         (lj--profile-get-friend-1 (cdr friends) username))))

(defun lj-profile-get-friend (profile friend-name)
  (lj--profile-get-friend-1 (lj-profile-friends profile) friend-name))

(defun lj-profile-get-friend-create (profile friend-name)
  (let ((friend (lj-profile-get-friend profile friend-name)))
    (if friend
        friend
      (lj-profile-add-friend profile friend-name))))

;;;###autoload
(defun lj-defprofile (system username)
  "Define a profile on SYSTEM (a symbol) for USERNAME (a string)."
  (let ((prosym (intern (concat username "-" (symbol-name system)))))
    (put prosym 'lj-profile (make-vector 10 nil))

    (lj-profile-set-system      prosym system)
    (lj-profile-set-username    prosym username)

    (add-to-list 'lj--profiles prosym)

    prosym))

(defun lj-profile-password (profile)
  "Fetch/cache the password for PROFILE."
  (interactive (list (lj--read-profile)))
  (or (get profile 'lj-password)
      (put profile 'lj-password
           (ljc-md5 (read-passwd (format "%s password for %s: "
                                         (lj-profile-system profile)
                                         (lj-profile-username profile))))))
  (lj--message 6 "%s password is %s"
               profile (get profile 'lj-password))
  (get profile 'lj-password))

(defun lj-profile-clear-password (profile)
  "Clear the cached password for PROFILE."
  (interactive (list (lj--read-profile)))
  (put profile 'lj-password nil)
  (setq lj--startup-modified-p t))

(defun lj--read-profile ()
  "Interactively read a profile from the user."
  (lj--initialize)
  (let ((defval (cond ((consp lj--current-profile)
                       (symbol-name (car lj--current-profile)))
                      (lj--current-profile
                       (symbol-name lj--current-profile))
                      ((consp lj-default-profile)
                       (symbol-name (car lj-default-profile)))
                      (lj-default-profile
                       (symbol-name lj-default-profile))
                      (t ""))))
    (intern (completing-read "LJ profile: "
                             (lj--profile-alist)
                             nil
                             t
                             defval))))

;;;###autoload
(defun lj-profile-change (new-profile)
  "Subsequent LJ stuff should post using NEW-PROFILE by default."
  (interactive (list (lj--read-profile)))
  (lj--initialize)
  (setq lj--current-profile new-profile))

(provide 'lj-objs)
;;; lj-objs.el ends here
