;;; lj-prefs.el --- preferences for ljupdate

;; Copyright (C) 2002  Edward O'Connor <ted@oconnor.cx>

;; Author: Edward O'Connor <ted@oconnor.cx>
;; Keywords: 

;; This file is part of ljupdate.

;; ljupdate is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License as
;; published by the Free Software Foundation; either version 2, or
;; {at your option} any later version.

;; ljupdate is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public
;; License along with ljupdate, or with your Emacs. See the file
;; COPYING, or, if you're using GNU Emacs, try typing C-h C-c to
;; bring it up. If you're using XEmacs, C-h C-l does this. If you
;; do not have a copy, you can obtain one by writing to the Free
;; Software Foundation at this address:

;;                Free Software Foundation, Inc.
;;                59 Temple Place, Suite 330
;;                Boston, MA  02111-1307
;;                USA

;;; Commentary:

;; 

;;; Code:

(require 'ljcompat)

(defcustom lj-remember-passwords-flag nil
  "*Non-nil means ljupdate should cache password hashes in `lj-startup-file'.

This should not be a security risk if the permissions on
`lj-startup-file' are set sanely, namely to 600 or something very
similar, and besides, it doesn't cache your actual password, but
merely the md5 hash of it (assuming `lj--use-md5-p' is non-nil,
which it really really should be). Nevertheless, we default to
nil to be on the safe side."
  :type '(choice (const :tag "Remember my passwords" t)
                 (const :tag "Don't remember my passwords" nil))
  :group 'ljupdate)

(defvar lj--initialized-p nil
  "Whether or not ljupdate has been initialized yet.")

;;;###autoload
(defun lj--initialize ()
  "Initialize various things."
  (unless lj--initialized-p
    (load lj-startup-file t)
    (or lj--current-profile
        (setq lj--current-profile lj-default-profile))
    (setq lj--initialized-p t)))

;;;###autoload
(defun lj--save-startup-file ()
  "Save your lj startup file."
  (lj--initialize)
  (when lj--startup-modified-p
    (with-current-buffer (get-buffer-create " *LiveJournal-startup*")
      (set (make-local-variable 'version-control) 'never)
      (setq buffer-file-name lj-startup-file)
      (setq default-directory (file-name-directory buffer-file-name))
      (buffer-disable-undo (current-buffer))
      (erase-buffer)
      (lj--message 2 "Saving %s..." lj-startup-file)

      (let ((print-quoted t)
            (print-escape-newlines t))
        (insert
         ";;; " lj-startup-file " --- LJ startup file. -*- mode: emacs-lisp -*-\n\n"
         ";; Generally, you should not edit this by hand unless you know what\n"
         ";; you're doing. Its format is subject to change without notice. I'm\n"
         ";; just saying this to sound ominous.\n\n")
        (let ((systems lj--systems)
              (profiles lj--profiles)
              (print-quoted t)
              (print-readably t)
              (print-escape-multibyte nil))
          (mapc (lambda (system)
                  (insert
                   (format "(put '%s 'lj-system %S)\n"
                           (symbol-name system)
                           (get system 'lj-system))))
                systems)
          (insert (format "(setq lj--systems '%S)\n"
                          lj--systems))
          (mapc (lambda (profile)
                  (insert
                   (format "(put '%s 'lj-profile %S)\n"
                           (symbol-name profile)
                           (get profile 'lj-profile))
                   (if lj-remember-passwords-flag
                       (format "(put '%s 'lj-password %S)\n"
                               (symbol-name profile)
                               (get profile 'lj-password))
                     "")))
                profiles)
          (insert (format "(setq lj--profiles '%S)\n"
                          lj--profiles))))

      (run-hooks 'lj-save-startup-hook)
      (save-buffer)
      (kill-buffer (current-buffer))
      (lj--message 2 "Saving %s...done" lj-startup-file)
      (setq lj--startup-modified-p nil))))

(provide 'lj-prefs)
;;; lj-prefs.el ends here
