;;; lj-proto.el --- lj protocol API

;; Copyright (C) 2002  Edward O'Connor <ted@oconnor.cx>

;; Author: Edward O'Connor <ted@oconnor.cx>
;; Keywords: convenience

;; This file is part of ljupdate.

;; ljupdate is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License as
;; published by the Free Software Foundation; either version 2, or
;; {at your option} any later version.

;; ljupdate is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public
;; License along with ljupdate, or with your Emacs. See the file
;; COPYING, or, if you're using GNU Emacs, try typing C-h C-c to
;; bring it up. If you're using XEmacs, C-h C-l does this. If you
;; do not have a copy, you can obtain one by writing to the Free
;; Software Foundation at this address:

;;                Free Software Foundation, Inc.
;;                59 Temple Place, Suite 330
;;                Boston, MA  02111-1307
;;                USA

;;; Commentary:

;; 

;;; Code:

(require 'ljcompat)

(defvar lj--protocol-version 0
  "Version of the LiveJournal protocol supported by ljupdate.")

(defvar lj--protocol-modes '()
  "List of defined LiveJournal protocol modes.")

(lj--deferror lj-unsupported-protocol-mode-error (lj-error)
  "Invalid or unsupported LJ protocol mode")

(lj--deferror lj-request-object-error (lj-error)
  "Object provided is not an LJ request object")

(defun lj--make-protocol-matcher-for-field (field-name)
  "Make a protocol matcher for FIELD-NAME (a symbol)."
  (with-temp-buffer
    (insert (symbol-name field-name))
    (goto-char (point-min))
    (let ((found-one nil))
      (while (search-forward "*" nil t)
        (setq found-one t)
        (replace-match ".*" nil t))
      (if found-one
          (buffer-substring-no-properties (point-min) (point-max))
        ;; Poor attempt at being efficient. :)
        (symbol-name field-name)))))

(defvar lj--protocol-universal-send-fields
  ;; MODE, USER, and one of the PASSWORD fields will be supplied
  ;; by the profile object when we make the request into a raw
  ;; request. VER is always the value of `lj--protocol-version'.
  '(mode user password hpassword ver)
  "Send fields that may appear in any protocol mode.")

(defvar lj--protocol-universal-recv-fields
  '(success errmsg)
  "Receive fields that may appear in any protocol mode.")

(defun lj--make-field-matcher-for-mode (mode sendp)
  "Blah."
  (mapconcat 'lj--make-protocol-matcher-for-field
             (if sendp
                 (append lj--protocol-universal-send-fields
                         (get mode 'lj-request-fields))
               (append lj--protocol-universal-recv-fields
                       (get mode 'lj-response-fields)))
             "\\|"))

;; This is something we should probably consider caching.
(defmacro lj--define-protocol-mode (mode supported-by docstring send-fields &optional recv-fields)
  "Blah."
  `(progn
     (add-to-list 'lj--protocol-modes ',mode)
     (put ',mode 'lj-protocol-mode ,supported-by)
     (put ',mode 'lj-protocol-mode-description ,docstring)
     (put ',mode 'lj-request-fields ,send-fields)
     (put ',mode 'lj-request-field-matcher
          (lj--make-field-matcher-for-mode ',mode t))
     (put ',mode 'lj-response-fields ,recv-fields)
     (put ',mode 'lj-response-field-matcher
          (lj--make-field-matcher-for-mode ',mode nil))
     ',mode))
(put 'lj--define-protocol-mode 'lisp-indent-function 2)

;;; Define the standard LiveJournal protocol modes.
;;; List of modes and docstrings ripped from here:
;;;     http://www.livejournal.com/developer/modelist.bml

(lj--define-protocol-mode checkfriends 'ljcheckf
  "Mode that clients can use to poll the server to see if their
friends list has been updated. This request is extremely quick,
and is the preferred way for users to see when their friends list
is updated, rather than pounding on reload in their browser, which
is stressful on the serves."
  '(lastupdate mask)
  '(lastupdate new interval))

;; Support for this should be merged with support for the
;; postevent mode. They're essentially identical, ne?
(lj--define-protocol-mode editevent nil
  "Edit or delete a user's past journal entry."
  '(itemid event lineendings subject security allowmask year mon
    day hour min prop_* usejournal))

;; To be supplied by lj-editf.el
(lj--define-protocol-mode editfriendgroups nil
  "Edit the user's defined groups of friends."
  '(editfriend_groupmask_* efg_delete_* efg_set_*_name
    efg_set_*_sort efg_set_*_public))

;; To be supplied by lj-editf.el
(lj--define-protocol-mode editfriends 'lj-editf
  "Add, edit, or delete friends from the user's friends list."
  '(editfriend_delete_* editfriend_add_*_user editfriend_add_*_fg
    editfriend_add_*_bg editfriend_add_*_groupmask)
  '(friends_added friend_*_user friend_*_name))

(lj--define-protocol-mode friendof nil
  "Returns a list of which other LiveJournal users list this user
as their friend."
  '(friendoflimit)
  '(friendof_count friendof_*_user friendof_*_name friendof_*_bg
    friendof_*_fg friendof_*_type friendof_*_status))

(lj--define-protocol-mode getdaycounts 'lj-entry
  "This mode retrieves the number of journal entries per day.
Useful for populating calendar widgets in GUI clients."
  '(usejournal)
  '(*-*-*))

(lj--define-protocol-mode getevents 'lj-entry
  "Download parts of the user's journal."
  '(truncate prefersubject noprops selecttype lastsync year month
    day howmany beforedate itemid lineendings usejournal)
  '(events_count events_*_itemid events_*_eventtime events_*_event
    events_*_security events_*_allowmask events_*_subject prop_count
    prop_*_itemid prop_*_name prop_*_value))

(lj--define-protocol-mode getfriendgroups nil
  "Retrieves a list of the user's defined groups of friends."
  '()
  '(frgrp_*_name frgrp_*_sortorder frgrp_*_public frgrp_maxnum))

(lj--define-protocol-mode getfriends 'lj-login
  "Returns a list of which other LiveJournal users this user lists
as their friend."
  '(includefriendof includegroups includebdays friendlimit)
  '(friend_count friend_*_user friend_*_name friend_*_birthday
    friend_*_bg friend_*_fg friend_*_groupmask friend_*_type
    friend_*_status))

(lj--define-protocol-mode login 'lj-login
  "Log in to the server, while announcing your client version. The
server returns with whether the password is good or not, the
user's name, an optional message to be displayed to the user, and
the list of the user's friend groups. (friend groups can also be
retrieved using the getfriendgroups mode)"
  '(clientversion getmoods getmenus getpickws getpickwurls)
  '(name message frgrp_*_name fgrp_*_sortorder frgrp_maxnum
    access_count access_* mood_count mood_*_id mood_*_name
    menu_*_count menu_*_*_text menu_*_*_url menu_*_*_sub pickw_count
    pickw_* pickwurl_count pickwurl_* fastserver))

(lj--define-protocol-mode postevent 'lj-post
  "The most important mode, this is how a user actually submits a
new log entry to the server."
  '(event lineendings subject security allowmask year mon day hour
    min prop_* usejournal)
  '(itemid))

(lj--define-protocol-mode syncitems nil
  "Returns a list (or part of a list) of all the items (journal
entries, to-do items, comments) that have been created or updated
on LiveJournal since you last downloaded them. Note that the items
themselves are not returned --- only the item type and the item
number. After you get this you have to go fetch the items using
another protocol mode. For journal entries (type \"L\"), use the
getevents mode with a selecttype of \"syncitems\"."
  '(lastsync)
  '(sync_total sync_count sync_*_item sync_*_action sync_*_time))

(defun lj--ensure-protocol-mode (mode)
  (or (get mode 'lj-protocol-mode)
      (signal 'lj-unsupported-protocol-mode-error mode)))

;;;###autoload
(defun lj--make-request (mode fields)
  "Make a request object for protocol MODE with FIELDS.
FIELDS should be an alist."
  (lj--ensure-protocol-mode mode)
  (list 'lj-request mode fields))
(put 'lj--make-request 'lisp-indent-function 1)

;; Example usage:

;; (lj--make-request 'login
;;   ;; mode, user, {,h}password, and version should all be magically
;;   ;; taken care of.
;;   '((clientversion . lj--client-version)
;;     (getmoods      . nil) ; should be converted to 0
;;     (getmenus      . nil)
;;     (getpickws     . nil)
;;     (getpickwurls  . nil)))

(defmacro lj--between-p (elt left right)
  "Is ELT numerically between LEFT and RIGHT?"
  `(and (>= ,elt ,left) (<= ,elt ,right)))

(defun lj--url-escape (string)
  "Escape the STRING."
  (lj--initialize)
  (when (stringp string)
    (lj--message 9 "The string is \"%s\"." string)
    (let ((escaped-list '()))
      (mapcar
       (lambda (elt)
	 (when (not (string-equal elt ""))
	   (let ((char (ljc-char-to-int (or (string-to-char elt)
                                            ?\ ))))
	     (cond ((or (lj--between-p char (ljc-char-int ?A) (ljc-char-int ?Z))
			(lj--between-p char (ljc-char-int ?a) (ljc-char-int ?z))
			(lj--between-p char (ljc-char-int ?0) (ljc-char-int ?9)))
		    (setq escaped-list (nconc escaped-list (list elt))))
		   ((= char ?\ )
		    (setq escaped-list (nconc escaped-list (list "+"))))
		   (t
		    (setq escaped-list
			  (nconc escaped-list
				 (list (format "%%%02x" char)))))))))
       ;; We don't do the obvious (split-string string "") here
       ;; because that infloops in XEmacs 21.1.
       (mapcar (lambda (elt)
                 (make-string 1 elt))
               (string-to-list string)))
      (apply 'concat escaped-list))))

(defun lj--ensure-string (thingy)
  "Return a string rendition of THINGY.

Just how we convert to a string is a little odd. This isn't
intended to be a general-purpose to-string thingy, that's why
`format' exists. This is mainly for supporting reasonable formats
of things when building up request packets."

  (lj--url-escape
   (cond ((stringp thingy) thingy)
         ;; If thingy is `t' or `nil', it's a boolean value, at
         ;; least as far as the LJ protocol stuff is concerned.
         ((eq thingy t) "1")
         ((null thingy) "0")
         ;; Symbols can be used for their names
         ((symbolp thingy) (symbol-name thingy))
         ((numberp thingy) (number-to-string thingy))
         (t (error "Don't know what to do with `%s' thingy" thingy)))))

(defun lj--make-raw-request (profile request)
  "Using PROFILE, make a raw request from REQUEST."
  (or (and (eq (car request) 'lj-request)
           (get (cadr request) 'lj-protocol-mode)
           (listp (nth 2 request)))
      (signal 'lj-request-object-error request))

  (let*
      ((mode-name (symbol-name (cadr request)))
       (raw-request-top
        (concat
         "mode=" mode-name
         "&user=" (lj--url-escape (lj-profile-username profile))
         lj--password-argname (lj--url-escape (lj-profile-password profile)))))

    (concat raw-request-top
            (mapconcat (lambda (field)
                         (concat "&"
                                 ;; Note that `lj--ensure-string'
                                 ;; calls `lj--url-escape'.
                                 (lj--ensure-string (car field))
                                 "=" (lj--ensure-string (cdr field))))
                       (nth 2 request)
                       "")
            ;; Put ver last because sometimes the ^M screws the
            ;; previous thing up
            "&ver=" (number-to-string lj--protocol-version)
            "\r\n")))

(defun lj--send-request (profile request)
  "As PROFILE, send REQUEST."
  (let ((raw-request (lj--make-raw-request profile request)))
    (lj--message 9 "Raw request is \"%s\"" raw-request)
    (lj--send-raw-request profile raw-request)))

(provide 'lj-proto)
;;; lj-proto.el ends here
