;;; lj-url.el --- generic URL building for ljupdate

;; Copyright (C) 2002  Edward O'Connor <ted@oconnor.cx>

;; Author: Edward O'Connor <ted@oconnor.cx>
;; Keywords: convenience

;; This file is part of ljupdate.

;; ljupdate is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License as
;; published by the Free Software Foundation; either version 2, or
;; {at your option} any later version.

;; ljupdate is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public
;; License along with ljupdate, or with your Emacs. See the file
;; COPYING, or, if you're using GNU Emacs, try typing C-h C-c to
;; bring it up. If you're using XEmacs, C-h C-l does this. If you
;; do not have a copy, you can obtain one by writing to the Free
;; Software Foundation at this address:

;;                Free Software Foundation, Inc.
;;                59 Temple Place, Suite 330
;;                Boston, MA  02111-1307
;;                USA

;;; Commentary:

;; 

;;; Code:

(require 'lj-objs)

(defun lj-url-system-homepage (system)
  "Return a URL pointing at SYSTEM."
  (concat "http://"
          (lj-system-hostname system)
          (let ((port (lj-system-port system)))
            (if (and port (numberp port) (/= 80 port))
                (format ":%d" port)
              ""))
          "/"))

(defun lj-url-system-stats (system)
  "Return a URL pointing to SYSTEM's stats page."
  (concat (lj-url-system-homepage system) "stats.bml"))

(defun lj-url-system-random-user-entries (system)
  "Return a URL pointing to SYSTEM's random user page."
  (concat (lj-url-system-homepage system) "random.bml"))

(defun lj-url-system-username-entries (system username)
  "Return a URL pointing to SYSTEM's entries page for USERNAME."
  (concat (lj-url-system-homepage system)
          "users/" username "/"))

(defun lj-url-profile-entries (profile)
  "Return a URL pointing at PROFILE's entries page."
  (lj-url-system-username-entries (lj-profile-system profile)
                                  (lj-profile-username profile)))

(defun lj-url-profile-friends (profile)
  "Return a URL pointing at PROFILE's friends page."
  (concat (lj-url-profile-entries profile) "friends/"))

(defun lj-url-profile-calendar (profile)
  "Return a URL pointing at PROFILE's entries page."
  (concat (lj-url-profile-entries profile) "calendar/"))

(defun lj-url-profile-userinfo (profile)
  "Return a URL pointing at PROFILE's userinfo page."
  (concat (lj-url-system-homepage (lj-profile-system profile))
        "userinfo.bml?user=" (lj-profile-username profile)))

(defun lj-url-profile-userinfo-full (profile)
  "Return a URL pointing at PROFILE's full userinfo page."
  (concat (lj-url-profile-userinfo profile) "&mode=full"))

(defun lj-url-profile-memories (profile)
  "Return a URL pointing at PROFILE's memories page."
  (concat (lj-url-system-homepage (lj-profile-system profile))
        "tools/memories.bml?user=" (lj-profile-username profile)))

(defun lj-url-profile-pics (profile)
  "Return a URL pointing at PROFILE's userpics list page."
  (concat (lj-url-system-homepage (lj-profile-system profile))
          "allpics.bml?user=" (lj-profile-username profile)))

(defun lj-url-friend-entries (profile friend)
  "Return a URL pointing to PROFILE's FRIEND's entries page."
  (concat (lj-url-system-homepage (lj-profile-system profile))
          "users/" (lj-friend-username friend) "/"))

(provide 'lj-url)
;;; lj-url.el ends here
