;;; ljserver.el --- routines for talking to LiveJournal servers

;; Copyright (C) 2002  Edward O'Connor <ted@oconnor.cx>

;; Author: Edward O'Connor <ted@oconnor.cx>
;; Keywords: convenience

;; This file is part of ljupdate.

;; ljupdate is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License as
;; published by the Free Software Foundation; either version 2, or
;; {at your option} any later version.

;; ljupdate is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public
;; License along with ljupdate, or with your Emacs. See the file
;; COPYING, or, if you're using GNU Emacs, try typing C-h C-c to
;; bring it up. If you're using XEmacs, C-h C-l does this. If you
;; do not have a copy, you can obtain one by writing to the Free
;; Software Foundation at this address:

;;                Free Software Foundation, Inc.
;;                59 Temple Place, Suite 330
;;                Boston, MA  02111-1307
;;                USA

;;; Commentary:

;; 

;;; Code:

(require 'ljcompat)

(defcustom lj-proxy nil
  "*If non-nil, the hostname of the proxy server to use.
If nil, don't use a proxy."
  :type '(choice string (const nil))
  :group 'ljupdate)

(defcustom lj-proxy-port 80
  "*Port on the proxy to use."
  :type 'integer
  :group 'ljupdate)

(defcustom lj-timeout-interval 180
  "*How long to wait for the LiveJournal server response. In seconds."
  :type 'integer
  :group 'ljupdate)(defvar lj--last-request nil
  "The last request sent to the LJ server.
This is for debugging purposes.")

(defvar lj--last-response nil
  "The last response sent to the LJ server.
This is for debugging purposes.")

(lj--deferror lj-network-error (lj-error)
  "An unspecified network error has occured")

(lj--deferror lj-network-timeout-error (lj-network-error lj-error)
  "Network connection timed out")

(lj--deferror lj-http-error (lj-network-error lj-error)
  "The server response was not 200 OK")

(lj--deferror lj-no-success-key-error (lj-network-error lj-error)
  "The server did not return a SUCCESS key")

(lj--deferror lj-server-error (lj-network-error lj-error)
  "The server has reported an error")

(defun lj--read-response-alist ()
  "Create an alist from the LiveJournal server response."
  (lj--initialize)
  (let  ((lj-response-alist nil)
         (here (point))
         (end (progn (goto-char (point-max))
                     (point)))
         key
         val)

    (goto-char here)
    (while (< (point) end)
      (beginning-of-line)
      (setq key (buffer-substring (point) (progn (end-of-line)
                                                 (point))))
      (forward-line 1)
      (beginning-of-line)
      (setq val (buffer-substring (point) (progn (end-of-line)
                                                 (point))))
      (forward-line 1)
      (add-to-list 'lj-response-alist (cons key val)))
    lj-response-alist))


;; This function handles sending things to the LiveJournal server and
;; parsing return values from it.

;;;###autoload
(defun lj--send-raw-request (profile msg)
  "As PROFILE, send MSG to the LiveJournal server.
Return value is a cons: the car is non-nil if the transaction worked,
and nil if it didn't. The cdr is an alist of values from the server."
  (lj--initialize)
  (setq lj--last-request msg)
  (let* ((system (lj-profile-system profile))
         (worked-or-not nil)
         (response-alist nil)
         (msg-to-send
          (if (featurep 'xemacs)
              msg
            (if (and (fboundp 'coding-system-p)
                     (coding-system-p 'utf-8))
                (encode-coding-string
                 (decode-coding-string msg 'emacs-mule)
                 'utf-8)
              msg)))
         (headers (concat "POST "
                          (if lj-proxy
                              (lj-url-system-homepage system)
                            "")
                          "/interface/flat HTTP/1.0\r\n"
                          "Host: "
                          (lj-system-hostname system)
                          "\r\n"
			  (if (lj-profile-fast-p profile)
			      "Cookie: ljfastserver=1\r\n"
			    "")
                          "Content-type:"
                          "application/x-www-form-urlencoded\r\n"))
         (proc (open-network-stream "LiveJournal"
                                    "*LiveJournal-Server-Response*"
                                    (or lj-proxy
                                        (lj-system-hostname system))
                                    (if lj-proxy
                                        lj-proxy-port
                                      (lj-system-port system))))
         (buf (process-buffer proc)))

    ;; Perhaps this will allow for multibyte posting? We'll see.
    (unless (featurep 'xemacs)
      (when (and (fboundp 'coding-system-p)
                 (coding-system-p 'utf-8))
        (set-process-coding-system proc 'utf-8 'utf-8)))

    (process-send-string proc
                         (concat headers "Content-length: "
                                 (format "%d" (length msg-to-send))
                                 "\r\n\r\n" msg-to-send))

    (lj--message 2 "%s %s %s" "Waiting for a response from" system "...")

    ;; Watch us spin and stop Emacs from doing anything else!
    (while (equal (process-status proc) 'open)
      (when (not (accept-process-output proc lj-timeout-interval))
        (delete-process proc)
        (signal 'lj-network-timeout-error nil)))

    (lj--message 2 "Response received; processing...")

    (condition-case error-cons

        (with-current-buffer buf
          (goto-char (point-min))
          (while (search-forward "\C-m" nil t)
            (replace-match "" nil t))
          (goto-char (point-min))

          (if (looking-at "^HTTP/1.* 200 OK$")
              (progn
                (setq worked-or-not t)

                (while (not (looking-at "^$"))
                  (forward-line 1))
                (forward-line 1)
                (let* ((lj-response-alist (lj--read-response-alist))
                       (val (assoc "success" lj-response-alist)))
                  (setq response-alist lj-response-alist)
                  (if val
                      (if (string-equal (cdr val) "OK")
                          (setq worked-or-not t)
                        (signal 'lj-server-error
                                (cdr (assoc "errmsg" lj-response-alist))))
                    (signal 'lj-no-success-key-error nil))))
            (signal 'lj-http-error nil)))

      (lj-server-error
       (setq worked-or-not nil)
       (lj--message 1 "%s: %s" (get (car error-cons) 'error-message)
                    (cdr error-cons)))

      (lj-network-error
       (setq worked-or-not nil)
       (lj--message 1 "%s: %s" (get (car error-cons) 'error-message)
                    (cdr error-cons))))

    (setq lj--last-response response-alist)

    (when worked-or-not
      (lj--message 2 "%s %s%s" "Successfully talked to"
                   (lj-system-hostname system)
                   ".")
      (kill-buffer buf))

    (cons worked-or-not response-alist)))

(provide 'ljserver)
;;; ljserver.el ends here
