
/* Program to dump the registers of the TCIC/2 PCCIA controller
   in human-readable form.

   Robert Bedichek, robertb@cs.washington.edu
*/

#include <stdio.h>
#include <stdlib.h>

#ifdef linux
#include <asm/io.h>
#include <sys/types.h>
#include <sys/file.h>
#else
#include <dos.h>
#define inb_p 		inp
#define outb_p(val, addr) outp(addr, val)
#endif

typedef int ioaddr_t;
typedef unsigned char ioval_t;
typedef unsigned short ioval_short_t;

#define	EM_WINDOWS		(8)
#define	IO_WINDOWS		(4)
#define SOCKET_CONFIG_REGS	(2)

/* Offsets of TCIC/2 registers for the TCIC/2 base address. */

#define OFF_DATA		(0)
#define	OFF_ADDR		(2)
#define OFF_SCTRL		(6)
#define OFF_SSTAT		(7)
#define OFF_MODE		(8)
#define OFF_PWR			(9)
#define OFF_EDC			(10)
#define OFF_ICSR		(12)
#define OFF_IENA		(13)
#define OFF_AUX			(14)

/* Codes that are put in the top three bits of the R_ODE register
   to select which auxiliary register is visiable at OFF_AUX. */

#define AUX_TCTL 		(0<<5)
#define AUX_PCTL		(1<<5)
#define AUX_WCTL		(2<<5)
#define AUX_EXTERN 		(3<<5)
#define AUX_PDATA 		(4<<5)
#define AUX_SYSCFG		(5<<5)
#define AUX_ILOCK		(6<<5)
#define AUX_TEST		(7<<5)

/* Forward declarations of the register value printing functions. */

static void display_data(ioaddr_t);
static void display_addr(ioaddr_t);
static void display_sctrl(ioaddr_t);
static void display_sstat(ioaddr_t);
static void display_mode(ioaddr_t);
static void display_pwr(ioaddr_t);
static void display_edc(ioaddr_t);
static void display_icsr(ioaddr_t);
static void display_iena(ioaddr_t);
static void display_mem_window(ioaddr_t, int);
static void display_io_window(ioaddr_t, int);
static void display_socket_reg(ioaddr_t, int);
static void display_aux_tctl(ioaddr_t);
static void display_aux_pctl(ioaddr_t);
static void display_aux_wctl(ioaddr_t);
static void display_aux_extern(ioaddr_t);
static void display_aux_pdata(ioaddr_t);
static void display_aux_syscfg(ioaddr_t);
static void display_aux_ilock(ioaddr_t);
static void display_aux_test(ioaddr_t);

static ioval_short_t inb_p_short(ioaddr_t);
static void outb_p_short(ioval_short_t val, ioaddr_t addr);

/* If set, we print every register we know how to. */

static int verbose = 0;

/* If an argument is passed, use that for the base I/O address
   instead of the Databook default. */

void main(int argc, char *argv[])
{
  int i;
  ioaddr_t pcic_base = (ioaddr_t)0x240;
  ioval_short_t addr_save_low, addr_save_high;

  if (argc == 2)
    pcic_base = (ioaddr_t)atoi(argv[1]);

#ifdef unix
  if (ioperm(pcic_base, 16, 1) || ioperm(0x80, 1, 1)) {
    perror("ioperm");
    exit(1);
  }
#endif

  addr_save_low = inb_p_short(pcic_base + OFF_ADDR);
  addr_save_high = inb_p_short(pcic_base + OFF_ADDR + 2);

  display_addr(pcic_base);
  display_data(pcic_base);

  printf("Setting the socket bit in R_ADDR to 0\n");
  outb_p(inb_p(pcic_base + OFF_ADDR + 3) & ~0x10, pcic_base + OFF_ADDR + 3);

  display_sctrl(pcic_base);
  display_sstat(pcic_base);
  display_mode(pcic_base);
  display_pwr(pcic_base);
  if (verbose)
    display_edc(pcic_base);
  display_icsr(pcic_base);
  display_iena(pcic_base);

  for (i = 0 ; i < EM_WINDOWS ; i++) {
    display_mem_window(pcic_base, i);
  }

  for (i = 0 ; i < IO_WINDOWS ; i++) {
    display_io_window(pcic_base, i);
  }

  for (i = 0 ; i < SOCKET_CONFIG_REGS ; i++) {
    display_socket_reg(pcic_base, i);
  }

  if (verbose) {
    display_aux_tctl(pcic_base);
    display_aux_pctl(pcic_base);
  }
  display_aux_wctl(pcic_base);
  if (verbose) {
    display_aux_extern(pcic_base);
    display_aux_pdata(pcic_base);
  }
  display_aux_syscfg(pcic_base);
  display_aux_ilock(pcic_base);
  if (verbose)
    display_aux_test(pcic_base); 

  /* Restore the address register. */

  outb_p_short(addr_save_low, pcic_base + OFF_ADDR);
  outb_p_short(addr_save_high, pcic_base + OFF_ADDR + 2);
}
static void display_data(ioaddr_t pcic_base)
{
  ioval_short_t data = inb_p_short(pcic_base + OFF_DATA);
  printf("[%d] Data: 0x%04x\n",
          OFF_DATA, 
          data); 
}

static void display_addr(ioaddr_t pcic_base)
{
  ioval_short_t addr_low = inb_p_short(pcic_base + OFF_ADDR);
  ioval_short_t addr_high = inb_p_short(pcic_base + OFF_ADDR + 2);

  printf("[%d] Address: 0x%08X %s%s%s%s\n",
          OFF_ADDR, 
          (addr_high << 16) | addr_low,
          addr_high & 0x8000 ? "</CREG assertd>" : "",
          addr_high & 0x1000 ? "<socket 1>" : "<socket 0>",
          addr_high & 0x0800 ? "<indirect>" : "<card access>",
          addr_high & 0x0400 ? "<AR_TEST>" : ""); 
}

/* Display the socket control register. */

static void display_sctrl(ioaddr_t pcic_base)
{
  ioval_t sctrl = inb_p(pcic_base + OFF_SCTRL);
  printf("[%d] Socket Control: 0x%02x %s%s%s%s\n",
          OFF_SCTRL, 
          sctrl, 
          sctrl & 0x80 ? "<reset issued>" : "<reset cleared>",
          sctrl & 0x20 ? "<CRC>" : "<Simple>",
          sctrl & 0x10 ? ((sctrl & 0x08) ? "<auto-inc>" : "<2-byte-inc>")
                       : ((sctrl & 0x08) ? "<2-byte-hold>" : "<byte-hold>"),
          sctrl & 1    ? "<enabled>" : "<disabled>");
}

/* Display the socket status register. */

static void display_sstat(ioaddr_t pcic_base)
{
  ioval_t sstat = inb_p(pcic_base + OFF_SSTAT);
  printf("[%d] Socket Status: 0x%02x %s%s%s%s%s%s%s%s\n",
          OFF_SSTAT, 
          sstat,
          sstat & 0x80 ? "<inserted>" : "<no card>",
          sstat & 0x40 ? "<16bit>" : "<8bit>",
          sstat & 0x20 ? "" : "<IREQ asserted>",
          sstat & 0x10 ? "<Audio-1>" : "<Audio-0>",
          sstat & 0x08 ? "<card-change>" : "",
          sstat & 0x04 ? "<expired>" : "<counting>",
          sstat & 0x02 ? "<10usec>" : "<10usec counting>",
          sstat & 0x02 ? "<6usec>" : "<6usec counting>");
}

/* Display the mode register. */

static void display_mode(ioaddr_t pcic_base)
{
  ioval_t mode = inb_p(pcic_base + OFF_MODE);
  static char *aux_strings[8] = {"tctl", "pctl", "wctl", "extern", 
                                 "pdata", "syscfg", "ilock", "test"};
  printf("[%d] Mode: 0x%02x aux=%s %s%s%s%s\n",
          OFF_MODE,
          mode,
          aux_strings[(mode >> 5) & 7],
          mode & 0x10 ? "<word prog>" : "<byte prog>",
          mode & 0x08 ? "<wrt-bs-en>" : "<wrt-bs-dis>",
          mode & 0x04 ? "<card en>" : "<card dis>",
          mode & 0x02 ? "</CRD 1>" : "</CRD 0>",
          mode & 0x01 ? "</CWR 1>" : "</CWR 0>");
}

/* Display the power control register. */

static void display_pwr(ioaddr_t pcic_base)
{
  ioval_t pwr = inb_p(pcic_base + OFF_PWR);
  printf("[%d] Power: 0x%02x %s%s VPPSEL1,0=[%d,%d] VCCSEL1,0=[%d,%d]\n",
          OFF_PWR,
          pwr,
          pwr & 0x80 ? "<limited>" : "<current-ok>",
          pwr & 0x40 ? "<limit enabled>" : "<no limit>",
          pwr & 0x10,
          pwr & 0x08,
          pwr & 0x02,
          pwr & 0x01);
}

/* Display the error detect code register. */

static void display_edc(ioaddr_t pcic_base)
{
  ioval_t edc1 = inb_p(pcic_base + OFF_EDC);
  ioval_t edc2 = inb_p(pcic_base + OFF_EDC + 1);
  printf("[%d] Error: 0x%04x\n",
          OFF_EDC,
          (edc2 << 8) | edc1);
}

/* Display the interrupt control/status register. */

static void display_icsr(ioaddr_t pcic_base)
{
  ioval_t icsr = inb_p(pcic_base + OFF_ICSR);
  printf("[%d] Interrupt Status/Control: 0x%02x %s%s%s%s%s\n",
          OFF_ICSR,
          icsr,
          icsr & 0x80 ? "<IOCHK>" : "",
          icsr & 0x40 ? "<ANY R_SSTAT>" : "",
          icsr & 0x20 ? "<ERR>" : "",
          icsr & 0x10 ? "<PLS EXP>" : "",
          icsr & 0x08 ? "</CILOCK CHG>" : "");
}

/* Display the interrupt enable register. */

static void display_iena(ioaddr_t pcic_base)
{
  ioval_t iena = inb_p(pcic_base + OFF_IENA);
  printf("[%d] Interrupt Enable: 0x%02x\n",
          OFF_IENA,
          iena,
          iena & 0x40 ? "<IRQen r_sstat>" : "",
          iena & 0x20 ? "<IRQen /OVERCUR or IOCHK>" : "",
          iena & 0x10 ? "<IRQen pulse>" : "",
          iena & 0x08 ? "<IRQen change socket>" : "",
          iena & 0x02 ? (iena & 1 ? "<en-actvH-totem>" : "<en_actvL-totem>")
                      : (iena & 1 ? "<en-actvL-opend>" : "<disabled>"));

}

/* Display the memory window registers for I/O window 'window'. */

static void display_mem_window(ioaddr_t pcic_base, int window)
{
  ioval_short_t mext, mbase, mmap, mctl;

  /* Set the indirect bit in the address register. */
  outb_p(inb_p(pcic_base + OFF_ADDR + 2) | 8, pcic_base + OFF_ADDR + 2);

  /* This register is unused in the current implementation. */
  if (verbose) {
    outb_p_short(0x100 + (window << 3), pcic_base + OFF_ADDR);
    mext = inb_p_short(pcic_base + OFF_DATA);
  }

  outb_p_short(0x102 + (window << 3), pcic_base + OFF_ADDR);
  mbase = inb_p_short(pcic_base + OFF_DATA);

  outb_p_short(0x104 + (window << 3), pcic_base + OFF_ADDR);
  mmap = inb_p_short(pcic_base + OFF_DATA);

  outb_p_short(0x106 + (window << 3), pcic_base + OFF_ADDR);
  mctl = inb_p_short(pcic_base + OFF_DATA);

  printf("[%d] Memory Map %d: mbase=0x%x mmap=0x%x %s mctl=0x%x %s%s%s%s%s%s%s%s\n",
          OFF_DATA, window, mbase, 
          mmap, 
          mmap & 0x8000 ? "<attribute>" : "", 
          mctl,
          mctl & 0x8000 ? "<en>" : "",
          mctl & 0x1000 ? "<s1>" : "<s0>",
          mctl & 0x0800 ? "<16>" : "<8>",
          mctl & 0x0400 ? "<edc>" : "",
          mctl & 0x0200 ? "<Cache>" : "",
          mctl & 0x0100 ? "<Acc>" : "",
          mctl & 0x0080 ? "<WP>" : "",
          mctl & 0x0040 ? "<Quiet>" : "");
  if (verbose)
    printf("                  : mext=0x%x\n", mext);
}

/* Display the I/O window registers for I/O window 'window'. */

static void display_io_window(ioaddr_t pcic_base, int window)
{
  ioval_short_t iobase, ioctl;

  /* Set the indirect bit in the address register. */
  outb_p(inb_p(pcic_base + OFF_ADDR + 2) | 8, pcic_base + OFF_ADDR + 2);

  outb_p_short(0x200 + (window << 2), pcic_base + OFF_ADDR);
  iobase = inb_p_short(pcic_base + OFF_DATA);

  outb_p_short(0x202 + (window << 2), pcic_base + OFF_ADDR);
  ioctl = inb_p_short(pcic_base + OFF_DATA);

  printf("[%d] I/O Map %d: base=0x%x ioctl=0x%x %s%s%s%s%s%s%s\n",
          OFF_DATA, window, iobase, ioctl,
          ioctl & 0x8000 ? "<en>" : "",
          ioctl & 0x1000 ? "<s1>" : "<s0>",
          ioctl & 0x0800 ? (ioctl & 0x400 ? "<ATA>" : "<8>") 
                         : (ioctl & 0x400 ? "<16>" : "<IOIS16>"),
          ioctl & 0x0200 ? "<1byte>" : "",
          ioctl & 0x0100 ? "<Acc>" : "",
          ioctl & 0x0080 ? "<1klimit>" : "",
          ioctl & 0x0040 ? "<Quiet>" : "");
}

/* Display indirect socket configuration registers for socket
   number 'socket'. */

static void display_socket_reg(ioaddr_t pcic_base, int socket)
{
  ioval_short_t scf1, scf2;

  /* Set the indirect bit in the address register. */
  outb_p(inb_p(pcic_base + OFF_ADDR + 2) | 8, pcic_base + OFF_ADDR + 2);

  outb_p_short((socket << 3), pcic_base + OFF_ADDR);
  scf1 = inb_p_short(pcic_base + OFF_DATA);

  outb_p_short((socket << 3) + 2, pcic_base + OFF_ADDR);
  scf2 = inb_p_short(pcic_base + OFF_DATA);

  printf("[%d] Indirect Socket Configuration %d: scf1=0x%x scf2=0x%x\n",
          OFF_DATA, socket, scf1, scf2);
}

/* Display the timing control auxiliary register. */

static void display_aux_tctl(ioaddr_t pcic_base)
{
  ioval_short_t tctl;
  outb_p(AUX_TCTL | (inb_p(pcic_base + OFF_MODE) & 0x1f), pcic_base + OFF_MODE);

  tctl = inb_p_short(pcic_base + OFF_AUX);
  printf("[%d] Interrupt Enable: 0x%04x\n",
          OFF_AUX,
          tctl);
}

/* Display the programming pulse auxiliary register. */

static void display_aux_pctl(ioaddr_t pcic_base)
{
  ioval_short_t pctl;
  outb_p(AUX_PCTL | (inb_p(pcic_base + OFF_MODE) & 0x1f), pcic_base + OFF_MODE);

  pctl = inb_p_short(pcic_base + OFF_AUX);
  printf("[%d] Programming Pulse: 0x%04x\n",
          OFF_AUX,
          pctl);
}

/* Display the wait-state control auxiliary register. */

static void display_aux_wctl(ioaddr_t pcic_base)
{
  ioval_short_t wctl;
  outb_p(AUX_WCTL | (inb_p(pcic_base + OFF_MODE) & 0x1f), pcic_base + OFF_MODE);

  wctl = inb_p_short(pcic_base + OFF_AUX);
  printf("[%d] Wait-State Control: 0x%04x %s%s%s%s%s\n",
          OFF_AUX,
          wctl,
          wctl & 0x400 ? "</CEL>" : "",
          wctl & 0x200 ? "</CRD>" : "",
          wctl & 0x100 ? "</CWR>" : "",
          wctl & 0x080 ? "<BCLK>" : "<CCLK>",
          wctl & 0x040 ? "<Rising>" : "<Falling>",
          wctl & 0x020 ? "<sync>" : "<async>");
}

/* Display the external access auxiliary register. */

static void display_aux_extern(ioaddr_t pcic_base)
{
  ioval_short_t r_extern;
  outb_p(AUX_EXTERN | (inb_p(pcic_base + OFF_MODE) & 0x1f), pcic_base + OFF_MODE);

  r_extern = inb_p_short(pcic_base + OFF_AUX);
  printf("[%d] External Access: 0x%04x\n",
          OFF_AUX,
          r_extern);
}

/* Display the programming data auxiliary register. */

static void display_aux_pdata(ioaddr_t pcic_base)
{
  ioval_short_t pdata;
  outb_p(AUX_PDATA | (inb_p(pcic_base + OFF_MODE) & 0x1f), pcic_base + OFF_MODE);

  pdata = inb_p_short(pcic_base + OFF_AUX);
  printf("[%d] Programming Data: 0x%04x\n",
          OFF_AUX,
          pdata);
}

/* Display the system configuration auxiliary register. */

static void display_aux_syscfg(ioaddr_t pcic_base)
{
  ioval_short_t syscfg;
  static char *irq_strings[16]= {"STKIRQ", "RESERVED", "IRQ3", "IRQ4", 
                               "IRQ5", "IRQ6", "IRQ7", "RESERVED", "RESERVED", 
                               "IRQ10", "RESERVED", "RESERVED", "RESEVED", 
                               "IRQ14", "RESERVED"};

  outb_p(AUX_SYSCFG | (inb_p(pcic_base + OFF_MODE) & 0x1f), pcic_base + OFF_MODE);
  syscfg = inb_p_short(pcic_base + OFF_AUX);
  printf("[%d] System Configuration: 0x%04x %s%s\n",
          OFF_AUX,
          syscfg,
          syscfg & 0x8000 ? "<card-access>" : "",
          syscfg & 0x4000 ? 
                     "</BUSYLED tied to d15>" : "</BUSYLED tied to socket>");

  printf("                         : %s%s%s\n",
          syscfg & 0x2000 ? "</ULTI not asserted>" : "</MULTI asserted>",
          syscfg & 0x0100 ? "" : "<Zero-power>",
          syscfg & 0x0080 ? "<HIO16 totem>" : "<HIO16 open drain>");

  printf("                         : %s%s%s<%s>\n",
          syscfg & 0x0040 ? "<HMEM16 totem>" : "<HMEM16 open drain>",
          syscfg & 0x0020 ? "<safe dec>" : "",
          syscfg & 0x0010 ? "<Fast>" : "<ISA>",
          irq_strings[syscfg&0xf]);
}

/* Display the interlock control/status auxiliary register. */

static void display_aux_ilock(ioaddr_t pcic_base)
{
  ioval_short_t ilock;
  outb_p(AUX_ILOCK | (inb_p(pcic_base + OFF_MODE) & 0x1f), pcic_base + OFF_MODE);

  ilock = inb_p_short(pcic_base + OFF_AUX);
  printf("[%d] Interlock Ctl: 0x%04x %s%s%s%s%s%s%s\n",
          OFF_AUX,
          ilock,
          ilock & 0x80 ? (ilock & 0x40 ? "<2-stage>" : "<RESERVED>")
                       : (ilock & 0x40 ? "<RESERVED>" : "<Min-hold>"),
          ilock & 0x20 ? "</CWAITS1>" : "",
          ilock & 0x10 ? "</CWAIT eff>" : "",
          ilock & 0x08 ? "<bit 2 eff>" : "</CRESET hiZ>",
          ilock & 0x04 ? "</CRESET-1>" : "",
          ilock & 0x02 ? "</CICLOCK low>" : "</CICLOCK high>",
          ilock & 0x01 ? "</ILCKOUT low>" : "</ILCKOUT high>");
}

/* Display the test auxiliary register. */

static void display_aux_test(ioaddr_t pcic_base)
{
  ioval_short_t test;
  outb_p(AUX_TEST | (inb_p(pcic_base + OFF_MODE) & 0x1f), pcic_base + OFF_MODE);

  test = inb_p_short(pcic_base + OFF_AUX);
  printf("[%d] Test Control: 0x%04x\n",
          OFF_AUX,
          test);
}

/* Return a 16-bit value from I/O address 'addr' */

static ioval_short_t inb_p_short(ioaddr_t addr)
{
#ifdef linux
  return (inb_p(addr+1) << 8) | inb_p(addr);
#else
  return inpw(addr);
#endif
}

/* Write a 16-bit value 'val' to address 'addr' */

static void outb_p_short(ioval_short_t val, ioaddr_t addr)
{
#ifdef linux
  outb_p(val & 0xff, addr);
  outb_p((val >> 8) & 0xff, addr+1);
#else
  outpw(addr, val);
#endif
}
