/*
 * Copyright (c) 1988 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef lint
static char sccsid[] = "@(#)logwtmp.c	5.7 (Berkeley) 2/25/91";
#endif /* not lint */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <ttyent.h>
#include <fcntl.h>
#include <utmp.h>
#include <paths.h>
#include <string.h>

#define UTMPFILE	_PATH_UTMP
#define	WTMPFILE	_PATH_WTMP

static int fd = -1;
static int utmpfd = -1, ptrvalid = 0;
static off_t ptr;

#define MAX(x,y) ((x > y) ? x : y)

loguwtmp(linepid, name, host)
	char *linepid, *name, *host;
{
  struct utmp ut;
  struct stat buf;
  time_t time();
  char *strncpy();
  int cc, ttynum;
  char line[10];
  static int ftpline = 0, firsttime = 1;

  if (firsttime)
    {
      if (utmpfd == -1)
	utmpfd = open(UTMPFILE, O_RDWR, 0);
      ptrvalid = 0; ptr = 0;

      /*
       * We find the entry in utmp we want to use only once,
       * then just continue using it until we exit.
       * Unfortunately, utmp handling doesn't work with
       * a general mechanism; it's dependent on assumptions
       * about ttys or small numbers of lines for both BSD
       * in general and in AIX convenience routines.
       */
      if (utmpfd >= 0)
	{
#ifdef _AIX
	  /*
	   * Under AIX, we run through all of utmp looking for
	   * the appropriate place to put an ftp entry.
	   */
#else
	  /*
	   * Under BSD, the utmp entry location is determined
	   * by where your tty is in /etc/ttys. But we aren't
	   * using a tty. So we figure out how many ttys there
	   * are, and then write to utmp BEYOND the location
	   * of the last tty so we don't collide.
	   */
	  setttyent();
	  ttynum = 20; /* Safety in case /etc/ttys grows... */
	  while (getttyent() != 0)
	    ttynum++;
	  endttyent();
	  ptr = ttynum * sizeof(ut);
#endif

	  /*
	   * Standard tty handling in BSD doesn't require utmp
	   * to be locked since the /dev/tty* files provide the
	   * required locking mechanism. We have no such luxury;
	   * furthermore, locking of the utmp file is required
	   * under AIX, even if IBM's own software doesn't do it.
	   */
	  flock(utmpfd, LOCK_EX);
	  lseek(utmpfd, ptr, L_SET);

	  /* Scan for a line with the name ftpX */
	  while ((cc = read(utmpfd, &ut, sizeof(ut))) == sizeof(ut))
	    {
	      if (!strncmp("ftp", ut.ut_line, 3))
		{
		  /* If the ftp line here is empty, we're set. Emptiness
		   * is determined by a null username under BSD, and
		   * type DEAD_PROCESS under AIX - because I say so.
		   * The AIX manpage is not rich in suggestions
		   * of how this _should_ be done, and other software
		   * varies. */
#ifdef _AIX
		  if (ut.ut_type == DEAD_PROCESS)
#else
		  if (ut.ut_name[0] == '\0')
#endif
		    break;
		  ftpline++;
		}
	      ptr += cc;
	    }
	  if (cc == 0) /* EOF: add a new entry; leave ptr unchanged */
	    {
	      if (!fstat(utmpfd, &buf))
		{
		  ptrvalid = 1;
		}
	      else
		{ /*
		   * The fstat should never fail. The only reason
		   * it is done here is to get the current length
		   * of the file should it later prove necessary
		   * to truncate it to its original length when
		   * a write only partially succeeds.
		   */
		  flock(utmpfd, LOCK_UN);
		  close(utmpfd);
		  utmpfd = -1;
		}
	    }
	  else
	    if (cc == sizeof(ut))
	      {
		ptrvalid = 1;
	      }
	    else /* The utmp file has a bad length. Don't touch it. */
	      {
		flock(utmpfd, LOCK_UN);
		close(utmpfd);
		utmpfd = -1;
	      }
	}
    }

  bzero(&ut, sizeof(ut));
  if (ptrvalid)
    {
      /* Do this if we got to deal with utmp and got a real line number */
      sprintf(line, "ftp%d", ftpline);
      (void)strncpy(ut.ut_line, line, sizeof(ut.ut_line));
    }
  else /* otherwise, use the passed in line for wtmp logging only */
    (void)strncpy(ut.ut_line, linepid, sizeof(ut.ut_line));
  (void)strncpy(ut.ut_name, name, sizeof(ut.ut_name));
  (void)strncpy(ut.ut_host, host, sizeof(ut.ut_host));
  (void)time(&ut.ut_time);
#ifdef _AIX
  /* Note that name is only \0 in the case where the program
   * is exiting. */
  ut.ut_type = (name[0] == '\0' ? DEAD_PROCESS : USER_PROCESS);
  ut.ut_exit.e_exit = 2;
  ut.ut_pid = getpid();
#endif

  if (ptrvalid)
    {
      lseek(utmpfd, ptr, L_SET);
      cc = write(utmpfd, &ut, sizeof(ut));
      if (firsttime && cc != sizeof(ut))
	{
	  (void)ftruncate(utmpfd, buf.st_size);
	  ptrvalid = 0;
	  flock(utmpfd, LOCK_UN);
	  close(utmpfd);
	  utmpfd = -1;
	}
      else
	if (firsttime)
	  flock(utmpfd, LOCK_UN);
      /* In case this was the first time around and we had it locked...
       * Note that the file lock is only necessary for when we allocate
       * our slot. Afterwards, until we release the slot, its ours.
       */
    }

  firsttime = 0;

  logwtmp(ut.ut_line, ut.ut_name, ut.ut_host);
}

int user_logged_in(who)
     char *who;
{
  struct utmp ut;
  off_t p = 0;
#ifdef _AIX
  static int pid = 0;

  if (pid == 0)
    pid = getpid();
#endif

  if (utmpfd == -1)
    {
      utmpfd = open(UTMPFILE, O_RDWR, 0);
      if (utmpfd == -1)
	return 0; /* should this be what we do? XXX */
    }

  lseek(utmpfd, p, L_SET);
  while (read(utmpfd, &ut, sizeof(ut)) == sizeof(ut))
    {
      if (!strcmp(ut.ut_name, who) && 
#ifdef _AIX
	  (ut.ut_type == USER_PROCESS) &&
	  (ut.ut_pid != pid))
#else
	  ((ptrvalid && p != ptr) || !ptrvalid))
#endif
	return 1;
      else
	p += sizeof(ut);
    }

  return 0;
}

/*
 * Modified version of logwtmp that holds wtmp file open
 * after first call, for use with ftp (which may chroot
 * after login, but before logout).
 */
logwtmp(line, name, host)
	char *line, *name, *host;
{
	struct utmp ut;
	struct stat buf;
	time_t time();
	char *strncpy();

	if (fd < 0 && (fd = open(_PATH_WTMP, O_WRONLY|O_APPEND, 0)) < 0)
		return;
	if (!fstat(fd, &buf)) {
	        bzero(&ut, sizeof(ut));
		(void)strncpy(ut.ut_line, line, sizeof(ut.ut_line));
		(void)strncpy(ut.ut_name, name, sizeof(ut.ut_name));
		(void)strncpy(ut.ut_host, host, sizeof(ut.ut_host));
		(void)time(&ut.ut_time);
#ifdef _AIX
		ut.ut_pid = getpid();
		if (name[0] != '\0')
		  ut.ut_type = USER_PROCESS;
		else
		  ut.ut_type = DEAD_PROCESS;
#endif /* _AIX */

		if (write(fd, (char *)&ut, sizeof(struct utmp)) !=
		    sizeof(struct utmp))
			(void)ftruncate(fd, buf.st_size);
	}
}
