# Copyright (C) 2003 Peter J. Verveer
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met: 
#
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above
#    copyright notice, this list of conditions and the following
#    disclaimer in the documentation and/or other materials provided
#    with the distribution.
#
# 3. The name of the author may not be used to endorse or promote
#    products derived from this software without specific prior
#    written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
# OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
# DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
# GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
# WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
# NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.      

import sys
import unittest
import getopt
import math
import numarray
import numarray.nd_image
import numarray.fft
import numarray.numinclude as numinclude

eps = 1e-12

def mean_squared_error(a, b, m = None):
    # calculate the mean squared error between two arrays, optionally
    # using a mask
    if type(a) != type(numarray.array(())):
        a = numarray.array(a)
    if type(b) != type(numarray.array(())):
        b = numarray.array(b)
        
    if a.shape != b.shape:
        return 1.0

    if 0 in a.shape + b.shape:
        return 0.0

    if (a.type() in [numarray.Complex32, numarray.Complex64] or
        b.type() in [numarray.Complex32, numarray.Complex64]):
        a = numarray.array(a,  numarray.Complex)
        b = numarray.array(b, numarray.Complex)
        e1 = mean_squared_error(a.real, b.real, m)
        e2 = mean_squared_error(a.imag, b.imag, m)
        return math.sqrt(e1 * e1 + e2 * e2)
    else:
        ac = a.astype(numarray.Float64)
        bc = b.astype(numarray.Float64)
        diff = ac - bc
        diff = diff * diff
        if isinstance(m, numarray.NumArray):
            m = numarray.where(m, 1.0, 0.0)
            diff = diff * m
        if isinstance(diff, numarray.NumArray):
            return math.sqrt(numarray.sum(numarray.ravel(diff)))
        else:
            return diff

class NDImageTest(unittest.TestCase):

    def setUp(self):
        # list of numarray data types
	self.types = [numarray.Int8, numarray.UInt8,   numarray.Int16,
        	      numarray.UInt16, numarray.Int32,   numarray.UInt32,
                      numarray.Int64,  numarray.Float32, numarray.Float64]
	if numinclude.hasUInt64:
            self.types.append( numarray.UInt64 )

        # list of boundary modes:
        self.modes = ['nearest', 'wrap', 'reflect', 'constant']
        
    def test_correlate1d1(self):
        "test line correlation"
        array = numarray.array([1, 2])
        weights = numarray.array([2])
        output = numarray.nd_image.correlate1d(array, weights, 0)
        error = mean_squared_error(output, [2, 4])
        self.failUnless(error < eps)
        
    def test_correlate1d2(self):
        "test line correlation"
        array = numarray.array([1])
        weights = numarray.array([1, 1])
        output = numarray.nd_image.correlate1d(array, weights, 0)
        error = mean_squared_error(output, [2])
        self.failUnless(error < eps)
        
    def test_correlate1d3(self):
        "test line correlation"
        array = numarray.array([1, 2])
        weights = numarray.array([1, 1])
        output = numarray.nd_image.correlate1d(array, weights, 0)
        error = mean_squared_error(output, [2, 3])
        self.failUnless(error < eps)
      
    def test_correlate1d4(self):
        "test line correlation"
        array = numarray.array([1, 2, 3])
        weights = numarray.array([1, 1])
        output = numarray.nd_image.correlate1d(array, weights, 0)
        error = mean_squared_error(output, [2, 3, 5])
        self.failUnless(error < eps)
                
    def test_correlate1d5(self):
        "test line correlation"
        array = numarray.array([1, 2, 3])
        weights = numarray.array([1, 2, 3])
        output = numarray.nd_image.correlate1d(array, weights, 0)
        error = mean_squared_error(output, [9 , 14, 17])
        self.failUnless(error < eps)
        
    def test_correlate1d6(self):
        "test line correlation"
        array = numarray.array([1, 2, 3])
        weights = numarray.array([1, 2, 1])
        output = numarray.nd_image.correlate1d(array, weights, 0)
        error = mean_squared_error(output, [5, 8, 11])
        self.failUnless(error < eps)
        
    def test_correlate1d7(self):
        "test line correlation"
        array = numarray.array([1, 2, 3])
        weights = numarray.array([1, 2, -1])
        output = numarray.nd_image.correlate1d(array, weights, 0)
        error = mean_squared_error(output, [1, 2, 5])
        self.failUnless(error < eps)
        
    def test_correlate1d8(self):
        "test line correlation"
        array = numarray.array([[1, 2, 3],
                                [2, 4, 6]])
        weights = numarray.array([1, 2, 1])
        output = numarray.nd_image.correlate1d(array, weights, 0)
        error = mean_squared_error(output, [[5, 10, 15],
                                            [7, 14, 21]])
        self.failUnless(error < eps)


    def test_correlate1d9(self):
        "test line correlation"
        weights = numarray.array([1, 2, 1])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [2, 4, 6]], type1)
            for type2 in self.types:
                output = numarray.zeros((2, 3), numarray.Float64)
                numarray.nd_image.correlate1d(array, weights, 0,
                                              output_type = type2,
                                              output = output)
                error = mean_squared_error(output, [[5, 10, 15],
                                                    [7, 14, 21]])
                self.failUnless(error < eps)

    def test_correlate1d10(self):
        "test line correlation"
        weights = numarray.array([1, 2, 1])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [2, 4, 6]], type1)
            for type2 in self.types:
                output = numarray.zeros((2, 3), type2)
                numarray.nd_image.correlate1d(array, weights, 0,
                                              output_type = type2,
                                              output = output)
                error = mean_squared_error(output, [[5, 10, 15],
                                                    [7, 14, 21]])
                self.failUnless(error < eps)


    def test_correlate1d13(self):
        "test line correlation"
        weights = numarray.array([1, 2, 1])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [2, 4, 6]], type1)
            for type2 in self.types:
                output = numarray.zeros((2, 3), type2)
                numarray.nd_image.correlate1d(array, weights, 0,
                                              output_type = type2,
                                              mode = 'wrap', output = output)
                error = mean_squared_error(output, [[6, 12, 18],
                                                    [6, 12, 18]])
                self.failUnless(error < eps)

    def test_correlate1d14(self):
        "test line correlation"
        weights = numarray.array([1, 2, 1])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [2, 4, 6]], type1)
            for type2 in self.types:
                output = numarray.zeros((2, 3), type2)
                numarray.nd_image.correlate1d(array, weights, 0,
                                              output_type = type2,
                                              mode = 'nearest',
                                              output = output)
                error = mean_squared_error(output, [[5, 10, 15],
                                                    [7, 14, 21]])
                self.failUnless(error < eps)


    def test_correlate1d15(self):
        "test line correlation"
        weights = numarray.array([1, 2, 1])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [2, 4, 6]], type1)
            for type2 in self.types:
                output = numarray.zeros((2, 3), type2)
                numarray.nd_image.correlate1d(array, weights, 0,
                                              output_type = type2,
                                              mode = 'nearest',
                                              output = output,
                                     origin = -1)
                error = mean_squared_error(output, [[7, 14, 21],
                                                    [8, 16, 24]])
                self.failUnless(error < eps)


    def test_correlate1d16(self):
        "test line correlation"
        weights = numarray.array([1, 2, 1])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [2, 4, 6]], type1)
            for type2 in self.types:
                output = numarray.zeros((2, 3), type2)
                numarray.nd_image.correlate1d(array, weights, 0,
                                              output_type = type2,
                                              mode = 'nearest',
                                              output = output,
                                     origin = 1)
                error = mean_squared_error(output, [[4, 8, 12],
                                                    [5, 10, 15]])
                self.failUnless(error < eps)


    def test_correlate1(self):
        "Test correlation"
        array = numarray.array([1, 2, 3])
        kernel = numarray.array([1])
        output = numarray.nd_image.correlate(array, kernel)
        error = mean_squared_error(array, output)
        self.failUnless(error < eps)
      
    def test_correlate2(self):
        "Test correlation"
        array = numarray.array([1, 2, 3])
        kernel = numarray.array([1, 1])
        output = numarray.nd_image.correlate(array, kernel)
        error = mean_squared_error([2, 3, 5], output)
        self.failUnless(error < eps)
      
    def test_correlate3(self):
        "Test correlation"
        array = []
        kernel = numarray.array([1, 1])
        output = numarray.nd_image.correlate(array, kernel)
        error = mean_squared_error([], output)
        self.failUnless(error < eps)

    def test_correlate4(self):
        "Test correlation"
        array = numarray.array([[1, 2, 3],
                                [4, 5, 6]])
        kernel = numarray.array([[1, 1],
                                 [1, 1]])
        output = numarray.nd_image.correlate(array, kernel)
        error = mean_squared_error([[4, 6, 10],
                                    [10, 12, 16]], output)
        self.failUnless(error < eps)

    def test_correlate5(self):
        "Test correlation"
        array = numarray.array([[1, 2, 3],
                                [4, 5, 6]])
        kernel = numarray.array([[1, 0],
                                 [0, 1]])
        output = numarray.nd_image.correlate(array, kernel)
        error = mean_squared_error([[2, 3, 5],
                                    [5, 6, 8]], output)
        self.failUnless(error < eps)

    def test_correlate6(self):
        "Test correlation"
        kernel = numarray.array([[1, 0],
                                 [0, 1]])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [4, 5, 6]], type1)
            for type2 in self.types:
                output = numarray.nd_image.correlate(array, kernel,
                                                     output_type = type2)
                error = mean_squared_error([[2, 3, 5],
                                            [5, 6, 8]], output)
                self.failUnless(error < eps and output.type() == type2)

    def test_correlate7(self):
        "Test correlation"
        kernel = numarray.array([[1, 0],
                                 [0, 1]])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [4, 5, 6]], type1)
            for type2 in self.types:
                output = numarray.zeros(array.shape, type2)
                numarray.nd_image.correlate(array, kernel, output_type = type2,
                                            output = output)
                error = mean_squared_error([[2, 3, 5],
                                            [5, 6, 8]], output)
                self.failUnless(error < eps and output.type() == type2)

    def test_correlate8(self):
        "Test correlation"
        kernel = numarray.array([[1, 0],
                                 [0, 1]])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [4, 5, 6]], type1)
            output = numarray.zeros(array.shape, numarray.Float64)
            numarray.nd_image.correlate(array, kernel,
                                        output_type = numarray.Float32,
                                        output = output)
            error = mean_squared_error([[2, 3, 5],
                                        [5, 6, 8]], output)
            self.failUnless(error < eps and output.type == numarray.Float32)

    def test_correlate9(self):
        "Test correlation"
        kernel = numarray.array([[0.5, 0  ],
                                 [0  , 0.5]])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [4, 5, 6]], type1)
            output = numarray.zeros(array.shape, numarray.Float64)
            numarray.nd_image.correlate(array, kernel,
                                        output_type = numarray.Float32,
                                        output = output)
            error = mean_squared_error([[1, 1.5, 2.5],
                                        [2.5, 3, 4]], output)
            self.failUnless(error < eps and output.type == numarray.Float32)

    def test_correlate10(self):
        "Test correlation"
        array = numarray.array([1, 2, 3])
        kernel = numarray.array([1, 1])
        output = numarray.nd_image.correlate(array, kernel, origin = -1)
        error = mean_squared_error([3, 5, 6], output)
        self.failUnless(error < eps)

    def test_correlate11(self):
        "Test correlation"
        kernel = numarray.array([[1, 0],
                                 [0, 1]])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [4, 5, 6]], type1)
            output = numarray.zeros(array.shape, numarray.Float64)
            numarray.nd_image.correlate(array, kernel,
                                        output_type = numarray.Float32,
                                        output = output,
                                        mode = 'nearest', origin = -1)
            error = mean_squared_error([[6, 8, 9],
                                        [9, 11, 12]], output)
            self.failUnless(error < eps and output.type == numarray.Float32)

    def test_correlate12(self):
        "Test correlation"
        kernel = numarray.array([[1, 0],
                                 [0, 1]])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [4, 5, 6]], type1)
            output = numarray.zeros(array.shape, numarray.Float64)
            numarray.nd_image.correlate(array, kernel,
                                        output_type = numarray.Float32,
                                        output = output, mode = 'nearest',
                                        origin = [-1, 0])
            error = mean_squared_error([[5, 6, 8],
                                        [8, 9, 11]], output)
            self.failUnless(error < eps and output.type == numarray.Float32)

    def test_convolve1d1(self):
        "test line correlation"
        array = numarray.array([1, 2])
        weights = numarray.array([2])
        output = numarray.nd_image.convolve1d(array, weights, 0)
        error = mean_squared_error(output, [2, 4])
        self.failUnless(error < eps)
        
    def test_convolve1d2(self):
        "test line correlation"
        array = numarray.array([1])
        weights = numarray.array([1, 1])
        output = numarray.nd_image.convolve1d(array, weights, 0)
        error = mean_squared_error(output, [2])
        self.failUnless(error < eps)        

    def test_convolve1d3(self):
        "test line correlation"
        array = numarray.array([1, 2])
        weights = numarray.array([1, 1])
        output = numarray.nd_image.convolve1d(array, weights, 0)
        error = mean_squared_error(output, [3, 4])
        self.failUnless(error < eps)

    def convolve1d4(self):
        "test line correlation"
        array = numarray.array([1, 2, 3])
        weights = numarray.array([1, 1])
        output = numarray.nd_image.convolve1d(array, weights, 0)
        error = mean_squared_error(output, [3, 5, 6])
        self.failUnless(error < eps)

    def test_convolve1d5(self):
        "test line correlation"
        array = numarray.array([1, 2, 3])
        weights = numarray.array([1, 2, 3])
        output = numarray.nd_image.convolve1d(array, weights, 0)
        error = mean_squared_error(output, [7, 10, 15])
        self.failUnless(error < eps)
        
    def test_convolve1d6(self):
        "test line correlation"
        array = numarray.array([1, 2, 3])
        weights = numarray.array([1, 2, 1])
        output = numarray.nd_image.convolve1d(array, weights, 0)
        error = mean_squared_error(output, [5, 8, 11])
        self.failUnless(error < eps)
        
    def test_convolve1d7(self):
        "test line correlation"
        array = numarray.array([1, 2, 3])
        weights = numarray.array([1, 2, -1])
        output = numarray.nd_image.convolve1d(array, weights, 0)
        error = mean_squared_error(output, [3, 6, 7])
        self.failUnless(error < eps)
        
    def test_convolve1d8(self):
        "test line correlation"
        array = numarray.array([[1, 2, 3],
                                [2, 4, 6]])
        weights = numarray.array([1, 2, 1])
        output = numarray.nd_image.convolve1d(array, weights, 0)
        error = mean_squared_error(output, [[5, 10, 15],
                                            [7, 14, 21]])
        self.failUnless(error < eps)


    def test_convolve1d9(self):
        "test line correlation"
        weights = numarray.array([1, 2, 1])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [2, 4, 6]], type1)
            for type2 in self.types:
                output = numarray.zeros((2, 3), numarray.Float64)
                numarray.nd_image.convolve1d(array, weights, 0,
                                             output_type = type2,
                                             output = output)
                error = mean_squared_error(output, [[5, 10, 15],
                                                    [7, 14, 21]])
                self.failUnless(error < eps)

    def test_convolve1d10(self):
        "test line correlation"
        weights = numarray.array([1, 2, 1])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [2, 4, 6]], type1)
            for type2 in self.types:
                output = numarray.zeros((2, 3), type2)
                numarray.nd_image.convolve1d(array, weights, 0,
                                             output_type = type2,
                                             output = output)
                error = mean_squared_error(output, [[5, 10, 15],
                                                    [7, 14, 21]])
                self.failUnless(error < eps)


    def test_convolve1d13(self):
        "test line correlation"
        weights = numarray.array([1, 2, 1])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [2, 4, 6]], type1)
            for type2 in self.types:
                output = numarray.zeros((2, 3), type2)
                numarray.nd_image.convolve1d(array, weights, 0,
                                             output_type = type2,
                                             mode = 'wrap', output = output)
                error = mean_squared_error(output, [[6, 12, 18],
                                                    [6, 12, 18]])
                self.failUnless(error < eps)

    def test_convolve1d14(self):
        "test line correlation"
        weights = numarray.array([1, 2, 1])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [2, 4, 6]], type1)
            for type2 in self.types:
                output = numarray.zeros((2, 3), type2)
                numarray.nd_image.convolve1d(array, weights, 0,
                                             output_type = type2,
                                             mode = 'nearest', output = output)
                error = mean_squared_error(output, [[5, 10, 15],
                                                    [7, 14, 21]])
                self.failUnless(error < eps)


    def test_convolve1d15(self):
        "test line correlation"
        weights = numarray.array([1, 2, 1])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [2, 4, 6]], type1)
            for type2 in self.types:
                output = numarray.zeros((2, 3), type2)
                numarray.nd_image.convolve1d(array, weights, 0,
                                             output_type = type2,
                                             mode = 'nearest', output = output,
                                             origin = 1)
                error = mean_squared_error(output, [[7, 14, 21],
                                                    [8, 16, 24]])
                self.failUnless(error < eps)


    def test_convolve1d16(self):
        "test line correlation"
        weights = numarray.array([1, 2, 1])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [2, 4, 6]], type1)
            for type2 in self.types:
                output = numarray.zeros((2, 3), type2)
                numarray.nd_image.convolve1d(array, weights, 0,
                                             output_type = type2,
                                             mode = 'nearest', output = output,
                                             origin = -1)
                error = mean_squared_error(output, [[4, 8, 12],
                                                    [5, 10, 15]])
                self.failUnless(error < eps)


    def test_convolve1(self):
        "Test correlation"
        array = numarray.array([1, 2, 3])
        kernel = numarray.array([1])
        output = numarray.nd_image.convolve(array, kernel)
        error = mean_squared_error(array, output)
        self.failUnless(error < eps)
      
    def test_convolve2(self):
        "Test correlation"
        array = numarray.array([1, 2, 3])
        kernel = numarray.array([1, 1])
        output = numarray.nd_image.convolve(array, kernel)
        error = mean_squared_error([3, 5, 6], output)
        self.failUnless(error < eps)
      
    def test_convolve3(self):
        "Test correlation"
        array = []
        kernel = numarray.array([1, 1])
        output = numarray.nd_image.convolve(array, kernel)
        error = mean_squared_error([], output)
        self.failUnless(error < eps)

    def test_convolve4(self):
        "Test correlation"
        array = numarray.array([[1, 2, 3],
                                [4, 5, 6]])
        kernel = numarray.array([[1, 1],
                                 [1, 1]])
        output = numarray.nd_image.convolve(array, kernel)
        error = mean_squared_error([[12, 16, 18],
                                    [18, 22, 24]], output)
        self.failUnless(error < eps)

    def test_convolve5(self):
        "Test correlation"
        array = numarray.array([[1, 2, 3],
                                [4, 5, 6]])
        kernel = numarray.array([[1, 0],
                                 [0, 1]])
        output = numarray.nd_image.convolve(array, kernel)
        error = mean_squared_error([[6, 8, 9],
                                    [9, 11, 12]], output)
        self.failUnless(error < eps)

    def test_convolve6(self):
        "Test correlation"
        kernel = numarray.array([[1, 0],
                                 [0, 1]])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [4, 5, 6]], type1)
            for type2 in self.types:
                output = numarray.nd_image.convolve(array, kernel,
                                                    output_type = type2)
                error = mean_squared_error([[6, 8, 9],
                                            [9, 11, 12]], output)
                self.failUnless(error < eps and output.type() == type2)

    def test_convolve7(self):
        "Test correlation"
        kernel = numarray.array([[1, 0],
                                 [0, 1]])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [4, 5, 6]], type1)
            for type2 in self.types:
                output = numarray.zeros(array.shape, type2)
                numarray.nd_image.convolve(array, kernel, output_type = type2,
                                           output = output)
                error = mean_squared_error([[6, 8, 9],
                                            [9, 11, 12]], output)
                self.failUnless(error < eps and output.type() == type2)

    def test_convolve8(self):
        "Test correlation"
        kernel = numarray.array([[1, 0],
                                 [0, 1]])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [4, 5, 6]], type1)
            output = numarray.zeros(array.shape, numarray.Float64)
            numarray.nd_image.convolve(array, kernel,
                                       output_type = numarray.Float32,
                                       output = output)
            error = mean_squared_error([[6, 8, 9],
                                        [9, 11, 12]], output)
            self.failUnless(error < eps and output.type == numarray.Float32)

    def test_convolve9(self):
        "Test correlation"
        kernel = numarray.array([[0.5, 0  ],
                                 [0  , 0.5]])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [4, 5, 6]], type1)
            output = numarray.zeros(array.shape, numarray.Float64)
            numarray.nd_image.convolve(array, kernel,
                                       output_type = numarray.Float32,
                                       output = output)
            error = mean_squared_error([[3, 4, 4.5],
                                        [4.5, 5.5, 6]], output)
            self.failUnless(error < eps and output.type == numarray.Float32)

    def test_convolve10(self):
        "Test correlation"
        array = numarray.array([1, 2, 3])
        kernel = numarray.array([1, 1])
        output = numarray.nd_image.convolve(array, kernel, origin = -1)
        error = mean_squared_error([2, 3, 5], output)
        self.failUnless(error < eps)

    def test_convolve11(self):
        "Test correlation"
        kernel = numarray.array([[1, 0],
                                 [0, 1]])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [4, 5, 6]], type1)
            output = numarray.zeros(array.shape, numarray.Float64)
            numarray.nd_image.convolve(array, kernel,
                                       output_type = numarray.Float32,
                                       output = output, mode = 'nearest',
                                       origin = -1)
            error = mean_squared_error([[2, 3, 5],
                                        [5, 6, 8]], output)
            self.failUnless(error < eps and output.type == numarray.Float32)

    def test_convolve12(self):
        "Test correlation"
        kernel = numarray.array([[1, 0],
                                 [0, 1]])
        for type1 in self.types:
            array = numarray.array([[1, 2, 3],
                                    [4, 5, 6]], type1)
            output = numarray.zeros(array.shape, numarray.Float64)
            numarray.nd_image.convolve(array, kernel,
                                       output_type = numarray.Float32,
                                       output = output, mode = 'nearest',
                                       origin = [-1, 0])
            error = mean_squared_error([[3, 5, 6],
                                        [6, 8, 9]], output)
            self.failUnless(error < eps and output.type == numarray.Float32)

    def test_gauss1(self):
        "Test gaussian filter"
        input = numarray.array([[1, 2, 3],
                                [2, 4, 6]], numarray.Float32)
        output = numarray.nd_image.gaussian_filter(input, 0)
        error = mean_squared_error(output, input)
        self.failUnless(error < eps)
        
    def test_gauss2(self):
        "Test gaussian filter"
        input = numarray.array([[1, 2, 3],
                                [2, 4, 6]], numarray.Float32)
        output = numarray.nd_image.gaussian_filter(input, [1.0, 1.0])
        self.failUnless(input.type() == output.type() and
                        input.shape == output.shape)
        
    def test_gauss3(self):
        "Test gaussian filter"
        input = numarray.arange(100 * 100).astype(numarray.Float32)
        input.shape = (100, 100)
        output = numarray.nd_image.gaussian_filter(input, [1.0, 1.0])
        error = mean_squared_error(input, output)
        self.failUnless(input.type() == output.type() and
                        input.shape == output.shape and
                        output.sum() - input.sum() < eps and error > 1.0)
        
    def test_gauss4(self):
        "Test gaussian filter"
        input = numarray.arange(100 * 100).astype(numarray.Float32)
        input.shape = (100, 100)
        otype = numarray.Float64
        output = numarray.nd_image.gaussian_filter(input, [1.0, 1.0],
                                                   output_type = otype)
        error = mean_squared_error(input, output)
        self.failUnless(output.type() == numarray.Float64 and
                        input.shape == output.shape and error > 1.0)
        
    def test_gauss5(self):
        "Test gaussian filter"
        input = numarray.arange(100 * 100).astype(numarray.Float32)
        input.shape = (100, 100)
        otype = numarray.Float64
        output = numarray.nd_image.gaussian_filter(input, [1.0, 1.0],
                                                   order = 1,
                                                   output_type = otype)
        error = mean_squared_error(input, output)
        self.failUnless(output.type() == numarray.Float64 and
                        input.shape == output.shape and error > 1.0)

    def test_boxcar1(self):
        "Test boxcar filter"
        array = numarray.array([1, 2, 3])
        filter_shape = [0]
        output = numarray.nd_image.boxcar_filter(array, filter_shape)
        error = mean_squared_error(array, output)
        self.failUnless(error < eps)
        
    def test_boxcar2(self):
        "Test boxcar filter"
        array = numarray.array([1, 2, 3])
        filter_shape = [1]
        output = numarray.nd_image.boxcar_filter(array, filter_shape)
        error = mean_squared_error(array, output)
        self.failUnless(error < eps)
      
    def test_boxcar3(self):
        "Test boxcar filter"
        array = numarray.array([2, 4, 6])
        filter_shape = [2]
        output = numarray.nd_image.boxcar_filter(array, filter_shape)
        error = mean_squared_error([2, 3, 5], output)
        self.failUnless(error < eps)
      
    def test_boxcar4(self):
        "Test boxcar filter"
        array = []
        filter_shape = [1]
        output = numarray.nd_image.boxcar_filter(array, filter_shape)
        error = mean_squared_error([], output)
        self.failUnless(error < eps)

    def test_boxcar5(self):
        "Test boxcar filter"
        filter_shape = [2, 2]
        for type1 in self.types:
            array = numarray.array([[4, 8, 12],
                                    [16, 20, 24]], type1)
            for type2 in self.types:
                output = numarray.nd_image.boxcar_filter(array, filter_shape,
                                                         output_type = type2)
                error = mean_squared_error([[4, 6, 10],
                                            [10, 12, 16]], output)
                self.failUnless(error < eps and output.type() == type2)

    def test_boxcar8(self):
        "Test boxcar filter"
        array = numarray.array([2, 4, 6])
        size = 2
        output = numarray.nd_image.boxcar_filter1d(array, size, 0, origin = -1)
        error = mean_squared_error([3, 5, 6], output)
        self.failUnless(error < eps)

    def test_min1(self):
        "Test min filter"
        array = numarray.array([1, 2, 3, 4, 5])
        filter_shape = numarray.array([2])
        output = numarray.nd_image.minimum_filter(array, filter_shape)
        error = mean_squared_error([1, 1, 2, 3, 4], output)
        self.failUnless(error < eps)

    def test_min2(self):
        "Test min filter"
        array = numarray.array([1, 2, 3, 4, 5])
        filter_shape = numarray.array([3])
        output = numarray.nd_image.minimum_filter(array, filter_shape)
        error = mean_squared_error([1, 1, 2, 3, 4], output)
        self.failUnless(error < eps)


    def test_min3(self):
        "Test min filter"
        array = numarray.array([3, 2, 5, 1, 4])
        filter_shape = numarray.array([2])
        output = numarray.nd_image.minimum_filter(array, filter_shape)
        error = mean_squared_error([3, 2, 2, 1, 1], output)
        self.failUnless(error < eps)

    def test_min4(self):
        "Test min filter"
        array = numarray.array([3, 2, 5, 1, 4])
        filter_shape = numarray.array([3])
        output = numarray.nd_image.minimum_filter(array, filter_shape)
        error = mean_squared_error([2, 2, 1, 1, 1], output)
        self.failUnless(error < eps)


    def test_min5(self):
        "Test min filter"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        filter_shape = numarray.array([2, 3])
        output = numarray.nd_image.minimum_filter(array, filter_shape)
        error = mean_squared_error([[2, 2, 1, 1, 1],
                                    [2, 2, 1, 1, 1],
                                    [5, 3, 3, 1, 1]], output)
        self.failUnless(error < eps)

    def test_min6(self):
        "Test min filter"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 1, 1], [1, 1, 1]]
        output = numarray.nd_image.minimum_filter(array, footprint = footprint)
        error = mean_squared_error([[2, 2, 1, 1, 1],
                                    [2, 2, 1, 1, 1],
                                    [5, 3, 3, 1, 1]], output)
        self.failUnless(error < eps)

    def test_min7(self):
        "Test min filter"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        output = numarray.nd_image.minimum_filter(array, footprint = footprint)
        error = mean_squared_error([[2, 2, 1, 1, 1],
                                    [2, 3, 1, 3, 1],
                                    [5, 5, 3, 3, 1]], output)
        self.failUnless(error < eps)

    def test_min8(self):
        "Test min filter"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        output = numarray.nd_image.minimum_filter(array, footprint = footprint,
                                                  origin = -1)
        error = mean_squared_error([[3, 1, 3, 1, 4],
                                    [5, 3, 3, 1, 1],
                                    [3, 3, 1, 1, 1]], output)
        self.failUnless(error < eps)

    def test_min9(self):
        "Test min filter"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        output = numarray.nd_image.minimum_filter(array, footprint = footprint,
                                                  origin = [-1, 0])
        error = mean_squared_error([[2, 3, 1, 3, 1],
                                    [5, 5, 3, 3, 1],
                                    [5, 3, 3, 1, 1]], output)
        self.failUnless(error < eps)

    def test_max1(self):
        "Test max filter"
        array = numarray.array([1, 2, 3, 4, 5])
        filter_shape = numarray.array([2])
        output = numarray.nd_image.maximum_filter(array, filter_shape)
        error = mean_squared_error([1, 2, 3, 4, 5], output)
        self.failUnless(error < eps)

    def test_max2(self):
        "Test max filter"
        array = numarray.array([1, 2, 3, 4, 5])
        filter_shape = numarray.array([3])
        output = numarray.nd_image.maximum_filter(array, filter_shape)
        error = mean_squared_error([2, 3, 4, 5, 5], output)
        self.failUnless(error < eps)


    def test_max3(self):
        "Test max filter"
        array = numarray.array([3, 2, 5, 1, 4])
        filter_shape = numarray.array([2])
        output = numarray.nd_image.maximum_filter(array, filter_shape)
        error = mean_squared_error([3, 3, 5, 5, 4], output)
        self.failUnless(error < eps)

    def test_max4(self):
        "Test max filter"
        array = numarray.array([3, 2, 5, 1, 4])
        filter_shape = numarray.array([3])
        output = numarray.nd_image.maximum_filter(array, filter_shape)
        error = mean_squared_error([3, 5, 5, 5, 4], output)
        self.failUnless(error < eps)


    def test_max5(self):
        "Test max filter"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        filter_shape = numarray.array([2, 3])
        output = numarray.nd_image.maximum_filter(array, filter_shape)
        error = mean_squared_error([[3, 5, 5, 5, 4],
                                    [7, 9, 9, 9, 5],
                                    [8, 9, 9, 9, 7]], output)
        self.failUnless(error < eps)


    def test_max6(self):
        "Test max filter"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 1, 1], [1, 1, 1]]
        output = numarray.nd_image.maximum_filter(array, footprint = footprint)
        error = mean_squared_error([[3, 5, 5, 5, 4],
                                    [7, 9, 9, 9, 5],
                                    [8, 9, 9, 9, 7]], output)
        self.failUnless(error < eps)


    def test_max7(self):
        "Test max filter"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        output = numarray.nd_image.maximum_filter(array, footprint = footprint)
        error = mean_squared_error([[3, 5, 5, 5, 4],
                                    [7, 7, 9, 9, 5],
                                    [7, 9, 8, 9, 7]], output)
        self.failUnless(error < eps)

    def test_max8(self):
        "Test max filter"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        output = numarray.nd_image.maximum_filter(array, footprint = footprint,
                                                  origin = -1)
        error = mean_squared_error([[7, 9, 9, 5, 5],
                                    [9, 8, 9, 7, 5],
                                    [8, 8, 7, 7, 1]], output)
        self.failUnless(error < eps)

    def test_max9(self):
        "Test max filter"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        output = numarray.nd_image.maximum_filter(array, footprint = footprint,
                                                  origin = [-1, 0])
        error = mean_squared_error([[7, 7, 9, 9, 5],
                                    [7, 9, 8, 9, 7],
                                    [8, 8, 8, 7, 7]], output)
        self.failUnless(error < eps)

    def test_rank1(self):
        "Test rank filter"
        array = numarray.array([1, 2, 3, 4, 5])
        kernel = 2
        output = numarray.nd_image.rank_filter(array, 1, size = kernel)
        error = mean_squared_error([1, 2, 3, 4, 5], output)
        self.failUnless(error < eps)

    def test_rank2(self):
        "Test rank filter"
        array = numarray.array([1, 2, 3, 4, 5])
        kernel = numarray.array([3])
        output = numarray.nd_image.rank_filter(array, 1, size = kernel)
        error = mean_squared_error([1, 2, 3, 4, 5], output)
        self.failUnless(error < eps)


    def test_rank3(self):
        "Test rank filter"
        array = numarray.array([3, 2, 5, 1, 4])
        kernel = [2]
        output = numarray.nd_image.rank_filter(array, 1, size = kernel)
        error = mean_squared_error([3, 3, 5, 5, 4], output)
        self.failUnless(error < eps)

    def test_rank4(self):
        "Test rank filter"
        array = numarray.array([3, 2, 5, 1, 4])
        kernel = 3
        output = numarray.nd_image.rank_filter(array, 1, size = kernel)
        error = mean_squared_error([3, 3, 2, 4, 4], output)
        self.failUnless(error < eps)

    def test_rank5(self):
        "Test rank filter"
        array = numarray.array([3, 2, 5, 1, 4])
        kernel = 3
        output = numarray.nd_image.rank_filter(array, -2, size = kernel)
        error = mean_squared_error([3, 3, 2, 4, 4], output)
        self.failUnless(error < eps)

    def test_rank6(self):
        "Test rank filter"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [5, 8, 3, 7, 1],
                                [5, 6, 9, 3, 5]])
        kernel = [2, 3]
        output = numarray.nd_image.rank_filter(array, 1, size = kernel)
        error = mean_squared_error([[2, 2, 1, 1, 1],
                                    [3, 3, 2, 1, 1],
                                    [5, 5, 3, 3, 1]], output)
        self.failUnless(error < eps)

    def test_rank7(self):
        "Test rank filter"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [5, 8, 3, 7, 1],
                                [5, 6, 9, 3, 5]])
        kernel = numarray.array([2, 3])
        output = numarray.nd_image.rank_filter(array, -2, size = kernel)
        error = mean_squared_error([[3, 5, 5, 5, 4],
                                    [5, 5, 7, 5, 4],
                                    [6, 8, 8, 7, 5]], output)
        self.failUnless(error < eps)

    def test_rank8(self):
        "Test rank filter"
        for type in self.types:
            array = numarray.array([[3, 2, 5, 1, 4],
                                    [5, 8, 3, 7, 1],
                                    [5, 6, 9, 3, 5]], type)
            footprint = [[1, 0, 1], [0, 1, 0]]
            output = numarray.nd_image.rank_filter(array, 1,
                                                   footprint = footprint)
            error = mean_squared_error([[3, 3, 2, 4, 4],
                                        [3, 5, 2, 5, 1],
                                        [5, 5, 8, 3, 5]], output)
            self.failUnless(error < eps)


    def test_rank9(self):
        "Test min filter, called by rank_filter"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        output = numarray.nd_image.rank_filter(array, 0, footprint = footprint)
        error = mean_squared_error([[2, 2, 1, 1, 1],
                                    [2, 3, 1, 3, 1],
                                    [5, 5, 3, 3, 1]], output)
        self.failUnless(error < eps)


    def test_rank10(self):
        "Test max filter,called by rank_filter"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        output = numarray.nd_image.rank_filter(array, -1,
                                               footprint = footprint)
        error = mean_squared_error([[3, 5, 5, 5, 4],
                                    [7, 7, 9, 9, 5],
                                    [7, 9, 8, 9, 7]], output)
        self.failUnless(error < eps)


    def test_rank11(self):
        "Test rank filter"
        for type in self.types:
            array = numarray.array([[3, 2, 5, 1, 4],
                                    [5, 8, 3, 7, 1],
                                    [5, 6, 9, 3, 5]], type)
            footprint = [[1, 0, 1], [0, 1, 0]]
            output = numarray.nd_image.rank_filter(array, 1,
                                                   footprint = footprint,
                                                   origin = -1)
            error = mean_squared_error([[5, 2, 5, 1, 4],
                                        [5, 8, 3, 5, 1],
                                        [6, 6, 5, 5, 5]], output)
            self.failUnless(error < eps)

    def test_rank12(self):
        "Test rank filter"
        for type in self.types:
            array = numarray.array([[3, 2, 5, 1, 4],
                                    [5, 8, 3, 7, 1],
                                    [5, 6, 9, 3, 5]], type)
            footprint = [[1, 0, 1], [0, 1, 0]]
            output = numarray.nd_image.rank_filter(array, 1,
                                                   footprint = footprint,
                                                   origin = [-1, 0])
            error = mean_squared_error([[3, 5, 2, 5, 1],
                                        [5, 5, 8, 3, 5],
                                        [5, 6, 6, 5, 5]], output)
            self.failUnless(error < eps)

    def test_percentile1(self):
        "Test percentile filter"
        array = numarray.array([1, 2, 3, 4, 5])
        kernel = 2
        output = numarray.nd_image.percentile_filter(array, 100, size = kernel)
        error = mean_squared_error([1, 2, 3, 4, 5], output)
        self.failUnless(error < eps)

    def test_percentile2(self):
        "Test percentile filter"
        array = numarray.array([1, 2, 3, 4, 5])
        kernel = 3
        output = numarray.nd_image.percentile_filter(array, 50, size = kernel)
        error = mean_squared_error([1, 2, 3, 4, 5], output)
        self.failUnless(error < eps)

    def test_percentile3(self):
        "Test percentile filter"
        array = numarray.array([3, 2, 5, 1, 4])
        kernel = 2
        output = numarray.nd_image.percentile_filter(array, 100, size = kernel)
        error = mean_squared_error([3, 3, 5, 5, 4], output)
        self.failUnless(error < eps)

    def test_percentile4(self):
        "Test percentile filter"
        array = numarray.array([3, 2, 5, 1, 4])
        kernel = 3
        output = numarray.nd_image.percentile_filter(array, 50, size = kernel)
        error = mean_squared_error([3, 3, 2, 4, 4], output)
        self.failUnless(error < eps)

    def test_percentile5(self):
        "Test percentile filter"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [5, 8, 3, 7, 1],
                                [5, 6, 9, 3, 5]])
        kernel = (2, 3)
        output = numarray.nd_image.percentile_filter(array, 17, size = kernel)
        error = mean_squared_error([[2, 2, 1, 1, 1],
                                    [3, 3, 2, 1, 1],
                                    [5, 5, 3, 3, 1]], output)
        self.failUnless(error < eps)

    def test_percentile6(self):
        "Test percentile filter"
        for type in self.types:
            array = numarray.array([[3, 2, 5, 1, 4],
                                    [5, 8, 3, 7, 1],
                                    [5, 6, 9, 3, 5]], type)
            footprint = numarray.array([[1, 0, 1], [0, 1, 0]])
            output = numarray.nd_image.percentile_filter(array, 35,
                                                         footprint = footprint)
            error = mean_squared_error([[3, 3, 2, 4, 4],
                                        [3, 5, 2, 5, 1],
                                        [5, 5, 8, 3, 5]], output)

            self.failUnless(error < eps)


    def test_percentile7(self):
        "Test min filter, called by percentile_filter"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        output = numarray.nd_image.percentile_filter(array, 0.0,
                                                     footprint = footprint)
        error = mean_squared_error([[2, 2, 1, 1, 1],
                                    [2, 3, 1, 3, 1],
                                    [5, 5, 3, 3, 1]], output)
        self.failUnless(error < eps)


    def test_percentile8(self):
        "Test max filter, called by percentile_filter"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        output = numarray.nd_image.percentile_filter(array, 100.0,
                                                     footprint = footprint)
        error = mean_squared_error([[3, 5, 5, 5, 4],
                                    [7, 7, 9, 9, 5],
                                    [7, 9, 8, 9, 7]], output)
        self.failUnless(error < eps)


    def test_median1(self):
        "Test median filter"
        array = numarray.array([1, 2, 3, 4, 5])
        kernel = 2
        output = numarray.nd_image.median_filter(array, kernel)
        error = mean_squared_error([1, 2, 3, 4, 5], output)
        self.failUnless(error < eps)

    def test_median2(self):
        "Test median filter"
        array = numarray.array([1, 2, 3, 4, 5])
        kernel = (3,)
        output = numarray.nd_image.median_filter(array, kernel)
        error = mean_squared_error([1, 2, 3, 4, 5], output)
        self.failUnless(error < eps)

    def test_median3(self):
        "Test median filter"
        array = numarray.array([3, 2, 5, 1, 4])
        kernel = 2
        output = numarray.nd_image.median_filter(array, kernel)
        error = mean_squared_error([3, 3, 5, 5, 4], output)
        self.failUnless(error < eps)

    def test_median4(self):
        "Test median filter"
        array = numarray.array([3, 2, 5, 1, 4])
        kernel = 3
        output = numarray.nd_image.median_filter(array, kernel)
        error = mean_squared_error([3, 3, 2, 4, 4], output)
        self.failUnless(error < eps)

    def test_median5(self):
        "Test median filter"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [5, 8, 3, 7, 1],
                                [5, 6, 9, 3, 5]])
        kernel = numarray.array([2, 3])
        output = numarray.nd_image.median_filter(array, kernel)
        error = mean_squared_error([[3, 3, 2, 4, 4],
                                    [5, 5, 5, 4, 4],
                                    [5, 6, 7, 5, 5]], output)
        self.failUnless(error < eps)

    def test_median6(self):
        "Test median filter"
        for type in self.types:
            array = numarray.array([[3, 2, 5, 1, 4],
                                    [5, 8, 3, 7, 1],
                                    [5, 6, 9, 3, 5]], type)
            footprint = [[1, 0, 1], [0, 1, 0]]
            output = numarray.nd_image.median_filter(array,
                                                     footprint = footprint)
            error = mean_squared_error([[3, 3, 2, 4, 4],
                                        [3, 5, 2, 5, 1],
                                        [5, 5, 8, 3, 5]], output)
            self.failUnless(error < eps)

    def test_laplace1(self):
        "test laplace filter"
        for type in self.types:
            array = numarray.array([[3, 2, 5, 1, 4],
                                    [5, 8, 3, 7, 1],
                                    [5, 6, 9, 3, 5]], type)
            tmp1 = numarray.nd_image.correlate1d(array, [1, -2, 1], 0)
            tmp2 = numarray.nd_image.correlate1d(array, [1, -2, 1], 1)
            output = numarray.nd_image.laplace(array)
            error = mean_squared_error(tmp1 + tmp2, output)
            self.failUnless(error < eps)
            
            
    def test_laplace2(self):
        "test laplace filter"
        for type in self.types:
            array = numarray.array([[3, 2, 5, 1, 4],
                                    [5, 8, 3, 7, 1],
                                    [5, 6, 9, 3, 5]], type)
            tmp1 = numarray.nd_image.correlate1d(array, [1, -2, 1], 0)
            tmp2 = numarray.nd_image.correlate1d(array, [1, -2, 1], 1)
            output = numarray.zeros(array.shape, type)
            numarray.nd_image.laplace(array, output = output)
            error = mean_squared_error(tmp1 + tmp2, output)
            self.failUnless(error < eps)
            
    def test_gaussian_laplace1(self):
        "test gaussian laplace filter"
        for type in self.types:
            array = numarray.array([[3, 2, 5, 1, 4],
                                    [5, 8, 3, 7, 1],
                                    [5, 6, 9, 3, 5]], type)
            tmp1 = numarray.nd_image.gaussian_filter(array, 1.0, [2, 0])
            tmp2 = numarray.nd_image.gaussian_filter(array, 1.0, [0, 2])
            output = numarray.nd_image.gaussian_laplace(array, 1.0)
            error = mean_squared_error(tmp1 + tmp2, output)
            self.failUnless(error < eps)
            
            
    def test_gaussian_laplace2(self):
        "test gaussian laplace filter"
        for type in self.types:
            array = numarray.array([[3, 2, 5, 1, 4],
                                    [5, 8, 3, 7, 1],
                                    [5, 6, 9, 3, 5]], type)
            tmp1 = numarray.nd_image.gaussian_filter(array, 1.0, [2, 0])
            tmp2 = numarray.nd_image.gaussian_filter(array, 1.0, [0, 2])
            output = numarray.zeros(array.shape, type)
            numarray.nd_image.gaussian_laplace(array, 1.0, output)
            error = mean_squared_error(tmp1 + tmp2, output)
            self.failUnless(error < eps)
            
            
    def test_gaussian_gradient_magnitude1(self):
        "test gaussian gradient magnitude"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [5, 8, 3, 7, 1],
                                [5, 6, 9, 3, 5]], numarray.Float64)
        tmp1 = numarray.nd_image.gaussian_filter(array, 1.0, [1, 0])
        tmp2 = numarray.nd_image.gaussian_filter(array, 1.0, [0, 1])
        output = numarray.nd_image.gaussian_gradient_magnitude(array, 1.0)
        true = numarray.sqrt(tmp1 * tmp1 + tmp2 * tmp2)
        error = mean_squared_error(true, output)
        self.failUnless(error < eps)
            
            
    def test_gaussian_gradient_magnitude2(self):
        "test gaussian gradient magnitude"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [5, 8, 3, 7, 1],
                                [5, 6, 9, 3, 5]], numarray.Float64)
        tmp1 = numarray.nd_image.gaussian_filter(array, 1.0, [1, 0])
        tmp2 = numarray.nd_image.gaussian_filter(array, 1.0, [0, 1])
        output = numarray.zeros(array.shape, numarray.Float64)
        numarray.nd_image.gaussian_gradient_magnitude(array, 1.0, output)
        true = numarray.sqrt(tmp1 * tmp1 + tmp2 * tmp2)
        error = mean_squared_error(true, output)
        self.failUnless(error < eps)
            
            
    def test_prewitt1(self):
        "test prewitt filter"
        for type in self.types:
            array = numarray.array([[3, 2, 5, 1, 4],
                                    [5, 8, 3, 7, 1],
                                    [5, 6, 9, 3, 5]], type)
            tmp = numarray.nd_image.correlate1d(array, [1.0, 0.0, -1.0], 0)
            tmp = numarray.nd_image.correlate1d(tmp, [1.0, 1.0, 1.0], 1)
            output = numarray.nd_image.prewitt(array, 0)
            error = mean_squared_error(tmp, output)
            self.failUnless(error < eps)
            
            
    def test_prewitt2(self):
        "test prewitt filter"
        for type in self.types:
            array = numarray.array([[3, 2, 5, 1, 4],
                                    [5, 8, 3, 7, 1],
                                    [5, 6, 9, 3, 5]], type)
            tmp = numarray.nd_image.correlate1d(array, [1.0, 0.0, -1.0], 0)
            tmp = numarray.nd_image.correlate1d(tmp, [1.0, 1.0, 1.0], 1)
            output = numarray.zeros(array.shape, type)
            numarray.nd_image.prewitt(array, 0, output)
            error = mean_squared_error(tmp, output)
            self.failUnless(error < eps)
            
            
    def test_sobel1(self):
        "test prewitt filter"
        for type in self.types:
            array = numarray.array([[3, 2, 5, 1, 4],
                                    [5, 8, 3, 7, 1],
                                    [5, 6, 9, 3, 5]], type)
            tmp = numarray.nd_image.correlate1d(array, [1.0, 0.0, -1.0], 0)
            tmp = numarray.nd_image.correlate1d(tmp, [1.0, 2.0, 1.0], 1)
            output = numarray.nd_image.sobel(array, 0)
            error = mean_squared_error(tmp, output)
            self.failUnless(error < eps)
            
            
    def test_sobel2(self):
        "test prewitt filter"
        for type in self.types:
            array = numarray.array([[3, 2, 5, 1, 4],
                                    [5, 8, 3, 7, 1],
                                    [5, 6, 9, 3, 5]], type)
            tmp = numarray.nd_image.correlate1d(array, [1.0, 0.0, -1.0], 0)
            tmp = numarray.nd_image.correlate1d(tmp, [1.0, 2.0, 1.0], 1)
            output = numarray.zeros(array.shape, type)
            numarray.nd_image.sobel(array, 0, output)
            error = mean_squared_error(tmp, output)
            self.failUnless(error < eps)
            
            
    def test_extend1(self):
        "Test line extension"
        array = numarray.array([1, 2, 3])
        weights = numarray.array([1, 0])
        true_values = [[1, 1, 2],
                       [3, 1, 2],
                       [1, 1, 2],
                       [0, 1, 2]]
        for mode, true_value in zip(self.modes, true_values):
            output = numarray.nd_image.correlate1d(array, weights, 0,
                                                   mode = mode, cval = 0)
            error = mean_squared_error(output, true_value)
            self.failUnless(error < eps)

    def test_extend2(self):
        "Test line extension"
        array = numarray.array([1, 2, 3])
        weights = numarray.array([1, 0, 0, 0, 0, 0, 0, 0])
        true_values = [[1, 1, 1],
                       [3, 1, 2],
                       [3, 3, 2],
                       [0, 0, 0]]
        for mode, true_value in zip(self.modes, true_values):
            output = numarray.nd_image.correlate1d(array, weights, 0,
                                                   mode = mode, cval = 0)
            error = mean_squared_error(output, true_value)
            self.failUnless(error < eps)

    def test_extend3(self):
        "Test line extension"
        array = numarray.array([1, 2, 3])
        weights = numarray.array([0, 0, 1])
        true_values = [[2, 3, 3],
                       [2, 3, 1],
                       [2, 3, 3],
                       [2, 3, 0]]
        for mode, true_value in zip(self.modes, true_values):
            output = numarray.nd_image.correlate1d(array, weights, 0,
                                                   mode = mode, cval = 0)
            error = mean_squared_error(output, true_value)
            self.failUnless(error < eps)

    def test_extend4(self):
        "Test line extension"
        array = numarray.array([1, 2, 3])
        weights = numarray.array([0, 0, 0, 0, 0, 0, 0, 0, 1])
        true_values = [[3, 3, 3],
                       [2, 3, 1],
                       [2, 1, 1],
                       [0, 0, 0]]
        for mode, true_value in zip(self.modes, true_values):
            output = numarray.nd_image.correlate1d(array, weights, 0,
                                                   mode = mode, cval = 0)
            error = mean_squared_error(output, true_value)
            self.failUnless(error < eps)


    def test_extend5(self):
        "Test line extension"
        array = numarray.array([[1, 2, 3],
                                [4, 5, 6],
                                [7, 8, 9]])
        weights = numarray.array([[1, 0], [0, 0]])
        true_values = [[[1, 1, 2], [1, 1, 2], [4, 4, 5]],
                       [[9, 7, 8], [3, 1, 2], [6, 4, 5]],
                       [[1, 1, 2], [1, 1, 2], [4, 4, 5]],
                       [[0, 0, 0], [0, 1, 2], [0, 4, 5]]]
        for mode, true_value in zip(self.modes, true_values):
            output = numarray.nd_image.correlate(array, weights,
                                                 mode = mode, cval = 0)
            error = mean_squared_error(output, true_value)
            self.failUnless(error < eps)


    def test_extend6(self):
        "Test line extension"
        array = numarray.array([[1, 2, 3],
                                [4, 5, 6],
                                [7, 8, 9]])
        weights = numarray.array([[0, 0, 0], [0, 0, 0], [0, 0, 1]])
        true_values = [[[5, 6, 6], [8, 9, 9], [8, 9, 9]],
                       [[5, 6, 4], [8, 9, 7], [2, 3, 1]],
                       [[5, 6, 6], [8, 9, 9], [8, 9, 9]],
                       [[5, 6, 0], [8, 9, 0], [0, 0, 0]]]
        for mode, true_value in zip(self.modes, true_values):
            output = numarray.nd_image.correlate(array, weights,
                                                 mode = mode, cval = 0)
            error = mean_squared_error(output, true_value)
            self.failUnless(error < eps)


    def test_extend7(self):
        "Test line extension"
        array = numarray.array([1, 2, 3])
        weights = numarray.array([0, 0, 0, 0, 0, 0, 0, 0, 1])
        true_values = [[3, 3, 3],
                       [2, 3, 1],
                       [2, 1, 1],
                       [0, 0, 0]]
        for mode, true_value in zip(self.modes, true_values):
            output = numarray.nd_image.correlate(array, weights, 
                                                 mode = mode, cval = 0)
            error = mean_squared_error(output, true_value)
            self.failUnless(error < eps)

    def test_extend8(self):
        "Test line extension"
        array = numarray.array([[1], [2], [3]])
        weights = numarray.array([[0], [0], [0], [0], [0], [0], [0], [0], [1]])
        true_values = [[[3], [3], [3]],
                       [[2], [3], [1]],
                       [[2], [1], [1]],
                       [[0], [0], [0]]]
        for mode, true_value in zip(self.modes, true_values):
            output = numarray.nd_image.correlate(array, weights, 
                                                 mode = mode, cval = 0)
            error = mean_squared_error(output, true_value)
            self.failUnless(error < eps)

    def test_extend9(self):
        "Test line extension"
        array = numarray.array([1, 2, 3])
        weights = numarray.array([0, 0, 0, 0, 0, 0, 0, 0, 1])
        true_values = [[3, 3, 3],
                       [2, 3, 1],
                       [2, 1, 1],
                       [0, 0, 0]]
        for mode, true_value in zip(self.modes, true_values):
            output = numarray.nd_image.correlate(array, weights,
                                                 mode = mode, cval = 0)
            error = mean_squared_error(output, true_value)
            self.failUnless(error < eps)

    def test_extend10(self):
        "Test line extension"
        array = numarray.array([[1], [2], [3]])
        weights = numarray.array([[0], [0], [0], [0], [0], [0], [0], [0], [1]])
        true_values = [[[3], [3], [3]],
                       [[2], [3], [1]],
                       [[2], [1], [1]],
                       [[0], [0], [0]]]
        for mode, true_value in zip(self.modes, true_values):
            output = numarray.nd_image.correlate(array, weights,
                                                 mode = mode, cval = 0)
            error = mean_squared_error(output, true_value)
            self.failUnless(error < eps)

    def test_generate_structure1(self):
        "test generation of a binary structure"
        struct = numarray.nd_image.generate_binary_structure(0, 1)
        error = mean_squared_error(struct, 1)
        self.failUnless(error < eps)
            
    def test_generate_structure2(self):
        "test generation of a binary structure"
        struct = numarray.nd_image.generate_binary_structure(1, 1)
        error = mean_squared_error(struct, [1, 1, 1])
        self.failUnless(error < eps)
            
    def test_generate_structure3(self):
        "test generation of a binary structure"
        struct = numarray.nd_image.generate_binary_structure(2, 1)
        error = mean_squared_error(struct, [[0, 1, 0],
                                            [1, 1, 1],
                                            [0, 1, 0]])
        self.failUnless(error < eps)
            
    def test_generate_structure4(self):
        "test generation of a binary structure"
        struct = numarray.nd_image.generate_binary_structure(2, 2)
        error = mean_squared_error(struct, [[1, 1, 1],
                                            [1, 1, 1],
                                            [1, 1, 1]])
        self.failUnless(error < eps)
        
    def test_iterate_structure1(self):
        "Test iterating a structure"
        struct = [[0, 1, 0],
                  [1, 1, 1],
                  [0, 1, 0]]
        out = numarray.nd_image.iterate_structure(struct, 2)
        error = mean_squared_error(out, [[0, 0, 1, 0, 0],
                                         [0, 1, 1, 1, 0],
                                         [1, 1, 1, 1, 1],
                                         [0, 1, 1, 1, 0],
                                         [0, 0, 1, 0, 0]])
        self.failUnless(error < eps)

    def test_iterate_structure2(self):
        "Test iterating a structure"
        struct = [[0, 1],
                  [1, 1],
                  [0, 1]]
        out = numarray.nd_image.iterate_structure(struct, 2)
        error = mean_squared_error(out, [[0, 0, 1],
                                         [0, 1, 1],
                                         [1, 1, 1],
                                         [0, 1, 1],
                                         [0, 0, 1]])
        self.failUnless(error < eps)

    def test_iterate_structure3(self):
        "Test iterating a structure"
        struct = [[0, 1, 0],
                  [1, 1, 1],
                  [0, 1, 0]]
        out = numarray.nd_image.iterate_structure(struct, 2, 1)
        error = mean_squared_error(out[0], [[0, 0, 1, 0, 0],
                                            [0, 1, 1, 1, 0],
                                            [1, 1, 1, 1, 1],
                                            [0, 1, 1, 1, 0],
                                            [0, 0, 1, 0, 0]])
        self.failUnless(error < eps and out[1] == [2, 2])

    def test_binary_erosion1(self):
        "test binary erosion"
        for type in self.types:
            data = numarray.ones([], type)
            out = numarray.nd_image.binary_erosion(data)
            error = mean_squared_error(out, 1)
            self.failUnless(error < eps)
            
            
    def test_binary_erosion2(self):
        "test binary erosion"
        for type in self.types:
            data = numarray.ones([], type)
            out = numarray.nd_image.binary_erosion(data, border_value = 1)
            error = mean_squared_error(out, 1)
            self.failUnless(error < eps)
            
            
    def test_binary_erosion3(self):
        "test binary erosion"
        for type in self.types:
            data = numarray.ones([1], type)
            out = numarray.nd_image.binary_erosion(data)
            error = mean_squared_error(out, [0])
            self.failUnless(error < eps)
            
            
    def test_binary_erosion4(self):
        "test binary erosion"
        for type in self.types:
            data = numarray.ones([1], type)
            out = numarray.nd_image.binary_erosion(data, border_value = 1)
            error = mean_squared_error(out, [1])
            self.failUnless(error < eps)
            
            
    def test_binary_erosion5(self):
        "test binary erosion"
        for type in self.types:
            data = numarray.ones([3], type)
            out = numarray.nd_image.binary_erosion(data)
            error = mean_squared_error(out, [0, 1, 0])
            self.failUnless(error < eps)
            
            
    def test_binary_erosion6(self):
        "test binary erosion"
        for type in self.types:
            data = numarray.ones([3], type)
            out = numarray.nd_image.binary_erosion(data, border_value = 1)
            error = mean_squared_error(out, [1, 1, 1])
            self.failUnless(error < eps)
            
            
    def test_binary_erosion7(self):
        "test binary erosion"
        for type in self.types:
            data = numarray.ones([5], type)
            out = numarray.nd_image.binary_erosion(data)
            error = mean_squared_error(out, [0, 1, 1, 1, 0])
            self.failUnless(error < eps)
            
            
    def test_binary_erosion8(self):
        "test binary erosion"
        for type in self.types:
            data = numarray.ones([5], type)
            out = numarray.nd_image.binary_erosion(data, border_value = 1)
            error = mean_squared_error(out, [1, 1, 1, 1, 1])
            self.failUnless(error < eps)
            
            
    def test_binary_erosion9(self):
        "test binary erosion"
        for type in self.types:
            data = numarray.ones([5], type)
            data[2] = 0
            out = numarray.nd_image.binary_erosion(data)
            error = mean_squared_error(out, [0, 0, 0, 0, 0])
            self.failUnless(error < eps)
            
            
    def test_binary_erosion10(self):
        "test binary erosion"
        for type in self.types:
            data = numarray.ones([5], type)
            data[2] = 0
            out = numarray.nd_image.binary_erosion(data, border_value = 1)
            error = mean_squared_error(out, [1, 0, 0, 0, 1])
            self.failUnless(error < eps)
            
            
    def test_binary_erosion11(self):
        "test binary erosion"
        for type in self.types:
            data = numarray.ones([1, 1], type)
            out = numarray.nd_image.binary_erosion(data, border_value = 1)
            error = mean_squared_error(out, [[1]])
            self.failUnless(error < eps)
            
            
    def test_binary_erosion12(self):
        "test binary erosion"
        for type in self.types:
            data = numarray.ones([1, 1], type)
            out = numarray.nd_image.binary_erosion(data)
            error = mean_squared_error(out, [[0]])
            self.failUnless(error < eps)
            
            
    def test_binary_erosion13(self):
        "test binary erosion"
        for type in self.types:
            data = numarray.ones([1, 3], type)
            out = numarray.nd_image.binary_erosion(data)
            error = mean_squared_error(out, [[0, 0, 0]])
            self.failUnless(error < eps)
            
            
    def test_binary_erosion14(self):
        "test binary erosion"
        for type in self.types:
            data = numarray.ones([1, 3], type)
            out = numarray.nd_image.binary_erosion(data, border_value = 1)
            error = mean_squared_error(out, [[1, 1, 1]])
            self.failUnless(error < eps)
            
            
    def test_binary_erosion15(self):
        "test binary erosion"
        for type in self.types:
            data = numarray.ones([3, 3], type)
            out = numarray.nd_image.binary_erosion(data)
            error = mean_squared_error(out, [[0, 0, 0],
                                             [0, 1, 0],
                                             [0, 0, 0]])
            self.failUnless(error < eps)
            
            
    def test_binary_erosion16(self):
        "test binary erosion"
        for type in self.types:
            data = numarray.ones([3, 3], type)
            out = numarray.nd_image.binary_erosion(data, border_value = 1)
            error = mean_squared_error(out, [[1, 1, 1],
                                             [1, 1, 1],
                                             [1, 1, 1]])
            self.failUnless(error < eps)
            
            
    def test_binary_erosion17(self):
        "test binary erosion"
        true = [[0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 1, 0, 0],
                [0, 0, 0, 1, 1, 0, 0, 0],
                [0, 0, 1, 0, 0, 1, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0]]
        for type in self.types:
            data = numarray.array([[0, 0, 0, 0, 0, 0, 0, 0],
                                   [0, 1, 0, 0, 0, 0, 0, 0],
                                   [0, 0, 0, 0, 0, 1, 1, 1],
                                   [0, 0, 1, 1, 1, 1, 1, 1],
                                   [0, 0, 1, 1, 1, 1, 0, 0],
                                   [0, 1, 1, 1, 1, 1, 1, 0],
                                   [0, 1, 1, 0, 0, 1, 1, 0],
                                   [0, 0, 0, 0, 0, 0, 0, 0]], type)
            out = numarray.nd_image.binary_erosion(data, border_value = 1)
            error = mean_squared_error(out, true)
            self.failUnless(error < eps)
            
            
    def test_binary_erosion18(self):
        "test binary erosion"
        struct = numarray.nd_image.generate_binary_structure(2, 2)
        true = [[0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 1, 1, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0]]
        for type in self.types:
            data = numarray.array([[0, 0, 0, 0, 0, 0, 0, 0],
                                   [0, 1, 0, 0, 0, 0, 0, 0],
                                   [0, 0, 0, 0, 0, 1, 1, 1],
                                   [0, 0, 1, 1, 1, 1, 1, 1],
                                   [0, 0, 1, 1, 1, 1, 0, 0],
                                   [0, 1, 1, 1, 1, 1, 1, 0],
                                   [0, 1, 1, 0, 0, 1, 1, 0],
                                   [0, 0, 0, 0, 0, 0, 0, 0]], type)
            out = numarray.nd_image.binary_erosion(data, struct,
                                                   border_value = 1)
            error = mean_squared_error(out, true)
            self.failUnless(error < eps)
            
            
    def test_binary_erosion19(self):
        "test binary erosion"
        struct = [[0, 1],
                  [1, 1]]
        true = [[0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 1, 1, 1],
                [0, 0, 0, 1, 1, 1, 0, 0],
                [0, 0, 1, 1, 1, 1, 0, 0],
                [0, 0, 1, 0, 0, 0, 1, 0],
                [0, 0, 0, 0, 0, 0, 0, 0]]
        for type in self.types:
            data = numarray.array([[0, 0, 0, 0, 0, 0, 0, 0],
                                   [0, 1, 0, 0, 0, 0, 0, 0],
                                   [0, 0, 0, 0, 0, 1, 1, 1],
                                   [0, 0, 1, 1, 1, 1, 1, 1],
                                   [0, 0, 1, 1, 1, 1, 0, 0],
                                   [0, 1, 1, 1, 1, 1, 1, 0],
                                   [0, 1, 1, 0, 0, 1, 1, 0],
                                   [0, 0, 0, 0, 0, 0, 0, 0]], type)
            out = numarray.nd_image.binary_erosion(data, struct,
                                                   border_value = 1)
            error = mean_squared_error(out, true)
            self.failUnless(error < eps)
            
            
    def test_binary_erosion20(self):
        "test binary erosion"
        struct = [[0, 1, 0],
                  [1, 0, 1],
                  [0, 1, 0]]
        true = [[0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 1, 0, 0],
                [0, 0, 0, 1, 0, 0, 0, 0],
                [0, 0, 1, 0, 0, 1, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0]]
        for type in self.types:
            data = numarray.array([[0, 0, 0, 0, 0, 0, 0, 0],
                                   [0, 1, 0, 0, 0, 0, 0, 0],
                                   [0, 0, 0, 0, 0, 1, 1, 1],
                                   [0, 0, 1, 1, 1, 0, 1, 1],
                                   [0, 0, 1, 0, 1, 1, 0, 0],
                                   [0, 1, 0, 1, 1, 1, 1, 0],
                                   [0, 1, 1, 0, 0, 1, 1, 0],
                                   [0, 0, 0, 0, 0, 0, 0, 0]], type)
            out = numarray.nd_image.binary_erosion(data, struct,
                                                   border_value = 1)
            error = mean_squared_error(out, true)
            self.failUnless(error < eps)
            
            
    def test_binary_erosion21(self):
        "test binary erosion"
        struct = [[0, 1, 0],
                  [1, 1, 1],
                  [0, 1, 0]]
        true = [[0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 1, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0]]
        data = numarray.array([[0, 0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 1, 0, 0, 0],
                               [0, 0, 1, 1, 1, 0, 0],
                               [0, 1, 1, 1, 1, 1, 0],
                               [0, 0, 1, 1, 1, 0, 0],
                               [0, 0, 0, 1, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0, 0]], numarray.Bool)
        out = numarray.nd_image.binary_erosion(data, struct, border_value = 1,
                                               iterations = 2)
        error = mean_squared_error(out, true)
        self.failUnless(error < eps)
       

    def test_binary_erosion22(self):
        "test binary erosion"
        struct = [[0, 1, 0],
                  [1, 1, 1],
                  [0, 1, 0]]
        true = [[0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 1, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0]]
        data = numarray.array([[0, 0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 1, 0, 0, 0],
                               [0, 0, 1, 1, 1, 0, 0],
                               [0, 1, 1, 1, 1, 1, 0],
                               [0, 0, 1, 1, 1, 0, 0],
                               [0, 0, 0, 1, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0, 0]], numarray.Bool)
        out = numarray.zeros(data.shape, numarray.Bool)
        numarray.nd_image.binary_erosion(data, struct, border_value = 1,
                                         iterations = 2, output = out)
        error = mean_squared_error(out, true)
        self.failUnless(error < eps)
       
    def test_binary_erosion23(self):
        "test binary erosion"
        struct = [[0, 1, 0],
                  [1, 1, 1],
                  [0, 1, 0]]
        true = [[0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 1, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0]]
        data = numarray.array([[0, 0, 0, 1, 0, 0, 0],
                               [0, 0, 1, 1, 1, 0, 0],
                               [0, 1, 1, 1, 1, 1, 0],
                               [1, 1, 1, 1, 1, 1, 1],
                               [0, 1, 1, 1, 1, 1, 0],
                               [0, 0, 1, 1, 1, 0, 0],
                               [0, 0, 0, 1, 0, 0, 0]], numarray.Bool)
        out = numarray.nd_image.binary_erosion(data, struct, border_value = 1,
                                               iterations = 3)
        error = mean_squared_error(out, true)
        self.failUnless(error < eps)
       

    def test_binary_erosion24(self):
        "test binary erosion"
        struct = [[0, 1, 0],
                  [1, 1, 1],
                  [0, 1, 0]]
        true = [[0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 1, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0]]
        data = numarray.array([[0, 0, 0, 1, 0, 0, 0],
                               [0, 0, 1, 1, 1, 0, 0],
                               [0, 1, 1, 1, 1, 1, 0],
                               [1, 1, 1, 1, 1, 1, 1],
                               [0, 1, 1, 1, 1, 1, 0],
                               [0, 0, 1, 1, 1, 0, 0],
                               [0, 0, 0, 1, 0, 0, 0]], numarray.Bool)
        out = numarray.zeros(data.shape, numarray.Bool)
        numarray.nd_image.binary_erosion(data, struct, border_value = 1,
                                         iterations = 3, output = out)
        error = mean_squared_error(out, true)
        self.failUnless(error < eps)       

       
    def test_binary_erosion25(self):
        "test binary erosion"
        struct = [[0, 1, 0],
                  [1, 1, 1],
                  [0, 1, 0]]
        true = [[0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 1, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0]]
        data = numarray.array([[0, 0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 1, 0, 0, 0],
                               [0, 0, 1, 1, 1, 0, 0],
                               [0, 1, 1, 1, 1, 1, 0],
                               [0, 0, 1, 1, 1, 0, 0],
                               [0, 0, 0, 1, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0, 0]], numarray.Bool)
        out = numarray.nd_image.binary_erosion(data, struct, border_value = 1,
                                               iterations = 2)
        error = mean_squared_error(out, true)
        self.failUnless(error < eps)
       

    def test_binary_erosion26(self):
        "test binary erosion"
        struct = [[0, 1, 0],
                  [1, 1, 1],
                  [0, 1, 0]]
        true = [[0, 0, 0, 0, 0, 1, 1],
                [0, 0, 0, 0, 0, 0, 1],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0]]
        mask = [[1, 1, 1, 1, 1, 0, 0],
                [1, 1, 1, 1, 1, 1, 0],
                [1, 1, 1, 1, 1, 1, 1],
                [1, 1, 1, 1, 1, 1, 1],
                [1, 1, 1, 1, 1, 1, 1],
                [1, 1, 1, 1, 1, 1, 1],
                [1, 1, 1, 1, 1, 1, 1]]
        data = numarray.array([[0, 0, 0, 0, 0, 1, 1],
                               [0, 0, 0, 1, 0, 0, 1],
                               [0, 0, 1, 1, 1, 0, 0],
                               [0, 0, 1, 1, 1, 0, 0],
                               [0, 0, 1, 1, 1, 0, 0],
                               [0, 0, 0, 1, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0, 0]], numarray.Bool)
        out = numarray.nd_image.binary_erosion(data, struct, border_value = 1,
                                               mask = mask, iterations = -1)
        error = mean_squared_error(out, true)
        self.failUnless(error < eps)
       

    def test_binary_erosion27(self):
        "test binary erosion"
        struct = [[0, 1, 0],
                  [1, 1, 1],
                  [0, 1, 0]]
        true = [[0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 1, 0, 0, 0],
                [0, 0, 0, 1, 0, 0, 0],
                [0, 1, 1, 1, 1, 1, 0],
                [0, 0, 0, 1, 0, 0, 0],
                [0, 0, 0, 1, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0]]
        mask = [[0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 1, 1, 1, 0, 0],
                [0, 0, 1, 0, 1, 0, 0],
                [0, 0, 1, 1, 1, 0, 0],
                [0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0]]
        data = numarray.array([[0, 0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 1, 0, 0, 0],
                               [0, 0, 1, 1, 1, 0, 0],
                               [0, 1, 1, 1, 1, 1, 0],
                               [0, 0, 1, 1, 1, 0, 0],
                               [0, 0, 0, 1, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0, 0]], numarray.Bool)
        out = numarray.nd_image.binary_erosion(data, struct, border_value = 1,
                                               mask = mask)
        error = mean_squared_error(out, true)
        self.failUnless(error < eps)


    def test_binary_dilation1(self):
        "test binary dilation"
        for type in self.types:
            data = numarray.ones([], type)
            out = numarray.nd_image.binary_dilation(data)
            error = mean_squared_error(out, 1)
            self.failUnless(error < eps)
            
            
    def test_binary_dilation2(self):
        "test binary dilation"
        for type in self.types:
            data = numarray.zeros([], type)
            out = numarray.nd_image.binary_dilation(data)
            error = mean_squared_error(out, 0)
            self.failUnless(error < eps)
            
            
    def test_binary_dilation3(self):
        "test binary dilation"
        for type in self.types:
            data = numarray.ones([1], type)
            out = numarray.nd_image.binary_dilation(data)
            error = mean_squared_error(out, [1])
            self.failUnless(error < eps)
            
            
    def test_binary_dilation4(self):
        "test binary dilation"
        for type in self.types:
            data = numarray.zeros([1], type)
            out = numarray.nd_image.binary_dilation(data)
            error = mean_squared_error(out, [0])
            self.failUnless(error < eps)
            
            
    def test_binary_dilation5(self):
        "test binary dilation"
        for type in self.types:
            data = numarray.ones([3], type)
            out = numarray.nd_image.binary_dilation(data)
            error = mean_squared_error(out, [1, 1, 1])
            self.failUnless(error < eps)
            
            
    def test_binary_dilation6(self):
        "test binary dilation"
        for type in self.types:
            data = numarray.zeros([3], type)
            out = numarray.nd_image.binary_dilation(data)
            error = mean_squared_error(out, [0, 0, 0])
            self.failUnless(error < eps)
            
            
    def test_binary_dilation7(self):
        "test binary dilation"
        struct = numarray.nd_image.generate_binary_structure(1, 1)
        for type in self.types:
            data = numarray.zeros([3], type)
            data[1] = 1
            out = numarray.nd_image.binary_dilation(data)
            error = mean_squared_error(out, [1, 1, 1])
            self.failUnless(error < eps)
                        
            
    def test_binary_dilation8(self):
        "test binary dilation"
        for type in self.types:
            data = numarray.zeros([5], type)
            data[1] = 1
            data[3] = 1
            out = numarray.nd_image.binary_dilation(data)
            error = mean_squared_error(out, [1, 1, 1, 1, 1])
            self.failUnless(error < eps)
            
            
    def test_binary_dilation9(self):
        "test binary dilation"
        for type in self.types:
            data = numarray.zeros([5], type)
            data[1] = 1
            out = numarray.nd_image.binary_dilation(data)
            error = mean_squared_error(out, [1, 1, 1, 0, 0])
            self.failUnless(error < eps)
            
    def test_binary_dilation10(self):
        "test binary dilation"
        for type in self.types:
            data = numarray.ones([1, 1], type)
            out = numarray.nd_image.binary_dilation(data)
            error = mean_squared_error(out, [[1]])
            self.failUnless(error < eps)
            
            
    def test_binary_dilation11(self):
        "test binary dilation"
        for type in self.types:
            data = numarray.zeros([1, 1], type)
            out = numarray.nd_image.binary_dilation(data)
            error = mean_squared_error(out, [[0]])
            self.failUnless(error < eps)
            
            
    def test_binary_dilation12(self):
        "test binary dilation"
        for type in self.types:
            data = numarray.ones([1, 3], type)
            out = numarray.nd_image.binary_dilation(data)
            error = mean_squared_error(out, [[1, 1, 1]])
            self.failUnless(error < eps)
            
            
    def test_binary_dilation13(self):
        "test binary dilation"
        for type in self.types:
            data = numarray.ones([3, 3], type)
            out = numarray.nd_image.binary_dilation(data)
            error = mean_squared_error(out, [[1, 1, 1],
                                             [1, 1, 1],
                                             [1, 1, 1]])
            self.failUnless(error < eps)
            
            
    def test_binary_dilation14(self):
        "test binary dilation"
        for type in self.types:
            data = numarray.zeros([3, 3], type)
            data[1, 1] = 1
            out = numarray.nd_image.binary_dilation(data)
            error = mean_squared_error(out, [[0, 1, 0],
                                             [1, 1, 1],
                                             [0, 1, 0]])
            self.failUnless(error < eps)
            
            
    def test_binary_dilation15(self):
        "test binary dilation"
        struct = numarray.nd_image.generate_binary_structure(2, 2)
        for type in self.types:
            data = numarray.zeros([3, 3], type)
            data[1, 1] = 1
            out = numarray.nd_image.binary_dilation(data, struct)
            error = mean_squared_error(out, [[1, 1, 1],
                                             [1, 1, 1],
                                             [1, 1, 1]])
            self.failUnless(error < eps)
            
            
    def test_binary_dilation16(self):
        "test binary dilation"
        true = [[0, 1, 0, 0, 0, 0, 0, 0],
                [1, 1, 1, 0, 0, 0, 0, 0],
                [0, 1, 0, 0, 0, 1, 0, 0],
                [0, 0, 0, 1, 1, 1, 1, 0],
                [0, 0, 1, 1, 1, 1, 0, 0],
                [0, 1, 1, 1, 1, 1, 1, 0],
                [0, 0, 1, 0, 0, 1, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0]]

        for type in self.types:
            data = numarray.array([[0, 0, 0, 0, 0, 0, 0, 0],
                                   [0, 1, 0, 0, 0, 0, 0, 0],
                                   [0, 0, 0, 0, 0, 0, 0, 0],
                                   [0, 0, 0, 0, 0, 1, 0, 0],
                                   [0, 0, 0, 1, 1, 0, 0, 0],
                                   [0, 0, 1, 0, 0, 1, 0, 0],
                                   [0, 0, 0, 0, 0, 0, 0, 0],
                                   [0, 0, 0, 0, 0, 0, 0, 0]], type)
            out = numarray.nd_image.binary_dilation(data)
            error = mean_squared_error(out, true)
            self.failUnless(error < eps)
            
            
    def test_binary_dilation17(self):
        "test binary dilation"
        struct = numarray.nd_image.generate_binary_structure(2, 2)
        true = [[1, 1, 1, 0, 0, 0, 0, 0],
                [1, 1, 1, 0, 0, 0, 0, 0],
                [1, 1, 1, 0, 1, 1, 1, 0],
                [0, 0, 1, 1, 1, 1, 1, 0],
                [0, 1, 1, 1, 1, 1, 1, 0],
                [0, 1, 1, 1, 1, 1, 1, 0],
                [0, 1, 1, 1, 1, 1, 1, 0],
                [0, 0, 0, 0, 0, 0, 0, 0]]

        for type in self.types:
            data = numarray.array([[0, 0, 0, 0, 0, 0, 0, 0],
                                   [0, 1, 0, 0, 0, 0, 0, 0],
                                   [0, 0, 0, 0, 0, 0, 0, 0],
                                   [0, 0, 0, 0, 0, 1, 0, 0],
                                   [0, 0, 0, 1, 1, 0, 0, 0],
                                   [0, 0, 1, 0, 0, 1, 0, 0],
                                   [0, 0, 0, 0, 0, 0, 0, 0],
                                   [0, 0, 0, 0, 0, 0, 0, 0]], type)
            out = numarray.nd_image.binary_dilation(data, struct)
            error = mean_squared_error(out, true)
            self.failUnless(error < eps)


    def test_binary_dilation18(self):
        "test binary dilation"
        struct = [[0, 1],
                  [1, 1]]
        true = [[0, 1, 0, 0, 0, 0, 0, 0],
                [1, 1, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 1, 0, 0],
                [0, 0, 0, 1, 1, 1, 0, 0],
                [0, 0, 1, 1, 1, 1, 0, 0],
                [0, 1, 1, 0, 1, 1, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0]]

        for type in self.types:
            data = numarray.array([[0, 0, 0, 0, 0, 0, 0, 0],
                                   [0, 1, 0, 0, 0, 0, 0, 0],
                                   [0, 0, 0, 0, 0, 0, 0, 0],
                                   [0, 0, 0, 0, 0, 1, 0, 0],
                                   [0, 0, 0, 1, 1, 0, 0, 0],
                                   [0, 0, 1, 0, 0, 1, 0, 0],
                                   [0, 0, 0, 0, 0, 0, 0, 0],
                                   [0, 0, 0, 0, 0, 0, 0, 0]], type)
            out = numarray.nd_image.binary_dilation(data, struct)
            error = mean_squared_error(out, true)
            self.failUnless(error < eps)


    def test_binary_dilation19(self):
        "test binary dilation"
        true = [[1, 1, 1, 1],
                [1, 0, 0, 1],
                [1, 0, 0, 1],
                [1, 1, 1, 1]]

        for type in self.types:
            data = numarray.array([[0, 0, 0, 0],
                                   [0, 0, 0, 0],
                                   [0, 0, 0, 0],
                                   [0, 0, 0, 0]], type)
            out = numarray.nd_image.binary_dilation(data, border_value = 1)
            error = mean_squared_error(out, true)
            self.failUnless(error < eps)

    def test_binary_dilation20(self):
        "test binary dilation"
        struct = [[0, 1],
                  [1, 1]]
        true = [[0, 0, 0, 0, 0],
                [0, 0, 0, 1, 0],
                [0, 0, 1, 1, 0],
                [0, 1, 1, 1, 0],
                [0, 0, 0, 0, 0]]

        data = numarray.array([[0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0],
                               [0, 0, 0, 1, 0],
                               [0, 0, 0, 0, 0]], numarray.Bool)
        out = numarray.nd_image.binary_dilation(data, struct, iterations = 2)
        error = mean_squared_error(out, true)
        self.failUnless(error < eps)

    def test_binary_dilation21(self):
        "test binary dilation"
        struct = [[0, 1],
                  [1, 1]]
        true = [[0, 0, 0, 0, 0],
                [0, 0, 0, 1, 0],
                [0, 0, 1, 1, 0],
                [0, 1, 1, 1, 0],
                [0, 0, 0, 0, 0]]

        data = numarray.array([[0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0],
                               [0, 0, 0, 1, 0],
                               [0, 0, 0, 0, 0]], numarray.Bool)
        out = numarray.zeros(data.shape, numarray.Bool)
        numarray.nd_image.binary_dilation(data, struct, iterations = 2,
                                          output = out)
        error = mean_squared_error(out, true)
        self.failUnless(error < eps)

    def test_binary_dilation22(self):
        "test binary dilation"
        struct = [[0, 1],
                  [1, 1]]
        true = [[0, 0, 0, 1, 0],
                [0, 0, 1, 1, 0],
                [0, 1, 1, 1, 0],
                [1, 1, 1, 1, 0],
                [0, 0, 0, 0, 0]]

        data = numarray.array([[0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0],
                               [0, 0, 0, 1, 0],
                               [0, 0, 0, 0, 0]], numarray.Bool)
        out = numarray.nd_image.binary_dilation(data, struct, iterations = 3)
        error = mean_squared_error(out, true)
        self.failUnless(error < eps)

    def test_binary_dilation23(self):
        "test binary dilation"
        struct = [[0, 1],
                  [1, 1]]
        true = [[0, 0, 0, 1, 0],
                [0, 0, 1, 1, 0],
                [0, 1, 1, 1, 0],
                [1, 1, 1, 1, 0],
                [0, 0, 0, 0, 0]]

        data = numarray.array([[0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0],
                               [0, 0, 0, 1, 0],
                               [0, 0, 0, 0, 0]], numarray.Bool)
        out = numarray.zeros(data.shape, numarray.Bool)
        numarray.nd_image.binary_dilation(data, struct, iterations = 3,
                                          output = out)
        error = mean_squared_error(out, true)
        self.failUnless(error < eps)

    def test_binary_dilation24(self):
        "test binary dilation"
        struct = [[0, 1, 0],
                  [1, 1, 1],
                  [0, 1, 0]]
        true = numarray.array([[0, 1, 0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 1, 1, 0, 0],
                               [0, 0, 1, 1, 1, 0, 0, 0],
                               [0, 1, 1, 0, 1, 1, 0, 0],
                               [0, 0, 0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0, 0, 0]], numarray.Bool)
        mask = numarray.array([[0, 1, 0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0, 1, 0],
                               [0, 0, 0, 0, 1, 1, 0, 0],
                               [0, 0, 1, 1, 1, 0, 0, 0],
                               [0, 1, 1, 0, 1, 1, 0, 0],
                               [0, 0, 0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0, 0, 0]], numarray.Bool)
        data = numarray.array([[0, 1, 0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0, 0, 0],
                               [0, 1, 0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0, 0, 0]], numarray.Bool)
        
        out = numarray.nd_image.binary_dilation(data, struct, iterations = -1,
                                                mask = mask, border_value = 0)
        error = mean_squared_error(out, true)
        self.failUnless(error < eps)

    def test_binary_dilation25(self):
        "test binary dilation"
        struct = [[0, 1, 0],
                  [1, 1, 1],
                  [0, 1, 0]]
        true = [[0, 1, 0, 0, 0, 0, 0, 0],
                [0, 1, 1, 0, 0, 0, 0, 0],
                [0, 0, 1, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0]]
        mask = numarray.array([[0, 1, 0, 0, 0, 0, 0, 0],
                               [0, 1, 1, 0, 0, 0, 0, 0],
                               [0, 0, 1, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 1, 0, 0],
                               [0, 0, 0, 1, 1, 0, 0, 0],
                               [0, 0, 1, 0, 0, 1, 0, 0],
                               [0, 0, 0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0, 0, 0]], numarray.Bool)
        data = numarray.zeros(mask.shape, numarray.Bool)
        out = numarray.nd_image.binary_dilation(data, struct, iterations = -1,
                                                mask = mask, border_value = 1)
        error = mean_squared_error(out, true)
        self.failUnless(error < eps)


    def test_binary_opening1(self):
        "test binary opening"
        true = [[0, 1, 0, 0, 0, 0, 0, 0],
                [1, 1, 1, 0, 0, 0, 0, 0],
                [0, 1, 0, 0, 0, 1, 0, 0],
                [0, 0, 0, 0, 1, 1, 1, 0],
                [0, 0, 1, 0, 0, 1, 0, 0],
                [0, 1, 1, 1, 1, 1, 1, 0],
                [0, 0, 1, 0, 0, 1, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0]]
        for type in self.types:
            data = numarray.array([[0, 1, 0, 0, 0, 0, 0, 0],
                                   [1, 1, 1, 0, 0, 0, 0, 0],
                                   [0, 1, 0, 0, 0, 1, 0, 0],
                                   [0, 0, 0, 1, 1, 1, 1, 0],
                                   [0, 0, 1, 1, 0, 1, 0, 0],
                                   [0, 1, 1, 1, 1, 1, 1, 0],
                                   [0, 0, 1, 0, 0, 1, 0, 0],
                                   [0, 0, 0, 0, 0, 0, 0, 0]], type)
            out = numarray.nd_image.binary_opening(data)
            error = mean_squared_error(out, true)
            self.failUnless(error < eps)
       
            
    def test_binary_opening2(self):
        "test binary opening"
        struct = numarray.nd_image.generate_binary_structure(2, 2)
        true = [[1, 1, 1, 0, 0, 0, 0, 0],
                [1, 1, 1, 0, 0, 0, 0, 0],
                [1, 1, 1, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 1, 1, 1, 0, 0, 0, 0],
                [0, 1, 1, 1, 0, 0, 0, 0],
                [0, 1, 1, 1, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0]]
        for type in self.types:
            data = numarray.array([[1, 1, 1, 0, 0, 0, 0, 0],
                                   [1, 1, 1, 0, 0, 0, 0, 0],
                                   [1, 1, 1, 1, 1, 1, 1, 0],
                                   [0, 0, 1, 1, 1, 1, 1, 0],
                                   [0, 1, 1, 1, 0, 1, 1, 0],
                                   [0, 1, 1, 1, 1, 1, 1, 0],
                                   [0, 1, 1, 1, 1, 1, 1, 0],
                                   [0, 0, 0, 0, 0, 0, 0, 0]], type)
            out = numarray.nd_image.binary_opening(data, struct)
            error = mean_squared_error(out, true)
            self.failUnless(error < eps)
       
            
    def test_binary_closing1(self):
        "test binary closing"
        true = [[0, 0, 0, 0, 0, 0, 0, 0],
                [0, 1, 1, 0, 0, 0, 0, 0],
                [0, 1, 1, 1, 0, 1, 0, 0],
                [0, 0, 1, 1, 1, 1, 1, 0],
                [0, 0, 1, 1, 1, 1, 0, 0],
                [0, 1, 1, 1, 1, 1, 1, 0],
                [0, 0, 1, 0, 0, 1, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0]]
        for type in self.types:
            data = numarray.array([[0, 1, 0, 0, 0, 0, 0, 0],
                                   [1, 1, 1, 0, 0, 0, 0, 0],
                                   [0, 1, 0, 0, 0, 1, 0, 0],
                                   [0, 0, 0, 1, 1, 1, 1, 0],
                                   [0, 0, 1, 1, 0, 1, 0, 0],
                                   [0, 1, 1, 1, 1, 1, 1, 0],
                                   [0, 0, 1, 0, 0, 1, 0, 0],
                                   [0, 0, 0, 0, 0, 0, 0, 0]], type)
            out = numarray.nd_image.binary_closing(data)
            error = mean_squared_error(out, true)
            self.failUnless(error < eps)
       
            
    def test_binary_closing2(self):
        "test binary closing"
        struct = numarray.nd_image.generate_binary_structure(2, 2)
        true = [[0, 0, 0, 0, 0, 0, 0, 0],
                [0, 1, 1, 0, 0, 0, 0, 0],
                [0, 1, 1, 1, 1, 1, 1, 0],
                [0, 1, 1, 1, 1, 1, 1, 0],
                [0, 1, 1, 1, 1, 1, 1, 0],
                [0, 1, 1, 1, 1, 1, 1, 0],
                [0, 1, 1, 1, 1, 1, 1, 0],
                [0, 0, 0, 0, 0, 0, 0, 0]]
        for type in self.types:
            data = numarray.array([[1, 1, 1, 0, 0, 0, 0, 0],
                                   [1, 1, 1, 0, 0, 0, 0, 0],
                                   [1, 1, 1, 1, 1, 1, 1, 0],
                                   [0, 0, 1, 1, 1, 1, 1, 0],
                                   [0, 1, 1, 1, 0, 1, 1, 0],
                                   [0, 1, 1, 1, 1, 1, 1, 0],
                                   [0, 1, 1, 1, 1, 1, 1, 0],
                                   [0, 0, 0, 0, 0, 0, 0, 0]], type)
            out = numarray.nd_image.binary_closing(data, struct)
            error = mean_squared_error(out, true)
            self.failUnless(error < eps)
       
            
    def test_grey_erosion1(self):
        "Test grey erosion"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        output = numarray.nd_image.grey_erosion(array, footprint = footprint)
        error = mean_squared_error([[2, 2, 1, 1, 1],
                                    [2, 3, 1, 3, 1],
                                    [5, 5, 3, 3, 1]], output)
        self.failUnless(error < eps)


    def test_grey_erosion2(self):
        "Test grey erosion"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        structure = [[0, 0, 0], [0, 0, 0]]
        output = numarray.nd_image.grey_erosion(array, footprint = footprint,
                                                structure = structure)
        error = mean_squared_error([[2, 2, 1, 1, 1],
                                    [2, 3, 1, 3, 1],
                                    [5, 5, 3, 3, 1]], output)
        self.failUnless(error < eps)


    def test_grey_erosion3(self):
        "Test grey erosion"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        structure = [[1, 1, 1], [1, 1, 1]]
        output = numarray.nd_image.grey_erosion(array, footprint = footprint,
                                                structure = structure)
        error = mean_squared_error([[1, 1, 0, 0, 0],
                                    [1, 2, 0, 2, 0],
                                    [4, 4, 2, 2, 0]], output)
        self.failUnless(error < eps)


    def test_grey_dilation1(self):
        "Test grey dilation"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[0, 1, 1], [1, 0, 1]]
        output = numarray.nd_image.grey_dilation(array, footprint = footprint)
        error = mean_squared_error([[7, 7, 9, 9, 5],
                                    [7, 9, 8, 9, 7],
                                    [8, 8, 8, 7, 7]], output)
        self.failUnless(error < eps)

    def test_grey_dilation2(self):
        "Test grey dilation"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[0, 1, 1], [1, 0, 1]]
        structure = [[0, 0, 0], [0, 0, 0]]
        output = numarray.nd_image.grey_dilation(array, footprint = footprint,
                                                 structure = structure)
        error = mean_squared_error([[7, 7, 9, 9, 5],
                                    [7, 9, 8, 9, 7],
                                    [8, 8, 8, 7, 7]], output)
        self.failUnless(error < eps)

    def test_grey_dilation3(self):
        "Test grey dilation"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[0, 1, 1], [1, 0, 1]]
        structure = [[1, 1, 1], [1, 1, 1]]
        output = numarray.nd_image.grey_dilation(array, footprint = footprint,
                                                 structure = structure)
        error = mean_squared_error([[8,  8, 10, 10, 6],
                                    [8, 10,  9, 10, 8],
                                    [9,  9,  9,  8, 8]], output)
        self.failUnless(error < eps)

    def test_grey_opening1(self):
        "Test grey opening"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        tmp = numarray.nd_image.grey_erosion(array, footprint = footprint)
        true = numarray.nd_image.grey_dilation(tmp, footprint = footprint)
        output = numarray.nd_image.grey_opening(array, footprint = footprint)
        error = mean_squared_error(true, output)
        self.failUnless(error < eps)
        

    def test_grey_opening2(self):
        "Test grey opening"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        structure = [[0, 0, 0], [0, 0, 0]]
        tmp = numarray.nd_image.grey_erosion(array, footprint = footprint,
                                             structure = structure)
        true = numarray.nd_image.grey_dilation(tmp, footprint = footprint,
                                               structure = structure)
        output = numarray.nd_image.grey_opening(array, footprint = footprint,
                                                structure = structure)
        error = mean_squared_error(true, output)
        self.failUnless(error < eps)
        

    def test_grey_closing1(self):
        "Test grey opening"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        tmp = numarray.nd_image.grey_dilation(array, footprint = footprint)
        true = numarray.nd_image.grey_erosion(tmp, footprint = footprint)
        output = numarray.nd_image.grey_closing(array, footprint = footprint)
        error = mean_squared_error(true, output)
        self.failUnless(error < eps)
        

    def test_grey_closing2(self):
        "Test grey opening"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        structure = [[0, 0, 0], [0, 0, 0]]
        tmp = numarray.nd_image.grey_dilation(array, footprint = footprint,
                                              structure = structure)
        true = numarray.nd_image.grey_erosion(tmp, footprint = footprint,
                                              structure = structure)
        output = numarray.nd_image.grey_closing(array, footprint = footprint,
                                                structure = structure)
        error = mean_squared_error(true, output)
        self.failUnless(error < eps)
        

    def test_morphological_gradient1(self):
        "Test grey opening"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        structure = [[0, 0, 0], [0, 0, 0]]
        tmp1 = numarray.nd_image.grey_dilation(array, footprint = footprint,
                                               structure = structure)
        tmp2 = numarray.nd_image.grey_erosion(array, footprint = footprint,
                                              structure = structure)
        true = tmp1 - tmp2
        output = numarray.zeros(array.shape, array.type())
        numarray.nd_image.morphological_gradient(array, footprint=footprint,
                                                 structure=structure,
                                                 output = output)
        error = mean_squared_error(true, output)
        self.failUnless(error < eps)
        

    def test_morphological_gradient2(self):
        "Test grey opening"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        structure = [[0, 0, 0], [0, 0, 0]]
        tmp1 = numarray.nd_image.grey_dilation(array, footprint = footprint,
                                               structure = structure)
        tmp2 = numarray.nd_image.grey_erosion(array, footprint = footprint,
                                              structure = structure)
        true = tmp1 - tmp2
        output = numarray.nd_image.morphological_gradient(array,
                                                          footprint=footprint,
                                                          structure=structure)
        error = mean_squared_error(true, output)
        self.failUnless(error < eps)

        
    def test_morphological_laplace1(self):
        "Test grey opening"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        structure = [[0, 0, 0], [0, 0, 0]]
        tmp1 = numarray.nd_image.grey_dilation(array, footprint = footprint,
                                               structure = structure)
        tmp2 = numarray.nd_image.grey_erosion(array, footprint = footprint,
                                              structure = structure)
        true = tmp1 + tmp2 - 2 * array
        output = numarray.zeros(array.shape, array.type())
        numarray.nd_image.morphological_laplace(array, footprint=footprint,
                                                structure=structure,
                                                output = output)
        error = mean_squared_error(true, output)
        self.failUnless(error < eps)
        

    def test_morphological_laplace2(self):
        "Test grey opening"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        structure = [[0, 0, 0], [0, 0, 0]]
        tmp1 = numarray.nd_image.grey_dilation(array, footprint = footprint,
                                               structure = structure)
        tmp2 = numarray.nd_image.grey_erosion(array, footprint = footprint,
                                              structure = structure)
        true = tmp1 + tmp2 - 2 * array
        output = numarray.nd_image.morphological_laplace(array,
                                                         footprint=footprint,
                                                         structure=structure)
        error = mean_squared_error(true, output)
        self.failUnless(error < eps)
        

    def test_white_tophat1(self):
        "Test grey opening"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        structure = [[0, 0, 0], [0, 0, 0]]
        tmp = numarray.nd_image.grey_opening(array, footprint = footprint,
                                             structure = structure)
        true = array - tmp
        output = numarray.zeros(array.shape, array.type())
        numarray.nd_image.white_tophat(array, footprint=footprint,
                                       structure=structure,
                                       output = output)
        error = mean_squared_error(true, output)
        self.failUnless(error < eps)
        

    def test_white_tophat2(self):
        "Test grey opening"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        structure = [[0, 0, 0], [0, 0, 0]]
        tmp = numarray.nd_image.grey_opening(array, footprint = footprint,
                                             structure = structure)
        true = array - tmp
        output = numarray.nd_image.white_tophat(array, footprint=footprint,
                                                structure=structure)
        error = mean_squared_error(true, output)
        self.failUnless(error < eps)
        

    def test_black_tophat1(self):
        "Test grey opening"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        structure = [[0, 0, 0], [0, 0, 0]]
        tmp = numarray.nd_image.grey_closing(array, footprint = footprint,
                                             structure = structure)
        true = tmp - array
        output = numarray.zeros(array.shape, array.type())
        numarray.nd_image.black_tophat(array, footprint=footprint,
                                       structure=structure,
                                       output = output)
        error = mean_squared_error(true, output)
        self.failUnless(error < eps)
        

    def test_black_tophat2(self):
        "Test grey opening"
        array = numarray.array([[3, 2, 5, 1, 4],
                                [7, 6, 9, 3, 5],
                                [5, 8, 3, 7, 1]])
        footprint = [[1, 0, 1], [1, 1, 0]]
        structure = [[0, 0, 0], [0, 0, 0]]
        tmp = numarray.nd_image.grey_closing(array, footprint = footprint,
                                             structure = structure)
        true = tmp - array
        output = numarray.nd_image.black_tophat(array, footprint=footprint,
                                                structure=structure)
        error = mean_squared_error(true, output)
        self.failUnless(error < eps)
        

    def test_hit_or_miss1(self):
        "test binary hit-or-miss transform"
        struct = [[0, 1, 0],
                  [1, 1, 1],
                  [0, 1, 0]]
        true = [[0, 0, 0, 0, 0],
                [0, 1, 0, 0, 0],
                [0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0]]
        for type in self.types:
            data = numarray.array([[0, 1, 0, 0, 0],
                                   [1, 1, 1, 0, 0],
                                   [0, 1, 0, 1, 1],
                                   [0, 0, 1, 1, 1],
                                   [0, 1, 1, 1, 0],
                                   [0, 1, 1, 1, 1],
                                   [0, 1, 1, 1, 1],
                                   [0, 0, 0, 0, 0]], type)
            out = numarray.zeros(data.shape, type)
            numarray.nd_image.binary_hit_or_miss(data, struct, output = out)
            error = mean_squared_error(out, true)
            self.failUnless(error < eps)
       
            
    def test_hit_or_miss2(self):
        "test binary hit-or-miss transform"
        struct = [[0, 1, 0],
                  [1, 1, 1],
                  [0, 1, 0]]
        true = [[0, 0, 0, 0, 0, 0, 0, 0],
                [0, 1, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0]]
        for type in self.types:
            data = numarray.array([[0, 1, 0, 0, 1, 1, 1, 0],
                                   [1, 1, 1, 0, 0, 1, 0, 0],
                                   [0, 1, 0, 1, 1, 1, 1, 0],
                                   [0, 0, 0, 0, 0, 0, 0, 0]], type)
            out = numarray.nd_image.binary_hit_or_miss(data, struct)
            error = mean_squared_error(out, true)
            self.failUnless(error < eps)
       
            
    def test_hit_or_miss3(self):
        "test binary hit-or-miss transform"
        struct1 = [[0, 0, 0],
                   [1, 1, 1],
                   [0, 0, 0]]
        struct2 = [[1, 1, 1],
                   [0, 0, 0],
                   [1, 1, 1]]
        true = [[0, 0, 0, 0, 0, 1, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 1, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0]]
        for type in self.types:
            data = numarray.array([[0, 1, 0, 0, 1, 1, 1, 0],
                                   [1, 1, 1, 0, 0, 0, 0, 0],
                                   [0, 1, 0, 1, 1, 1, 1, 0],
                                   [0, 0, 1, 1, 1, 1, 1, 0],
                                   [0, 1, 1, 1, 0, 1, 1, 0],
                                   [0, 0, 0, 0, 1, 1, 1, 0],
                                   [0, 1, 1, 1, 1, 1, 1, 0],
                                   [0, 0, 0, 0, 0, 0, 0, 0]], type)
            out = numarray.nd_image.binary_hit_or_miss(data, struct1, struct2)
            error = mean_squared_error(out, true)
            self.failUnless(error < eps)
       
            
    def test_label1(self):
        "test label"
        data = numarray.ones([])
        out, n = numarray.nd_image.label(data)
        error = mean_squared_error(out, 1)
        self.failUnless(error < eps and n == 1)
            
            
    def test_label2(self):
        "test label"
        data = numarray.zeros([])
        out, n = numarray.nd_image.label(data)
        error = mean_squared_error(out, 0)
        self.failUnless(error < eps and n == 0)
            
    def test_label3(self):
        "test label"
        data = numarray.ones([1])
        out, n = numarray.nd_image.label(data)
        error = mean_squared_error(out, [1])
        self.failUnless(error < eps and n == 1)
            
            
    def test_label4(self):
        "test label"
        data = numarray.zeros([1])
        out, n = numarray.nd_image.label(data)
        error = mean_squared_error(out, [0])
        self.failUnless(error < eps and n == 0)
            
            
    def test_label5(self):
        "test label"
        data = numarray.ones([5])
        out, n = numarray.nd_image.label(data)
        error = mean_squared_error(out, [1, 1, 1, 1, 1])
        self.failUnless(error < eps and n == 1)
            
            
    def test_label6(self):
        "test label"
        data = numarray.array([1, 0, 1, 1, 0, 1])
        out, n = numarray.nd_image.label(data)
        error = mean_squared_error(out, [1, 0, 2, 2, 0, 3])
        self.failUnless(error < eps and n == 3)
            
    def test_label7(self):
        "test label"
        data = numarray.array([[0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0]])
        out, n = numarray.nd_image.label(data)
        error = mean_squared_error(out, [[0, 0, 0, 0, 0, 0],
                                         [0, 0, 0, 0, 0, 0],
                                         [0, 0, 0, 0, 0, 0],
                                         [0, 0, 0, 0, 0, 0],
                                         [0, 0, 0, 0, 0, 0],
                                         [0, 0, 0, 0, 0, 0]])
        self.failUnless(error < eps and n == 0)
            
    def test_label8(self):
        "test label"
        data = numarray.array([[1, 0, 0, 0, 0, 0],
                               [0, 0, 1, 1, 0, 0],
                               [0, 0, 1, 1, 1, 0],
                               [1, 1, 0, 0, 0, 0],
                               [1, 1, 0, 0, 0, 0],
                               [0, 0, 0, 1, 1, 0]])
        out, n = numarray.nd_image.label(data)
        error = mean_squared_error(out, [[1, 0, 0, 0, 0, 0],
                                         [0, 0, 2, 2, 0, 0],
                                         [0, 0, 2, 2, 2, 0],
                                         [3, 3, 0, 0, 0, 0],
                                         [3, 3, 0, 0, 0, 0],
                                         [0, 0, 0, 4, 4, 0]])
        self.failUnless(error < eps and n == 4)

    def test_label9(self):
        "test label"
        data = numarray.array([[1, 0, 0, 0, 0, 0],
                               [0, 0, 1, 1, 0, 0],
                               [0, 0, 1, 1, 1, 0],
                               [1, 1, 0, 0, 0, 0],
                               [1, 1, 0, 0, 0, 0],
                               [0, 0, 0, 1, 1, 0]])
        struct = numarray.nd_image.generate_binary_structure(2, 2)
        out, n = numarray.nd_image.label(data, struct)
        error = mean_squared_error(out, [[1, 0, 0, 0, 0, 0],
                                         [0, 0, 2, 2, 0, 0],
                                         [0, 0, 2, 2, 2, 0],
                                         [2, 2, 0, 0, 0, 0],
                                         [2, 2, 0, 0, 0, 0],
                                         [0, 0, 0, 3, 3, 0]])
        self.failUnless(error < eps and n == 3)
        
    def test_label10(self):
        "test label"
        data = numarray.array([[0, 0, 0, 0, 0, 0],
                               [0, 1, 1, 0, 1, 0],
                               [0, 1, 1, 1, 1, 0],
                               [0, 0, 0, 0, 0, 0]])
        struct = numarray.nd_image.generate_binary_structure(2, 2)
        out, n = numarray.nd_image.label(data, struct)
        error = mean_squared_error(out, [[0, 0, 0, 0, 0, 0],
                                         [0, 1, 1, 0, 1, 0],
                                         [0, 1, 1, 1, 1, 0],
                                         [0, 0, 0, 0, 0, 0]])
        self.failUnless(error < eps and n == 1)
        
    def test_label11(self):
        "test label"
        for type in self.types:
            data = numarray.array([[1, 0, 0, 0, 0, 0],
                                   [0, 0, 1, 1, 0, 0],
                                   [0, 0, 1, 1, 1, 0],
                                   [1, 1, 0, 0, 0, 0],
                                   [1, 1, 0, 0, 0, 0],
                                   [0, 0, 0, 1, 1, 0]], type)
            out, n = numarray.nd_image.label(data)
            error = mean_squared_error(out, [[1, 0, 0, 0, 0, 0],
                                             [0, 0, 2, 2, 0, 0],
                                             [0, 0, 2, 2, 2, 0],
                                             [3, 3, 0, 0, 0, 0],
                                             [3, 3, 0, 0, 0, 0],
                                             [0, 0, 0, 4, 4, 0]])
            self.failUnless(error < eps and n == 4)

    def test_find_objects1(self):
        "test find_objects"
        data = numarray.ones([])
        out = numarray.nd_image.find_objects(data)
        self.failUnless(out == [()])
            
            
    def test_find_objects2(self):
        "test find_objects"
        data = numarray.zeros([])
        out = numarray.nd_image.find_objects(data)
        self.failUnless(out == [])
            
    def test_find_objects3(self):
        "test find_objects"
        data = numarray.ones([1])
        out = numarray.nd_image.find_objects(data)
        self.failUnless(out == [(slice(0, 1, None),)])
            
    def test_find_objects4(self):
        "test find_objects"
        data = numarray.zeros([1])
        out = numarray.nd_image.find_objects(data)
        self.failUnless(out == [])
                        
            
    def test_find_objects5(self):
        "test find_objects"
        data = numarray.ones([5])
        out = numarray.nd_image.find_objects(data)
        self.failUnless(out == [(slice(0, 5, None),)])
            
            
    def test_find_objects6(self):
        "test find_objects"
        data = numarray.array([1, 0, 2, 2, 0, 3])
        out = numarray.nd_image.find_objects(data)
        self.failUnless(out == [(slice(0, 1, None),),
                                (slice(2, 4, None),),
                                (slice(5, 6, None),)])
            
    def test_find_objects7(self):
        "test find_objects"
        data = numarray.array([[0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0]])
        out = numarray.nd_image.find_objects(data)
        self.failUnless(out == []),
            
    def test_find_objects8(self):
        "test find_objects"
        data = numarray.array([[1, 0, 0, 0, 0, 0],
                               [0, 0, 2, 2, 0, 0],
                               [0, 0, 2, 2, 2, 0],
                               [3, 3, 0, 0, 0, 0],
                               [3, 3, 0, 0, 0, 0],
                               [0, 0, 0, 4, 4, 0]])
        out = numarray.nd_image.find_objects(data)
        self.failUnless(out == [(slice(0, 1, None), slice(0, 1, None)),
                                (slice(1, 3, None), slice(2, 5, None)),
                                (slice(3, 5, None), slice(0, 2, None)),
                                (slice(5, 6, None), slice(3, 5, None))])

        
    def test_find_objects9(self):
        "test find_objects"
        data = numarray.array([[1, 0, 0, 0, 0, 0],
                               [0, 0, 2, 2, 0, 0],
                               [0, 0, 2, 2, 2, 0],
                               [0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 0, 0, 0],
                               [0, 0, 0, 4, 4, 0]])
        out = numarray.nd_image.find_objects(data)
        self.failUnless(out == [(slice(0, 1, None), slice(0, 1, None)),
                                (slice(1, 3, None), slice(2, 5, None)),
                                None,
                                (slice(5, 6, None), slice(3, 5, None))])

    
    def test_spline1(self):
        "test spline filter"
        for type in self.types:
            data = numarray.ones([], type)
            for order in range(2, 6):
                out = numarray.nd_image.spline_filter(data, order = order)
                error = mean_squared_error(out, 1)
                self.failUnless(error < eps and out.type() == numarray.Float64)
            
    def test_spline2(self):
        "test spline filter"
        for type in self.types:
            data = numarray.array([1])
            for order in range(2, 6):
                out = numarray.nd_image.spline_filter(data, order = order)
                error = mean_squared_error(out, [1])
                self.failUnless(error < eps and out.type() == numarray.Float64)
            
    def test_spline3(self):
        "test spline filter"
        for type in self.types:
            data = numarray.ones([], type)
            for order in range(2, 6):
                out = numarray.nd_image.spline_filter(data, order,
                                                      output_type = type)
                error = mean_squared_error(out, 1)
                self.failUnless(error < eps and
                                out.type() == type)
            
    def test_spline4(self):
        "test spline filter"
        for type in self.types:
            data = numarray.ones([4], type)
            for order in range(2, 6):
                out = numarray.nd_image.spline_filter(data, order)
                error = mean_squared_error(out, [1, 1, 1, 1])
                self.failUnless(error < eps)
                        

    def test_spline5(self):
        "test spline filter"
        for type in self.types:
            data = numarray.ones([4, 4], type)
            for order in range(2, 6):
                out = numarray.nd_image.spline_filter(data, order = order)
                error = mean_squared_error(out, [[1, 1, 1, 1],
                                                 [1, 1, 1, 1],
                                                 [1, 1, 1, 1],
                                                 [1, 1, 1, 1]])
                self.failUnless(error < eps)
        
    def test_geometric_transform1(self):
        "test geometrical_transform"
        data = numarray.array([1])
        def mapping(x):
            return x
        for order in range(0, 6):
            out = numarray.nd_image.geometric_transform(data, mapping,
                                                        data.shape, order=order)
            error = mean_squared_error(out, [1])
            self.failUnless(error < eps)

    def test_geometric_transform2(self):
        "test geometric_transform filter"
        data = numarray.ones([4])
        def mapping(x):
            return x
        for order in range(0, 6):
            out = numarray.nd_image.geometric_transform(data, mapping,
                                                        data.shape, order=order)
            error = mean_squared_error(out, [1, 1, 1, 1])
            self.failUnless(error < eps)

    def test_geometric_transform3(self):
        "test geometric_transform filter"
        data = numarray.ones([4])
        def mapping(x):
            return (x[0] - 1,)
        for order in range(0, 6):
            out = numarray.nd_image.geometric_transform(data, mapping,
                                                        data.shape,
                                                        order=order)
            error = mean_squared_error(out, [0, 1, 1, 1])
            self.failUnless(error < eps)
                        
    def test_geometric_transform4(self):
        "test geometric_transform filter"
        data = numarray.array([4, 1, 3, 2])
        def mapping(x):
            return (x[0] - 1,)
        for order in range(0, 6):
            out = numarray.nd_image.geometric_transform(data, mapping,
                                                        data.shape,
                                                        order=order)
            error = mean_squared_error(out, [0, 4, 1, 3])
            self.failUnless(error < eps)
                                
    def test_geometric_transform5(self):
        "test geometric_transform filter"
        data = numarray.array([[1, 1, 1, 1],
                               [1, 1, 1, 1], 
                               [1, 1, 1, 1]])
        def mapping(x):
            return (x[0], x[1] - 1)
        for order in range(0, 6):
            out = numarray.nd_image.geometric_transform(data, mapping,
                                                        data.shape,
                                                        order=order)
            error = mean_squared_error(out, [[0, 1, 1, 1],
                                             [0, 1, 1, 1],
                                             [0, 1, 1, 1]])
            self.failUnless(error < eps)
                                

    def test_geometric_transform6(self):
        "test geometric_transform filter"
        data = numarray.array([[4, 1, 3, 2],
                               [7, 6, 8, 5], 
                               [3, 5, 3, 6]])
        def mapping(x):
            return (x[0], x[1] - 1)
        for order in range(0, 6):
            out = numarray.nd_image.geometric_transform(data, mapping,
                                                        data.shape,
                                                        order=order)
            error = mean_squared_error(out, [[0, 4, 1, 3],
                                             [0, 7, 6, 8],
                                             [0, 3, 5, 3]])
            self.failUnless(error < eps)
                                

    def test_geometric_transform7(self):
        "test geometric_transform filter"
        data = numarray.array([[4, 1, 3, 2],
                               [7, 6, 8, 5], 
                               [3, 5, 3, 6]])
        def mapping(x):
            return (x[0] - 1, x[1])
        for order in range(0, 6):
            out = numarray.nd_image.geometric_transform(data, mapping, 
                                                        data.shape,
                                                        order=order)
            error = mean_squared_error(out, [[0, 0, 0, 0],
                                             [4, 1, 3, 2],
                                             [7, 6, 8, 5]])
            self.failUnless(error < eps)
            
                                
    def test_geometric_transform8(self):
        "test geometric_transform filter"
        data = numarray.array([[4, 1, 3, 2],
                               [7, 6, 8, 5], 
                               [3, 5, 3, 6]])
        def mapping(x):
            return (x[0] - 1, x[1] - 1)
        for order in range(0, 6):
            out = numarray.nd_image.geometric_transform(data, mapping, 
                                                        data.shape,
                                                        order=order)
            error = mean_squared_error(out, [[0, 0, 0, 0],
                                             [0, 4, 1, 3],
                                             [0, 7, 6, 8]])
            self.failUnless(error < eps)
            

    def test_geometric_transform9(self):
        "test geometric_transform filter"
        data = numarray.array([[4, 1, 3, 2],
                               [7, 6, 8, 5], 
                               [3, 5, 3, 6]])
        def mapping(x):
            return (x[0] - 1, x[1] - 1)
        for order in range(0, 6):
            if (order > 1):
                filtered = numarray.nd_image.spline_filter(data, order=order)
            else:
                filtered = data
            out = numarray.nd_image.geometric_transform(filtered, mapping,
                                                        data.shape,
                                                        order=order,
                                                        prefilter = False)
            error = mean_squared_error(out, [[0, 0, 0, 0],
                                             [0, 4, 1, 3],
                                             [0, 7, 6, 8]])
            self.failUnless(error < eps)


    def test_geometric_transform10(self):
        "test geometric_transform filter"
        data = numarray.ones([2], numarray.Float64)
        def mapping(x):
            return (x[0] / 2,)
        for order in range(0, 6):
            out = numarray.nd_image.geometric_transform(data, mapping,
                                                        [4], order=order)
            error = mean_squared_error(out, [1, 1, 1, 1])
            self.failUnless(error < eps)
            
                        
    def test_geometric_transform11(self):
        "test geometric_transform filter"
        data = [1, 5, 2, 6, 3, 7, 4, 4]
        def mapping(x):
            return (2 * x[0],)
        for order in range(0, 6):
            out = numarray.nd_image.geometric_transform(data, mapping,
                                                        [4], order=order)
            error = mean_squared_error(out, [1, 2, 3, 4])
            self.failUnless(error < eps)
            
                                
    def test_geometric_transform12(self):
        "test geometric_transform filter"
        data = [1, 2, 3, 4]
        def mapping(x):
            return (x[0] / 2,)
        for order in range(0, 6):
            out = numarray.nd_image.geometric_transform(data, mapping,
                                                        [8], order=order)
            error = mean_squared_error(out[::2], [1, 2, 3, 4])
            self.failUnless(error < eps)
                                                                

    def test_geometric_transform13(self):
        "test geometric_transform filter"
        data = [[1, 2, 3, 4],
                [5, 6, 7, 8],
                [9.0, 10, 11, 12]]
        def mapping(x):
            return (x[0], x[1] * 2)
        for order in range(0, 6):
            out = numarray.nd_image.geometric_transform(data, mapping,
                                                        (3, 2), order=order)
            error = mean_squared_error(out, [[1, 3],
                                             [5, 7],
                                             [9, 11]])
            self.failUnless(error < eps)
            

    def test_geometric_transform20(self):
        "test geometric_transform filter"
        data = [[1, 2, 3, 4],
                [5, 6, 7, 8],
                [9, 10, 11, 12]]
        def mapping(x):
            return (x[0] * 2, x[1])
        for order in range(0, 6):
            out = numarray.nd_image.geometric_transform(data, mapping,
                                                        (1, 4), order=order)
            error = mean_squared_error(out, [[1, 2, 3, 4]])
            self.failUnless(error < eps)
            
            
    def test_geometric_transform21(self):
        "test geometric_transform filter"
        data = [[1, 2, 3, 4],
                [5, 6, 7, 8],
                [9, 10, 11, 12]]
        def mapping(x):
            return (x[0] * 2, x[1] * 2)
        for order in range(0, 6):
            out = numarray.nd_image.geometric_transform(data, mapping,
                                                        (1, 2), order=order)
            error = mean_squared_error(out, [[1, 3]])
            self.failUnless(error < eps)


    def test_geometric_transform22(self):
        "test geometric_transform filter"
        data = [[1, 2, 3, 4],
                [5, 6, 7, 8],
                [9, 10, 11, 12]]
        def mapping(x):
            return (x[0], x[1] / 2)
        for order in range(0, 6):
            out = numarray.nd_image.geometric_transform(data, mapping,
                                                        (3, 8), order=order)
            error = mean_squared_error(out[..., ::2], data)
            self.failUnless(error < eps)
            

    def test_geometric_transform23(self):
        "test geometric_transform filter"
        data = [[1, 2, 3, 4],
                [5, 6, 7, 8],
                [9, 10, 11, 12]]
        def mapping(x):
            return (x[0] / 2, x[1])
        for order in range(0, 6):
            out = numarray.nd_image.geometric_transform(data, mapping,
                                                        (6, 4), order=order)
            error = mean_squared_error(out[::2, ...], data)
            self.failUnless(error < eps)


    def test_geometric_transform24(self):
        "test geometric_transform filter"
        data = [[1, 2, 3, 4],
                [5, 6, 7, 8],
                [9, 10, 11, 12]]
        def mapping(x):
            return (x[0] / 2, x[1] / 2)
        for order in range(0, 6):
            out = numarray.nd_image.geometric_transform(data, mapping,
                                                        (6, 8), order=order)
            error = mean_squared_error(out[::2, ::2], data)
            self.failUnless(error < eps)
            

    def test_geometric_transform25(self):
        "test geometric_transform filter"
        data = numarray.array([[1, 2, 3, 4],
                               [5, 6, 7, 8],
                               [9, 10, 11, 12]], numarray.Float64)
        def mapping1(x):
            return (x[0] / 2, x[1] / 2)
        def mapping2(x):
            return (x[0] * 2, x[1] * 2)
        for order in range(0, 6):
            out = numarray.nd_image.geometric_transform(data, mapping1,
                                                        (6, 8), order=order)
            out = numarray.nd_image.geometric_transform(out, mapping2,
                                                        (3, 4), order=order)
            error = mean_squared_error(out, data)
            self.failUnless(error < eps)


    def test_geometric_transform26(self):
        "test geometric_transform filter"
        data = [[1, 2, 3, 4],
                [5, 6, 7, 8],
                [9, 10, 11, 12]]
        def mapping(x):
            return (1, x[0] * 2)
        for order in range(0, 6):
            out = numarray.nd_image.geometric_transform(data, mapping,
                                                        (2,), order=order)
            error = mean_squared_error(out, [5, 7])
            self.failUnless(error < eps)

    def test_affine_transform1(self):
        "test affine_transform"
        data = numarray.array([1])
        for order in range(0, 6):
            out = numarray.nd_image.affine_transform(data, [[1]], order=order)
            error = mean_squared_error(out, [1])
            self.failUnless(error < eps)

    def test_affine_transform2(self):
        "test affine_transform filter"
        data = numarray.ones([4])
        for order in range(0, 6):
            out = numarray.nd_image.affine_transform(data, [[1]], order=order)
            error = mean_squared_error(out, [1, 1, 1, 1])
            self.failUnless(error < eps)

    def test_affine_transform3(self):
        "test affine_transform filter"
        data = numarray.ones([4])
        for order in range(0, 6):
            out = numarray.nd_image.affine_transform(data, [[1]], -1,
                                                     order=order)
            error = mean_squared_error(out, [0, 1, 1, 1])
            self.failUnless(error < eps)
                        
    def test_affine_transform4(self):
        "test affine_transform filter"
        data = numarray.array([4, 1, 3, 2])
        for order in range(0, 6):
            out = numarray.nd_image.affine_transform(data, [[1]], -1,
                                                     order=order)
            error = mean_squared_error(out, [0, 4, 1, 3])
            self.failUnless(error < eps)
                                
    def test_affine_transform5(self):
        "test affine_transform filter"
        data = numarray.array([[1, 1, 1, 1],
                               [1, 1, 1, 1], 
                               [1, 1, 1, 1]])
        for order in range(0, 6):
            out = numarray.nd_image.affine_transform(data, [[1, 0], [0, 1]],
                                                     [0, -1], order=order)
            error = mean_squared_error(out, [[0, 1, 1, 1],
                                             [0, 1, 1, 1],
                                             [0, 1, 1, 1]])
            self.failUnless(error < eps)
                                

    def test_affine_transform6(self):
        "test affine_transform filter"
        data = numarray.array([[4, 1, 3, 2],
                               [7, 6, 8, 5], 
                               [3, 5, 3, 6]])
        for order in range(0, 6):
            out = numarray.nd_image.affine_transform(data, [[1, 0], [0, 1]],
                                                     [0, -1], order=order)
            error = mean_squared_error(out, [[0, 4, 1, 3],
                                             [0, 7, 6, 8],
                                             [0, 3, 5, 3]])
            self.failUnless(error < eps)
                                

    def test_affine_transform7(self):
        "test affine_transform filter"
        data = numarray.array([[4, 1, 3, 2],
                               [7, 6, 8, 5], 
                               [3, 5, 3, 6]])
        for order in range(0, 6):
            out = numarray.nd_image.affine_transform(data, [[1, 0], [0, 1]],
                                                     [-1, 0], order=order)
            error = mean_squared_error(out, [[0, 0, 0, 0],
                                             [4, 1, 3, 2],
                                             [7, 6, 8, 5]])
            self.failUnless(error < eps)
            
                                
    def test_affine_transform8(self):
        "test affine_transform filter"
        data = numarray.array([[4, 1, 3, 2],
                               [7, 6, 8, 5], 
                               [3, 5, 3, 6]])
        for order in range(0, 6):
            out = numarray.nd_image.affine_transform(data, [[1, 0], [0, 1]],
                                                     [-1, -1], order=order)
            error = mean_squared_error(out, [[0, 0, 0, 0],
                                             [0, 4, 1, 3],
                                             [0, 7, 6, 8]])
            self.failUnless(error < eps)
            

    def test_affine_transform9(self):
        "test affine_transform filter"
        data = numarray.array([[4, 1, 3, 2],
                               [7, 6, 8, 5], 
                               [3, 5, 3, 6]])
        for order in range(0, 6):
            if (order > 1):
                filtered = numarray.nd_image.spline_filter(data, order=order)
            else:
                filtered = data
            out = numarray.nd_image.affine_transform(filtered,[[1, 0], [0, 1]],
                                                     [-1, -1], order=order,
                                                     prefilter = False)
            error = mean_squared_error(out, [[0, 0, 0, 0],
                                             [0, 4, 1, 3],
                                             [0, 7, 6, 8]])
            self.failUnless(error < eps)


    def test_affine_transform10(self):
        "test affine_transform filter"
        data = numarray.ones([2], numarray.Float64)
        for order in range(0, 6):
            out = numarray.nd_image.affine_transform(data, [[0.5]],
                                                     output_shape = (4,),
                                                     order=order)
            error = mean_squared_error(out, [1, 1, 1, 1])
            self.failUnless(error < eps)
            
                        
    def test_affine_transform11(self):
        "test affine_transform filter"
        data = [1, 5, 2, 6, 3, 7, 4, 4]
        for order in range(0, 6):
            out = numarray.nd_image.affine_transform(data, [[2]], 0, (4,),
                                                     order=order)
            error = mean_squared_error(out, [1, 2, 3, 4])
            self.failUnless(error < eps)
            
                                
    def test_affine_transform12(self):
        "test affine_transform filter"
        data = [1, 2, 3, 4]
        for order in range(0, 6):
            out = numarray.nd_image.affine_transform(data, [[0.5]], 0, (8,),
                                                     order=order)
            error = mean_squared_error(out[::2], [1, 2, 3, 4])
            self.failUnless(error < eps)
                                                                

    def test_affine_transform13(self):
        "test affine_transform filter"
        data = [[1, 2, 3, 4],
                [5, 6, 7, 8],
                [9.0, 10, 11, 12]]
        for order in range(0, 6):
            out = numarray.nd_image.affine_transform(data, [[1, 0], [0, 2]], 0,
                                                     (3, 2), order=order)
            error = mean_squared_error(out, [[1, 3],
                                             [5, 7],
                                             [9, 11]])
            self.failUnless(error < eps)
            

    def test_affine_transform20(self):
        "test affine_transform filter"
        data = [[1, 2, 3, 4],
                [5, 6, 7, 8],
                [9, 10, 11, 12]]
        for order in range(0, 6):
            out = numarray.nd_image.affine_transform(data, [[2, 0], [0, 1]], 0,
                                                     (1, 4), order=order)
            error = mean_squared_error(out, [[1, 2, 3, 4]])
            self.failUnless(error < eps)
            
            
    def test_affine_transform21(self):
        "test affine_transform filter"
        data = [[1, 2, 3, 4],
                [5, 6, 7, 8],
                [9, 10, 11, 12]]
        for order in range(0, 6):
            out = numarray.nd_image.affine_transform(data, [[2, 0], [0, 2]], 0,
                                                     (1, 2), order=order)
            error = mean_squared_error(out, [[1, 3]])
            self.failUnless(error < eps)


    def test_affine_transform22(self):
        "test affine_transform filter"
        data = [[1, 2, 3, 4],
                [5, 6, 7, 8],
                [9, 10, 11, 12]]
        for order in range(0, 6):
            out = numarray.nd_image.affine_transform(data, [[1, 0], [0, 0.5]],
                                                     0, (3, 8), order=order)
            error = mean_squared_error(out[..., ::2], data)
            self.failUnless(error < eps)
            

    def test_affine_transform23(self):
        "test affine_transform filter"
        data = [[1, 2, 3, 4],
                [5, 6, 7, 8],
                [9, 10, 11, 12]]
        for order in range(0, 6):
            out = numarray.nd_image.affine_transform(data, [[0.5, 0], [0, 1]],
                                                     0, (6, 4), order=order)
            error = mean_squared_error(out[::2, ...], data)
            self.failUnless(error < eps)


    def test_affine_transform24(self):
        "test affine_transform filter"
        data = [[1, 2, 3, 4],
                [5, 6, 7, 8],
                [9, 10, 11, 12]]
        for order in range(0, 6):
            out = numarray.nd_image.affine_transform(data,
                                                     [[0.5, 0], [0, 0.5]], 0,
                                                     (6, 8), order=order)
            error = mean_squared_error(out[::2, ::2], data)
            self.failUnless(error < eps)
            

    def test_affine_transform25(self):
        "test affine_transform filter"
        data = numarray.array([[1, 2, 3, 4],
                               [5, 6, 7, 8],
                               [9, 10, 11, 12]], numarray.Float64)
        for order in range(0, 6):
            out = numarray.nd_image.affine_transform(data,
                                                     [[0.5, 0], [0, 0.5]], 0,
                                                     (6, 8), order=order)
            out = numarray.nd_image.affine_transform(out,
                                                     [[2.0, 0], [0, 2.0]], 0,
                                                     (3, 4), order=order)
            error = mean_squared_error(out, data)
            self.failUnless(error < eps)


    def test_affine_transform26(self):
        "test affine_transform filter"
        data = [[1, 2, 3, 4],
                [5, 6, 7, 8],
                [9, 10, 11, 12]]
        for order in range(0, 6):
            out = numarray.nd_image.affine_transform(data, [[0], [2]], 0,
                                                     (2,), order=order)
            error = mean_squared_error(out, [1, 3])
            self.failUnless(error < eps)


    def test_affine_transform27(self):
        "test affine_transform filter"
        data = [[1, 2, 3, 4],
                [5, 6, 7, 8],
                [9, 10, 11, 12]]
        for order in range(0, 6):
            out = numarray.nd_image.affine_transform(data, [[2], [0]], 0,
                                                     (2,), order=order)
            error = mean_squared_error(out, [1, 9])
            self.failUnless(error < eps)


    def test_shift1(self):
        "test shift"
        data = numarray.array([1])
        for order in range(0, 6):
            out = numarray.nd_image.shift(data, [1], order=order)
            error = mean_squared_error(out, [0])
            self.failUnless(error < eps)

    def test_shift2(self):
        "test shift filter"
        data = numarray.ones([4])
        for order in range(0, 6):
            out = numarray.nd_image.shift(data, [1], order=order)
            error = mean_squared_error(out, [0, 1, 1, 1])
            self.failUnless(error < eps)

    def test_shift3(self):
        "test shift filter"
        data = numarray.ones([4])
        for order in range(0, 6):
            out = numarray.nd_image.shift(data, -1, order=order)
            error = mean_squared_error(out, [1, 1, 1, 0])
            self.failUnless(error < eps)
                        
    def test_shift4(self):
        "test shift filter"
        data = numarray.array([4, 1, 3, 2])
        for order in range(0, 6):
            out = numarray.nd_image.shift(data, 1, order=order)
            error = mean_squared_error(out, [0, 4, 1, 3])
            self.failUnless(error < eps)
                                
    def test_shift5(self):
        "test shift filter"
        data = numarray.array([[1, 1, 1, 1],
                               [1, 1, 1, 1], 
                               [1, 1, 1, 1]])
        for order in range(0, 6):
            out = numarray.nd_image.shift(data, [0, 1], order=order)
            error = mean_squared_error(out, [[0, 1, 1, 1],
                                             [0, 1, 1, 1],
                                             [0, 1, 1, 1]])
            self.failUnless(error < eps)
                                

    def test_shift6(self):
        "test shift filter"
        data = numarray.array([[4, 1, 3, 2],
                               [7, 6, 8, 5], 
                               [3, 5, 3, 6]])
        for order in range(0, 6):
            out = numarray.nd_image.shift(data, [0, 1], order=order)
            error = mean_squared_error(out, [[0, 4, 1, 3],
                                             [0, 7, 6, 8],
                                             [0, 3, 5, 3]])
            self.failUnless(error < eps)
                                

    def test_shift7(self):
        "test shift filter"
        data = numarray.array([[4, 1, 3, 2],
                               [7, 6, 8, 5], 
                               [3, 5, 3, 6]])
        for order in range(0, 6):
            out = numarray.nd_image.shift(data, [1, 0], order=order)
            error = mean_squared_error(out, [[0, 0, 0, 0],
                                             [4, 1, 3, 2],
                                             [7, 6, 8, 5]])
            self.failUnless(error < eps)
            
                                
    def test_shift8(self):
        "test shift filter"
        data = numarray.array([[4, 1, 3, 2],
                               [7, 6, 8, 5], 
                               [3, 5, 3, 6]])
        for order in range(0, 6):
            out = numarray.nd_image.shift(data, [1, 1], order=order)
            error = mean_squared_error(out, [[0, 0, 0, 0],
                                             [0, 4, 1, 3],
                                             [0, 7, 6, 8]])
            self.failUnless(error < eps)
            

    def test_shift9(self):
        "test shift filter"
        data = numarray.array([[4, 1, 3, 2],
                               [7, 6, 8, 5], 
                               [3, 5, 3, 6]])
        for order in range(0, 6):
            if (order > 1):
                filtered = numarray.nd_image.spline_filter(data, order=order)
            else:
                filtered = data
            out = numarray.nd_image.shift(filtered, [1, 1], order=order,
                                          prefilter = False)
            error = mean_squared_error(out, [[0, 0, 0, 0],
                                             [0, 4, 1, 3],
                                             [0, 7, 6, 8]])
            self.failUnless(error < eps)


    def test_zoom1(self):
        "test zoom filter"
        data = numarray.ones([2], numarray.Float64)
        for order in range(0, 6):
            out = numarray.nd_image.zoom(data, 2.0, order=order)
            error = mean_squared_error(out, [1, 1, 1, 1])
            self.failUnless(error < eps)
            
                        
    def test_zoom2(self):
        "test zoom filter"
        data = [1, 5, 2, 6, 3, 7, 4, 4]
        for order in range(0, 6):
            out = numarray.nd_image.zoom(data, 0.5, order=order)
            error = mean_squared_error(out, [1, 2, 3, 4])
            self.failUnless(error < eps)
            
                                
    def test_zoom3(self):
        "test zoom filter"
        data = [1, 2, 3, 4]
        for order in range(0, 6):
            out = numarray.nd_image.zoom(data, 2, order=order)
            error = mean_squared_error(out[::2], [1, 2, 3, 4])
            self.failUnless(error < eps)
                                                                

    def test_zoom4(self):
        "test zoom filter"
        data = [[1, 2, 3, 4],
                [5, 6, 7, 8],
                [9.0, 10, 11, 12]]
        for order in range(0, 6):
            out = numarray.nd_image.zoom(data, [1, 0.5], order=order)
            error = mean_squared_error(out, [[1, 3],
                                             [5, 7],
                                             [9, 11]])
            self.failUnless(error < eps)
            

    def test_zoom5(self):
        "test zoom filter"
        data = [[1, 2, 3, 4],
                [5, 6, 7, 8],
                [9, 10, 11, 12]]
        for order in range(0, 6):
            out = numarray.nd_image.zoom(data, [0.5, 1], order=order)
            error = mean_squared_error(out, [[1, 2, 3, 4]])
            self.failUnless(error < eps)
            
            
    def test_zoom6(self):
        "test zoom filter"
        data = [[1, 2, 3, 4],
                [5, 6, 7, 8],
                [9, 10, 11, 12]]
        for order in range(0, 6):
            out = numarray.nd_image.zoom(data, [0.5, 0.5], order=order)
            error = mean_squared_error(out, [[1, 3]])
            self.failUnless(error < eps)


    def test_zoom7(self):
        "test zoom filter"
        data = [[1, 2, 3, 4],
                [5, 6, 7, 8],
                [9, 10, 11, 12]]
        for order in range(0, 6):
            out = numarray.nd_image.zoom(data, [1, 2], order=order)
            error = mean_squared_error(out[..., ::2], data)
            self.failUnless(error < eps)
            

    def test_zoom8(self):
        "test zoom filter"
        data = [[1, 2, 3, 4],
                [5, 6, 7, 8],
                [9, 10, 11, 12]]
        for order in range(0, 6):
            out = numarray.nd_image.zoom(data, [2, 1], order=order)
            error = mean_squared_error(out[::2, ...], data)
            self.failUnless(error < eps)


    def test_zoom9(self):
        "test zoom filter"
        data = [[1, 2, 3, 4],
                [5, 6, 7, 8],
                [9, 10, 11, 12]]
        for order in range(0, 6):
            out = numarray.nd_image.zoom(data, [2, 2], order=order)
            error = mean_squared_error(out[::2, ::2], data)
            self.failUnless(error < eps)
            

    def test_zoom10(self):
        "test zoom filter"
        data = numarray.array([[1, 2, 3, 4],
                               [5, 6, 7, 8],
                               [9, 10, 11, 12]], numarray.Float64)
        for order in range(0, 6):
            out = numarray.nd_image.zoom(data, [2, 2], order=order)
            out = numarray.nd_image.zoom(out, [0.5, 0.5], order=order)
            error = mean_squared_error(out, data)
            self.failUnless(error < eps)


    def test_zoom_affine1(self):
        "test zoom filter by affine transformation"
        data = [[1, 2, 3, 4],
                [5, 6, 7, 8],
                [9, 10, 11, 12]]
        for order in range(0, 6):
            out = numarray.nd_image.affine_transform(data, [0.5, 0.5], 0,
                                                     (6, 8), order=order)
            error = mean_squared_error(out[::2, ::2], data)
            self.failUnless(error < eps)


    def test_map_indices1(self):
        "test map indices"
        data = numarray.array([[4, 1, 3, 2],
                               [7, 6, 8, 5], 
                               [3, 5, 3, 6]])
        idx = numarray.indices(data.shape)
        idx -= 1
        for order in range(0, 6):
            out = numarray.nd_image.map_indices(data, idx, order=order)
            error = mean_squared_error(out, [[0, 0, 0, 0],
                                             [0, 4, 1, 3],
                                             [0, 7, 6, 8]])
            self.failUnless(error < eps)

    def test_map_indices2(self):
        "test map indices"
        data = numarray.array([[4, 1, 3, 2],
                               [7, 6, 8, 5], 
                               [3, 5, 3, 6]])
        idx = numarray.indices(data.shape, numarray.Float64)
        idx -= 0.5
        for order in range(0, 6):
            out1 = numarray.nd_image.shift(data, 0.5, order=order)
            out2 = numarray.nd_image.map_indices(data, idx, order=order)
            error = mean_squared_error(out1, out2)
            self.failUnless(error < eps)

    def test_rotate1(self):
        "test array rotate"
        data = numarray.array([[0, 0, 0, 0],
                               [0, 1, 1, 0],
                               [0, 0, 0, 0]], numarray.Float64)
        for order in range(0, 6):
            out = numarray.nd_image.rotate(data, 0)
            error = mean_squared_error(out, data)
            self.failUnless(error < eps)
            
    def test_rotate2(self):
        "test array rotate"
        data = numarray.array([[0, 0, 0, 0],
                               [0, 1, 0, 0],
                               [0, 0, 0, 0]], numarray.Float64)
        true = numarray.array([[0, 0, 0],
                               [0, 0, 0],
                               [0, 1, 0],
                               [0, 0, 0]], numarray.Float64)
        for order in range(0, 6):
            out = numarray.nd_image.rotate(data, 90)
            error = mean_squared_error(out, true)
            self.failUnless(error < eps)
            
    def test_rotate3(self):
        "test array rotate"
        data = numarray.array([[0, 0, 0, 0, 0],
                               [0, 1, 1, 0, 0],
                               [0, 0, 0, 0, 0]], numarray.Float64)
        true = numarray.array([[0, 0, 0],
                               [0, 0, 0],
                               [0, 1, 0],
                               [0, 1, 0],
                               [0, 0, 0]], numarray.Float64)
        for order in range(0, 6):
            out = numarray.nd_image.rotate(data, 90)
            error = mean_squared_error(out, true)
            self.failUnless(error < eps)
            
    def test_sum1(self):
        "Test sum of an empty array"
        for type in self.types:
            input = numarray.array([], type)
            output = numarray.nd_image.sum(input)
            self.failUnless(output == 0.0)
            
    def test_sum2(self):
        "Test sum of an empty array"
        for type in self.types:
            input = numarray.zeros([0, 4], type)
            output = numarray.nd_image.sum(input)
            self.failUnless(output == 0.0)
            
    def test_sum3(self):
        "Test sum of a scalar array"
        for type in self.types:
            input = numarray.ones([], type)
            output = numarray.nd_image.sum(input)
            self.failUnless(output == 1.0)

    def test_sum4(self):
        "Test sum of a one-dimensional array"
        for type in self.types:
            input = numarray.array([1, 2], type)
            output = numarray.nd_image.sum(input)
            self.failUnless(output == 3.0)

    def test_sum5(self):
        "Test sum of a two-dimensional array"
        for type in self.types:
            input = numarray.array([[1, 2], [3, 4]], type)
            output = numarray.nd_image.sum(input)
            self.failUnless(output == 10.0)

    def test_sum6(self):
        "Test sum of an empty array"
        labels = numarray.array([], numarray.Bool)
        for type in self.types:
            input = numarray.array([], type)
            output = numarray.nd_image.sum(input, labels = labels)
            self.failUnless(output == 0.0)
            
    def test_sum7(self):
        "Test sum of an empty array"
        labels = numarray.ones([0, 4], numarray.Bool)
        for type in self.types:
            input = numarray.zeros([0, 4], type)
            output = numarray.nd_image.sum(input, labels = labels)
            self.failUnless(output == 0.0)
            
    def test_sum8(self):
        "Test sum of a one-dimensional array"
        labels = numarray.array([1, 0], numarray.Bool)
        for type in self.types:
            input = numarray.array([1, 2], type)
            output = numarray.nd_image.sum(input, labels = labels)
            self.failUnless(output == 1.0)

    def test_sum9(self):
        "Test sum of a two-dimensional array"
        labels = numarray.array([1, 0], numarray.Bool)
        for type in self.types:
            input = numarray.array([[1, 2], [3, 4]], type)
            output = numarray.nd_image.sum(input, labels = labels)
            self.failUnless(output == 4.0)

    def test_sum10(self):
        "Test sum of a two-dimensional array"
        labels = numarray.array([1, 0], numarray.Bool)
        input = numarray.array([[1, 2], [3, 4]], numarray.Bool)
        output = numarray.nd_image.sum(input, labels = labels)
        self.failUnless(output == 2.0)

    def test_sum11(self):
        "Test sum of a two-dimensional array"
        labels = numarray.array([1, 2], numarray.Int8)
        for type in self.types:
            input = numarray.array([[1, 2], [3, 4]], type)
            output = numarray.nd_image.sum(input, labels = labels, index = 2)
            self.failUnless(output == 6.0)


    def test_sum12(self):
        "Test sum of a two-dimensional array"
        labels = numarray.array([[1, 2], [2, 4]], numarray.Int8)
        for type in self.types:
            input = numarray.array([[1, 2], [3, 4]], type)
            output = numarray.nd_image.sum(input, labels = labels,
                                            index = [4, 8, 2])
            self.failUnless(output == [4.0, 0.0, 5.0])


    def test_mean1(self):
        "Test mean of a two-dimensional array"
        labels = numarray.array([1, 0], numarray.Bool)
        for type in self.types:
            input = numarray.array([[1, 2], [3, 4]], type)
            output = numarray.nd_image.mean(input, labels = labels)
            self.failUnless(output == 2.0)

    def test_mean2(self):
        "Test mean of a two-dimensional array"
        labels = numarray.array([1, 0], numarray.Bool)
        input = numarray.array([[1, 2], [3, 4]], numarray.Bool)
        output = numarray.nd_image.mean(input, labels = labels)
        self.failUnless(output == 1.0)
        

    def test_mean3(self):
        "Test mean of a two-dimensional array"
        labels = numarray.array([1, 2])
        for type in self.types:
            input = numarray.array([[1, 2], [3, 4]], type)
            output = numarray.nd_image.mean(input, labels = labels, index = 2)
            self.failUnless(output == 3.0)


    def test_mean4(self):
        "Test sum of a two-dimensional array"
        labels = numarray.array([[1, 2], [2, 4]], numarray.Int8)
        for type in self.types:
            input = numarray.array([[1, 2], [3, 4]], type)
            output = numarray.nd_image.mean(input, labels = labels,
                                            index = [4, 8, 2])
            self.failUnless(output == [4.0, 0.0, 2.5])

    def test_minimum1(self):
        "Test minimum of a two-dimensional array"
        labels = numarray.array([1, 0], numarray.Bool)
        for type in self.types:
            input = numarray.array([[1, 2], [3, 4]], type)
            output = numarray.nd_image.minimum(input, labels = labels)
            self.failUnless(output == 1.0)

    def test_minimum2(self):
        "Test minimum of a two-dimensional array"
        labels = numarray.array([1, 0], numarray.Bool)
        input = numarray.array([[2, 2], [2, 4]], numarray.Bool)
        output = numarray.nd_image.minimum(input, labels = labels)
        self.failUnless(output == 1.0)

    def test_minimum3(self):
        "Test minimum of a two-dimensional array"
        labels = numarray.array([1, 2])
        for type in self.types:
            input = numarray.array([[1, 2], [3, 4]], type)
            output = numarray.nd_image.minimum(input, labels = labels,
                                               index = 2)
            self.failUnless(output == 2.0)

    def test_minimum4(self):
        "Test minimum of a two-dimensional array"
        labels = numarray.array([[1, 2], [2, 3]])
        for type in self.types:
            input = numarray.array([[1, 2], [3, 4]], type)
            output = numarray.nd_image.minimum(input, labels = labels,
                                               index = [2, 3, 8])
            self.failUnless(output == [2.0, 4.0, 0.0])

    def test_maximum1(self):
        "Test maximum of a two-dimensional array"
        labels = numarray.array([1, 0], numarray.Bool)
        for type in self.types:
            input = numarray.array([[1, 2], [3, 4]], type)
            output = numarray.nd_image.maximum(input, labels = labels)
            self.failUnless(output == 3.0)

    def test_maximum2(self):
        "Test maximum of a two-dimensional array"
        labels = numarray.array([1, 0], numarray.Bool)
        input = numarray.array([[2, 2], [2, 4]], numarray.Bool)
        output = numarray.nd_image.maximum(input, labels = labels)
        self.failUnless(output == 1.0)
        
    def test_maximum3(self):
        "Test maximum of a two-dimensional array"
        labels = numarray.array([1, 2])
        for type in self.types:
            input = numarray.array([[1, 2], [3, 4]], type)
            output = numarray.nd_image.maximum(input, labels = labels,
                                               index = 2)
            self.failUnless(output == 4.0)

    def test_maximum4(self):
        "Test maximum of a two-dimensional array"
        labels = numarray.array([[1, 2], [2, 3]])
        for type in self.types:
            input = numarray.array([[1, 2], [3, 4]], type)
            output = numarray.nd_image.maximum(input, labels = labels,
                                               index = [2, 3, 8])
            self.failUnless(output == [3.0, 4.0, 0.0])

    def test_variance1(self):
        "Test variance"
        for type in self.types:
            input = numarray.array([], type)
            output = numarray.nd_image.variance(input)
            self.failUnless(float(output) == 0.0)

    def test_variance2(self):
        "Test variance"
        for type in self.types:
            input = numarray.array([1], type)
            output = numarray.nd_image.variance(input)
            self.failUnless(float(output) == 0.0)

    def test_variance3(self):
        "Test variance"
        for type in self.types:
            input = numarray.array([1, 3], type)
            output = numarray.nd_image.variance(input)
            self.failUnless(output == 2.0)

    def test_variance4(self):
        "Test variance"
        input = numarray.array([1, 0], numarray.Bool)
        output = numarray.nd_image.variance(input)
        self.failUnless(output == 0.5)

    def test_variance5(self):
        "Test variance"
        labels = [2, 2, 3]
        for type in self.types:
            input = numarray.array([1, 3, 8], type)
            output = numarray.nd_image.variance(input, labels, 2)
            self.failUnless(output == 2.0)

    def test_variance6(self):
        "Test variance"
        labels = [2, 2, 3, 3, 4]
        for type in self.types:
            input = numarray.array([1, 3, 8, 10, 8], type)
            output = numarray.nd_image.variance(input, labels, [2, 3, 4])
            self.failUnless(output == [2.0, 2.0, 0.0])

    def test_standard_deviation1(self):
        "Test standard deviation"
        for type in self.types:
            input = numarray.array([], type)
            output = numarray.nd_image.standard_deviation(input)
            self.failUnless(float(output) == 0.0)

    def test_standard_deviation2(self):
        "Test standard deviation"
        for type in self.types:
            input = numarray.array([1], type)
            output = numarray.nd_image.standard_deviation(input)
            self.failUnless(float(output) == 0.0)

    def test_standard_deviation3(self):
        "Test standard deviation"
        for type in self.types:
            input = numarray.array([1, 3], type)
            output = numarray.nd_image.standard_deviation(input)
            self.failUnless(output == math.sqrt(2.0))

    def test_standard_deviation4(self):
        "Test standard deviation"
        input = numarray.array([1, 0], numarray.Bool)
        output = numarray.nd_image.standard_deviation(input)
        self.failUnless(output == math.sqrt(0.5))

    def test_standard_deviation5(self):
        "Test standard deviation"
        labels = [2, 2, 3]
        for type in self.types:
            input = numarray.array([1, 3, 8], type)
            output = numarray.nd_image.standard_deviation(input, labels, 2)
            self.failUnless(output == math.sqrt(2.0))

    def test_standard_deviation6(self):
        "Test variance"
        labels = [2, 2, 3, 3, 4]
        for type in self.types:
            input = numarray.array([1, 3, 8, 10, 8], type)
            output = numarray.nd_image.standard_deviation(input, labels,
                                                          [2, 3, 4])
            self.failUnless(output == [math.sqrt(2.0), math.sqrt(2.0), 0.0])

    def test_minimum_position1(self):
        "Test minimum position of a two-dimensional array"
        labels = numarray.array([1, 0], numarray.Bool)
        for type in self.types:
            input = numarray.array([[1, 2], [3, 4]], type)
            output = numarray.nd_image.minimum_position(input, labels = labels)
            self.failUnless(output == (0, 0))

    def test_minimum_position2(self):
        "Test minimum position of a two-dimensional array"
        for type in self.types:
            input = numarray.array([[5, 4, 2, 5],
                                    [3, 7, 0, 2],
                                    [1, 5, 1, 1]], type)
            output = numarray.nd_image.minimum_position(input)
            self.failUnless(output == (1, 2))

    def test_minimum_position3(self):
        "Test minimum position of a two-dimensional array"
        input = numarray.array([[5, 4, 2, 5],
                                [3, 7, 0, 2],
                                [1, 5, 1, 1]], numarray.Bool)
        output = numarray.nd_image.minimum_position(input)
        self.failUnless(output == (1, 2))

    def test_minimum_position4(self):
        "Test minimum position of a two-dimensional array"
        input = numarray.array([[5, 4, 2, 5],
                                [3, 7, 1, 2],
                                [1, 5, 1, 1]], numarray.Bool)
        output = numarray.nd_image.minimum_position(input)
        self.failUnless(output == (0, 0))

    def test_minimum_position5(self):
        "Test minimum position of a two-dimensional array"
        labels = [1, 2, 0, 4]
        for type in self.types:
            input = numarray.array([[5, 4, 2, 5],
                                    [3, 7, 0, 2],
                                    [1, 5, 2, 3]], type)
            output = numarray.nd_image.minimum_position(input, labels)
            self.failUnless(output == (2, 0))

    def test_minimum_position6(self):
        "Test minimum position of a two-dimensional array"
        labels = [1, 2, 3, 4]
        for type in self.types:
            input = numarray.array([[5, 4, 2, 5],
                                    [3, 7, 0, 2],
                                    [1, 5, 1, 1]], type)
            output = numarray.nd_image.minimum_position(input, labels, 2)
            self.failUnless(output == (0, 1))


    def test_minimum_position7(self):
        "Test minimum position of a two-dimensional array"
        labels = [1, 2, 3, 4]
        for type in self.types:
            input = numarray.array([[5, 4, 2, 5],
                                    [3, 7, 0, 2],
                                    [1, 5, 1, 1]], type)
            output = numarray.nd_image.minimum_position(input, labels, [2, 3])
            self.failUnless(output == [(0, 1), (1, 2)])


    def test_maximum_position1(self):
        "Test maximum position of a two-dimensional array"
        labels = numarray.array([1, 0], numarray.Bool)
        for type in self.types:
            input = numarray.array([[1, 2], [3, 4]], type)
            output = numarray.nd_image.maximum_position(input, labels = labels)
            self.failUnless(output == (1, 0))

    def test_maximum_position2(self):
        "Test maximum position of a two-dimensional array"
        for type in self.types:
            input = numarray.array([[5, 4, 2, 5],
                                    [3, 7, 8, 2],
                                    [1, 5, 1, 1]], type)
            output = numarray.nd_image.maximum_position(input)
            self.failUnless(output == (1, 2))

    def test_maximum_position3(self):
        "Test maximum position of a two-dimensional array"
        input = numarray.array([[5, 4, 2, 5],
                                [3, 7, 8, 2],
                                [1, 5, 1, 1]], numarray.Bool)
        output = numarray.nd_image.maximum_position(input)
        self.failUnless(output == (0, 0))

    def test_maximum_position4(self):
        "Test maximum position of a two-dimensional array"
        labels = [1, 2, 0, 4]
        for type in self.types:
            input = numarray.array([[5, 4, 2, 5],
                                    [3, 7, 8, 2],
                                    [1, 5, 1, 1]], type)
            output = numarray.nd_image.maximum_position(input, labels)
            self.failUnless(output == (1, 1))

    def test_maximum_position5(self):
        "Test maximum position of a two-dimensional array"
        labels = [1, 2, 0, 4]
        for type in self.types:
            input = numarray.array([[5, 4, 2, 5],
                                    [3, 7, 8, 2],
                                    [1, 5, 1, 1]], type)
            output = numarray.nd_image.maximum_position(input, labels, 1)
            self.failUnless(output == (0, 0))

    def test_maximum_position6(self):
        "Test maximum position of a two-dimensional array"
        labels = [1, 2, 0, 4]
        for type in self.types:
            input = numarray.array([[5, 4, 2, 5],
                                    [3, 7, 8, 2],
                                    [1, 5, 1, 1]], type)
            output = numarray.nd_image.maximum_position(input, labels, [1, 2])
            self.failUnless(output == [(0, 0), (1, 1)])

    def test_center_of_mass1(self):
        "Test center of mass of a two-dimensional array"
        true = [0.0, 0.0]
        for type in self.types:
            input = numarray.array([[1, 0], [0, 0]], type)
            output = numarray.nd_image.center_of_mass(input)
            e = mean_squared_error(true, output)
            self.failUnless(e < eps)

    def test_center_of_mass2(self):
        "Test center of mass of a two-dimensional array"
        true = [1, 0]
        for type in self.types:
            input = numarray.array([[0, 0], [1, 0]], type)
            output = numarray.nd_image.center_of_mass(input)
            e = mean_squared_error(true, output)
            self.failUnless(e < eps)

    def test_center_of_mass3(self):
        "Test center of mass of a two-dimensional array"
        true = [0, 1]
        for type in self.types:
            input = numarray.array([[0, 1], [0, 0]], type)
            output = numarray.nd_image.center_of_mass(input)
            e = mean_squared_error(true, output)
            self.failUnless(e < eps)

    def test_center_of_mass4(self):
        "Test center of mass of a two-dimensional array"
        true = [1, 1]
        for type in self.types:
            input = numarray.array([[0, 0], [0, 1]], type)
            output = numarray.nd_image.center_of_mass(input)
            e = mean_squared_error(true, output)
            self.failUnless(e < eps)

    def test_center_of_mass5(self):
        "Test center of mass of a two-dimensional array"
        true = [0.5, 0.5]
        for type in self.types:
            input = numarray.array([[1, 1], [1, 1]], type)
            output = numarray.nd_image.center_of_mass(input)
            e = mean_squared_error(true, output)
            self.failUnless(e < eps)

    def test_center_of_mass6(self):
        "Test center of mass of a two-dimensional array"
        true = [0.5, 0.5]
        input = numarray.array([[1, 2], [3, 1]], numarray.Bool)
        output = numarray.nd_image.center_of_mass(input)
        e = mean_squared_error(true, output)
        self.failUnless(e < eps)

    def test_center_of_mass7(self):
        "Test center of mass of a two-dimensional array"
        labels = [1, 0]
        true = [0.5, 0.0]
        input = numarray.array([[1, 2], [3, 1]], numarray.Bool)
        output = numarray.nd_image.center_of_mass(input, labels)
        e = mean_squared_error(true, output)
        self.failUnless(e < eps)

    def test_center_of_mass8(self):
        "Test center of mass of a two-dimensional array"
        labels = [1, 2]
        true = [0.5, 1.0]
        input = numarray.array([[5, 2], [3, 1]], numarray.Bool)
        output = numarray.nd_image.center_of_mass(input, labels, 2)
        e = mean_squared_error(true, output)
        self.failUnless(e < eps)


    def test_center_of_mass9(self):
        "Test center of mass of a two-dimensional array"
        labels = [1, 2]
        true = [(0.5, 0.0), (0.5, 1.0)]
        input = numarray.array([[1, 2], [1, 1]], numarray.Bool)
        output = numarray.nd_image.center_of_mass(input, labels, [1, 2])
        e = mean_squared_error(true, output)
        self.failUnless(e < eps)


    def test_fourier_gaussian_real1(self):
        "test gaussian fourier filter for real transforms"
        for shape in [(32, 16), (31, 15)]:
            for type in [numarray.Float32, numarray.Float64]:
                a = numarray.zeros(shape, type)
                a[0, 0] = 1.0
                a = numarray.fft.real_fft(a, shape[0], 0)
                a = numarray.fft.fft(a, shape[1], 1)
                a = numarray.nd_image.fourier_gaussian(a, [5.0, 2.5],
                                                       shape[0], 0)
                a = numarray.fft.inverse_fft(a, shape[1], 1)
                a = numarray.fft.inverse_real_fft(a, shape[0], 0)
                error = mean_squared_error(numarray.nd_image.sum(a), 1.0)
                self.failUnless(error < eps)
                

    def test_fourier_gaussian_complex1(self):
        "test gaussian fourier filter for complex transforms"
        for shape in [(32, 16), (31, 15)]:
            for type in [numarray.Complex32, numarray.Complex64]:
                a = numarray.zeros(shape, type)
                a[0, 0] = 1.0
                a = numarray.fft.fft(a, shape[0], 0)
                a = numarray.fft.fft(a, shape[1], 1)
                a = numarray.nd_image.fourier_gaussian(a, [5.0, 2.5], -1, 0)
                a = numarray.fft.inverse_fft(a, shape[1], 1)
                a = numarray.fft.inverse_fft(a, shape[0], 0)
                error = mean_squared_error(numarray.nd_image.sum(a.real), 1.0)
                self.failUnless(error < eps)
        
        
    def test_fourier_boxcar_real1(self):
        "test boxcar fourier filter for real transforms"
        for shape in [(32, 16), (31, 15)]:
            for type in [numarray.Float32, numarray.Float64]:
                a = numarray.zeros(shape, type)
                a[0, 0] = 1.0
                a = numarray.fft.real_fft(a, shape[0], 0)
                a = numarray.fft.fft(a, shape[1], 1)
                a = numarray.nd_image.fourier_boxcar(a, [5.0, 2.5],
                                                     shape[0], 0)
                a = numarray.fft.inverse_fft(a, shape[1], 1)
                a = numarray.fft.inverse_real_fft(a, shape[0], 0)
                error = mean_squared_error(numarray.nd_image.sum(a), 1.0)
                self.failUnless(error < eps)
                

    def test_fourier_boxcar_complex1(self):
        "test boxcar fourier filter for complex transforms"
        for shape in [(32, 16), (31, 15)]:
            for type in [numarray.Complex32, numarray.Complex64]:
                a = numarray.zeros(shape, type)
                a[0, 0] = 1.0
                a = numarray.fft.fft(a, shape[0], 0)
                a = numarray.fft.fft(a, shape[1], 1)
                a = numarray.nd_image.fourier_boxcar(a, [5.0, 2.5], -1, 0)
                a = numarray.fft.inverse_fft(a, shape[1], 1)
                a = numarray.fft.inverse_fft(a, shape[0], 0)
                error = mean_squared_error(numarray.nd_image.sum(a.real), 1.0)
                self.failUnless(error < eps)
        
        
    def test_fourier_ellipsoid_real1(self):
        "test ellipsoid fourier filter for real transforms"
        for shape in [(32, 16), (31, 15)]:
            for type in [numarray.Float32, numarray.Float64]:
                a = numarray.zeros(shape, type)
                a[0, 0] = 1.0
                a = numarray.fft.real_fft(a, shape[0], 0)
                a = numarray.fft.fft(a, shape[1], 1)
                a = numarray.nd_image.fourier_ellipsoid(a, [5.0, 2.5],
                                                        shape[0], 0)
                a = numarray.fft.inverse_fft(a, shape[1], 1)
                a = numarray.fft.inverse_real_fft(a, shape[0], 0)
                error = mean_squared_error(numarray.nd_image.sum(a), 1.0)
                self.failUnless(error < eps)
                

    def test_fourier_ellipsoid_complex1(self):
        "test ellipsoid fourier filter for complex transforms"
        for shape in [(32, 16), (31, 15)]:
            for type in [numarray.Complex32, numarray.Complex64]:
                a = numarray.zeros(shape, type)
                a[0, 0] = 1.0
                a = numarray.fft.fft(a, shape[0], 0)
                a = numarray.fft.fft(a, shape[1], 1)
                a = numarray.nd_image.fourier_ellipsoid(a, [5.0, 2.5], -1, 0)
                a = numarray.fft.inverse_fft(a, shape[1], 1)
                a = numarray.fft.inverse_fft(a, shape[0], 0)
                error = mean_squared_error(numarray.nd_image.sum(a.real), 1.0)
                self.failUnless(error < eps)
        
        
    def test_fourier_shift_real1(self):
        "test shift filter for real transforms"
        for shape in [(32, 16), (31, 15)]:
            for type in [numarray.Float32, numarray.Float64]:
                true = numarray.arange(shape[0] * shape[1],
                                       shape = shape, type = type)
                a = numarray.fft.real_fft(true, shape[0], 0)
                a = numarray.fft.fft(a, shape[1], 1)
                a = numarray.nd_image.fourier_shift(a, [1, 1], shape[0], 0)
                a = numarray.fft.inverse_fft(a, shape[1], 1)
                a = numarray.fft.inverse_real_fft(a, shape[0], 0)
                error1 = mean_squared_error(a[1:, 1:], true[:-1, :-1])
                error2 = mean_squared_error(a.imag, numarray.zeros(shape))
                self.failUnless(error1 < 1e-10 and error2 < 1e-10)


    def test_fourier_shift_complex1(self):
        "test shift filter for complex transforms"
        for shape in [(32, 16), (31, 15)]:
            for type in [numarray.Complex32, numarray.Complex64]:
                true = numarray.arange(shape[0] * shape[1],
                                       shape = shape, type = type)
                a = numarray.fft.fft(true, shape[0], 0)
                a = numarray.fft.fft(a, shape[1], 1)
                a = numarray.nd_image.fourier_shift(a, [1, 1], -1, 0)
                a = numarray.fft.inverse_fft(a, shape[1], 1)
                a = numarray.fft.inverse_fft(a, shape[0], 0)
                error1 = mean_squared_error(a.real[1:, 1:], true[:-1, :-1])
                error2 = mean_squared_error(a.imag, numarray.zeros(shape))
                self.failUnless(error1 < 1e-10 and error2 < 1e-10)


class NDImageTestResult(unittest.TestResult):
    separator1 = '=' * 70 + '\n'
    separator2 = '-' * 70 + '\n'

    def __init__(self, stream, verbose):
        unittest.TestResult.__init__(self)
        self.stream = stream
        self.verbose = verbose

    def getDescription(self, test):
        return test.shortDescription() or str(test)

    def startTest(self, test):
        unittest.TestResult.startTest(self, test)
        if self.verbose:
            self.stream.write(self.getDescription(test))
            self.stream.write(" ... ")

    def addSuccess(self, test):
        unittest.TestResult.addSuccess(self, test)
        if self.verbose:
            self.stream.write("ok\n")

    def addError(self, test, err):
        unittest.TestResult.addError(self, test, err)
        if self.verbose:
            self.stream.write("ERROR\n")

    def addFailure(self, test, err):
        unittest.TestResult.addFailure(self, test, err)
        if self.verbose:
            self.stream.write("FAIL\n")

    def printErrors(self):
        self.printErrorList('ERROR', self.errors)
        self.printErrorList('FAIL', self.failures)

    def printErrorList(self, flavour, errors):
        for test, err in errors:
            self.stream.write(self.separator1)
            description = self.getDescription(test)
            self.stream.write("%s: %s\n" % (flavour, description))
            self.stream.write(self.separator2)
            self.stream.write(err)


def test():
    options, args = getopt.getopt(sys.argv[1:], 'v')
    verbose = 0
    for option, value in options:
        if option == '-v':
            verbose = 1
            break
    suite = unittest.TestSuite()
    suite.addTest(unittest.makeSuite(NDImageTest))
    result = NDImageTestResult(sys.stdout, verbose)
    suite(result)
    result.printErrors()
    return len(result.failures), result.testsRun


if __name__ == '__main__':
    unittest.main()
