/* Copyright (C) 2003 Peter J. Verveer
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met: 
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.      
 */

#include "ni_support.h"
#include <stdlib.h>
#include <math.h>
#include <assert.h>

#include "numconfig.h"

#if !defined(M_PI)
#define M_PI 3.14159265358979323846
#endif

int NI_FourierGaussian(PyArrayObject *input, double* isigmas, int n,
		       int axis, PyArrayObject** output, PyObject* output_in)
{
  NI_Iterator ii, io;
  char *pi, *po;
  double *sigmas = NULL, **params = NULL;
  int kk, hh, oshape[NI_MAXDIM], size, irank, itype, idims[NI_MAXDIM];
  NumarrayType otype;

  assert(input != NULL);
  assert(output != NULL);
  assert(isigmas != NULL);

  irank = NI_GetArrayRank(input);

  /* support negative axis specification: */
  if (axis < 0)
    axis += irank;

  /* check axis specification: */
  if (irank > 0 && (axis < 0 || axis >= irank)) {
    PyErr_SetString(PyExc_RuntimeError, "invalid axis specified");
    goto exit;
  }

  /* output shape is equal to the input shape: */
  NI_GetArrayDimensions(input, idims);
  for(kk = 0; kk < irank; kk++)
    oshape[kk] = idims[kk];
  
  /* precalculate the sigmas: */
  sigmas = (double*)malloc(irank * sizeof(double));
  if (!sigmas) {
    PyErr_NoMemory();
    goto exit;
  }
  for(kk = 0; kk < irank; kk++) {
    /* along the direction of the real transform we must use the given
       length of that dimensons, unless a complex transform is assumed
       (n < 0): */
    int shape = kk == axis ? (n < 0 ? oshape[kk] : n) : oshape[kk];
    sigmas[kk] = isigmas[kk] * M_PI / (double)shape;
    sigmas[kk] = -2.0 * sigmas[kk] * sigmas[kk];
  }

  /* allocate memory for tables: */
  params = (double**) malloc(irank * sizeof(double*));
  if (!params) {
    PyErr_NoMemory();
    goto exit;
  }
  for(kk = 0; kk < irank; kk++) 
    params[kk] = NULL;
  for(kk = 0; kk < irank; kk++) {
    if (oshape[kk] > 0) {
      params[kk] = (double*)malloc(oshape[kk] * sizeof(double));
      if (!params[kk]) {
	PyErr_NoMemory();
	goto exit;
      }
    }
  }
  
  /* calculate the tables of exponentials: */
  for (hh = 0; hh < irank; hh++) {
    if (oshape[hh] > 1) {
      if (hh == axis && n >= 0) {
	for(kk = 0; kk < oshape[hh]; kk++) {
	  double tmp = sigmas[hh] * kk * kk;
	  params[hh][kk] = fabs(tmp) > 50.0 ? 0.0 : exp(tmp);
	}
      } else {
	int jj = 0;
	for(kk = 0; kk < (oshape[hh] + 1) / 2; kk++) {
	  double tmp = sigmas[hh] * kk * kk;
	  params[hh][jj++] = fabs(tmp) > 50.0 ? 0.0 : exp(tmp);
	}
	for(kk = -(oshape[hh] / 2); kk < 0; kk++) {
	  double tmp = sigmas[hh] * kk * kk;
	  params[hh][jj++] = fabs(tmp) > 50.0 ? 0.0 : exp(tmp);
	}
      }
    } else if (oshape[hh] > 0) {
      params[hh][0] = 1.0;
    }
  }
  
  itype = NI_GetArrayType(input);
  switch(itype) {
  case tBool:
  case tUInt8:
  case tUInt16:
  case tUInt32:
#if HAS_UINT64
  case tUInt64:
#endif
  case tInt8:
  case tInt16:
  case tInt32:
  case tInt64:
    otype = tFloat64;
    break;
  case tFloat32:
  case tFloat64:
  case tComplex32:
  case tComplex64:
    otype = itype;
    break;
  default:
    PyErr_SetString(PyExc_RuntimeError, "array type not supported");
    goto exit;
  }

  /* allocate the output array: */
  if (!NI_OutputArray(otype, irank, oshape, output_in, output))
    goto exit;

  /* initialize input element iterator: */
  if (!NI_InitPointIterator(input, &ii))
    goto exit;

  /* initialize output element iterator: */
  if (!NI_InitPointIterator(*output, &io))
    goto exit;

  pi = NI_GetArrayData(input);
  po = NI_GetArrayData(*output);
  size = NI_Elements(input);

  /* iterator over the elements: */
  for(hh = 0; hh < size; hh++) {
    double tmp = params[0][ii.coordinates[0]];
    for(kk = 1; kk < irank; kk++)
      tmp *= params[kk][ii.coordinates[kk]];
    switch (itype) {
    case tBool:
      *(Float64*)po = *(Bool*)pi * tmp;
      break;
    case tUInt8:
      *(Float64*)po = *(UInt8*)pi * tmp;
      break;
    case tUInt16:
      *(Float64*)po = *(UInt16*)pi * tmp;
      break;
    case tUInt32:
      *(Float64*)po = *(UInt32*)pi * tmp;
      break;
#if HAS_UINT64
    case tUInt64:
      *(Float64*)po = *(UInt64*)pi * tmp;
      break;
#endif
    case tInt8:
      *(Float64*)po = *(Int8*)pi * tmp;
      break;
    case tInt16:
      *(Float64*)po = *(Int16*)pi * tmp;
      break;
    case tInt32:
      *(Float64*)po = *(Int32*)pi * tmp;
      break;
    case tInt64:
      *(Float64*)po = *(Int64*)pi * tmp;
      break;
    case tFloat32:
      *(Float32*)po = *(Float32*)pi * tmp;
      break;
    case tFloat64:
      *(Float64*)po = *(Float64*)pi * tmp;
      break;
    case tComplex32:
      (*(Complex32*)po).r = (*(Complex32*)pi).r * tmp;
      (*(Complex32*)po).i = (*(Complex32*)pi).i * tmp;
      break;
    case tComplex64:
      (*(Complex64*)po).r = (*(Complex64*)pi).r * tmp;
      (*(Complex64*)po).i = (*(Complex64*)pi).i * tmp;
      break;
    default:
      PyErr_SetString(PyExc_RuntimeError, "data type not supported");
      goto exit;
    }
    NI_ITERATOR_NEXT2(ii, io, pi, po);
  }
  
 exit:
  if (sigmas)
    free(sigmas);
  if (params) {
    for(kk = 0; kk < irank; kk++)
      if (params[kk])
	free(params[kk]);
    free(params);
  }
  return PyErr_Occurred() ? 0 : 1;
}


int NI_FourierBoxcar(PyArrayObject *input, double* sizes, int n,
		     int axis, PyArrayObject** output, PyObject* output_in)
{
  NI_Iterator ii, io;
  char *pi, *po;
  double **params = NULL;
  int kk, hh, oshape[NI_MAXDIM], size, irank, itype, idims[NI_MAXDIM];
  NumarrayType otype;

  assert(input != NULL);
  assert(output != NULL);
  assert(sizes != NULL);

  irank = NI_GetArrayRank(input);

  /* support negative axis specification: */
  if (axis < 0)
    axis += irank;

  /* check axis specification: */
  if (irank > 0 && (axis < 0 || axis >= irank)) {
    PyErr_SetString(PyExc_RuntimeError, "invalid axis specified");
    goto exit;
  }

  /* output shape is equal to the input shape: */
  NI_GetArrayDimensions(input, idims);
  for(kk = 0; kk < irank; kk++)
    oshape[kk] = idims[kk];
  
  /* allocate memory for tables: */
  params = (double**) malloc(irank * sizeof(double*));
  if (!params) {
    PyErr_NoMemory();
    goto exit;
  }
  for(kk = 0; kk < irank; kk++) 
    params[kk] = NULL;
  for(kk = 0; kk < irank; kk++) {
    if (oshape[kk] > 0) {
      params[kk] = (double*)malloc(oshape[kk] * sizeof(double));
      if (!params[kk]) {
	PyErr_NoMemory();
	goto exit;
      }
    }
  }
  
  /* calculate the tables of parameters: */
  for (hh = 0; hh < irank; hh++) {
    if (oshape[hh] > 1) {
      params[hh][0] = 1.0;
      if (hh == axis && n >= 0) {
	double tmp = M_PI * sizes[hh] / n;
	for(kk = 1; kk < oshape[hh]; kk++)
	  params[hh][kk] = tmp > 0.0 ? sin(tmp * kk) / (tmp * kk) : 0.0;
      } else {
	double tmp = M_PI * sizes[hh] / oshape[hh];
	int jj = 1;
	for(kk = 1; kk < (oshape[hh] + 1) / 2; kk++)
	  params[hh][jj++] = tmp > 0.0 ? sin(tmp * kk) / (tmp * kk) : 0.0;
	for(kk = -(oshape[hh] / 2); kk < 0; kk++)
	  params[hh][jj++] = tmp > 0.0 ? sin(tmp * kk) / (tmp * kk) : 0.0;
      }
    } else if (oshape[hh] > 0) {
      params[hh][0] = 1.0;
    }
  }

  itype = NI_GetArrayType(input);
  switch(itype) {
  case tBool:
  case tUInt8:
  case tUInt16:
  case tUInt32:
#if HAS_UINT64
  case tUInt64:
#endif
  case tInt8:
  case tInt16:
  case tInt32:
  case tInt64:
    otype = tFloat64;
    break;
  case tFloat32:
  case tFloat64:
  case tComplex32:
  case tComplex64:
    otype = itype;
    break;
  default:
    PyErr_SetString(PyExc_RuntimeError, "array type not supported");
    goto exit;
  }

  /* allocate the output array: */
  if (!NI_OutputArray(otype, irank, oshape, output_in, output))
    goto exit;

  /* initialize input element iterator: */
  if (!NI_InitPointIterator(input, &ii))
    goto exit;

  /* initialize output element iterator: */
  if (!NI_InitPointIterator(*output, &io))
    goto exit;

  pi = NI_GetArrayData(input);
  po = NI_GetArrayData(*output);
  size = NI_Elements(input);

  /* iterator over the elements: */
  for(hh = 0; hh < size; hh++) {
    double tmp = params[0][ii.coordinates[0]];
    for(kk = 1; kk < irank; kk++)
      tmp *= params[kk][ii.coordinates[kk]];
    switch (itype) {
    case tBool:
      *(Float64*)po = *(Bool*)pi * tmp;
      break;
    case tUInt8:
      *(Float64*)po = *(UInt8*)pi * tmp;
      break;
    case tUInt16:
      *(Float64*)po = *(UInt16*)pi * tmp;
      break;
    case tUInt32:
      *(Float64*)po = *(UInt32*)pi * tmp;
      break;
#if HAS_UINT64
    case tUInt64:
      *(Float64*)po = *(UInt64*)pi * tmp;
      break;
#endif
    case tInt8:
      *(Float64*)po = *(Int8*)pi * tmp;
      break;
    case tInt16:
      *(Float64*)po = *(Int16*)pi * tmp;
      break;
    case tInt32:
      *(Float64*)po = *(Int32*)pi * tmp;
      break;
    case tInt64:
      *(Float64*)po = *(Int64*)pi * tmp;
      break;
    case tFloat32:
      *(Float32*)po = *(Float32*)pi * tmp;
      break;
    case tFloat64:
      *(Float64*)po = *(Float64*)pi * tmp;
      break;
    case tComplex32:
      (*(Complex32*)po).r = (*(Complex32*)pi).r * tmp;
      (*(Complex32*)po).i = (*(Complex32*)pi).i * tmp;
      break;
    case tComplex64:
      (*(Complex64*)po).r = (*(Complex64*)pi).r * tmp;
      (*(Complex64*)po).i = (*(Complex64*)pi).i * tmp;
      break;
    default:
      PyErr_SetString(PyExc_RuntimeError, "data type not supported");
      goto exit;
    }
    NI_ITERATOR_NEXT2(ii, io, pi, po);
  }
  
 exit:
  if (params) {
    for(kk = 0; kk < irank; kk++)
      if (params[kk])
	free(params[kk]);
    free(params);
  }
  return PyErr_Occurred() ? 0 : 1;
}

static double polevl(double x, const double coef[], int N)
{
  double ans;
  const double *p = coef;
  int i = N;

  ans = *p++;
  do
    ans = ans * x + *p++;
  while(--i);

  return ans ;
}

double p1evl(double x, const double coef[], int N)
{
  double ans;
  const double *p = coef;
  int i = N - 1;

  ans = x + *p++;
  do
    ans = ans * x + *p++;
  while(--i);

  return ans;
}

#define THPIO4 2.35619449019234492885
#define SQ2OPI .79788456080286535588
#define Z1 1.46819706421238932572E1
#define Z2 4.92184563216946036703E1

static double _bessel_j1(double x)
{
  double w, z, p, q, xn;
  const double RP[4] = {
    -8.99971225705559398224E8,
    4.52228297998194034323E11,
    -7.27494245221818276015E13,
    3.68295732863852883286E15,
  };
  const double RQ[8] = {
    6.20836478118054335476E2,
    2.56987256757748830383E5,
    8.35146791431949253037E7,
    2.21511595479792499675E10,
    4.74914122079991414898E12,
    7.84369607876235854894E14,
    8.95222336184627338078E16,
    5.32278620332680085395E18,
  };
  const double PP[7] = {
    7.62125616208173112003E-4,
    7.31397056940917570436E-2,
    1.12719608129684925192E0,
    5.11207951146807644818E0,
    8.42404590141772420927E0,
    5.21451598682361504063E0,
    1.00000000000000000254E0,
  };
  const double PQ[7] = {
    5.71323128072548699714E-4,
    6.88455908754495404082E-2,
    1.10514232634061696926E0,
    5.07386386128601488557E0,
    8.39985554327604159757E0,
    5.20982848682361821619E0,
    9.99999999999999997461E-1,
  };
  const double QP[8] = {
    5.10862594750176621635E-2,
    4.98213872951233449420E0,
    7.58238284132545283818E1,
    3.66779609360150777800E2,
    7.10856304998926107277E2,
    5.97489612400613639965E2,
    2.11688757100572135698E2,
    2.52070205858023719784E1,
  };
  const double QQ[7] = {
    7.42373277035675149943E1,
    1.05644886038262816351E3,
    4.98641058337653607651E3,
    9.56231892404756170795E3,
    7.99704160447350683650E3,
    2.82619278517639096600E3,
    3.36093607810698293419E2,
  };

  w = x;
  if (x < 0)
    w = -x;
  
  if (w <= 5.0) {
    z = x * x;	
    w = polevl(z, RP, 3) / p1evl(z, RQ, 8);
    w = w * x * (z - Z1) * (z - Z2);
    return w ;
  }
  
  w = 5.0 / x;
  z = w * w;
  p = polevl(z, PP, 6) / polevl(z, PQ, 6);
  q = polevl(z, QP, 7) / p1evl(z, QQ, 7);
  xn = x - THPIO4;
  p = p * cos(xn) - w * q * sin(xn);
  return p * SQ2OPI / sqrt(x);
}


int NI_FourierEllipsoid(PyArrayObject *input, double* sizes, int n,
			int axis, PyArrayObject** output, PyObject* output_in)
{
  NI_Iterator ii, io;
  char *pi, *po;
  double **params = NULL;
  int kk, hh, oshape[NI_MAXDIM], size, irank, itype, idims[NI_MAXDIM];
  NumarrayType otype;

  assert(input != NULL);
  assert(output != NULL);
  assert(sizes != NULL);

  irank = NI_GetArrayRank(input);

  /* check dimensionality */
  if (irank > 3) {
    PyErr_SetString(PyExc_RuntimeError, "rank > 3 not supported");
    goto exit;
  }

  /* support negative axis specification: */
  if (axis < 0)
    axis += irank;

  /* check axis specification: */
  if (irank > 0 && (axis < 0 || axis >= irank)) {
    PyErr_SetString(PyExc_RuntimeError, "invalid axis specified");
    goto exit;
  }

  /* output shape is equal to the input shape: */
  NI_GetArrayDimensions(input, idims);
  for(kk = 0; kk < irank; kk++)
    oshape[kk] = idims[kk];
  
  /* allocate memory for tables: */
  params = (double**) malloc(irank * sizeof(double*));
  if (!params) {
    PyErr_NoMemory();
    goto exit;
  }
  for(kk = 0; kk < irank; kk++) 
    params[kk] = NULL;
  for(kk = 0; kk < irank; kk++) {
    if (oshape[kk] > 0) {
      params[kk] = (double*)malloc(oshape[kk] * sizeof(double));
      if (!params[kk]) {
	PyErr_NoMemory();
	goto exit;
      }
    }
  }
  
  /* calculate the tables of parameters: */
  for (hh = 0; hh < irank; hh++) {
    if (oshape[hh] > 1) {
      params[hh][0] = 1.0;
      if (hh == axis && n >= 0) {
	double tmp = M_PI * sizes[hh] / n;
	for(kk = 0; kk < oshape[hh]; kk++)
	  params[hh][kk] = (double)kk * tmp;
      } else {
	double tmp = M_PI * sizes[hh] / oshape[hh];
	int jj = 0;
	for(kk = 0; kk < (oshape[hh] + 1) / 2; kk++)
	  params[hh][jj++] = (double)kk * tmp;
	for(kk = -(oshape[hh] / 2); kk < 0; kk++)
	  params[hh][jj++] = (double)kk * tmp;
      }
    } else if (oshape[hh] > 0) {
      params[hh][0] = 1.0;
    }
  }
  if (irank > 1)
    for(hh = 0; hh < irank; hh++)
      for(kk = 0; kk < oshape[hh]; kk++)
	params[hh][kk] = params[hh][kk] * params[hh][kk];

  itype = NI_GetArrayType(input);
  switch(itype) {
  case tBool:
  case tUInt8:
  case tUInt16:
  case tUInt32:
#if HAS_UINT64
  case tUInt64:
#endif
  case tInt8:
  case tInt16:
  case tInt32:
  case tInt64:
    otype = tFloat64;
    break;
  case tFloat32:
  case tFloat64:
  case tComplex32:
  case tComplex64:
    otype = itype;
    break;
  default:
    PyErr_SetString(PyExc_RuntimeError, "array type not supported");
    goto exit;
  }

  /* allocate the output array: */
  if (!NI_OutputArray(otype, irank, oshape, output_in, output))
    goto exit;

  /* initialize input element iterator: */
  if (!NI_InitPointIterator(input, &ii))
    goto exit;

  /* initialize output element iterator: */
  if (!NI_InitPointIterator(*output, &io))
    goto exit;

  pi = NI_GetArrayData(input);
  po = NI_GetArrayData(*output);
  size = NI_Elements(input);

  /* iterator over the elements: */
  for(hh = 0; hh < size; hh++) {
    double tmp = 1.0;
    switch (irank) {
    case 1:
      tmp = params[0][ii.coordinates[0]];
      tmp = tmp > 0.0 ? sin(tmp) / (tmp) : 1.0;
      break;
    case 2:
      tmp = 0.0;
      for(kk = 0; kk < 2; kk++)
	tmp += params[kk][ii.coordinates[kk]];
      tmp = sqrt(tmp);
      tmp = tmp > 0.0 ? 2.0 * _bessel_j1(tmp) / tmp : 1.0;
      break;
    case 3:
      {
	double r = 0.0;
	for(kk = 0; kk < 3; kk++)
	  r += params[kk][ii.coordinates[kk]];
	r = sqrt(r);
	if (r > 0.0) {
	  tmp = 3.0 * (sin(r) - r * cos(r));
	  tmp /= r * r * r;
	} else {
	  tmp = 1.0;
	}
      }
      break;
    }
    switch (itype) {
    case tBool:
      *(Float64*)po = *(Bool*)pi * tmp;
      break;
    case tUInt8:
      *(Float64*)po = *(UInt8*)pi * tmp;
      break;
    case tUInt16:
      *(Float64*)po = *(UInt16*)pi * tmp;
      break;
    case tUInt32:
      *(Float64*)po = *(UInt32*)pi * tmp;
      break;
#if HAS_UINT64
    case tUInt64:
      *(Float64*)po = *(UInt64*)pi * tmp;
      break;
#endif
    case tInt8:
      *(Float64*)po = *(Int8*)pi * tmp;
      break;
    case tInt16:
      *(Float64*)po = *(Int16*)pi * tmp;
      break;
    case tInt32:
      *(Float64*)po = *(Int32*)pi * tmp;
      break;
    case tInt64:
      *(Float64*)po = *(Int64*)pi * tmp;
      break;
    case tFloat32:
      *(Float32*)po = *(Float32*)pi * tmp;
      break;
    case tFloat64:
      *(Float64*)po = *(Float64*)pi * tmp;
      break;
    case tComplex32:
      (*(Complex32*)po).r = (*(Complex32*)pi).r * tmp;
      (*(Complex32*)po).i = (*(Complex32*)pi).i * tmp;
      break;
    case tComplex64:
      (*(Complex64*)po).r = (*(Complex64*)pi).r * tmp;
      (*(Complex64*)po).i = (*(Complex64*)pi).i * tmp;
      break;
    default:
      PyErr_SetString(PyExc_RuntimeError, "data type not supported");
      goto exit;
    }
    NI_ITERATOR_NEXT2(ii, io, pi, po);
  }
  
 exit:
  if (params) {
    for(kk = 0; kk < irank; kk++)
      if (params[kk])
	free(params[kk]);
    free(params);
  }
  return PyErr_Occurred() ? 0 : 1;
}


int NI_FourierShift(PyArrayObject *input, double* ishift, int n, int axis, 
		    PyArrayObject** output, PyObject* output_in)
{
  NI_Iterator ii, io;
  char *pi, *po;
  double *shifts = NULL, **params = NULL;
  int kk, hh, oshape[NI_MAXDIM], size, irank, itype, idims[NI_MAXDIM];
  NumarrayType otype;

  assert(input != NULL);
  assert(output != NULL);
  assert(ishift != NULL);

  irank = NI_GetArrayRank(input);

  /* support negative axis specification: */
  if (axis < 0)
    axis += irank;

  /* check axis specification: */
  if (irank > 0 && (axis < 0 || axis >= irank)) {
    PyErr_SetString(PyExc_RuntimeError, "invalid axis specified");
    goto exit;
  }

  /* output shape is equal to the input shape: */
  NI_GetArrayDimensions(input, idims);
  for(kk = 0; kk < irank; kk++)
    oshape[kk] = idims[kk];
  
  /* precalculate the shifts: */
  shifts = (double*)malloc(irank * sizeof(double));
  if (!shifts) {
    PyErr_NoMemory();
    goto exit;
  }
  for(kk = 0; kk < irank; kk++) {
    /* along the direction of the real transform we must use the given
       length of that dimensons, unless a complex transform is assumed
       (n < 0): */
    int shape = kk == axis ? (n < 0 ? oshape[kk] : n) : oshape[kk];
    shifts[kk] = -2.0 * M_PI * ishift[kk] / (double)shape;
  }

  /* allocate memory for tables: */
  params = (double**) malloc(irank * sizeof(double*));
  if (!params) {
    PyErr_NoMemory();
    goto exit;
  }
  for(kk = 0; kk < irank; kk++) 
    params[kk] = NULL;
  for(kk = 0; kk < irank; kk++) {
    if (oshape[kk] > 0) {
      params[kk] = (double*)malloc(oshape[kk] * sizeof(double));
      if (!params[kk]) {
	PyErr_NoMemory();
	goto exit;
      }
    }
  }
  
  /* calculate the tables of parameters: */
  for (hh = 0; hh < irank; hh++) {
    if (hh == axis && n >= 0) {
      for(kk = 0; kk < oshape[hh]; kk++)
	params[hh][kk] = shifts[hh] * kk;
    } else {
      int jj = 0;
      for(kk = 0; kk < (oshape[hh] + 1) / 2; kk++) {
	params[hh][jj++] = shifts[hh] * kk;
      }
      for(kk = -(oshape[hh] / 2); kk < 0; kk++) {
	params[hh][jj++] = shifts[hh] * kk;
      }
    }
  }

  itype = NI_GetArrayType(input);
  otype = itype;
  if (otype == tFloat32)
    otype = tComplex32;
  if (otype == tFloat64)
    otype = tComplex64;
  switch(itype) {
  case tBool:
  case tUInt8:
  case tUInt16:
  case tUInt32:
#if HAS_UINT64
  case tUInt64:
#endif
  case tInt8:
  case tInt16:
  case tInt32:
  case tInt64:
    otype = tComplex64;
    break;
  case tFloat32:
    otype = tComplex32;
    break;
  case tFloat64:
    otype = tComplex64;
    break;
  case tComplex32:
    otype = tComplex32;
    break;
  case tComplex64:
    otype = tComplex64;
    break;
  default:
    PyErr_SetString(PyExc_RuntimeError, "array type not supported");
    goto exit;
  }

  /* allocate the output array: */
  if (!NI_OutputArray(otype, irank, oshape, output_in, output))
    goto exit;

  /* initialize input element iterator: */
  if (!NI_InitPointIterator(input, &ii))
    goto exit;

  /* initialize output element iterator: */
  if (!NI_InitPointIterator(*output, &io))
    goto exit;

  pi = NI_GetArrayData(input);
  po = NI_GetArrayData(*output);
  size = NI_Elements(input);

  /* iterator over the elements: */
  for(hh = 0; hh < size; hh++) {
    double tmp = 0.0, sint, cost;
    for(kk = 0; kk < irank; kk++)
      tmp += params[kk][ii.coordinates[kk]];
    sint = sin(tmp);
    cost = cos(tmp);
    switch (itype) {
    case tBool:
      tmp = *(Bool*)pi;
      (*(Complex64*)po).r = tmp * cost;
      (*(Complex64*)po).i = tmp * sint;
      break;
    case tUInt8:
      tmp = *(UInt8*)pi;
      (*(Complex64*)po).r = tmp * cost;
      (*(Complex64*)po).i = tmp * sint;
      break;
    case tUInt16:
      tmp = *(UInt16*)pi;
      (*(Complex64*)po).r = tmp * cost;
      (*(Complex64*)po).i = tmp * sint;
      break;
    case tUInt32:
      tmp = *(UInt32*)pi;
      (*(Complex64*)po).r = tmp * cost;
      (*(Complex64*)po).i = tmp * sint;
      break;
#if HAS_UINT64
    case tUInt64:
      tmp = *(UInt64*)pi;
      (*(Complex64*)po).r = tmp * cost;
      (*(Complex64*)po).i = tmp * sint;
      break;
#endif
    case tInt8:
      tmp = *(Int8*)pi;
      (*(Complex64*)po).r = tmp * cost;
      (*(Complex64*)po).i = tmp * sint;
      break;
    case tInt16:
      tmp = *(Int16*)pi;
      (*(Complex64*)po).r = tmp * cost;
      (*(Complex64*)po).i = tmp * sint;
      break;
    case tInt32:
      tmp = *(Int32*)pi;
      (*(Complex64*)po).r = tmp * cost;
      (*(Complex64*)po).i = tmp * sint;
      break;
    case tInt64:
      tmp = *(Int64*)pi;
      (*(Complex64*)po).r = tmp * cost;
      (*(Complex64*)po).i = tmp * sint;
      break;
    case tFloat32:
      tmp = *(Float32*)pi;
      (*(Complex32*)po).r = tmp * cost;
      (*(Complex32*)po).i = tmp * sint;
      break;
    case tFloat64:
      tmp = *(Float64*)pi;
      (*(Complex64*)po).r = tmp * cost;
      (*(Complex64*)po).i = tmp * sint;
      break;
    case tComplex32:
      {
	double r = (*(Complex32*)pi).r * cost - (*(Complex32*)pi).i * sint;
	double i = (*(Complex32*)pi).r * sint + (*(Complex32*)pi).i * cost;
	(*(Complex32*)po).r = r;
	(*(Complex32*)po).i = i;
      }
      break;
    case tComplex64:
      {
	double r = (*(Complex64*)pi).r * cost - (*(Complex64*)pi).i * sint;
	double i = (*(Complex64*)pi).r * sint + (*(Complex64*)pi).i * cost;
	(*(Complex64*)po).r = r;
	(*(Complex64*)po).i = i;
      }
      break;
    default:
      PyErr_SetString(PyExc_RuntimeError, "data type not supported");
      goto exit;
    }
    NI_ITERATOR_NEXT2(ii, io, pi, po);
  }
  
 exit:
  if (shifts)
    free(shifts);
  if (params) {
    for(kk = 0; kk < irank; kk++)
      if (params[kk])
	free(params[kk]);
    free(params);
  }
  return PyErr_Occurred() ? 0 : 1;
}
