/* $Id: _numarraymodule.c,v 1.46 2004/03/06 21:07:11 jaytmiller Exp $ */

#include <Python.h>
#include <stdio.h>
#include <math.h>
#include <signal.h>

#include "arrayobject.h"
#include "structmember.h"

/* Compile _numarray type for Pythons >= 2.2 */
char *_numarray__doc__ = 
"_numarray is a module which supplies C-helpers for numarray, including\n"
"the _numarray baseclass of numarray for Python version >= 2.2.  \n"
"Class _numarray is used to accelerate selected methods of _numarray by \n"
"handling simple cases directly in C.\n";

#define DEFERRED_ADDRESS(ADDR) 0

staticforward PyTypeObject _numarray_type;

static int
_numarray_init(PyArrayObject *self, PyObject *args, PyObject *kwds)
{
        static char *kwlist[] = {"shape", "type", "buffer", "byteoffset", 
				 "bytestride", "byteorder", "aligned", 
				 "real", "imag", NULL};
	PyObject *shape = NULL;
	PyObject *type  = NULL;
	PyObject *buffer = Py_None;
	int       byteoffset = 0;
	PyObject *bytestride = Py_None;
	char     *byteorder = NULL;
	int       aligned = 1;
	PyObject *real = Py_None, *imag=Py_None;

	int       typeno;
	PyObject *args2;

	if (!PyArg_ParseTupleAndKeywords(args, kwds, "|OOOiOsiOO", kwlist, 
					 &shape, &type, &buffer, &byteoffset, 
					 &bytestride, &byteorder, &aligned,
					 &real, &imag))
		return -1;

	if (type) {
		type = NA_getType(type);
		if (!type) return -1;
		if ((typeno = NA_typeObjectToTypeNo( type )) < 0) {
			PyErr_Format(PyExc_RuntimeError, 
				     "_numarray_init: can't get typeno for type");
			return -1;
		}
		Py_DECREF(type);
	} else {
		typeno = tAny;
	}
	
	if (!(self->descr = PyArray_DescrFromType( typeno ))) {
		PyErr_Format(PyExc_RuntimeError, 
			     "_numarray_init: bad type number");
		return -1;
	}
	
	if (byteorder) {
		if (!strcmp(byteorder, "little"))
			self->byteorder = NUM_LITTLE_ENDIAN;
		else if (!strcmp(byteorder, "big"))
			self->byteorder = NUM_BIG_ENDIAN;
		else {
			PyErr_Format(PyExc_ValueError, 
		     "_numarray_init: byteorder must be 'little' or 'big'");
			return -1;
		}
	} else
		self->byteorder = NA_ByteOrder();
	NA_updateByteswap(self);
	

	args2 = Py_BuildValue("OiOiOi", shape, self->descr->elsize, buffer,
			      byteoffset, bytestride, aligned);
	if (!args2) return -1;

	if (_numarray_type.tp_base->tp_init((PyObject *)self, args2, NULL) < 0)
		return -1;

	Py_DECREF(args2);
	
	self->_shadows = NULL;

	/* Since the follow attrs are defined in Python, _numarray is abstract */
	if (real != Py_None) {
		if (PyObject_SetAttrString(
			    (PyObject *) self, "real", real) < 0)
			return -1;
	}
	if (imag != Py_None) {
		if (PyObject_SetAttrString(
			    (PyObject *) self, "imag", imag) < 0)
			return -1;
	}

	return 0;
}

static PyObject *
_numarray_shadows_get(PyArrayObject *self)
{
	if (self->_shadows) {
		Py_INCREF(self->_shadows);
		return self->_shadows;
	} else {
		Py_INCREF(Py_None);
		return Py_None;
	}
}

static int
_numarray_shadows_set(PyArrayObject *self, PyObject *s)
{
	Py_XDECREF(self->_shadows);
	if (s) Py_INCREF(s);
	self->_shadows = s;
	return 0;
}

static PyObject *
_numarray_type_get(PyArrayObject *self)
{
	return NA_typeNoToTypeObject(self->descr->type_num);
}

static int
_numarray_type_set(PyArrayObject *self, PyObject *s)
{
	PyObject *item;
	long ntype;
	if (!s) { 
		PyErr_Format(PyExc_RuntimeError, "can't delete _type"); 
		return -1; 
	}
	if (!(item = PyObject_GetAttrString(s, "name"))) 
		return -1;
	if (!PyString_Check(item)) {
		PyErr_Format(PyExc_TypeError, "type name is not a string");
		return -1;
	}	       
	ntype = NA_nameToTypeNo( PyString_AsString(item) );
	if (ntype < 0) {
		PyErr_Format(PyExc_ValueError, "_numarray_type_set: unknown type:'%s'",
			     PyString_AsString(item));
		return -1;
	}
	Py_DECREF(item);
	self->descr = PyArray_DescrFromType( ntype );
	return 0;
}

static PyObject *
_numarray_byteorder_get(PyArrayObject *self)
{
	if (self->byteorder)
		return PyString_FromString("big");
	else
		return PyString_FromString("little");
}

static int
_numarray_byteorder_set(PyArrayObject *self, PyObject *s)
{
	char *order;
	if (!s) { PyErr_Format(PyExc_RuntimeError, "can't delete _byteorder"); return -1; }
	if (!PyString_Check(s)) {
		PyErr_Format(PyExc_TypeError, 
			     "_numarray_byteorder_set: must be 'little' or 'big'");
		return -1;
	}
	order = PyString_AsString(s);
	if (!strcmp(order, "big"))
		self->byteorder = NUM_BIG_ENDIAN;
	else if (!strcmp(order, "little"))
		self->byteorder = NUM_LITTLE_ENDIAN;
	else {
		PyErr_Format(PyExc_ValueError, 
			     "_numarray_byteorder_set: only accepts 'little' or 'big'");
		return -1;
	}
	NA_updateByteswap(self); 
	return 0;
}

static PyObject *
_numarray_check_overflow_get(PyArrayObject *self)
{
	return PyInt_FromLong((self->flags & CHECKOVERFLOW) != 0);
}

static int
_numarray_check_overflow_set(PyArrayObject *self, PyObject *s)
{
	if (!s) { 
		PyErr_Format(
			PyExc_RuntimeError, "can't delete _check_overflow"); 
		return -1; 
	}
	if  (!PyInt_Check(s)) {
		PyErr_Format(
			PyExc_TypeError, "_check_overflow must be an integer.");
		return -1;
	}
	if (PyInt_AsLong(s)) {
		self->flags |= CHECKOVERFLOW;
	} else {
		self->flags &= ~CHECKOVERFLOW;
	}
	return 0;
}

static PyGetSetDef _numarray_getsets[] = {
 	{"_shadows", 
	 (getter)_numarray_shadows_get, 
	 (setter) _numarray_shadows_set, "numeric shadows object"}, 
 	{"_type", 
	 (getter)_numarray_type_get, 
	 (setter) _numarray_type_set, "numeric type object"}, 
	{"_byteorder", 
	 (getter)_numarray_byteorder_get, 
	 (setter) _numarray_byteorder_set, "byteorder/endian-ness of array, 'big' or 'little'"}, 
	{"_check_overflow", 
	 (getter)_numarray_check_overflow_get, 
	 (setter) _numarray_check_overflow_set, "byteorder/endian-ness of array, 'big' or 'little'"}, 
	{0}
};

static PyObject *
_numarray_isbyteswapped(PyArrayObject *self, PyObject *args)
{
	NA_updateByteswap(self);
	return PyInt_FromLong((self->flags & NOTSWAPPED) == 0);
}

static PyObject *
fromlist(PyObject *self, PyObject *args)
{
	PyObject *seq;

	if (!PyArg_ParseTuple(args, "O:fromlist", &seq))
		return NULL;

	return NA_setArrayFromSequence((PyArrayObject *)self, seq);
}

static PyObject *p_copyFromAndConvert;
static PyObject *p_copyBytes[16];
static PyObject *p_copyNbytes;

static PyObject *
_getCopyByte(int n)
{
	char name[80];
	PyObject *dict, *function;
	if (n <= ELEM(p_copyBytes))
		sprintf(name, "copy%dbytes", n);
	else
		sprintf(name, "copyNbytes");
	dict = NA_initModuleGlobal("numarray._bytes", "functionDict");
	if (!dict) return NULL;
	function = PyDict_GetItemString(dict, name);
	Py_DECREF(dict);
	Py_INCREF(function);
	return function;
}
		
static int
deferred_numarray_init(void)
{
	int i;
	static int initialized;
		
	if (initialized) return 0;
	p_copyFromAndConvert = 
		NA_initModuleGlobal("numarray.ufunc", "_copyFromAndConvert");
	if (!p_copyFromAndConvert) return -1;
	
	p_copyNbytes = _getCopyByte(ELEM(p_copyBytes)+1);
	if (!p_copyNbytes) return -1;
	
	for(i=0; i<ELEM(p_copyBytes); i++) {
		p_copyBytes[i] = p_copyNbytes;
		Py_INCREF(p_copyNbytes);
	}
	for(i=1; i<=ELEM(p_copyBytes)+1; i*=2) {
		Py_DECREF(p_copyBytes[i-1]);
		p_copyBytes[i-1] = _getCopyByte(i);
		if (!p_copyBytes[i-1]) return -1;
	}
	initialized = 1;
	return 0;
}

static int
_noZeros(int n, maybelong *array)
{
	int i;
	for(i=0; i<n; i++)
		if (!array[i]) return 0;
	return 1;
}

static PyObject *
_copyFrom(PyObject *self, PyObject *arr0)
{
	PyObject *arr, *barr, *result;
	PyArrayObject *selfa = (PyArrayObject *) self;
	PyArrayObject *arra;

	if (deferred_numarray_init() < 0)
		return NULL;

	arra = NA_InputArray(arr0, tAny, 0);
	if (!arra) return NULL;
	arr = (PyObject *) arra;

	if (NA_NumArrayCheck(arr)) {
		if ((selfa->descr->type_num == arra->descr->type_num) &&
		    NA_ShapeEqual(selfa, arra) &&
		    (selfa->byteorder == arra->byteorder) &&
		    PyArray_ISALIGNED(self) &&
		    PyArray_ISALIGNED(arr) &&
		    _noZeros(arra->nstrides, arra->strides))
		{
			PyObject *cfunc;
			if (selfa->itemsize <= ELEM(p_copyBytes))
				cfunc = p_copyBytes[ selfa->itemsize - 1 ];
			else
				cfunc = p_copyNbytes;
			result = NA_callStrideConvCFuncCore(
				cfunc, selfa->nd, selfa->dimensions,
				arra->_data, arra->byteoffset, arra->nstrides, arra->strides,
				selfa->_data, selfa->byteoffset, selfa->nstrides, selfa->strides,
				selfa->itemsize);
			Py_DECREF(arr);
			return result;
		}
	} 
	barr = PyObject_CallMethod(
		self, "_broadcast", "(O)", arr);
	Py_DECREF(arr);
	if (!barr) return NULL;
	result = PyObject_CallFunction(
		p_copyFromAndConvert, "(OO)", barr, self);
	Py_DECREF(barr);
	return result;
}

static PyObject *
_Py_copyFrom(PyObject *self, PyObject *args)
{
	PyObject *a;

	if (!PyArg_ParseTuple(args, "O:_copyFrom", &a))
		return NULL;
	return _copyFrom(self, a);
}

static PyObject *
_numarray_getitem(PyObject *self, PyObject *args)
{
	PyArrayObject *me = (PyArrayObject *) self;
	long offset;
	if (!PyArg_ParseTuple(args, "l:_getitem", &offset))
		return NULL;
	if (!NA_updateDataPtr(me)) 
		return NULL;
	return NA_getPythonScalar(me, offset-me->byteoffset);
}

static PyObject *
_numarray_setitem(PyObject *self, PyObject *args)
{
	PyArrayObject *me = (PyArrayObject *) self;
	PyObject *value;
	long offset;
	if (!PyArg_ParseTuple(args, "lO:_setitem", &offset, &value))
		return NULL;
	if (!NA_updateDataPtr(me)) 
		return NULL;
	if (NA_setFromPythonScalar(me, offset-me->byteoffset, value) < 0)
		return NULL;
	Py_INCREF(Py_None);
	return Py_None;
}

static PyMethodDef _numarray_methods[] = {
	{"isbyteswapped", (PyCFunction)_numarray_isbyteswapped, METH_VARARGS,
	 "isbyteswapped() -> 0 if data is in natural machine order, 1 otherwise."},
	{"_getitem", (PyCFunction) _numarray_getitem, METH_VARARGS, 
	 "_getitem(offset) -> scalar value at offset"},
	{"_setitem", (PyCFunction) _numarray_setitem, METH_VARARGS, 
	 "_setitem(offset, value) sets array at offset to scalar value"},
	{"fromlist", fromlist, METH_VARARGS,
	 "fromlist()   loads sequence into array."},
	{"_copyFrom", _Py_copyFrom, METH_VARARGS,
	 "_copyFrom(a)  copies 'a' onto self."},
	{NULL,	NULL},
};

static PyTypeObject _numarray_type = {
	PyObject_HEAD_INIT(DEFERRED_ADDRESS(&PyType_Type))
	0,
	"_numarray._numarray",
	sizeof(PyArrayObject),
        0,
	0,			                /* tp_dealloc */
	0,					/* tp_print */
	0,					/* tp_getattr */
	0,					/* tp_setattr */
	0,					/* tp_compare */
	0,					/* tp_repr */
	0,					/* tp_as_number */
        0,                                      /* tp_as_sequence */
	0,                       		/* tp_as_mapping */
	0,					/* tp_hash */
	0,					/* tp_call */
	0,					/* tp_str */
	0,					/* tp_getattro */
	0,					/* tp_setattro */
	0,					/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT |
	Py_TPFLAGS_BASETYPE,                    /* tp_flags */
	0,					/* tp_doc */
	0,					/* tp_traverse */
	0,					/* tp_clear */
	0,					/* tp_richcompare */
	0,					/* tp_weaklistoffset */
	0,					/* tp_iter */
	0,					/* tp_iternext */
	_numarray_methods,			/* tp_methods */
	0,					/* tp_members */
	_numarray_getsets,			/* tp_getset */
	DEFERRED_ADDRESS(&_ndarray_type),       /* tp_base */
	0,					/* tp_dict */
	0,					/* tp_descr_get */
	0,					/* tp_descr_set */
	0,					/* tp_dictoffset */
	(initproc)_numarray_init,		/* tp_init */
	0,					/* tp_alloc */
	0,      				/* tp_new */
};

typedef void Sigfunc(int);

static PyObject *
_maxtype(PyObject *module, PyObject *args)
{
	long maxtype;
	PyObject *seq;
	if (!PyArg_ParseTuple(args, "O", &seq))
		return NULL;
	maxtype = NA_maxType(seq);
	if (maxtype < 0) 
		return NULL;
	else
		return PyInt_FromLong(maxtype);
}

static PyMethodDef _numarray_functions[] = {
	{"_maxtype", _maxtype, METH_VARARGS, 
	 "_maxtype(seq) returns an integer code corresponding to the maximum type in seq.  0:Int, 1:Long, 2:Float, 3:Complex"},
	{NULL,      NULL}        /* Sentinel */
};

DL_EXPORT(void)
	init_numarray(void) {
	PyObject *m;
	PyObject *nm, *nd, *nt;

	if (!(nm = PyImport_ImportModule("numarray._ndarray")))  /* NEW */
		Py_FatalError("_numarray: can't import ndarraytype extension.");

	nd = PyModule_GetDict(nm);                          /* BOR */
	if (!(nt = PyDict_GetItemString(nd, "_ndarray")))   /* BOR */
		Py_FatalError("_numarray: can't get type _ndarray._ndarray");
	if (!PyType_Check(nt))
		Py_FatalError("_numarray: _ndarray._ndarray isn't a type object");

	Py_DECREF(nm); 
	Py_INCREF(nt);
	_numarray_type.tp_base = (PyTypeObject *) nt;

	if (PyType_Ready(&_numarray_type) < 0)
		return;

	m = Py_InitModule3("_numarray",
			   _numarray_functions,
			   _numarray__doc__);
	if (m == NULL)
		return;

	Py_INCREF(&_numarray_type);
	if (PyModule_AddObject(m, "_numarray", (PyObject *) &_numarray_type) < 0)
		return;

	import_libnumarray();
}
