
#include <Python.h>
#include <stdio.h>
#include <math.h>
#include "libnumarray.h"

#ifdef MS_WIN32
#pragma warning(once : 4244)
#endif

#define logical_and(arg1, arg2) (arg1 != 0) & (arg2 != 0)
#define logical_or(arg1, arg2)  (arg1 != 0) | (arg2 != 0)
#define logical_xor(arg1, arg2) ((arg1 != 0) ^ (arg2 != 0)) & 1
#define ufmaximum(arg1, arg2) (((temp1=arg1) > (temp2=arg2)) ? temp1 : temp2)
#define ufminimum(arg1, arg2) (((temp1=arg1) < (temp2=arg2)) ? temp1 : temp2)

static PyObject *_Error;

/*********************  minus  *********************/

static int minus_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = -*tin;
    }
    return 0;
}

UFUNC_DESCR2(minus_Float64_vector, sizeof(Float64), sizeof(Float64));
/*********************  add  *********************/

static int add_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Float64  *tin2  = (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 + *tin2;
    }
    return 0;
}

UFUNC_DESCR3(add_Float64_vector_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int add_Float64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin1     = (Float64 *) buffers[0];
    Float64 tscalar   = *(Float64 *) buffers[1];
    Float64 *tout    = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 + tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(add_Float64_vector_scalar, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int add_Float64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 tscalar = *(Float64 *) buffers[0];
    Float64 *tin2   =  (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar + *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(add_Float64_scalar_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int add_Float64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64 *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout = (Float64 *) ((char *) output + outboffset);
    Float64 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            tout = (Float64 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval + *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            add_Float64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(add_Float64_accumulate, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));

static int add_Float64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64  *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout  = (Float64 *) ((char *) output + outboffset);
    Float64 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            net = net + *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            add_Float64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(add_Float64_reduce, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));
/*********************  subtract  *********************/

static int subtract_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Float64  *tin2  = (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 - *tin2;
    }
    return 0;
}

UFUNC_DESCR3(subtract_Float64_vector_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int subtract_Float64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin1     = (Float64 *) buffers[0];
    Float64 tscalar   = *(Float64 *) buffers[1];
    Float64 *tout    = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 - tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(subtract_Float64_vector_scalar, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int subtract_Float64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 tscalar = *(Float64 *) buffers[0];
    Float64 *tin2   =  (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar - *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(subtract_Float64_scalar_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int subtract_Float64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64 *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout = (Float64 *) ((char *) output + outboffset);
    Float64 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            tout = (Float64 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval - *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            subtract_Float64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(subtract_Float64_accumulate, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));

static int subtract_Float64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64  *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout  = (Float64 *) ((char *) output + outboffset);
    Float64 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            net = net - *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            subtract_Float64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(subtract_Float64_reduce, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/

static int multiply_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Float64  *tin2  = (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 * *tin2;
    }
    return 0;
}

UFUNC_DESCR3(multiply_Float64_vector_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int multiply_Float64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin1     = (Float64 *) buffers[0];
    Float64 tscalar   = *(Float64 *) buffers[1];
    Float64 *tout    = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 * tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(multiply_Float64_vector_scalar, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int multiply_Float64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 tscalar = *(Float64 *) buffers[0];
    Float64 *tin2   =  (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar * *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(multiply_Float64_scalar_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int multiply_Float64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64 *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout = (Float64 *) ((char *) output + outboffset);
    Float64 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            tout = (Float64 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval * *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            multiply_Float64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(multiply_Float64_accumulate, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));

static int multiply_Float64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64  *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout  = (Float64 *) ((char *) output + outboffset);
    Float64 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            net = net * *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            multiply_Float64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(multiply_Float64_reduce, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));
/*********************  divide  *********************/
/*********************  divide  *********************/

static int divide_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Float64  *tin2  = (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 / *tin2;
    }
    return 0;
}

UFUNC_DESCR3(divide_Float64_vector_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int divide_Float64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin1     = (Float64 *) buffers[0];
    Float64 tscalar   = *(Float64 *) buffers[1];
    Float64 *tout    = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 / tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(divide_Float64_vector_scalar, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int divide_Float64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 tscalar = *(Float64 *) buffers[0];
    Float64 *tin2   =  (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar / *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(divide_Float64_scalar_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int divide_Float64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64 *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout = (Float64 *) ((char *) output + outboffset);
    Float64 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            tout = (Float64 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval / *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            divide_Float64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(divide_Float64_accumulate, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));

static int divide_Float64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64  *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout  = (Float64 *) ((char *) output + outboffset);
    Float64 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            net = net / *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            divide_Float64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(divide_Float64_reduce, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));
/*********************  floor_divide  *********************/
/*********************  floor_divide  *********************/

static int floor_divide_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Float64  *tin2  = (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        NUM_FLOORDIVIDE(*tin1, *tin2, *tout);
    }
    return 0;
}

UFUNC_DESCR3(floor_divide_Float64_vector_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int floor_divide_Float64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin1     = (Float64 *) buffers[0];
    Float64 tscalar   = *(Float64 *) buffers[1];
    Float64 *tout    = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        NUM_FLOORDIVIDE(*tin1, tscalar, *tout);
    }
    return 0;
}

UFUNC_DESCR3vs(floor_divide_Float64_vector_scalar, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int floor_divide_Float64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 tscalar = *(Float64 *) buffers[0];
    Float64 *tin2   =  (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        NUM_FLOORDIVIDE(tscalar, *tin2, *tout);
    }
    return 0;
}

UFUNC_DESCR3sv(floor_divide_Float64_scalar_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int floor_divide_Float64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64 *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout = (Float64 *) ((char *) output + outboffset);
    Float64 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            tout = (Float64 *) ((char *) tout + outbstrides[dim]);
            NUM_FLOORDIVIDE(lastval ,*tin, *tout);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            floor_divide_Float64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(floor_divide_Float64_accumulate, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));

static int floor_divide_Float64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64  *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout  = (Float64 *) ((char *) output + outboffset);
    Float64 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            NUM_FLOORDIVIDE(net, *tin, net);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            floor_divide_Float64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(floor_divide_Float64_reduce, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));
/*********************  true_divide  *********************/
/*********************  true_divide  *********************/

static int true_divide_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Float64  *tin2  = (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 / *tin2;
    }
    return 0;
}

UFUNC_DESCR3(true_divide_Float64_vector_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int true_divide_Float64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin1     = (Float64 *) buffers[0];
    Float64 tscalar   = *(Float64 *) buffers[1];
    Float64 *tout    = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 / tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(true_divide_Float64_vector_scalar, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int true_divide_Float64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 tscalar = *(Float64 *) buffers[0];
    Float64 *tin2   =  (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar / *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(true_divide_Float64_scalar_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int true_divide_Float64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64 *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout = (Float64 *) ((char *) output + outboffset);
    Float64 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            tout = (Float64 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval / *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            true_divide_Float64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(true_divide_Float64_accumulate, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));

static int true_divide_Float64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64  *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout  = (Float64 *) ((char *) output + outboffset);
    Float64 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            net = net / *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            true_divide_Float64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(true_divide_Float64_reduce, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));
/*********************  remainder  *********************/
/*********************  remainder  *********************/

static int remainder_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Float64  *tin2  = (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = fmod(*tin1, *tin2);
    }
    return 0;
}

UFUNC_DESCR3(remainder_Float64_vector_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int remainder_Float64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin1     = (Float64 *) buffers[0];
    Float64 tscalar   = *(Float64 *) buffers[1];
    Float64 *tout    = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = fmod(*tin1, tscalar);
    }
    return 0;
}

UFUNC_DESCR3vs(remainder_Float64_vector_scalar, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int remainder_Float64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 tscalar = *(Float64 *) buffers[0];
    Float64 *tin2   =  (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = fmod(tscalar, *tin2);
    }
    return 0;
}

UFUNC_DESCR3sv(remainder_Float64_scalar_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int remainder_Float64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64 *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout = (Float64 *) ((char *) output + outboffset);
    Float64 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            tout = (Float64 *) ((char *) tout + outbstrides[dim]);
            *tout = fmod(lastval ,*tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            remainder_Float64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(remainder_Float64_accumulate, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));

static int remainder_Float64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64  *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout  = (Float64 *) ((char *) output + outboffset);
    Float64 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            net = fmod(net, *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            remainder_Float64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(remainder_Float64_reduce, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));
/*********************  power  *********************/

static int power_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Float64  *tin2  = (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = num_pow(*tin1, *tin2);
    }
    return 0;
}

UFUNC_DESCR3(power_Float64_vector_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int power_Float64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin1     = (Float64 *) buffers[0];
    Float64 tscalar   = *(Float64 *) buffers[1];
    Float64 *tout    = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = num_pow(*tin1, tscalar);
    }
    return 0;
}

UFUNC_DESCR3vs(power_Float64_vector_scalar, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int power_Float64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 tscalar = *(Float64 *) buffers[0];
    Float64 *tin2   =  (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = num_pow(tscalar, *tin2);
    }
    return 0;
}

UFUNC_DESCR3sv(power_Float64_scalar_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int power_Float64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64 *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout = (Float64 *) ((char *) output + outboffset);
    Float64 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            tout = (Float64 *) ((char *) tout + outbstrides[dim]);
            *tout = num_pow(lastval ,*tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            power_Float64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(power_Float64_accumulate, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));

static int power_Float64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64  *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout  = (Float64 *) ((char *) output + outboffset);
    Float64 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            net = num_pow(net, *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            power_Float64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(power_Float64_reduce, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));
/*********************  abs  *********************/

static int abs_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = fabs(*tin);
    }
    return 0;
}

UFUNC_DESCR2(abs_Float64_vector, sizeof(Float64), sizeof(Float64));
/*********************  sin  *********************/

static int sin_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = sin(*tin);
    }
    return 0;
}

UFUNC_DESCR2(sin_Float64_vector, sizeof(Float64), sizeof(Float64));
/*********************  cos  *********************/

static int cos_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = cos(*tin);
    }
    return 0;
}

UFUNC_DESCR2(cos_Float64_vector, sizeof(Float64), sizeof(Float64));
/*********************  tan  *********************/

static int tan_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = tan(*tin);
    }
    return 0;
}

UFUNC_DESCR2(tan_Float64_vector, sizeof(Float64), sizeof(Float64));
/*********************  arcsin  *********************/

static int arcsin_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = asin(*tin);
    }
    return 0;
}

UFUNC_DESCR2(arcsin_Float64_vector, sizeof(Float64), sizeof(Float64));
/*********************  arccos  *********************/

static int arccos_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = acos(*tin);
    }
    return 0;
}

UFUNC_DESCR2(arccos_Float64_vector, sizeof(Float64), sizeof(Float64));
/*********************  arctan  *********************/

static int arctan_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = atan(*tin);
    }
    return 0;
}

UFUNC_DESCR2(arctan_Float64_vector, sizeof(Float64), sizeof(Float64));
/*********************  arctan2  *********************/

static int arctan2_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Float64  *tin2  = (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = atan2(*tin1, *tin2);
    }
    return 0;
}

UFUNC_DESCR3(arctan2_Float64_vector_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int arctan2_Float64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin1     = (Float64 *) buffers[0];
    Float64 tscalar   = *(Float64 *) buffers[1];
    Float64 *tout    = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = atan2(*tin1, tscalar);
    }
    return 0;
}

UFUNC_DESCR3vs(arctan2_Float64_vector_scalar, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int arctan2_Float64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 tscalar = *(Float64 *) buffers[0];
    Float64 *tin2   =  (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = atan2(tscalar, *tin2);
    }
    return 0;
}

UFUNC_DESCR3sv(arctan2_Float64_scalar_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int arctan2_Float64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64 *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout = (Float64 *) ((char *) output + outboffset);
    Float64 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            tout = (Float64 *) ((char *) tout + outbstrides[dim]);
            *tout = atan2(lastval ,*tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            arctan2_Float64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(arctan2_Float64_accumulate, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));

static int arctan2_Float64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64  *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout  = (Float64 *) ((char *) output + outboffset);
    Float64 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            net = atan2(net, *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            arctan2_Float64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(arctan2_Float64_reduce, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));
/*********************  log  *********************/

static int log_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_log(*tin);
    }
    return 0;
}

UFUNC_DESCR2(log_Float64_vector, sizeof(Float64), sizeof(Float64));
/*********************  log10  *********************/

static int log10_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_log10(*tin);
    }
    return 0;
}

UFUNC_DESCR2(log10_Float64_vector, sizeof(Float64), sizeof(Float64));
/*********************  exp  *********************/

static int exp_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = exp(*tin);
    }
    return 0;
}

UFUNC_DESCR2(exp_Float64_vector, sizeof(Float64), sizeof(Float64));
/*********************  sinh  *********************/

static int sinh_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = sinh(*tin);
    }
    return 0;
}

UFUNC_DESCR2(sinh_Float64_vector, sizeof(Float64), sizeof(Float64));
/*********************  cosh  *********************/

static int cosh_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = cosh(*tin);
    }
    return 0;
}

UFUNC_DESCR2(cosh_Float64_vector, sizeof(Float64), sizeof(Float64));
/*********************  tanh  *********************/

static int tanh_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = tanh(*tin);
    }
    return 0;
}

UFUNC_DESCR2(tanh_Float64_vector, sizeof(Float64), sizeof(Float64));
/*********************  arcsinh  *********************/

static int arcsinh_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_asinh(*tin);
    }
    return 0;
}

UFUNC_DESCR2(arcsinh_Float64_vector, sizeof(Float64), sizeof(Float64));
/*********************  arccosh  *********************/

static int arccosh_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_acosh(*tin);
    }
    return 0;
}

UFUNC_DESCR2(arccosh_Float64_vector, sizeof(Float64), sizeof(Float64));
/*********************  arctanh  *********************/

static int arctanh_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_atanh(*tin);
    }
    return 0;
}

UFUNC_DESCR2(arctanh_Float64_vector, sizeof(Float64), sizeof(Float64));
/*********************  ieeemask  *********************/
/*********************  ieeemask  *********************/

static int ieeemask_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Int32  *tin2  = (Int32 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = NA_IeeeSpecial64(tin1, tin2);
    }
    return 0;
}

UFUNC_DESCR3(ieeemask_Float64_vector_vector, sizeof(Float64), sizeof(Int32), sizeof(Bool));

/*********************  sqrt  *********************/

static int sqrt_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = sqrt(*tin);
    }
    return 0;
}

UFUNC_DESCR2(sqrt_Float64_vector, sizeof(Float64), sizeof(Float64));
/*********************  equal  *********************/

static int equal_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Float64  *tin2  = (Float64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 == *tin2;
    }
    return 0;
}

UFUNC_DESCR3(equal_Float64_vector_vector, sizeof(Float64), sizeof(Float64), sizeof(Bool));


static int equal_Float64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin1     = (Float64 *) buffers[0];
    Float64 tscalar   = *(Float64 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 == tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(equal_Float64_vector_scalar, sizeof(Float64), sizeof(Float64), sizeof(Bool));


static int equal_Float64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 tscalar = *(Float64 *) buffers[0];
    Float64 *tin2   =  (Float64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar == *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(equal_Float64_scalar_vector, sizeof(Float64), sizeof(Float64), sizeof(Bool));

/*********************  not_equal  *********************/

static int not_equal_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Float64  *tin2  = (Float64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 != *tin2;
    }
    return 0;
}

UFUNC_DESCR3(not_equal_Float64_vector_vector, sizeof(Float64), sizeof(Float64), sizeof(Bool));


static int not_equal_Float64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin1     = (Float64 *) buffers[0];
    Float64 tscalar   = *(Float64 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 != tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(not_equal_Float64_vector_scalar, sizeof(Float64), sizeof(Float64), sizeof(Bool));


static int not_equal_Float64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 tscalar = *(Float64 *) buffers[0];
    Float64 *tin2   =  (Float64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar != *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(not_equal_Float64_scalar_vector, sizeof(Float64), sizeof(Float64), sizeof(Bool));

/*********************  greater  *********************/

static int greater_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Float64  *tin2  = (Float64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 > *tin2;
    }
    return 0;
}

UFUNC_DESCR3(greater_Float64_vector_vector, sizeof(Float64), sizeof(Float64), sizeof(Bool));


static int greater_Float64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin1     = (Float64 *) buffers[0];
    Float64 tscalar   = *(Float64 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 > tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(greater_Float64_vector_scalar, sizeof(Float64), sizeof(Float64), sizeof(Bool));


static int greater_Float64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 tscalar = *(Float64 *) buffers[0];
    Float64 *tin2   =  (Float64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar > *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(greater_Float64_scalar_vector, sizeof(Float64), sizeof(Float64), sizeof(Bool));

/*********************  greater_equal  *********************/

static int greater_equal_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Float64  *tin2  = (Float64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 >= *tin2;
    }
    return 0;
}

UFUNC_DESCR3(greater_equal_Float64_vector_vector, sizeof(Float64), sizeof(Float64), sizeof(Bool));


static int greater_equal_Float64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin1     = (Float64 *) buffers[0];
    Float64 tscalar   = *(Float64 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 >= tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(greater_equal_Float64_vector_scalar, sizeof(Float64), sizeof(Float64), sizeof(Bool));


static int greater_equal_Float64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 tscalar = *(Float64 *) buffers[0];
    Float64 *tin2   =  (Float64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar >= *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(greater_equal_Float64_scalar_vector, sizeof(Float64), sizeof(Float64), sizeof(Bool));

/*********************  less  *********************/

static int less_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Float64  *tin2  = (Float64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 < *tin2;
    }
    return 0;
}

UFUNC_DESCR3(less_Float64_vector_vector, sizeof(Float64), sizeof(Float64), sizeof(Bool));


static int less_Float64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin1     = (Float64 *) buffers[0];
    Float64 tscalar   = *(Float64 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 < tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(less_Float64_vector_scalar, sizeof(Float64), sizeof(Float64), sizeof(Bool));


static int less_Float64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 tscalar = *(Float64 *) buffers[0];
    Float64 *tin2   =  (Float64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar < *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(less_Float64_scalar_vector, sizeof(Float64), sizeof(Float64), sizeof(Bool));

/*********************  less_equal  *********************/

static int less_equal_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Float64  *tin2  = (Float64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 <= *tin2;
    }
    return 0;
}

UFUNC_DESCR3(less_equal_Float64_vector_vector, sizeof(Float64), sizeof(Float64), sizeof(Bool));


static int less_equal_Float64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin1     = (Float64 *) buffers[0];
    Float64 tscalar   = *(Float64 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 <= tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(less_equal_Float64_vector_scalar, sizeof(Float64), sizeof(Float64), sizeof(Bool));


static int less_equal_Float64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 tscalar = *(Float64 *) buffers[0];
    Float64 *tin2   =  (Float64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar <= *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(less_equal_Float64_scalar_vector, sizeof(Float64), sizeof(Float64), sizeof(Bool));

/*********************  logical_and  *********************/

static int logical_and_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Float64  *tin2  = (Float64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = logical_and(*tin1, *tin2);
    }
    return 0;
}

UFUNC_DESCR3(logical_and_Float64_vector_vector, sizeof(Float64), sizeof(Float64), sizeof(Bool));


static int logical_and_Float64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin1     = (Float64 *) buffers[0];
    Float64 tscalar   = *(Float64 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = logical_and(*tin1, tscalar);
    }
    return 0;
}

UFUNC_DESCR3vs(logical_and_Float64_vector_scalar, sizeof(Float64), sizeof(Float64), sizeof(Bool));


static int logical_and_Float64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 tscalar = *(Float64 *) buffers[0];
    Float64 *tin2   =  (Float64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = logical_and(tscalar, *tin2);
    }
    return 0;
}

UFUNC_DESCR3sv(logical_and_Float64_scalar_vector, sizeof(Float64), sizeof(Float64), sizeof(Bool));

/*********************  logical_or  *********************/

static int logical_or_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Float64  *tin2  = (Float64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = logical_or(*tin1, *tin2);
    }
    return 0;
}

UFUNC_DESCR3(logical_or_Float64_vector_vector, sizeof(Float64), sizeof(Float64), sizeof(Bool));


static int logical_or_Float64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin1     = (Float64 *) buffers[0];
    Float64 tscalar   = *(Float64 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = logical_or(*tin1, tscalar);
    }
    return 0;
}

UFUNC_DESCR3vs(logical_or_Float64_vector_scalar, sizeof(Float64), sizeof(Float64), sizeof(Bool));


static int logical_or_Float64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 tscalar = *(Float64 *) buffers[0];
    Float64 *tin2   =  (Float64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = logical_or(tscalar, *tin2);
    }
    return 0;
}

UFUNC_DESCR3sv(logical_or_Float64_scalar_vector, sizeof(Float64), sizeof(Float64), sizeof(Bool));

/*********************  logical_xor  *********************/

static int logical_xor_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Float64  *tin2  = (Float64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = logical_xor(*tin1, *tin2);
    }
    return 0;
}

UFUNC_DESCR3(logical_xor_Float64_vector_vector, sizeof(Float64), sizeof(Float64), sizeof(Bool));


static int logical_xor_Float64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin1     = (Float64 *) buffers[0];
    Float64 tscalar   = *(Float64 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = logical_xor(*tin1, tscalar);
    }
    return 0;
}

UFUNC_DESCR3vs(logical_xor_Float64_vector_scalar, sizeof(Float64), sizeof(Float64), sizeof(Bool));


static int logical_xor_Float64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 tscalar = *(Float64 *) buffers[0];
    Float64 *tin2   =  (Float64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = logical_xor(tscalar, *tin2);
    }
    return 0;
}

UFUNC_DESCR3sv(logical_xor_Float64_scalar_vector, sizeof(Float64), sizeof(Float64), sizeof(Bool));

/*********************  logical_not  *********************/

static int logical_not_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Bool *tout = (Bool *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = !*tin;
    }
    return 0;
}

UFUNC_DESCR2(logical_not_Float64_vector, sizeof(Float64), sizeof(Bool));
/*********************  bitwise_and  *********************/
/*********************  bitwise_or  *********************/
/*********************  bitwise_xor  *********************/
/*********************  bitwise_not  *********************/
/*********************  rshift  *********************/
/*********************  lshift  *********************/
/*********************  floor  *********************/
/*********************  floor  *********************/

static int floor_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = floor(*tin);
    }
    return 0;
}

UFUNC_DESCR2(floor_Float64_vector, sizeof(Float64), sizeof(Float64));
/*********************  ceil  *********************/
/*********************  ceil  *********************/

static int ceil_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = ceil(*tin);
    }
    return 0;
}

UFUNC_DESCR2(ceil_Float64_vector, sizeof(Float64), sizeof(Float64));
/*********************  maximum  *********************/

static int maximum_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Float64  *tin2  = (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    Float64 temp1, temp2;
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = ufmaximum(*tin1, *tin2);
    }
    return 0;
}

UFUNC_DESCR3(maximum_Float64_vector_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int maximum_Float64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin1     = (Float64 *) buffers[0];
    Float64 tscalar   = *(Float64 *) buffers[1];
    Float64 *tout    = (Float64 *) buffers[2];
    Float64 temp1, temp2;
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = ufmaximum(*tin1, tscalar);
    }
    return 0;
}

UFUNC_DESCR3vs(maximum_Float64_vector_scalar, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int maximum_Float64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 tscalar = *(Float64 *) buffers[0];
    Float64 *tin2   =  (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    Float64 temp1, temp2;
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = ufmaximum(tscalar, *tin2);
    }
    return 0;
}

UFUNC_DESCR3sv(maximum_Float64_scalar_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int maximum_Float64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64 *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout = (Float64 *) ((char *) output + outboffset);
    Float64 lastval;
    Float64 temp1, temp2;
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            tout = (Float64 *) ((char *) tout + outbstrides[dim]);
            *tout = ufmaximum(lastval ,*tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            maximum_Float64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(maximum_Float64_accumulate, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));

static int maximum_Float64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64  *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout  = (Float64 *) ((char *) output + outboffset);
    Float64 net;
    Float64 temp1, temp2;
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            net = ufmaximum(net, *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            maximum_Float64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(maximum_Float64_reduce, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));
/*********************  minimum  *********************/

static int minimum_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Float64  *tin2  = (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    Float64 temp1, temp2;
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = ufminimum(*tin1, *tin2);
    }
    return 0;
}

UFUNC_DESCR3(minimum_Float64_vector_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int minimum_Float64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin1     = (Float64 *) buffers[0];
    Float64 tscalar   = *(Float64 *) buffers[1];
    Float64 *tout    = (Float64 *) buffers[2];
    Float64 temp1, temp2;
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = ufminimum(*tin1, tscalar);
    }
    return 0;
}

UFUNC_DESCR3vs(minimum_Float64_vector_scalar, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int minimum_Float64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 tscalar = *(Float64 *) buffers[0];
    Float64 *tin2   =  (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    Float64 temp1, temp2;
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = ufminimum(tscalar, *tin2);
    }
    return 0;
}

UFUNC_DESCR3sv(minimum_Float64_scalar_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int minimum_Float64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64 *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout = (Float64 *) ((char *) output + outboffset);
    Float64 lastval;
    Float64 temp1, temp2;
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            tout = (Float64 *) ((char *) tout + outbstrides[dim]);
            *tout = ufminimum(lastval ,*tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            minimum_Float64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(minimum_Float64_accumulate, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));

static int minimum_Float64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64  *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout  = (Float64 *) ((char *) output + outboffset);
    Float64 net;
    Float64 temp1, temp2;
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            net = ufminimum(net, *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            minimum_Float64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(minimum_Float64_reduce, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));
/*********************  fabs  *********************/

static int fabs_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = fabs(*tin);
    }
    return 0;
}

UFUNC_DESCR2(fabs_Float64_vector, sizeof(Float64), sizeof(Float64));
/*********************  _round  *********************/

static int _round_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_round(*tin);
    }
    return 0;
}

UFUNC_DESCR2(_round_Float64_vector, sizeof(Float64), sizeof(Float64));
/*********************  hypot  *********************/

static int hypot_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Float64  *tin2  = (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = hypot(*tin1, *tin2);
    }
    return 0;
}

UFUNC_DESCR3(hypot_Float64_vector_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int hypot_Float64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin1     = (Float64 *) buffers[0];
    Float64 tscalar   = *(Float64 *) buffers[1];
    Float64 *tout    = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = hypot(*tin1, tscalar);
    }
    return 0;
}

UFUNC_DESCR3vs(hypot_Float64_vector_scalar, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int hypot_Float64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 tscalar = *(Float64 *) buffers[0];
    Float64 *tin2   =  (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = hypot(tscalar, *tin2);
    }
    return 0;
}

UFUNC_DESCR3sv(hypot_Float64_scalar_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int hypot_Float64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64 *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout = (Float64 *) ((char *) output + outboffset);
    Float64 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            tout = (Float64 *) ((char *) tout + outbstrides[dim]);
            *tout = hypot(lastval ,*tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            hypot_Float64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(hypot_Float64_accumulate, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));

static int hypot_Float64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64  *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout  = (Float64 *) ((char *) output + outboffset);
    Float64 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            net = hypot(net, *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            hypot_Float64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(hypot_Float64_reduce, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));
/*********************  minus  *********************/
/*********************  add  *********************/
/*********************  subtract  *********************/
/*********************  multiply  *********************/
/*********************  divide  *********************/
/*********************  true_divide  *********************/
/*********************  remainder  *********************/
/*********************  power  *********************/
/*********************  abs  *********************/
/*********************  sin  *********************/
/*********************  cos  *********************/
/*********************  tan  *********************/
/*********************  arcsin  *********************/
/*********************  arccos  *********************/
/*********************  arctan  *********************/
/*********************  arcsinh  *********************/
/*********************  arccosh  *********************/
/*********************  arctanh  *********************/
/*********************  log  *********************/
/*********************  log10  *********************/
/*********************  exp  *********************/
/*********************  sinh  *********************/
/*********************  cosh  *********************/
/*********************  tanh  *********************/
/*********************  sqrt  *********************/
/*********************  equal  *********************/
/*********************  not_equal  *********************/
/*********************  logical_and  *********************/
/*********************  logical_or  *********************/
/*********************  logical_xor  *********************/
/*********************  logical_not  *********************/
/*********************  floor  *********************/
/*********************  ceil  *********************/
/*********************  maximum  *********************/
/*********************  minimum  *********************/
/*********************  fabs  *********************/
/*********************  _round  *********************/
/*********************  hypot  *********************/

static PyMethodDef _ufuncFloat64Methods[] = {

	{NULL,      NULL}        /* Sentinel */
};

static PyObject *init_funcDict(void) {
    PyObject *dict;
    dict = PyDict_New();
    NA_add_cfunc(dict, "('minus', 'vector', (('Float64',), ('Float64',)))", (void *) &minus_Float64_vector_descr);
    NA_add_cfunc(dict, "('add', 'vector_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &add_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('add', 'vector_scalar', (('Float64', 'Float64'), ('Float64',)))", (void *) &add_Float64_vector_scalar_descr);
    NA_add_cfunc(dict, "('add', 'scalar_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &add_Float64_scalar_vector_descr);
    NA_add_cfunc(dict, "('add', 'reduce', (('Float64',), ('Float64',)))", (void *) &add_Float64_reduce_descr);
    NA_add_cfunc(dict, "('add', 'accumulate', (('Float64',), ('Float64',)))", (void *) &add_Float64_accumulate_descr);
    NA_add_cfunc(dict, "('subtract', 'vector_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &subtract_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('subtract', 'vector_scalar', (('Float64', 'Float64'), ('Float64',)))", (void *) &subtract_Float64_vector_scalar_descr);
    NA_add_cfunc(dict, "('subtract', 'scalar_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &subtract_Float64_scalar_vector_descr);
    NA_add_cfunc(dict, "('subtract', 'reduce', (('Float64',), ('Float64',)))", (void *) &subtract_Float64_reduce_descr);
    NA_add_cfunc(dict, "('subtract', 'accumulate', (('Float64',), ('Float64',)))", (void *) &subtract_Float64_accumulate_descr);
    NA_add_cfunc(dict, "('multiply', 'vector_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &multiply_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('multiply', 'vector_scalar', (('Float64', 'Float64'), ('Float64',)))", (void *) &multiply_Float64_vector_scalar_descr);
    NA_add_cfunc(dict, "('multiply', 'scalar_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &multiply_Float64_scalar_vector_descr);
    NA_add_cfunc(dict, "('multiply', 'reduce', (('Float64',), ('Float64',)))", (void *) &multiply_Float64_reduce_descr);
    NA_add_cfunc(dict, "('multiply', 'accumulate', (('Float64',), ('Float64',)))", (void *) &multiply_Float64_accumulate_descr);
    NA_add_cfunc(dict, "('divide', 'vector_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &divide_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('divide', 'vector_scalar', (('Float64', 'Float64'), ('Float64',)))", (void *) &divide_Float64_vector_scalar_descr);
    NA_add_cfunc(dict, "('divide', 'scalar_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &divide_Float64_scalar_vector_descr);
    NA_add_cfunc(dict, "('divide', 'reduce', (('Float64',), ('Float64',)))", (void *) &divide_Float64_reduce_descr);
    NA_add_cfunc(dict, "('divide', 'accumulate', (('Float64',), ('Float64',)))", (void *) &divide_Float64_accumulate_descr);
    NA_add_cfunc(dict, "('floor_divide', 'vector_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &floor_divide_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('floor_divide', 'vector_scalar', (('Float64', 'Float64'), ('Float64',)))", (void *) &floor_divide_Float64_vector_scalar_descr);
    NA_add_cfunc(dict, "('floor_divide', 'scalar_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &floor_divide_Float64_scalar_vector_descr);
    NA_add_cfunc(dict, "('floor_divide', 'reduce', (('Float64',), ('Float64',)))", (void *) &floor_divide_Float64_reduce_descr);
    NA_add_cfunc(dict, "('floor_divide', 'accumulate', (('Float64',), ('Float64',)))", (void *) &floor_divide_Float64_accumulate_descr);
    NA_add_cfunc(dict, "('true_divide', 'vector_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &true_divide_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('true_divide', 'vector_scalar', (('Float64', 'Float64'), ('Float64',)))", (void *) &true_divide_Float64_vector_scalar_descr);
    NA_add_cfunc(dict, "('true_divide', 'scalar_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &true_divide_Float64_scalar_vector_descr);
    NA_add_cfunc(dict, "('true_divide', 'reduce', (('Float64',), ('Float64',)))", (void *) &true_divide_Float64_reduce_descr);
    NA_add_cfunc(dict, "('true_divide', 'accumulate', (('Float64',), ('Float64',)))", (void *) &true_divide_Float64_accumulate_descr);
    NA_add_cfunc(dict, "('remainder', 'vector_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &remainder_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('remainder', 'vector_scalar', (('Float64', 'Float64'), ('Float64',)))", (void *) &remainder_Float64_vector_scalar_descr);
    NA_add_cfunc(dict, "('remainder', 'scalar_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &remainder_Float64_scalar_vector_descr);
    NA_add_cfunc(dict, "('remainder', 'reduce', (('Float64',), ('Float64',)))", (void *) &remainder_Float64_reduce_descr);
    NA_add_cfunc(dict, "('remainder', 'accumulate', (('Float64',), ('Float64',)))", (void *) &remainder_Float64_accumulate_descr);
    NA_add_cfunc(dict, "('power', 'vector_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &power_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('power', 'vector_scalar', (('Float64', 'Float64'), ('Float64',)))", (void *) &power_Float64_vector_scalar_descr);
    NA_add_cfunc(dict, "('power', 'scalar_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &power_Float64_scalar_vector_descr);
    NA_add_cfunc(dict, "('power', 'reduce', (('Float64',), ('Float64',)))", (void *) &power_Float64_reduce_descr);
    NA_add_cfunc(dict, "('power', 'accumulate', (('Float64',), ('Float64',)))", (void *) &power_Float64_accumulate_descr);
    NA_add_cfunc(dict, "('abs', 'vector', (('Float64',), ('Float64',)))", (void *) &abs_Float64_vector_descr);
    NA_add_cfunc(dict, "('sin', 'vector', (('Float64',), ('Float64',)))", (void *) &sin_Float64_vector_descr);
    NA_add_cfunc(dict, "('cos', 'vector', (('Float64',), ('Float64',)))", (void *) &cos_Float64_vector_descr);
    NA_add_cfunc(dict, "('tan', 'vector', (('Float64',), ('Float64',)))", (void *) &tan_Float64_vector_descr);
    NA_add_cfunc(dict, "('arcsin', 'vector', (('Float64',), ('Float64',)))", (void *) &arcsin_Float64_vector_descr);
    NA_add_cfunc(dict, "('arccos', 'vector', (('Float64',), ('Float64',)))", (void *) &arccos_Float64_vector_descr);
    NA_add_cfunc(dict, "('arctan', 'vector', (('Float64',), ('Float64',)))", (void *) &arctan_Float64_vector_descr);
    NA_add_cfunc(dict, "('arctan2', 'vector_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &arctan2_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('arctan2', 'vector_scalar', (('Float64', 'Float64'), ('Float64',)))", (void *) &arctan2_Float64_vector_scalar_descr);
    NA_add_cfunc(dict, "('arctan2', 'scalar_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &arctan2_Float64_scalar_vector_descr);
    NA_add_cfunc(dict, "('arctan2', 'reduce', (('Float64',), ('Float64',)))", (void *) &arctan2_Float64_reduce_descr);
    NA_add_cfunc(dict, "('arctan2', 'accumulate', (('Float64',), ('Float64',)))", (void *) &arctan2_Float64_accumulate_descr);
    NA_add_cfunc(dict, "('log', 'vector', (('Float64',), ('Float64',)))", (void *) &log_Float64_vector_descr);
    NA_add_cfunc(dict, "('log10', 'vector', (('Float64',), ('Float64',)))", (void *) &log10_Float64_vector_descr);
    NA_add_cfunc(dict, "('exp', 'vector', (('Float64',), ('Float64',)))", (void *) &exp_Float64_vector_descr);
    NA_add_cfunc(dict, "('sinh', 'vector', (('Float64',), ('Float64',)))", (void *) &sinh_Float64_vector_descr);
    NA_add_cfunc(dict, "('cosh', 'vector', (('Float64',), ('Float64',)))", (void *) &cosh_Float64_vector_descr);
    NA_add_cfunc(dict, "('tanh', 'vector', (('Float64',), ('Float64',)))", (void *) &tanh_Float64_vector_descr);
    NA_add_cfunc(dict, "('arcsinh', 'vector', (('Float64',), ('Float64',)))", (void *) &arcsinh_Float64_vector_descr);
    NA_add_cfunc(dict, "('arccosh', 'vector', (('Float64',), ('Float64',)))", (void *) &arccosh_Float64_vector_descr);
    NA_add_cfunc(dict, "('arctanh', 'vector', (('Float64',), ('Float64',)))", (void *) &arctanh_Float64_vector_descr);
    NA_add_cfunc(dict, "('ieeemask', 'vector_vector', (('Float64', 'Int32'), ('Bool',)))", (void *) &ieeemask_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('sqrt', 'vector', (('Float64',), ('Float64',)))", (void *) &sqrt_Float64_vector_descr);
    NA_add_cfunc(dict, "('equal', 'vector_vector', (('Float64', 'Float64'), ('Bool',)))", (void *) &equal_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('equal', 'vector_scalar', (('Float64', 'Float64'), ('Bool',)))", (void *) &equal_Float64_vector_scalar_descr);
    NA_add_cfunc(dict, "('equal', 'scalar_vector', (('Float64', 'Float64'), ('Bool',)))", (void *) &equal_Float64_scalar_vector_descr);
    NA_add_cfunc(dict, "('not_equal', 'vector_vector', (('Float64', 'Float64'), ('Bool',)))", (void *) &not_equal_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('not_equal', 'vector_scalar', (('Float64', 'Float64'), ('Bool',)))", (void *) &not_equal_Float64_vector_scalar_descr);
    NA_add_cfunc(dict, "('not_equal', 'scalar_vector', (('Float64', 'Float64'), ('Bool',)))", (void *) &not_equal_Float64_scalar_vector_descr);
    NA_add_cfunc(dict, "('greater', 'vector_vector', (('Float64', 'Float64'), ('Bool',)))", (void *) &greater_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('greater', 'vector_scalar', (('Float64', 'Float64'), ('Bool',)))", (void *) &greater_Float64_vector_scalar_descr);
    NA_add_cfunc(dict, "('greater', 'scalar_vector', (('Float64', 'Float64'), ('Bool',)))", (void *) &greater_Float64_scalar_vector_descr);
    NA_add_cfunc(dict, "('greater_equal', 'vector_vector', (('Float64', 'Float64'), ('Bool',)))", (void *) &greater_equal_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('greater_equal', 'vector_scalar', (('Float64', 'Float64'), ('Bool',)))", (void *) &greater_equal_Float64_vector_scalar_descr);
    NA_add_cfunc(dict, "('greater_equal', 'scalar_vector', (('Float64', 'Float64'), ('Bool',)))", (void *) &greater_equal_Float64_scalar_vector_descr);
    NA_add_cfunc(dict, "('less', 'vector_vector', (('Float64', 'Float64'), ('Bool',)))", (void *) &less_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('less', 'vector_scalar', (('Float64', 'Float64'), ('Bool',)))", (void *) &less_Float64_vector_scalar_descr);
    NA_add_cfunc(dict, "('less', 'scalar_vector', (('Float64', 'Float64'), ('Bool',)))", (void *) &less_Float64_scalar_vector_descr);
    NA_add_cfunc(dict, "('less_equal', 'vector_vector', (('Float64', 'Float64'), ('Bool',)))", (void *) &less_equal_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('less_equal', 'vector_scalar', (('Float64', 'Float64'), ('Bool',)))", (void *) &less_equal_Float64_vector_scalar_descr);
    NA_add_cfunc(dict, "('less_equal', 'scalar_vector', (('Float64', 'Float64'), ('Bool',)))", (void *) &less_equal_Float64_scalar_vector_descr);
    NA_add_cfunc(dict, "('logical_and', 'vector_vector', (('Float64', 'Float64'), ('Bool',)))", (void *) &logical_and_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('logical_and', 'vector_scalar', (('Float64', 'Float64'), ('Bool',)))", (void *) &logical_and_Float64_vector_scalar_descr);
    NA_add_cfunc(dict, "('logical_and', 'scalar_vector', (('Float64', 'Float64'), ('Bool',)))", (void *) &logical_and_Float64_scalar_vector_descr);
    NA_add_cfunc(dict, "('logical_or', 'vector_vector', (('Float64', 'Float64'), ('Bool',)))", (void *) &logical_or_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('logical_or', 'vector_scalar', (('Float64', 'Float64'), ('Bool',)))", (void *) &logical_or_Float64_vector_scalar_descr);
    NA_add_cfunc(dict, "('logical_or', 'scalar_vector', (('Float64', 'Float64'), ('Bool',)))", (void *) &logical_or_Float64_scalar_vector_descr);
    NA_add_cfunc(dict, "('logical_xor', 'vector_vector', (('Float64', 'Float64'), ('Bool',)))", (void *) &logical_xor_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('logical_xor', 'vector_scalar', (('Float64', 'Float64'), ('Bool',)))", (void *) &logical_xor_Float64_vector_scalar_descr);
    NA_add_cfunc(dict, "('logical_xor', 'scalar_vector', (('Float64', 'Float64'), ('Bool',)))", (void *) &logical_xor_Float64_scalar_vector_descr);
    NA_add_cfunc(dict, "('logical_not', 'vector', (('Float64',), ('Bool',)))", (void *) &logical_not_Float64_vector_descr);
    NA_add_cfunc(dict, "('floor', 'vector', (('Float64',), ('Float64',)))", (void *) &floor_Float64_vector_descr);
    NA_add_cfunc(dict, "('ceil', 'vector', (('Float64',), ('Float64',)))", (void *) &ceil_Float64_vector_descr);
    NA_add_cfunc(dict, "('maximum', 'vector_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &maximum_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('maximum', 'vector_scalar', (('Float64', 'Float64'), ('Float64',)))", (void *) &maximum_Float64_vector_scalar_descr);
    NA_add_cfunc(dict, "('maximum', 'scalar_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &maximum_Float64_scalar_vector_descr);
    NA_add_cfunc(dict, "('maximum', 'reduce', (('Float64',), ('Float64',)))", (void *) &maximum_Float64_reduce_descr);
    NA_add_cfunc(dict, "('maximum', 'accumulate', (('Float64',), ('Float64',)))", (void *) &maximum_Float64_accumulate_descr);
    NA_add_cfunc(dict, "('minimum', 'vector_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &minimum_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('minimum', 'vector_scalar', (('Float64', 'Float64'), ('Float64',)))", (void *) &minimum_Float64_vector_scalar_descr);
    NA_add_cfunc(dict, "('minimum', 'scalar_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &minimum_Float64_scalar_vector_descr);
    NA_add_cfunc(dict, "('minimum', 'reduce', (('Float64',), ('Float64',)))", (void *) &minimum_Float64_reduce_descr);
    NA_add_cfunc(dict, "('minimum', 'accumulate', (('Float64',), ('Float64',)))", (void *) &minimum_Float64_accumulate_descr);
    NA_add_cfunc(dict, "('fabs', 'vector', (('Float64',), ('Float64',)))", (void *) &fabs_Float64_vector_descr);
    NA_add_cfunc(dict, "('_round', 'vector', (('Float64',), ('Float64',)))", (void *) &_round_Float64_vector_descr);
    NA_add_cfunc(dict, "('hypot', 'vector_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &hypot_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('hypot', 'vector_scalar', (('Float64', 'Float64'), ('Float64',)))", (void *) &hypot_Float64_vector_scalar_descr);
    NA_add_cfunc(dict, "('hypot', 'scalar_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &hypot_Float64_scalar_vector_descr);
    NA_add_cfunc(dict, "('hypot', 'reduce', (('Float64',), ('Float64',)))", (void *) &hypot_Float64_reduce_descr);
    NA_add_cfunc(dict, "('hypot', 'accumulate', (('Float64',), ('Float64',)))", (void *) &hypot_Float64_accumulate_descr);
    return dict;
}

/* platform independent*/
#ifdef MS_WIN32
__declspec(dllexport)
#endif
void init_ufuncFloat64(void) {
    PyObject *m, *d;
    m = Py_InitModule("_ufuncFloat64", _ufuncFloat64Methods);
    d = PyModule_GetDict(m);
    _Error = PyErr_NewException("_ufuncFloat64.error", NULL, NULL);
    PyDict_SetItemString(d, "error", _Error);
    import_libnumarray();
    PyDict_SetItemString(d, "functionDict", init_funcDict());
}
