
#include <Python.h>
#include <stdio.h>
#include <math.h>
#include "libnumarray.h"

#ifdef MS_WIN32
#pragma warning(once : 4244)
#endif

#define logical_and(arg1, arg2) (arg1 != 0) & (arg2 != 0)
#define logical_or(arg1, arg2)  (arg1 != 0) | (arg2 != 0)
#define logical_xor(arg1, arg2) ((arg1 != 0) ^ (arg2 != 0)) & 1
#define ufmaximum(arg1, arg2) (((temp1=arg1) > (temp2=arg2)) ? temp1 : temp2)
#define ufminimum(arg1, arg2) (((temp1=arg1) < (temp2=arg2)) ? temp1 : temp2)

static PyObject *_Error;

/*********************  minus  *********************/

static int minus_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    UInt64 *tout = (UInt64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = -*tin;
    }
    return 0;
}

UFUNC_DESCR2(minus_UInt64_vector, sizeof(UInt64), sizeof(UInt64));
/*********************  add  *********************/

static int add_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 + *tin2;
    }
    return 0;
}

UFUNC_DESCR3(add_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int add_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    UInt64 *tout    = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 + tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(add_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int add_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar + *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(add_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int add_UInt64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64 *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout = (UInt64 *) ((char *) output + outboffset);
    UInt64 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            tout = (UInt64 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval + *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            add_UInt64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(add_UInt64_accumulate, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));

static int add_UInt64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64  *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout  = (UInt64 *) ((char *) output + outboffset);
    UInt64 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            net = net + *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            add_UInt64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(add_UInt64_reduce, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));
/*********************  subtract  *********************/

static int subtract_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 - *tin2;
    }
    return 0;
}

UFUNC_DESCR3(subtract_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int subtract_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    UInt64 *tout    = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 - tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(subtract_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int subtract_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar - *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(subtract_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int subtract_UInt64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64 *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout = (UInt64 *) ((char *) output + outboffset);
    UInt64 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            tout = (UInt64 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval - *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            subtract_UInt64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(subtract_UInt64_accumulate, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));

static int subtract_UInt64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64  *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout  = (UInt64 *) ((char *) output + outboffset);
    UInt64 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            net = net - *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            subtract_UInt64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(subtract_UInt64_reduce, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/

static int multiply_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    Int64 temp;
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        temp = ((UInt64) *tin1) * ((UInt64) *tin2);
    if (umult64_overflow(*tin1, *tin2)) temp = (Float64) int_overflow_error(9223372036854775807.);
    *tout = (UInt64) temp;
    }
    return 0;
}

UFUNC_DESCR3(multiply_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int multiply_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    UInt64 *tout    = (UInt64 *) buffers[2];
    Int64 temp;
    for (i=0; i<niter; i++, tin1++, tout++) {
        temp = ((UInt64) *tin1) * ((UInt64) tscalar);
    if (umult64_overflow(*tin1, tscalar)) temp = (Float64) int_overflow_error(9223372036854775807.);
    *tout = (UInt64) temp;
    }
    return 0;
}

UFUNC_DESCR3vs(multiply_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int multiply_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    Int64 temp;
    for (i=0; i<niter; i++, tin2++, tout++) {
        temp = ((UInt64) tscalar) * ((UInt64) *tin2);
    if (umult64_overflow(tscalar, *tin2)) temp = (Float64) int_overflow_error(9223372036854775807.);
    *tout = (UInt64) temp;
    }
    return 0;
}

UFUNC_DESCR3sv(multiply_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int multiply_UInt64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64 *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout = (UInt64 *) ((char *) output + outboffset);
    UInt64 lastval;
    Int64 temp;
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            tout = (UInt64 *) ((char *) tout + outbstrides[dim]);
            temp = ((UInt64) lastval) * ((UInt64) *tin);
    if (umult64_overflow(lastval, *tin)) temp = (Float64) int_overflow_error(9223372036854775807.);
    *tout = (UInt64) temp;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            multiply_UInt64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(multiply_UInt64_accumulate, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));

static int multiply_UInt64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64  *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout  = (UInt64 *) ((char *) output + outboffset);
    UInt64 net;
    Int64 temp;
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            temp = ((UInt64) net) * ((UInt64) *tin);
    if (umult64_overflow(net, *tin)) temp = (Float64) int_overflow_error(9223372036854775807.);
    net = (UInt64) temp;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            multiply_UInt64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(multiply_UInt64_reduce, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));
/*********************  multiply  *********************/
/*********************  divide  *********************/

static int divide_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = ((*tin2==0) ? int_dividebyzero_error(*tin2, *tin1) : *tin1 / *tin2);
    }
    return 0;
}

UFUNC_DESCR3(divide_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int divide_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    UInt64 *tout    = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = ((tscalar==0) ? int_dividebyzero_error(tscalar, *tin1) : *tin1 / tscalar);
    }
    return 0;
}

UFUNC_DESCR3vs(divide_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int divide_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = ((*tin2==0) ? int_dividebyzero_error(*tin2, 0) : tscalar / *tin2);
    }
    return 0;
}

UFUNC_DESCR3sv(divide_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int divide_UInt64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64 *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout = (UInt64 *) ((char *) output + outboffset);
    UInt64 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            tout = (UInt64 *) ((char *) tout + outbstrides[dim]);
            *tout = ((*tin==0) ? int_dividebyzero_error(*tin, 0) : lastval / *tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            divide_UInt64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(divide_UInt64_accumulate, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));

static int divide_UInt64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64  *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout  = (UInt64 *) ((char *) output + outboffset);
    UInt64 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            net = ((*tin==0) ? int_dividebyzero_error(*tin, 0) : net / *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            divide_UInt64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(divide_UInt64_reduce, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));
/*********************  divide  *********************/
/*********************  floor_divide  *********************/

static int floor_divide_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = ((*tin2==0) ? int_dividebyzero_error(*tin2, *tin1) : floor(*tin1 / (double) *tin2));
    }
    return 0;
}

UFUNC_DESCR3(floor_divide_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int floor_divide_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    UInt64 *tout    = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = ((tscalar==0) ? int_dividebyzero_error(tscalar, *tin1) : floor(*tin1 / (double) tscalar));
    }
    return 0;
}

UFUNC_DESCR3vs(floor_divide_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int floor_divide_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = ((*tin2==0) ? int_dividebyzero_error(*tin2, 0) : floor(tscalar / (double) *tin2));
    }
    return 0;
}

UFUNC_DESCR3sv(floor_divide_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int floor_divide_UInt64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64 *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout = (UInt64 *) ((char *) output + outboffset);
    UInt64 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            tout = (UInt64 *) ((char *) tout + outbstrides[dim]);
            *tout = ((*tin==0) ? int_dividebyzero_error(*tin, 0) : floor(lastval / (double) *tin));
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            floor_divide_UInt64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(floor_divide_UInt64_accumulate, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));

static int floor_divide_UInt64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64  *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout  = (UInt64 *) ((char *) output + outboffset);
    UInt64 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            net = ((*tin==0) ? int_dividebyzero_error(*tin, 0) : floor(net / (double) *tin));
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            floor_divide_UInt64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(floor_divide_UInt64_reduce, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));
/*********************  floor_divide  *********************/
/*********************  true_divide  *********************/

static int true_divide_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = ((*tin2==0) ? int_dividebyzero_error(*tin2, *tin1) : *tin1 / (double) *tin2);
    }
    return 0;
}

UFUNC_DESCR3(true_divide_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Float64));


static int true_divide_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    Float64 *tout    = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = ((tscalar==0) ? int_dividebyzero_error(tscalar, *tin1) : *tin1 / (double) tscalar);
    }
    return 0;
}

UFUNC_DESCR3vs(true_divide_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(Float64));


static int true_divide_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = ((*tin2==0) ? int_dividebyzero_error(*tin2, 0) : tscalar / (double) *tin2);
    }
    return 0;
}

UFUNC_DESCR3sv(true_divide_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Float64));

/*********************  true_divide  *********************/
/*********************  remainder  *********************/

static int remainder_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = ((*tin2==0) ? int_dividebyzero_error(*tin2, *tin1) : *tin1  %  *tin2);
    }
    return 0;
}

UFUNC_DESCR3(remainder_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int remainder_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    UInt64 *tout    = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = ((tscalar==0) ? int_dividebyzero_error(tscalar, *tin1) : *tin1  %  tscalar);
    }
    return 0;
}

UFUNC_DESCR3vs(remainder_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int remainder_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = ((*tin2==0) ? int_dividebyzero_error(*tin2, 0) : tscalar  %  *tin2);
    }
    return 0;
}

UFUNC_DESCR3sv(remainder_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int remainder_UInt64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64 *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout = (UInt64 *) ((char *) output + outboffset);
    UInt64 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            tout = (UInt64 *) ((char *) tout + outbstrides[dim]);
            *tout = ((*tin==0) ? int_dividebyzero_error(*tin, 0) : lastval  %  *tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            remainder_UInt64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(remainder_UInt64_accumulate, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));

static int remainder_UInt64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64  *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout  = (UInt64 *) ((char *) output + outboffset);
    UInt64 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            net = ((*tin==0) ? int_dividebyzero_error(*tin, 0) : net  %  *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            remainder_UInt64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(remainder_UInt64_reduce, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));
/*********************  remainder  *********************/
/*********************  power  *********************/

static int power_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = num_pow(*tin1, *tin2);
    }
    return 0;
}

UFUNC_DESCR3(power_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int power_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    UInt64 *tout    = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = num_pow(*tin1, tscalar);
    }
    return 0;
}

UFUNC_DESCR3vs(power_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int power_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = num_pow(tscalar, *tin2);
    }
    return 0;
}

UFUNC_DESCR3sv(power_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int power_UInt64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64 *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout = (UInt64 *) ((char *) output + outboffset);
    UInt64 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            tout = (UInt64 *) ((char *) tout + outbstrides[dim]);
            *tout = num_pow(lastval ,*tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            power_UInt64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(power_UInt64_accumulate, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));

static int power_UInt64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64  *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout  = (UInt64 *) ((char *) output + outboffset);
    UInt64 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            net = num_pow(net, *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            power_UInt64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(power_UInt64_reduce, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));
/*********************  abs  *********************/

static int abs_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    UInt64 *tout = (UInt64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = fabs(*tin);
    }
    return 0;
}

UFUNC_DESCR2(abs_UInt64_vector, sizeof(UInt64), sizeof(UInt64));
/*********************  sin  *********************/

static int sin_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = sin(*tin);
    }
    return 0;
}

UFUNC_DESCR2(sin_UInt64_vector, sizeof(UInt64), sizeof(Float64));
/*********************  cos  *********************/

static int cos_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = cos(*tin);
    }
    return 0;
}

UFUNC_DESCR2(cos_UInt64_vector, sizeof(UInt64), sizeof(Float64));
/*********************  tan  *********************/

static int tan_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = tan(*tin);
    }
    return 0;
}

UFUNC_DESCR2(tan_UInt64_vector, sizeof(UInt64), sizeof(Float64));
/*********************  arcsin  *********************/

static int arcsin_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = asin(*tin);
    }
    return 0;
}

UFUNC_DESCR2(arcsin_UInt64_vector, sizeof(UInt64), sizeof(Float64));
/*********************  arccos  *********************/

static int arccos_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = acos(*tin);
    }
    return 0;
}

UFUNC_DESCR2(arccos_UInt64_vector, sizeof(UInt64), sizeof(Float64));
/*********************  arctan  *********************/

static int arctan_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = atan(*tin);
    }
    return 0;
}

UFUNC_DESCR2(arctan_UInt64_vector, sizeof(UInt64), sizeof(Float64));
/*********************  arctan2  *********************/

static int arctan2_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = atan2(*tin1, *tin2);
    }
    return 0;
}

UFUNC_DESCR3(arctan2_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Float64));


static int arctan2_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    Float64 *tout    = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = atan2(*tin1, tscalar);
    }
    return 0;
}

UFUNC_DESCR3vs(arctan2_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(Float64));


static int arctan2_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = atan2(tscalar, *tin2);
    }
    return 0;
}

UFUNC_DESCR3sv(arctan2_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Float64));

/*********************  log  *********************/

static int log_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_log(*tin);
    }
    return 0;
}

UFUNC_DESCR2(log_UInt64_vector, sizeof(UInt64), sizeof(Float64));
/*********************  log10  *********************/

static int log10_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_log10(*tin);
    }
    return 0;
}

UFUNC_DESCR2(log10_UInt64_vector, sizeof(UInt64), sizeof(Float64));
/*********************  exp  *********************/

static int exp_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = exp(*tin);
    }
    return 0;
}

UFUNC_DESCR2(exp_UInt64_vector, sizeof(UInt64), sizeof(Float64));
/*********************  sinh  *********************/

static int sinh_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = sinh(*tin);
    }
    return 0;
}

UFUNC_DESCR2(sinh_UInt64_vector, sizeof(UInt64), sizeof(Float64));
/*********************  cosh  *********************/

static int cosh_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = cosh(*tin);
    }
    return 0;
}

UFUNC_DESCR2(cosh_UInt64_vector, sizeof(UInt64), sizeof(Float64));
/*********************  tanh  *********************/

static int tanh_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = tanh(*tin);
    }
    return 0;
}

UFUNC_DESCR2(tanh_UInt64_vector, sizeof(UInt64), sizeof(Float64));
/*********************  arcsinh  *********************/

static int arcsinh_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_asinh(*tin);
    }
    return 0;
}

UFUNC_DESCR2(arcsinh_UInt64_vector, sizeof(UInt64), sizeof(Float64));
/*********************  arccosh  *********************/

static int arccosh_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_acosh(*tin);
    }
    return 0;
}

UFUNC_DESCR2(arccosh_UInt64_vector, sizeof(UInt64), sizeof(Float64));
/*********************  arctanh  *********************/

static int arctanh_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_atanh(*tin);
    }
    return 0;
}

UFUNC_DESCR2(arctanh_UInt64_vector, sizeof(UInt64), sizeof(Float64));
/*********************  ieeemask  *********************/
/*********************  ieeemask  *********************/
/*********************  sqrt  *********************/

static int sqrt_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = sqrt(*tin);
    }
    return 0;
}

UFUNC_DESCR2(sqrt_UInt64_vector, sizeof(UInt64), sizeof(Float64));
/*********************  equal  *********************/

static int equal_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 == *tin2;
    }
    return 0;
}

UFUNC_DESCR3(equal_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));


static int equal_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 == tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(equal_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));


static int equal_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar == *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(equal_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));

/*********************  not_equal  *********************/

static int not_equal_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 != *tin2;
    }
    return 0;
}

UFUNC_DESCR3(not_equal_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));


static int not_equal_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 != tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(not_equal_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));


static int not_equal_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar != *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(not_equal_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));

/*********************  greater  *********************/

static int greater_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 > *tin2;
    }
    return 0;
}

UFUNC_DESCR3(greater_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));


static int greater_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 > tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(greater_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));


static int greater_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar > *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(greater_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));

/*********************  greater_equal  *********************/

static int greater_equal_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 >= *tin2;
    }
    return 0;
}

UFUNC_DESCR3(greater_equal_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));


static int greater_equal_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 >= tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(greater_equal_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));


static int greater_equal_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar >= *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(greater_equal_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));

/*********************  less  *********************/

static int less_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 < *tin2;
    }
    return 0;
}

UFUNC_DESCR3(less_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));


static int less_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 < tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(less_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));


static int less_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar < *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(less_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));

/*********************  less_equal  *********************/

static int less_equal_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 <= *tin2;
    }
    return 0;
}

UFUNC_DESCR3(less_equal_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));


static int less_equal_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 <= tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(less_equal_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));


static int less_equal_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar <= *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(less_equal_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));

/*********************  logical_and  *********************/

static int logical_and_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = logical_and(*tin1, *tin2);
    }
    return 0;
}

UFUNC_DESCR3(logical_and_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));


static int logical_and_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = logical_and(*tin1, tscalar);
    }
    return 0;
}

UFUNC_DESCR3vs(logical_and_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));


static int logical_and_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = logical_and(tscalar, *tin2);
    }
    return 0;
}

UFUNC_DESCR3sv(logical_and_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));

/*********************  logical_or  *********************/

static int logical_or_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = logical_or(*tin1, *tin2);
    }
    return 0;
}

UFUNC_DESCR3(logical_or_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));


static int logical_or_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = logical_or(*tin1, tscalar);
    }
    return 0;
}

UFUNC_DESCR3vs(logical_or_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));


static int logical_or_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = logical_or(tscalar, *tin2);
    }
    return 0;
}

UFUNC_DESCR3sv(logical_or_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));

/*********************  logical_xor  *********************/

static int logical_xor_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = logical_xor(*tin1, *tin2);
    }
    return 0;
}

UFUNC_DESCR3(logical_xor_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));


static int logical_xor_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = logical_xor(*tin1, tscalar);
    }
    return 0;
}

UFUNC_DESCR3vs(logical_xor_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));


static int logical_xor_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = logical_xor(tscalar, *tin2);
    }
    return 0;
}

UFUNC_DESCR3sv(logical_xor_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Bool));

/*********************  logical_not  *********************/

static int logical_not_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    Bool *tout = (Bool *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = !*tin;
    }
    return 0;
}

UFUNC_DESCR2(logical_not_UInt64_vector, sizeof(UInt64), sizeof(Bool));
/*********************  bitwise_and  *********************/

static int bitwise_and_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 & *tin2;
    }
    return 0;
}

UFUNC_DESCR3(bitwise_and_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int bitwise_and_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    UInt64 *tout    = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 & tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(bitwise_and_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int bitwise_and_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar & *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(bitwise_and_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int bitwise_and_UInt64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64 *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout = (UInt64 *) ((char *) output + outboffset);
    UInt64 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            tout = (UInt64 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval & *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            bitwise_and_UInt64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(bitwise_and_UInt64_accumulate, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));

static int bitwise_and_UInt64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64  *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout  = (UInt64 *) ((char *) output + outboffset);
    UInt64 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            net = net & *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            bitwise_and_UInt64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(bitwise_and_UInt64_reduce, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));
/*********************  bitwise_or  *********************/

static int bitwise_or_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 | *tin2;
    }
    return 0;
}

UFUNC_DESCR3(bitwise_or_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int bitwise_or_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    UInt64 *tout    = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 | tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(bitwise_or_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int bitwise_or_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar | *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(bitwise_or_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int bitwise_or_UInt64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64 *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout = (UInt64 *) ((char *) output + outboffset);
    UInt64 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            tout = (UInt64 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval | *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            bitwise_or_UInt64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(bitwise_or_UInt64_accumulate, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));

static int bitwise_or_UInt64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64  *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout  = (UInt64 *) ((char *) output + outboffset);
    UInt64 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            net = net | *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            bitwise_or_UInt64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(bitwise_or_UInt64_reduce, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));
/*********************  bitwise_xor  *********************/

static int bitwise_xor_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 ^ *tin2;
    }
    return 0;
}

UFUNC_DESCR3(bitwise_xor_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int bitwise_xor_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    UInt64 *tout    = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 ^ tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(bitwise_xor_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int bitwise_xor_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar ^ *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(bitwise_xor_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int bitwise_xor_UInt64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64 *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout = (UInt64 *) ((char *) output + outboffset);
    UInt64 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            tout = (UInt64 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval ^ *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            bitwise_xor_UInt64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(bitwise_xor_UInt64_accumulate, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));

static int bitwise_xor_UInt64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64  *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout  = (UInt64 *) ((char *) output + outboffset);
    UInt64 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            net = net ^ *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            bitwise_xor_UInt64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(bitwise_xor_UInt64_reduce, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));
/*********************  bitwise_not  *********************/

static int bitwise_not_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    UInt64 *tout = (UInt64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = ~*tin;
    }
    return 0;
}

UFUNC_DESCR2(bitwise_not_UInt64_vector, sizeof(UInt64), sizeof(UInt64));
/*********************  rshift  *********************/

static int rshift_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 >> *tin2;
    }
    return 0;
}

UFUNC_DESCR3(rshift_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int rshift_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    UInt64 *tout    = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 >> tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(rshift_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int rshift_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar >> *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(rshift_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int rshift_UInt64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64 *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout = (UInt64 *) ((char *) output + outboffset);
    UInt64 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            tout = (UInt64 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval >> *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            rshift_UInt64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(rshift_UInt64_accumulate, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));

static int rshift_UInt64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64  *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout  = (UInt64 *) ((char *) output + outboffset);
    UInt64 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            net = net >> *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            rshift_UInt64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(rshift_UInt64_reduce, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));
/*********************  lshift  *********************/

static int lshift_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 << *tin2;
    }
    return 0;
}

UFUNC_DESCR3(lshift_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int lshift_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    UInt64 *tout    = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 << tscalar;
    }
    return 0;
}

UFUNC_DESCR3vs(lshift_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int lshift_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar << *tin2;
    }
    return 0;
}

UFUNC_DESCR3sv(lshift_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int lshift_UInt64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64 *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout = (UInt64 *) ((char *) output + outboffset);
    UInt64 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            tout = (UInt64 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval << *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            lshift_UInt64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(lshift_UInt64_accumulate, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));

static int lshift_UInt64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64  *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout  = (UInt64 *) ((char *) output + outboffset);
    UInt64 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            net = net << *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            lshift_UInt64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(lshift_UInt64_reduce, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));
/*********************  floor  *********************/

static int floor_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    UInt64 *tout = (UInt64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = (*tin);
    }
    return 0;
}

UFUNC_DESCR2(floor_UInt64_vector, sizeof(UInt64), sizeof(UInt64));
/*********************  floor  *********************/
/*********************  ceil  *********************/

static int ceil_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    UInt64 *tout = (UInt64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = (*tin);
    }
    return 0;
}

UFUNC_DESCR2(ceil_UInt64_vector, sizeof(UInt64), sizeof(UInt64));
/*********************  ceil  *********************/
/*********************  maximum  *********************/

static int maximum_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    UInt64 temp1, temp2;
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = ufmaximum(*tin1, *tin2);
    }
    return 0;
}

UFUNC_DESCR3(maximum_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int maximum_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    UInt64 *tout    = (UInt64 *) buffers[2];
    UInt64 temp1, temp2;
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = ufmaximum(*tin1, tscalar);
    }
    return 0;
}

UFUNC_DESCR3vs(maximum_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int maximum_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    UInt64 temp1, temp2;
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = ufmaximum(tscalar, *tin2);
    }
    return 0;
}

UFUNC_DESCR3sv(maximum_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int maximum_UInt64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64 *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout = (UInt64 *) ((char *) output + outboffset);
    UInt64 lastval;
    UInt64 temp1, temp2;
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            tout = (UInt64 *) ((char *) tout + outbstrides[dim]);
            *tout = ufmaximum(lastval ,*tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            maximum_UInt64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(maximum_UInt64_accumulate, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));

static int maximum_UInt64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64  *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout  = (UInt64 *) ((char *) output + outboffset);
    UInt64 net;
    UInt64 temp1, temp2;
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            net = ufmaximum(net, *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            maximum_UInt64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(maximum_UInt64_reduce, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));
/*********************  minimum  *********************/

static int minimum_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    UInt64 temp1, temp2;
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = ufminimum(*tin1, *tin2);
    }
    return 0;
}

UFUNC_DESCR3(minimum_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int minimum_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    UInt64 *tout    = (UInt64 *) buffers[2];
    UInt64 temp1, temp2;
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = ufminimum(*tin1, tscalar);
    }
    return 0;
}

UFUNC_DESCR3vs(minimum_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int minimum_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    UInt64 *tout  = (UInt64 *) buffers[2];
    UInt64 temp1, temp2;
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = ufminimum(tscalar, *tin2);
    }
    return 0;
}

UFUNC_DESCR3sv(minimum_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(UInt64));


static int minimum_UInt64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64 *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout = (UInt64 *) ((char *) output + outboffset);
    UInt64 lastval;
    UInt64 temp1, temp2;
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            tout = (UInt64 *) ((char *) tout + outbstrides[dim]);
            *tout = ufminimum(lastval ,*tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            minimum_UInt64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(minimum_UInt64_accumulate, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));

static int minimum_UInt64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    UInt64  *tin   = (UInt64 *) ((char *) input  + inboffset);
    UInt64 *tout  = (UInt64 *) ((char *) output + outboffset);
    UInt64 net;
    UInt64 temp1, temp2;
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (UInt64 *) ((char *) tin + inbstrides[dim]);
            net = ufminimum(net, *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            minimum_UInt64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(minimum_UInt64_reduce, CHECK_ALIGN, sizeof(UInt64), sizeof(UInt64));
/*********************  fabs  *********************/

static int fabs_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = fabs(*tin);
    }
    return 0;
}

UFUNC_DESCR2(fabs_UInt64_vector, sizeof(UInt64), sizeof(Float64));
/*********************  _round  *********************/

static int _round_UInt64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin = (UInt64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_round(*tin);
    }
    return 0;
}

UFUNC_DESCR2(_round_UInt64_vector, sizeof(UInt64), sizeof(Float64));
/*********************  hypot  *********************/

static int hypot_UInt64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64  *tin1  = (UInt64 *) buffers[0];
    UInt64  *tin2  = (UInt64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = hypot(*tin1, *tin2);
    }
    return 0;
}

UFUNC_DESCR3(hypot_UInt64_vector_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Float64));


static int hypot_UInt64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 *tin1     = (UInt64 *) buffers[0];
    UInt64 tscalar   = *(UInt64 *) buffers[1];
    Float64 *tout    = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = hypot(*tin1, tscalar);
    }
    return 0;
}

UFUNC_DESCR3vs(hypot_UInt64_vector_scalar, sizeof(UInt64), sizeof(UInt64), sizeof(Float64));


static int hypot_UInt64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt64 tscalar = *(UInt64 *) buffers[0];
    UInt64 *tin2   =  (UInt64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = hypot(tscalar, *tin2);
    }
    return 0;
}

UFUNC_DESCR3sv(hypot_UInt64_scalar_vector, sizeof(UInt64), sizeof(UInt64), sizeof(Float64));

/*********************  minus  *********************/
/*********************  add  *********************/
/*********************  subtract  *********************/
/*********************  multiply  *********************/
/*********************  divide  *********************/
/*********************  true_divide  *********************/
/*********************  remainder  *********************/
/*********************  power  *********************/
/*********************  abs  *********************/
/*********************  sin  *********************/
/*********************  cos  *********************/
/*********************  tan  *********************/
/*********************  arcsin  *********************/
/*********************  arccos  *********************/
/*********************  arctan  *********************/
/*********************  arcsinh  *********************/
/*********************  arccosh  *********************/
/*********************  arctanh  *********************/
/*********************  log  *********************/
/*********************  log10  *********************/
/*********************  exp  *********************/
/*********************  sinh  *********************/
/*********************  cosh  *********************/
/*********************  tanh  *********************/
/*********************  sqrt  *********************/
/*********************  equal  *********************/
/*********************  not_equal  *********************/
/*********************  logical_and  *********************/
/*********************  logical_or  *********************/
/*********************  logical_xor  *********************/
/*********************  logical_not  *********************/
/*********************  floor  *********************/
/*********************  ceil  *********************/
/*********************  maximum  *********************/
/*********************  minimum  *********************/
/*********************  fabs  *********************/
/*********************  _round  *********************/
/*********************  hypot  *********************/

static PyMethodDef _ufuncUInt64Methods[] = {

	{NULL,      NULL}        /* Sentinel */
};

static PyObject *init_funcDict(void) {
    PyObject *dict;
    dict = PyDict_New();
    NA_add_cfunc(dict, "('minus', 'vector', (('UInt64',), ('UInt64',)))", (void *) &minus_UInt64_vector_descr);
    NA_add_cfunc(dict, "('add', 'vector_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &add_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('add', 'vector_scalar', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &add_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('add', 'scalar_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &add_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('add', 'reduce', (('UInt64',), ('UInt64',)))", (void *) &add_UInt64_reduce_descr);
    NA_add_cfunc(dict, "('add', 'accumulate', (('UInt64',), ('UInt64',)))", (void *) &add_UInt64_accumulate_descr);
    NA_add_cfunc(dict, "('subtract', 'vector_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &subtract_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('subtract', 'vector_scalar', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &subtract_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('subtract', 'scalar_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &subtract_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('subtract', 'reduce', (('UInt64',), ('UInt64',)))", (void *) &subtract_UInt64_reduce_descr);
    NA_add_cfunc(dict, "('subtract', 'accumulate', (('UInt64',), ('UInt64',)))", (void *) &subtract_UInt64_accumulate_descr);
    NA_add_cfunc(dict, "('multiply', 'vector_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &multiply_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('multiply', 'vector_scalar', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &multiply_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('multiply', 'scalar_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &multiply_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('multiply', 'reduce', (('UInt64',), ('UInt64',)))", (void *) &multiply_UInt64_reduce_descr);
    NA_add_cfunc(dict, "('multiply', 'accumulate', (('UInt64',), ('UInt64',)))", (void *) &multiply_UInt64_accumulate_descr);
    NA_add_cfunc(dict, "('divide', 'vector_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &divide_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('divide', 'vector_scalar', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &divide_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('divide', 'scalar_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &divide_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('divide', 'reduce', (('UInt64',), ('UInt64',)))", (void *) &divide_UInt64_reduce_descr);
    NA_add_cfunc(dict, "('divide', 'accumulate', (('UInt64',), ('UInt64',)))", (void *) &divide_UInt64_accumulate_descr);
    NA_add_cfunc(dict, "('floor_divide', 'vector_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &floor_divide_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('floor_divide', 'vector_scalar', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &floor_divide_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('floor_divide', 'scalar_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &floor_divide_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('floor_divide', 'reduce', (('UInt64',), ('UInt64',)))", (void *) &floor_divide_UInt64_reduce_descr);
    NA_add_cfunc(dict, "('floor_divide', 'accumulate', (('UInt64',), ('UInt64',)))", (void *) &floor_divide_UInt64_accumulate_descr);
    NA_add_cfunc(dict, "('true_divide', 'vector_vector', (('UInt64', 'UInt64'), ('Float64',)))", (void *) &true_divide_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('true_divide', 'vector_scalar', (('UInt64', 'UInt64'), ('Float64',)))", (void *) &true_divide_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('true_divide', 'scalar_vector', (('UInt64', 'UInt64'), ('Float64',)))", (void *) &true_divide_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('remainder', 'vector_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &remainder_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('remainder', 'vector_scalar', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &remainder_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('remainder', 'scalar_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &remainder_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('remainder', 'reduce', (('UInt64',), ('UInt64',)))", (void *) &remainder_UInt64_reduce_descr);
    NA_add_cfunc(dict, "('remainder', 'accumulate', (('UInt64',), ('UInt64',)))", (void *) &remainder_UInt64_accumulate_descr);
    NA_add_cfunc(dict, "('power', 'vector_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &power_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('power', 'vector_scalar', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &power_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('power', 'scalar_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &power_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('power', 'reduce', (('UInt64',), ('UInt64',)))", (void *) &power_UInt64_reduce_descr);
    NA_add_cfunc(dict, "('power', 'accumulate', (('UInt64',), ('UInt64',)))", (void *) &power_UInt64_accumulate_descr);
    NA_add_cfunc(dict, "('abs', 'vector', (('UInt64',), ('UInt64',)))", (void *) &abs_UInt64_vector_descr);
    NA_add_cfunc(dict, "('sin', 'vector', (('UInt64',), ('Float64',)))", (void *) &sin_UInt64_vector_descr);
    NA_add_cfunc(dict, "('cos', 'vector', (('UInt64',), ('Float64',)))", (void *) &cos_UInt64_vector_descr);
    NA_add_cfunc(dict, "('tan', 'vector', (('UInt64',), ('Float64',)))", (void *) &tan_UInt64_vector_descr);
    NA_add_cfunc(dict, "('arcsin', 'vector', (('UInt64',), ('Float64',)))", (void *) &arcsin_UInt64_vector_descr);
    NA_add_cfunc(dict, "('arccos', 'vector', (('UInt64',), ('Float64',)))", (void *) &arccos_UInt64_vector_descr);
    NA_add_cfunc(dict, "('arctan', 'vector', (('UInt64',), ('Float64',)))", (void *) &arctan_UInt64_vector_descr);
    NA_add_cfunc(dict, "('arctan2', 'vector_vector', (('UInt64', 'UInt64'), ('Float64',)))", (void *) &arctan2_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('arctan2', 'vector_scalar', (('UInt64', 'UInt64'), ('Float64',)))", (void *) &arctan2_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('arctan2', 'scalar_vector', (('UInt64', 'UInt64'), ('Float64',)))", (void *) &arctan2_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('log', 'vector', (('UInt64',), ('Float64',)))", (void *) &log_UInt64_vector_descr);
    NA_add_cfunc(dict, "('log10', 'vector', (('UInt64',), ('Float64',)))", (void *) &log10_UInt64_vector_descr);
    NA_add_cfunc(dict, "('exp', 'vector', (('UInt64',), ('Float64',)))", (void *) &exp_UInt64_vector_descr);
    NA_add_cfunc(dict, "('sinh', 'vector', (('UInt64',), ('Float64',)))", (void *) &sinh_UInt64_vector_descr);
    NA_add_cfunc(dict, "('cosh', 'vector', (('UInt64',), ('Float64',)))", (void *) &cosh_UInt64_vector_descr);
    NA_add_cfunc(dict, "('tanh', 'vector', (('UInt64',), ('Float64',)))", (void *) &tanh_UInt64_vector_descr);
    NA_add_cfunc(dict, "('arcsinh', 'vector', (('UInt64',), ('Float64',)))", (void *) &arcsinh_UInt64_vector_descr);
    NA_add_cfunc(dict, "('arccosh', 'vector', (('UInt64',), ('Float64',)))", (void *) &arccosh_UInt64_vector_descr);
    NA_add_cfunc(dict, "('arctanh', 'vector', (('UInt64',), ('Float64',)))", (void *) &arctanh_UInt64_vector_descr);
    NA_add_cfunc(dict, "('sqrt', 'vector', (('UInt64',), ('Float64',)))", (void *) &sqrt_UInt64_vector_descr);
    NA_add_cfunc(dict, "('equal', 'vector_vector', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &equal_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('equal', 'vector_scalar', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &equal_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('equal', 'scalar_vector', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &equal_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('not_equal', 'vector_vector', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &not_equal_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('not_equal', 'vector_scalar', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &not_equal_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('not_equal', 'scalar_vector', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &not_equal_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('greater', 'vector_vector', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &greater_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('greater', 'vector_scalar', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &greater_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('greater', 'scalar_vector', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &greater_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('greater_equal', 'vector_vector', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &greater_equal_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('greater_equal', 'vector_scalar', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &greater_equal_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('greater_equal', 'scalar_vector', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &greater_equal_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('less', 'vector_vector', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &less_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('less', 'vector_scalar', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &less_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('less', 'scalar_vector', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &less_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('less_equal', 'vector_vector', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &less_equal_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('less_equal', 'vector_scalar', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &less_equal_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('less_equal', 'scalar_vector', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &less_equal_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('logical_and', 'vector_vector', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &logical_and_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('logical_and', 'vector_scalar', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &logical_and_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('logical_and', 'scalar_vector', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &logical_and_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('logical_or', 'vector_vector', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &logical_or_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('logical_or', 'vector_scalar', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &logical_or_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('logical_or', 'scalar_vector', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &logical_or_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('logical_xor', 'vector_vector', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &logical_xor_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('logical_xor', 'vector_scalar', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &logical_xor_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('logical_xor', 'scalar_vector', (('UInt64', 'UInt64'), ('Bool',)))", (void *) &logical_xor_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('logical_not', 'vector', (('UInt64',), ('Bool',)))", (void *) &logical_not_UInt64_vector_descr);
    NA_add_cfunc(dict, "('bitwise_and', 'vector_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &bitwise_and_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('bitwise_and', 'vector_scalar', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &bitwise_and_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('bitwise_and', 'scalar_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &bitwise_and_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('bitwise_and', 'reduce', (('UInt64',), ('UInt64',)))", (void *) &bitwise_and_UInt64_reduce_descr);
    NA_add_cfunc(dict, "('bitwise_and', 'accumulate', (('UInt64',), ('UInt64',)))", (void *) &bitwise_and_UInt64_accumulate_descr);
    NA_add_cfunc(dict, "('bitwise_or', 'vector_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &bitwise_or_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('bitwise_or', 'vector_scalar', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &bitwise_or_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('bitwise_or', 'scalar_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &bitwise_or_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('bitwise_or', 'reduce', (('UInt64',), ('UInt64',)))", (void *) &bitwise_or_UInt64_reduce_descr);
    NA_add_cfunc(dict, "('bitwise_or', 'accumulate', (('UInt64',), ('UInt64',)))", (void *) &bitwise_or_UInt64_accumulate_descr);
    NA_add_cfunc(dict, "('bitwise_xor', 'vector_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &bitwise_xor_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('bitwise_xor', 'vector_scalar', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &bitwise_xor_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('bitwise_xor', 'scalar_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &bitwise_xor_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('bitwise_xor', 'reduce', (('UInt64',), ('UInt64',)))", (void *) &bitwise_xor_UInt64_reduce_descr);
    NA_add_cfunc(dict, "('bitwise_xor', 'accumulate', (('UInt64',), ('UInt64',)))", (void *) &bitwise_xor_UInt64_accumulate_descr);
    NA_add_cfunc(dict, "('bitwise_not', 'vector', (('UInt64',), ('UInt64',)))", (void *) &bitwise_not_UInt64_vector_descr);
    NA_add_cfunc(dict, "('rshift', 'vector_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &rshift_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('rshift', 'vector_scalar', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &rshift_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('rshift', 'scalar_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &rshift_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('rshift', 'reduce', (('UInt64',), ('UInt64',)))", (void *) &rshift_UInt64_reduce_descr);
    NA_add_cfunc(dict, "('rshift', 'accumulate', (('UInt64',), ('UInt64',)))", (void *) &rshift_UInt64_accumulate_descr);
    NA_add_cfunc(dict, "('lshift', 'vector_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &lshift_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('lshift', 'vector_scalar', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &lshift_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('lshift', 'scalar_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &lshift_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('lshift', 'reduce', (('UInt64',), ('UInt64',)))", (void *) &lshift_UInt64_reduce_descr);
    NA_add_cfunc(dict, "('lshift', 'accumulate', (('UInt64',), ('UInt64',)))", (void *) &lshift_UInt64_accumulate_descr);
    NA_add_cfunc(dict, "('floor', 'vector', (('UInt64',), ('UInt64',)))", (void *) &floor_UInt64_vector_descr);
    NA_add_cfunc(dict, "('ceil', 'vector', (('UInt64',), ('UInt64',)))", (void *) &ceil_UInt64_vector_descr);
    NA_add_cfunc(dict, "('maximum', 'vector_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &maximum_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('maximum', 'vector_scalar', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &maximum_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('maximum', 'scalar_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &maximum_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('maximum', 'reduce', (('UInt64',), ('UInt64',)))", (void *) &maximum_UInt64_reduce_descr);
    NA_add_cfunc(dict, "('maximum', 'accumulate', (('UInt64',), ('UInt64',)))", (void *) &maximum_UInt64_accumulate_descr);
    NA_add_cfunc(dict, "('minimum', 'vector_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &minimum_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('minimum', 'vector_scalar', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &minimum_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('minimum', 'scalar_vector', (('UInt64', 'UInt64'), ('UInt64',)))", (void *) &minimum_UInt64_scalar_vector_descr);
    NA_add_cfunc(dict, "('minimum', 'reduce', (('UInt64',), ('UInt64',)))", (void *) &minimum_UInt64_reduce_descr);
    NA_add_cfunc(dict, "('minimum', 'accumulate', (('UInt64',), ('UInt64',)))", (void *) &minimum_UInt64_accumulate_descr);
    NA_add_cfunc(dict, "('fabs', 'vector', (('UInt64',), ('Float64',)))", (void *) &fabs_UInt64_vector_descr);
    NA_add_cfunc(dict, "('_round', 'vector', (('UInt64',), ('Float64',)))", (void *) &_round_UInt64_vector_descr);
    NA_add_cfunc(dict, "('hypot', 'vector_vector', (('UInt64', 'UInt64'), ('Float64',)))", (void *) &hypot_UInt64_vector_vector_descr);
    NA_add_cfunc(dict, "('hypot', 'vector_scalar', (('UInt64', 'UInt64'), ('Float64',)))", (void *) &hypot_UInt64_vector_scalar_descr);
    NA_add_cfunc(dict, "('hypot', 'scalar_vector', (('UInt64', 'UInt64'), ('Float64',)))", (void *) &hypot_UInt64_scalar_vector_descr);
    return dict;
}

/* platform independent*/
#ifdef MS_WIN32
__declspec(dllexport)
#endif
void init_ufuncUInt64(void) {
    PyObject *m, *d;
    m = Py_InitModule("_ufuncUInt64", _ufuncUInt64Methods);
    d = PyModule_GetDict(m);
    _Error = PyErr_NewException("_ufuncUInt64.error", NULL, NULL);
    PyDict_SetItemString(d, "error", _Error);
    import_libnumarray();
    PyDict_SetItemString(d, "functionDict", init_funcDict());
}
