#include <glib.h>
#include <string.h>
#include "entity.h"


ENode *enode_root = NULL;

/* Number of nodes to allocate at a time */
#define ENODE_ALLOC_NUM 50
static EMemChunk *enode_chunk_admin = NULL;


void
enode_free (ENode * node)
{
    ECHECK_RET (node != NULL);

    if (node->element)
	ebuf_free (node->element);
    if (node->children)
	g_slist_free (node->children);
    if (node->data)
	ebuf_free (node->data);
    if (node->entity_data)
	g_hash_table_destroy (node->entity_data);
    if (node->attribs) {
	GSList *tmp;
	tmp = node->attribs;
	while (tmp) {
	    EBuf *buf = tmp->data;
	    ebuf_free (buf);
	    tmp = tmp->next;
	}

	g_slist_free (node->attribs);
    }

    eutils_memchunk_free (enode_chunk_admin, node);
}

static ENode *
enode_alloc (EBuf * type)
{
    ENode *node;

    ECHECK_RETVAL (type != NULL, NULL);

    node = eutils_memchunk_alloc (enode_chunk_admin);
    node->refcount = 1;
    node->attribs = NULL;
    node->event_watchers = NULL;
    node->element = type;

    /* Special case for norender tag.  This sets the flag to * not render
     * children of this node */
    if (ebuf_equal_str (node->element, "norender")) {
	node->flags |= ENODE_NO_RENDER_CHILDREN;
    }

    return (node);
}

static EBuf *
enode_assigned_name (void)
{
    EBuf *name;
    gchar name_buf[128];
    static gint unique_enode_id = 0;

    g_snprintf (name_buf, 128, "_%d", unique_enode_id++);

    name = ebuf_new_with_str (name_buf);

    return (name);
}

ENode *
enode_root_node (void)
{
    return (enode_root);
}

/* Initialize enode interface */
void
enode_init (void)
{
    ENode *node;
    static gint initialized = FALSE;

    if (!initialized) {
	enode_chunk_admin = eutils_memchunk_admin_new (sizeof (ENode),
						       ENODE_ALLOC_NUM);

	node = enode_alloc (ebuf_new_with_str ("root"));
	ENODE_SET_FLAG (node, ENODE_RENDERED);
	enode_root = node;

	initialized = TRUE;
    } else {
	g_warning ("Icky!  enode_init() called twice ?!");
    }
}

ENode *
enode_new_child_norender (ENode * node, EBuf * type, GSList * attribs)
{
    ENode *child;
    EBuf *name;

    child = enode_alloc (type);
    child->parent = node;

    node->children =
	g_slist_append_tail (node->children, child, &node->children_tail);

    child->attribs = attribs;
    if (child->attribs)
	child->attribs_tail = g_slist_last (child->attribs);

    name = enode_attrib (child, "name", NULL);

    if (ebuf_empty (name))
	enode_attrib (child, "name", enode_assigned_name ());

    return (child);
}

/* Helper function for enode_new_child () to split apart the basename and the 
 * name.  Return is the type and name */
static void
enode_name_and_type (gchar * basename, EBuf ** type, EBuf ** name)
{
    gchar *dot;
    gint typelen;

    *name = NULL;

    dot = strstr (basename, ".");
    if (dot)
	typelen = dot - basename;
    else
	typelen = strlen (basename);

    *type = ebuf_new_with_data (basename, typelen);

    if (dot) {
	*name = ebuf_new_with_str (&basename[typelen + 1]);
    } else {
	*name = NULL;
    }
}


/* Create new child node */
ENode *
enode_new_child (ENode * node, gchar * basename, GSList * attribs)
{
    ENode *child;
    EBuf *type;
    EBuf *name;

    ECHECK_RETVAL (node != NULL, NULL);
    ECHECK_RETVAL (basename != NULL, NULL);

    /* Split apart type and name */
    enode_name_and_type (basename, &type, &name);

    /* Special case for <object>'s, if it's a new object, we wrap it in an
     * <instance> tag to avoid duplicate naming in the same parent */
    if (ebuf_equal_str (type, "object")) {
	node = enode_new_child (node, "instance", NULL);
	ENODE_SET_FLAG (node, ENODE_INSTANCE_PLACEHOLDER);
    }

    child = enode_new_child_norender (node, type, attribs);

    if (name) {
	enode_attrib (child, "name", name);
    }

    enode_event_render (child);
    enode_event_parent (node, child);
    return (child);
}


/* Returns the element type */
EBufConst *
enode_type (ENode * node)
{
    ECHECK_RETVAL (node != NULL, NULL);
    return (node->element);
}


/* Returns the path to a node */
EBufFreeMe *
enode_path (ENode * node)
{
    ENode *tmp_node;
    GSList *node_path = NULL;
    GSList *tmp;
    EBuf *path;

    ECHECK_RETVAL (node != NULL, NULL);

    path = ebuf_new ();

    if (node == enode_root_node ()) {
	ebuf_append_char (path, '/');
	return (path);
    }

    /* first we build a list of nodes that follow from one * parent to the
     * next */
    tmp_node = node;
    while (tmp_node) {
	/* don't append the root node */
	if (tmp_node->parent)
	    node_path = g_slist_prepend (node_path, tmp_node);
	tmp_node = enode_parent (tmp_node, NULL);
    }

    /* Now iterate through them in the opposite direction (starting at the
     * root), and build the path */
    tmp = node_path;
    while (tmp) {
	EBuf *basename;
	ENode *curnode = tmp->data;

	ebuf_append_char (path, '/');
	basename = enode_basename (curnode);
	ebuf_append_ebuf (path, basename);
	ebuf_free (basename);

	tmp = tmp->next;
    }

    g_slist_free (node_path);

    return (path);
}

/* Return type.name for node - Allocates new ebuf */
EBufFreeMe *
enode_basename (ENode * node)
{
    EBuf *buf;

    ECHECK_RETVAL (node != NULL, NULL);

    buf = ebuf_new_with_ebuf (enode_type (node));
    ebuf_append_char (buf, '.');
    ebuf_append_ebuf (buf, enode_attrib (node, "name", NULL));

    return (buf);
}


/* Return a description of this node type */
gchar *
enode_description (ENode * node)
{
    Element *element;

    ECHECK_RETVAL (node != NULL, NULL);

    element = element_lookup_element (node->element);

    if (element)
	return (element->description);
    else
	return (NULL);
}


