#include <glib.h>
#include "entity.h"

#define INDENT_STRING "  "
#define ATTRIBUTE_WRAP_LENGTH 50
#define ATTRIBUTE_INDENTATION " "


static void
enode_xml_entity_encode (EBuf * src, EBuf * dst, gint encode_whitespace)
{
    guchar *str;
    guchar buf[20];
    gint i;

    if (!src)
	return;

    str = src->str;

    for (i = 0; i < src->len; i++) {
	if (str[i] == '<') {
	    ebuf_append_str (dst, "&lt;");
	} else if (str[i] == '>') {
	    ebuf_append_str (dst, "&gt;");
	} else if (str[i] == '&') {
	    ebuf_append_str (dst, "&amp;");
	} else if (str[i] == '"') {
	    ebuf_append_str (dst, "&quot;");
	} else if (str[i] == '\'') {
	    ebuf_append_str (dst, "&apos;");
	} else if (!encode_whitespace &&
		   ((str[i] == '\n') || (str[i] == '\r') || (str[i] == '\t'))) {
	    ebuf_append_char (dst, str[i]);
	} else if ((str[i] >= 0x20) && (str[i] < 0x80)) {
	    ebuf_append_char (dst, str[i]);
	} else {
	    g_snprintf (buf, 20, "&#%d;", str[i]);
	    ebuf_append_str (dst, buf);
	}
    }
}

static void
enode_xml_stream_indent (EBuf * xmlstream, gint level)
{
    gint i;

    for (i = 0; i < level; i++) {
	ebuf_append_str (xmlstream, INDENT_STRING);
    }
}

static gint
data_ignorable (ENode * node)
{
    if (!node || !node->data)
	return (1);
    return (ebuf_is_whitespace (node->data));
}


static void
enode_xml_stream_start_node (ENode * curnode, EBuf * xmlstream, gint level)
{
    EBuf *data;
    GSList *tmp;
    gint linelen = 0;

    enode_xml_stream_indent (xmlstream, level);
    ebuf_append_char (xmlstream, '<');
    ebuf_append_ebuf (xmlstream, curnode->element);

    tmp = curnode->attribs;

    while (tmp) {
	EBuf *attr;
	EBuf *value;

	attr = tmp->data;
	tmp = tmp->next;
	value = tmp->data;
	tmp = tmp->next;

	/* do not print if attribute or value are NULL, or value has no
	 * length. * Or, if attribute starts with '_', or it's a name
	 * attribute that has a value * that starts with '_' (reserved for
	 * unique, system-given names) */
	if (!attr || !value || !value->len ||
	    attr->str[0] == '_' ||
	    (ebuf_equal_str (attr, "name") && value->str[0] == '_'))
	    continue;

	linelen += attr->len + value->len;
	if (linelen > ATTRIBUTE_WRAP_LENGTH) {
	    ebuf_append_char (xmlstream, '\n');
	    enode_xml_stream_indent (xmlstream, level);
	    ebuf_append_str (xmlstream, ATTRIBUTE_INDENTATION);
	    linelen = 0;
	} else {
	    ebuf_append_char (xmlstream, ' ');
	}

	ebuf_append_ebuf (xmlstream, attr);
	ebuf_append_str (xmlstream, "=\"");
	/* Append value as entity encoded */
	enode_xml_entity_encode (value, xmlstream, TRUE);
	ebuf_append_char (xmlstream, '\"');
    }

    data = enode_get_data (curnode);

    if (!data_ignorable (curnode) || curnode->children)
	ebuf_append_str (xmlstream, ">\n");
    else
	ebuf_append_str (xmlstream, "/>\n");

    if (!data_ignorable (curnode)) {
	enode_xml_stream_indent (xmlstream, level);

	ebuf_append_str (xmlstream, "<![CDATA[");
	ebuf_append_ebuf (xmlstream, data);
	ebuf_append_str (xmlstream, "]]>\n");
    }
}

static void
enode_xml_stream_end_node (ENode * curnode, EBuf * xmlstream, gint level)
{
    if (!data_ignorable (curnode) || curnode->children) {
	enode_xml_stream_indent (xmlstream, level);
	ebuf_append_str (xmlstream, "</");
	ebuf_append_ebuf (xmlstream, curnode->element);
	ebuf_append_str (xmlstream, ">\n");
    }
}


/* Return XML for your children */
static void
enode_xml_append_child_xml (ENode * topnode, EBuf * xmlstream, gint level)
{
    ENode *curnode;
    ENode *parentnode;
    GQueue *q;
    GQueue *cq;
    GSList *child;

    if (!topnode)
	topnode = enode_root_node ();

    q = g_queue_create ();
    cq = g_queue_create ();

    g_queue_push_tail (q, NULL);

    /* now that we have the node, we have to keep walking through until we
     * render everything. */
    curnode = topnode;

    child = curnode->children;
    g_queue_push_tail (cq, child);

    while (TRUE) {
	while (child) {
	    /* save the current parent */
	    g_queue_push_tail (q, curnode);
	    /* save the current child */
	    g_queue_push_tail (cq, child);

	    /* delve into next child */
	    curnode = child->data;

	    if (!ENODE_FLAG_ISSET (curnode, ENODE_INSTANCE_PLACEHOLDER)) {
		enode_xml_stream_start_node (curnode, xmlstream, level);
		level++;
	    }
	    /* find first child and descend */
	    child = curnode->children;
	}

	/* when we end up here, it means we've descended all the way down a
	 * fork. Now we climb back out and set it to the next child if it's
	 * found */
	parentnode = g_queue_pop_tail (q);

	/* if we've returned back beyond the top, we've completed the run. */
	if (parentnode == NULL) {
	    g_queue_free (q);
	    g_queue_free (cq);
	    return;
	}

	if (!ENODE_FLAG_ISSET (curnode, ENODE_INSTANCE_PLACEHOLDER)) {
	    level--;
	    enode_xml_stream_end_node (curnode, xmlstream, level);
	}

	/* find the next child in the parents list */
	child = g_queue_pop_tail (cq);
	if (child)
	    child = child->next;

	curnode = parentnode;
    }
}


/* Return XML for yourself and all children */
EBufFreeMe *
enode_get_xml (ENode * topnode)
{
    EBuf *xmlstream;

    ECHECK_RETVAL (topnode != NULL, NULL);

    xmlstream = ebuf_new_sized (2048);

    enode_xml_stream_start_node (topnode, xmlstream, 0);
    enode_xml_append_child_xml (topnode, xmlstream, 1);
    enode_xml_stream_end_node (topnode, xmlstream, 0);

    return (xmlstream);
}

/* Return XML for your children */
EBufFreeMe *
enode_get_child_xml (ENode * node)
{
    EBuf *xmlstream;

    ECHECK_RETVAL (node != NULL, NULL);

    xmlstream = ebuf_new_sized (2048);
    enode_xml_append_child_xml (node, xmlstream, 0);
    return (xmlstream);
}

/* Append xml to children of yourself.  This will instantiate * the ENodes
 * into the tree */
void
enode_append_xml (ENode * node, EBuf * xml)
{
    ECHECK_RET (node != NULL);
    ECHECK_RET (xml != NULL);

    xml_parse_string (node, xml);
}


