#include <glib.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>

#include "entity.h"
#include "modulegen.h"
#include "entity-exec-class.h"

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif				/* HAVE_GETOPT_H */

static void
print_usage (void)
{
    g_print ("Usage: %s  [OPTION]... [FILE]\n"
	     "  -V           Print version and exit.\n"
	     "  -g           Turn on debug messages. "
	     "Accepts domain as argument, or 'all'\n"
	     "  -a           Load all files on command line into Entity.\n"
	     "  -c           Execution class to bind this file to.\n"
	     "  -m           Generate module config files.\n"
	     "  -l           Specify alternate location for libraries and configuration directories.\n"
	     "  -h           Print this and die.\n"
	     "%s %s, compiled: %s\n"
	     "Report bugs to <entity-devel@lists.sourceforge.net>\n",
	     PACKAGE, PACKAGE, VERSION, __DATE__);
}

static gchar *
full_path (gchar * filename)
{
    gchar *full_path;
    gchar buf[PATH_MAX];

    if (filename[0] == '/') {
	full_path = g_strdup (filename);
    } else {
	getcwd (buf, PATH_MAX);
	full_path = g_strconcat (buf, "/", filename, NULL);
    }

    return (full_path);
}

int
main (int argc, gchar * argv[])
{
    gchar *wd;
    gint c;
    GSList *files = NULL;
    gint load_all_files = FALSE;

    entity_init (&argc, &argv);

    while (TRUE) {		/* Get all the args. */
	const char *optstr = "asg:hVmc:l:";
	c = getopt (argc, argv, optstr);

	if (c == -1)
	    break;

	switch (c) {
	case 'g':		/* Debugging flag: -d<domains>. */
	    edebug_enable ((optarg));
	    break;

	case 'a':		/* Load all files on command line into entity. */
	    load_all_files = TRUE;
	    break;

	case 'h':		/* Help: -h or -\?. */
	    /* Fall through. */
	case '?':
	    print_usage ();
	    exit (0);
	    /* NOTREACHED */
	    break;

	case 'V':		/* Print the version. */
	    g_print ("%s %s\n", PACKAGE, VERSION);
	    exit (0);
	    /* NOTREACHED */
	    break;
#ifndef WIN32
	case 'm':		/* Create a config file from loaded libs. */
	    modulegen (NULL);
	    break;
	case 't':		/* Create a config file from loaded libs. */
	    modulegen (NULL);
	    break;
#endif				/* ifndef WIN32 */
	case 'l':		/* Set alternative directory for config tree. 
				 */
	    econfig_set_attr ("config-location", g_strdup (optarg));
	    break;
	case 'c':		/* The execution class to start file on:
				 * -c<class>. */
	    econfig_set_attr ("exec-class", g_strstrip (optarg));
	    break;

	default:		/* Faulty argument... */
	    g_print ("?? getopt returned character code 0%o ??\n", c);

	}			/* End switch. */

    }				/* Args while(TRUE)... */



    
    while (optind < argc) {

#ifdef WIN32
	/* On win32 we just beleive the path they gave us.. no magic with
	 * changing working dirs etc.  Also, determining if it's really a
	 * full path is a lot harder. */
	files = g_slist_append (files, argv[optind]);
#else
	files = g_slist_append (files, full_path (argv[optind]));
#endif
	if (load_all_files == TRUE) {
	    optind++;
	} else {
	    break;
	}
    }

#ifndef WIN32
    /* Set working directory */
    wd = g_strconcat (g_get_home_dir (), NULL);
    chdir (wd);
    g_free (wd);
#endif				/* WIN32 */

    /* Load this file up in the execution class specified.  This also ends up
     * calling the mainloop. */
    if (files) {
	entity_exec_class_load (files);
    } else {
	print_usage ();
	exit (0);
    }

    g_slist_free (files);
    entity_mainloop_run ();	/* Loop waiting on the tree. */

    return (0);
}


