#include <glib.h>
#include <gmodule.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>

#ifndef WIN32
/* glib.h defines everything in dirent.h for win32 * don't feel like hacking
 * GLIB just yet ;-) * - Kirk Ismay */
#include <dirent.h>
#endif

#include <errno.h>

#include "entity.h"


static void renderers_register_dynaload_element (void);
static void renderers_dynamic_init (void);
static void renderers_static_init ();
static ENode *dynaload_node = NULL;

/* C libs. */
static void renderer_dynaload_clib_element_render (ENode * node);
/* Entity libs using userrend. */
static void renderer_dynaload_elib_element_render (ENode * node);


void
renderers_init (void)
{
    renderers_static_init ();

    /* Create a node to hold the dynaload config entries under */
    dynaload_node = enode_new_child (enode_root_node (), "dynaloaders", NULL);
    renderers_register_dynaload_element ();
    renderers_dynamic_init ();
    return;
}

static gint			/* FALSE on failure. */
renderer_init_elib (ENode * loader_node)
{
    EBuf *lib;
    gchar *elibfile;		/* .e file with the <renderer> tag. */
    ENode *parent;

    if (!loader_node)
	return FALSE;

    parent = enode_parent (loader_node, NULL);
    lib = enode_attrib (parent, "library", NULL);
    if (ebuf_empty (lib))
	return FALSE;

    EDEBUG (("renderers", "Trying to init elib (%s)\n", lib->str));

    /* elibfile will look something like: *
     * /usr/local/share/entity/elib/name-of-lib.e */
    elibfile =
	g_strconcat (econfig_get_attr ("config-location"), "/elib/", lib->str,
		     ".e", NULL);

    /* The new elib node will live under the dynaloader node... * so we need
     * to be sure that the dynaloader node doesn't get killed. */
    xml_parse_file (loader_node, elibfile);

    g_free (elibfile);

    return TRUE;		/* We made it.  Lets hope that the tag is now 
				 * defined. */
}

static gint			/* FALSE means it didn't init. */
renderer_init_clib (gchar * modname)
{
    GModule *module;
    InitFunc initfunc;

    /* g_print ("Loading clib %s\n", modname); */

    EDEBUG (("renderers", "Loading module %s", modname));

    module = eutils_load_module (modname);

    if (!module) {
	g_warning ("Unable to load module %s", modname);
	return FALSE;
    }

    if (!g_module_symbol (module, "renderer_init", (gpointer *) & initfunc)) {
	g_warning ("Unable to initialize module %s", modname);
	return FALSE;
    }

    initfunc (RENDERER_REGISTER | RENDERER_INIT);
    return TRUE;
}

/* This is used to keep track of the last xml node that defined the mainloop */
static ENode *main_loop_node = NULL;

static void
renderer_library_render (ENode * node)
{
    EBuf *loopname;

    /* As soon as something is loaded that sets the control func, we have *
     * to respond by loading that library and setting up the control func
     * ASAP */
    loopname = enode_attrib (node, "mainloop", NULL);

    /* If there is a main loop for this module then we need to store this
     * information for later.  The mainloop library has to be loaded after
     * all these are registered, because it may register its own node types,
     * and needs to override the autoloader ones. */
    if (ebuf_not_empty (loopname)) {
	/* We could issue a warning here if the mainloop has already * been
	 * defined, but we'll let the mainloop registration deal * with that. 
	 * It may also be special cased for baby_main or * such */
	main_loop_node = node;
    }
}

/* For each <dynaload-element> that we load from the file, this gets called.
 * This registers the element the dynaload-element is referring to, setting
 * it to run the renderer_dynaload_element_render() function when it is
 * render'd in your program. This in turn will load the required library for
 * this element, replacing this dummy entry. */

static void
renderer_element_render (ENode * node)
{
    Element *element;
    ENode *parent;
    EBuf *type;

    /* Your usual registration of the dynaloader to get the * module loaded
     * when it's required */

    parent = enode_parent (node, "dynaload-library");
    if (!parent) {
	g_warning ("No dynaload-library parent tag for node %s, giving up.",
		   node->element->str);
	return;
    }

    element = g_new0 (Element, 1);

    type = enode_attrib (parent, "type", NULL);

    if (ebuf_empty (type) || ebuf_equal_str (type, "clib")) {	/* On NOT_SET 
								 * assumes
								 * its clib. */
	element->render_func = renderer_dynaload_clib_element_render;
    } else if (ebuf_not_empty (type) && ebuf_equal_str (type, "elib")) {
	element->render_func = renderer_dynaload_elib_element_render;
    } else
	g_warning
	    ("While attempting to load element implementation, type=\"%s\", which is an unknown type.",
	     type->str);

    type = enode_attrib (node, "tag", NULL);
    if (ebuf_empty (type)) {
	g_warning ("Attribute 'tag' is unset in the '%s' node",
		   node->element->str);
	g_free (element);
	return;
    }

    element->tag = g_strdup (type->str);
    element->dynaloader_node = node;
    element_register (element);
}

/* Now that the above has been called, it uses this function below * to load
 * up the module that provides this element.  When it registers, * it
 * overwrites the previous "dummy" entry that was made for all * elements it
 * provides. * * We have become an itermedate renderer for the regular tag.
 *  * Once we are called we will be overwritten by the real renderer. */
static void
renderer_dynaload_clib_element_render (ENode * node)
{
    EBuf *modname;
    Element *element;
    ENode *libnode;

    /* And now we should be able to cleanup that tree */
    element = element_lookup_element (node->element);
    libnode = enode_parent (element->dynaloader_node, "dynaload-library");
    if (!libnode)
	return;

    modname = enode_attrib (libnode, "library", NULL);

    if (ebuf_empty (modname))
	return;

    if (renderer_init_clib (modname->str) == FALSE)
	return;			/* Didn't init correctly. */

    /* And now that we have the renderer in place, we re-render the node. */
    ENODE_UNSET_FLAG (node, ENODE_RENDERED);
    enode_event_render (node);

    /* Now delete this tree, as we don't need to keep it around anymore. */
    if (libnode)
	enode_destroy (libnode);

    return;			/* Lib is now loaded and tag is ready to be
				 * rendered. */
}

static void
renderer_dynaload_elib_element_render (ENode * node)
{
    Element *element;

    element = element_lookup_element (node->element);

    if (!element->dynaloader_node)	/* Can't do any thing for them... */
	return;

    if (renderer_init_elib (element->dynaloader_node) == FALSE)
	return;			/* Failed loading. */

    /* And now that we have the renderer in place, we re-render the node.
     * We have to reset the flag though first. */
    ENODE_UNSET_FLAG (node, ENODE_RENDERED);
    enode_event_render (node);

    return;
}

/* 1. find & load .conf files, registering their elements and *
 * attributes. *  2. register as handlers for those elements stubs that will 
 * *     automatically load the right .so and pass control to it's stub. */
static int
conffile (const struct dirent *d)
{
    gchar *filename = d->d_name;
    int len = strlen (filename);

    if (len >= 5)
	return (strcmp (".conf", filename + (len - 5)) == 0 ? 1 : 0);
    else
	return (0);
}


/* Below are all the init functions. */

static void
renderers_dynamic_init (void)
{
    gchar *filename;
    DIR *dir;
    struct dirent *entry;
    gchar *confdir;

    confdir =
	g_strconcat (econfig_get_attr ("config-location"), "/config", NULL);
    dir = opendir (confdir);

    if (!dir)
	g_error ("Error opening directory '%s' for listing: %s", confdir,
		 g_strerror (errno));

    while ((entry = readdir (dir))) {
	if (conffile (entry)) {
	    EDEBUG (
		    ("renderers", "Loading module conf file %s",
		     entry->d_name));

	    filename = g_strconcat (confdir, "/", entry->d_name, NULL);
	    xml_parse_file (dynaload_node, filename);
	    g_free (filename);
	}
    }

    closedir (dir);
    g_free (confdir);

    if (main_loop_node && !entity_mainloop) {
	ENode *node = main_loop_node;
	EBuf *lib;
	EBuf *loopname;
	EBuf *type;

	loopname = enode_attrib (node, "mainloop", NULL);

	/* Do a little sanity checking based on the lib. */
	lib = enode_attrib (node, "library", NULL);
	if (ebuf_empty (lib)) {
	    g_warning ("Configuration sets mainloop (%s), but does not \n"
		       "specify library that contains it!", loopname->str);
	    return;
	}

	/* I'm requiring that all mainloop code be written in C. MW */
	type = enode_attrib (node, "type", NULL);

	if (ebuf_not_empty (type) && !ebuf_equal_str (type, "clib")) {	/* On 
									 * NULL, 
									 * we 
									 * assume 
									 * clib. 
									 */
	    g_warning ("mainloop (%s) is not of clib type\n", loopname->str);
	    return;
	}

	/* This module isn't inited yet, so init it now because we * needs
	 * its mainloop to be defined. */
	renderer_init_clib (lib->str);

	return;
    }
}

static void
renderers_static_init ()
{
    /* The following are always built in */
}

void
renderers_register_dynaload_element (void)
{
    Element *element;
    ElementAttr *e_attr;

    element = g_new0 (Element, 1);
    element->render_func = renderer_element_render;
    element->destroy_func = NULL;
    element->description =
	"Used internally to keep track of elements that should be dynamically loaded.";
    element->tag = "dynaload-element";
    element_register (element);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "tag";
    e_attr->description = "The element's tag string";
    e_attr->value_desc = "string";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);

    element = g_new0 (Element, 1);
    element->render_func = renderer_library_render;
    element->destroy_func = NULL;
    element->description =
	"Used internally to keep track of which libraries contain which elements.";
    element->tag = "dynaload-library";
    element_register (element);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "mainloop";
    e_attr->description = "Required event loop and main loop handler function";
    e_attr->value_desc = "string";
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "library";
    e_attr->description = "The name of the library providing these renders.";
    e_attr->value_desc = "string";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "type";
    e_attr->description = "The library type for the tag.";
    e_attr->value_desc = "string";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);
}


