#include <glib.h>
#include <stdio.h>
#include <string.h>
#include <xmlparse.h>
#include <errno.h>
#include "glib.h"
#include "entity.h"

typedef struct {
    ENode *top_node;

    ENode *working_node;
    ENode *top_of_new_tree;

    GSList *new_node_list;
    GSList *new_node_list_tail;
    gchar *source_filename;
} ParserState;


static void
start_element (void *user_data, char *type, char **atts)
{
    ParserState *state = user_data;
    GSList *attribs = NULL;
    GSList *attribs_tail = NULL;
    ENode *new_child;
    gint i;

    for (i = 0; atts[i]; i += 2) {
	if (!atts[i] || !atts[i + 1])
	    break;
	attribs = g_slist_append_tail (attribs, ebuf_new_with_str (atts[i]),
				       &attribs_tail);
	attribs = g_slist_append_tail (attribs, ebuf_new_with_str (atts[i + 1]),
				       &attribs_tail);
    }

    /* If we're rendering in a new object, wrap it in an <instance> tag, but
     * only if there isn't one already there.. this can happen if the trees
     * been loaded and saved with the instance tags */
    if (g_str_equal (type, "object") &&
	!ebuf_equal_str (state->working_node->element, "instance")) {
	new_child = enode_new_child_norender (state->working_node,
					      ebuf_new_with_str ("instance"),
					      NULL);

	ENODE_SET_FLAG (new_child, ENODE_INSTANCE_PLACEHOLDER);

	state->new_node_list = g_slist_append_tail (state->new_node_list,
						    state->working_node,
						    &state->new_node_list_tail);
	state->new_node_list = g_slist_append_tail (state->new_node_list,
						    new_child,
						    &state->new_node_list_tail);

	/* Update the "top of new tree" if this is the first new node */
	if (!state->top_of_new_tree)
	    state->top_of_new_tree = new_child;

	state->working_node = new_child;

	/* And put the filename it was loaded from into the attributes for
	 * the new object. */
	if (state->source_filename) {
	    attribs =
		g_slist_append_tail (attribs, ebuf_new_with_str ("__filename"),
				     &attribs_tail);
	    attribs =
		g_slist_append_tail (attribs,
				     ebuf_new_with_str (state->source_filename),
				     &attribs_tail);
	}
    }

    /* insert the node into the tree */
    new_child = enode_new_child_norender (state->working_node, 
					  ebuf_new_with_str (type),
					  attribs);

    /* Update the "top of new tree" if this is the first new node */
    if (!state->top_of_new_tree)
	state->top_of_new_tree = new_child;

    state->new_node_list = g_slist_append_tail (state->new_node_list,
						state->working_node,
						&state->new_node_list_tail);
    state->new_node_list = g_slist_append_tail (state->new_node_list,
						new_child,
						&state->new_node_list_tail);
    state->working_node = new_child;
}

static void
end_element (void *user_data, const char *name)
{
    ParserState *state = user_data;

    /* Destroy non-cdata, whitespace-only data */
    if (state->working_node->data &&
	ebuf_is_whitespace (state->working_node->data) &&
	!ENODE_FLAG_ISSET (state->working_node, ENODE_HAS_CDATA)) {
	ebuf_free (state->working_node->data);
	state->working_node->data = NULL;
    }

    state->working_node = enode_parent (state->working_node, NULL);

    /* One special case, since the node_parent will create a new <instance>
     * for app tags, we have to walk over that here or we get screwed up */
    if (ENODE_FLAG_ISSET (state->working_node, ENODE_INSTANCE_PLACEHOLDER)) {
	state->working_node = enode_parent (state->working_node, NULL);
	/* Always kill instance data.. should never have any */
	if (state->working_node->data) {
	    ebuf_free (state->working_node->data);
	    state->working_node->data = NULL;
	}
    }
}

static void
cdata_start_section_handler (void *user_data)
{
    ParserState *state = user_data;

    ENODE_SET_FLAG (state->working_node, ENODE_HAS_CDATA);
    ENODE_SET_FLAG (state->working_node, ENODE_IN_CDATA_SECTION);

    /* If there's CDATA, we're not keeping the rest of it.. */
    /* I'm cheating here because I know it's not rendered yet.. :) */

    if (state->working_node->data)
	ebuf_truncate (state->working_node->data, 0);
}

static void
cdata_end_section_handler (void *user_data)
{
    ParserState *state = user_data;

    ENODE_UNSET_FLAG (state->working_node, ENODE_IN_CDATA_SECTION);
}

static void
character_data_handler (void *user_data, XML_Char * s, int len)
{
    ParserState *state = user_data;

    /* If we had a cdata section, and we're not out of it, don't append data */
    if (ENODE_FLAG_ISSET (state->working_node, ENODE_HAS_CDATA) &&
	!ENODE_FLAG_ISSET (state->working_node, ENODE_IN_CDATA_SECTION))
	return;

    /* otherwise, we're good to go */
    if (state->working_node->data)
	ebuf_append_data (state->working_node->data, s, len);
    else
	state->working_node->data = ebuf_new_with_data (s, len);
}

static void
processing_instruction_handler (void *user_data, XML_Char * target,
				XML_Char * data)
{
    /* Cheat and pull it in as cdata on a tag */
    gchar *atts[] = { NULL };
    start_element (user_data, target, atts);
    cdata_start_section_handler (user_data);
    character_data_handler (user_data, data, strlen (data));
    cdata_end_section_handler (user_data);
    end_element (user_data, target);
}


static void
xml_parser_state_render (ParserState * state)
{
    GSList *tmp;

    tmp = state->new_node_list;
    while (tmp) {
	ENode *parent;
	ENode *child;

	parent = tmp->data;
	tmp = tmp->next;
	child = tmp->data;
	tmp = tmp->next;

	enode_event_render (child);
	enode_event_parent (parent, child);
    }
}

/* Check on various errors if it's a good idea to exit or not */
static void
xml_parser_check_for_exit (void)
{
    ENode *root;
    GSList *children;
    GSList *tmp;
    gint found = FALSE;

    root = enode_root_node ();

    /* Get a list of immediate children */
    children = enode_children (root, NULL);

    /* If there are no child nodes, or the only node is the
     * dynaloaders node, then we should exit */
    tmp = children;
    while (tmp) {
	ENode *node = tmp->data;
	
	if (!ebuf_equal_str (enode_type (node), "dynaloaders")) {
	    found = TRUE;
	}
	EDEBUG (("xml-parser", "Checking for exit - node type was '%s'",
		 node->element->str));

	tmp = tmp->next;
    }

    if (found == FALSE) 
	entity_mainloop_exit ();
    
    g_slist_free (children);
}




void
xml_parser_state_cleanup_from_error (ParserState * state)
{
    GSList *tmp;
    
    tmp = state->new_node_list;
    if (tmp->data) {
	ENode *node;
	node = tmp->data;
	enode_destroy (node);
    }

    /* If there are no children, exit */
    xml_parser_check_for_exit ();
}


/* Returns TRUE on success, FALSE on failure */
static gint
xml_parse_string_chunk (ParserState * state, gchar * xml,
			gint xml_len, gint done)
{
    static gint initialized = FALSE;
    static XML_Parser parser = NULL;

    if (!initialized) {
	parser = XML_ParserCreate (NULL);
	XML_Parse (parser, "<entity>", strlen ("<entity>"), 0);
	XML_SetUserData (parser, state);

	XML_SetElementHandler (parser, (gpointer) start_element, end_element);
	XML_SetCharacterDataHandler (parser, (gpointer) character_data_handler);
	XML_SetCdataSectionHandler (parser, cdata_start_section_handler,
				    cdata_end_section_handler);
	XML_SetProcessingInstructionHandler (parser,
					     (gpointer)
					     processing_instruction_handler);

	initialized = TRUE;
    }

    EDEBUG (("xml-parser", "Parsing chunk '%s'\n", xml));

    /* Do the actual parsing */
    if (!XML_Parse (parser, xml, xml_len, FALSE)) {
	g_warning ("While parsing file '%s' %s at line %d",
		   state->source_filename ? state->
		   source_filename : "(Unknown)",
		   XML_ErrorString (XML_GetErrorCode (parser)),
		   XML_GetCurrentLineNumber (parser));

	/* FIXME: Should clean up here.. delete nodes etc! */
	return (FALSE);
    }


    if (done) {
	XML_SetElementHandler (parser, NULL, NULL);
	XML_SetCharacterDataHandler (parser, NULL);
	XML_SetProcessingInstructionHandler (parser, NULL);
	XML_Parse (parser, "</entity>", strlen ("</entity>"), TRUE);

	XML_ParserFree (parser);

	parser = NULL;
	initialized = FALSE;
    }
    return (TRUE);
}

static ParserState *
xml_parser_state_new (ENode * top_node)
{
    ParserState *state;

    state = g_new0 (ParserState, 1);

    if (top_node == NULL)
	state->top_node = enode_root_node ();
    else
	state->top_node = top_node;

    state->working_node = state->top_node;

    return (state);
}

static void
xml_parser_state_free (ParserState * state)
{
    g_slist_free (state->new_node_list);
    g_free (state);
}


/* Returns the top node from the parsing */
ENode *
xml_parse_file (ENode * top_node, gchar * filename)
{
    char buf[BUFSIZ];
    int done;
    FILE *fp;
    ENode *top;
    EBuf *filename_ebuf;
    ParserState *state;

    fp = fopen (filename, "r");

    if (!fp) {
	g_warning ("Unable to open file %s: %s", filename, g_strerror (errno));
	xml_parser_check_for_exit ();
	return (NULL);
    }

    /* Do a little peak to see if we're running from command line interp. */
    fgets (buf, BUFSIZ, fp);

    /* if the first char is a '#', just leave it, else rewind fp to parse *
     * entire file. */
    if (buf[0] != '#')
	rewind (fp);

    state = xml_parser_state_new (top_node);
    state->source_filename = filename;

    do {
	size_t len = fread (buf, 1, sizeof (buf), fp);
	done = len < sizeof (buf);
	if (xml_parse_string_chunk (state, buf, len, done) == FALSE) {
	    /* Parser failure */
	    xml_parser_state_cleanup_from_error (state);
	    xml_parser_state_free (state);
	    return (NULL);
	}
    }
    while (!done);

    top = state->top_of_new_tree;

    filename_ebuf = ebuf_new_with_str (filename);
    enode_attrib_quiet (top, "__filename", filename_ebuf);

    xml_parser_state_render (state);
    xml_parser_state_free (state);

    return (top);
}


ENode *
xml_parse_string (ENode * top_node, EBuf * xml)
{
    ENode *top;
    ParserState *state;

    if (top_node && xml) {
	state = xml_parser_state_new (top_node);
	if (xml_parse_string_chunk (state, xml->str, xml->len, TRUE) == FALSE) {
	    /* Parser failure */
	    xml_parser_state_cleanup_from_error (state);
	    xml_parser_state_free (state);
	    return (NULL);
	}
	xml_parser_state_render (state);
	top = state->top_of_new_tree;
	xml_parser_state_free (state);
	return (top);
    }
    return (NULL);
}


