/* csink.c * * Description: * * Author(s): Cory Stone * * Created:
 * 05/25/2000 * * Last Modified: $Id: csink.c,v 1.16 2000/10/10 10:10:27
 * imain Exp $ * */

#include "csink.h"
#include <sys/time.h>
#include <stdio.h>


/* This should have a string table for the errors... */
gchar *
csink_errstr (CSink * sink)
{
    return sink->error_record->description;
}


/* User callback setting */

void
csink_set_new_data_func (CSink * sink, CSinkCallbackFunc func)
{
    sink->on_new_data = func;
}

void
csink_set_empty_send_queue_func (CSink * sink, CSinkCallbackFunc func)
{
    sink->on_empty_send_queue = func;
}

void
csink_set_error_func (CSink * sink, CSinkCallbackFunc func)
{
    sink->on_error = func;
}

void
csink_set_close_func (CSink * sink, CSinkCallbackFunc func)
{
    sink->on_close = func;
}

void
csink_set_connect_func (CSink * sink, CSinkCallbackFunc func)
{
    sink->on_connect = func;
}

void *
csink_get_user_data (CSink * sink)
{
    return sink->user_data;
}

void
csink_set_user_data (CSink * sink, void *data)
{
    sink->user_data = data;
}


/* Error handling */

gint
csink_error (CSink * sink)
{
    if (TRUE == sink->err) {
	sink->err = FALSE;
	return TRUE;
    }

    return FALSE;
}

void
csink_on_error (CSink * sink, char *err_name)
{
    CSink_Error *e = csink_error_table;
    CSink_Error *found = NULL;

    CDEBUG (("csink", "Asked to signal error: %s", err_name));

    while (e->name != NULL) {
	if (strcmp (e->name, err_name) == 0) {
	    found = e;
	    break;
	}
	e++;
    }

    sink->err = TRUE;
    if (found) {
	sink->error_record = found;

	CDEBUG (("csink", "Got error: %s\n", sink->error_record->name));
	CDEBUG (("csink", "err string: %s\n", csink_errstr (sink)));
    } else {
	sink->error_record = &csink_error_internal_error;

	CDEBUG (("csink", "'%s' is not a registered error.", err_name));
    }

    if (sink->on_error) {
	sink->on_error (sink);
    }
}

gchar *
csink_erormsg (CSink * sink)
{
    /* Do a lookup of sink->err and print the appropriate mesg. */
    CDEBUG (("csink", "Sorry, no errors yet!\n"));
    return NULL;
}

void
csink_on_new_data (CSink * sink)
{
    if (sink->on_new_data)
	sink->on_new_data (sink);
}

CBuf *
csink_read (CSink * sink)
{
    CBuf *message = NULL;

    CDEBUG(("csink","in read"));

    if (sink->read (sink)) {
      CDEBUG(("csink","got a return..."));
      message = (CBuf *) g_ptr_array_index (sink->inQueue, 0);
      CDEBUG(("csink","the message is %s", message->str));
      g_ptr_array_remove_index (sink->inQueue, 0);
      CDEBUG(("csink","cleared the queue"));
    }

    CDEBUG(("csink","returning"));
    return message;
}

gint
csink_write (CSink * sink, CBuf * data)
{
    return (sink->write (sink, data));
}


void
csink_close (CSink * sink)
{
    CDEBUG (("csink", "calling sink close function."));
    sink->close (sink);
    if (sink->on_close)
	sink->on_close (sink);

}

void
csink_free (CSink * sink)
{
    CDEBUG (("csink", "freeing a csink."));
    sink->free (sink);
}



/* get the queue sizes in bytes */

gint
csink_send_queue_size (CSink * sink)
{
    int i,
     size;
    for (i = 0, size = 0; i < sink->outQueue->len; i++)
	size += ((CBuf *) g_ptr_array_index (sink->outQueue, i))->len;

    return size;
}

gint
csink_receive_queue_size (CSink * sink)
{
    int i,
     size;
    for (i = 0, size = 0; i < sink->outQueue->len; i++)
	size += ((CBuf *) g_ptr_array_index (sink->inQueue, i))->len;

    return size;
}



/* do a fast select() to see if anything needs handling right now. hmm. no.
 * 
 * the user should give US the functions to watch the fds
 * 
 * so they could call like, csink_set_add_fd_watcher_func
 * (func_with_known_interface) csink_set_remove_fd_watcher_func
 * (func_with_known_interface)
 * 
 * and we would use those.
 * 
 * we'd supply a csink_init_with_glib
 * 
 * which would set us up with glib's fd watching.
 * 
 * also, we can have a poll-style function which does our fd watching and
 * dispatching for us. */

/* memo to self: when building, try to detect if we're building with entity.
 * if we are, then we should use entity's facilities for this shit. */


CSinkAddFDFunc csink_fd_add_func = NULL;
CSinkRemoveFDFunc csink_fd_remove_func = NULL;

void
csink_init_funcs (CSinkAddFDFunc fd_add_func, CSinkRemoveFDFunc fd_remove_func)
{
    csink_fd_add_func = fd_add_func;
    csink_fd_remove_func = fd_remove_func;
}


/*************************************************************
  Below is an implementation of the fd watching interface; call
  csink_set_polling_fd_funcs and be sure to call csink_do_poll every once
  in a while (like in your main loop)
*************************************************************/

typedef struct {
    CSinkCallbackFunc func;
    CSink *sink;
} WrapData;

int
wrap_func (int fd, CSinkFDCondition cond, void *data)
{
    WrapData *wrap_data = (WrapData *) data;
    wrap_data->func (wrap_data->sink);
    return 1;			// i guess?
}

void *
csink_add_fd (int fd, CSinkFDCondition cond, CSinkCallbackFunc func,
	      CSink * sink)
{
    WrapData *wrap_data;

    if (csink_fd_add_func) {
	wrap_data = g_new0 (WrapData, 1);
	wrap_data->func = func;
	wrap_data->sink = sink;
	return csink_fd_add_func (fd, cond, wrap_func, (void *) wrap_data);
    }

    CDEBUG (("csink", "aw fuck! there are no fd watchers registered!"));
    return NULL;
}

void
csink_remove_fd (void *tag)
{
    if (csink_fd_remove_func)
	csink_fd_remove_func (tag);
    else
	CDEBUG (("csink", "aw fuck! there are no fd watchers registered!"));
}


typedef struct _csink_polled_fd CSinkPolledFD;
struct _csink_polled_fd {
    void *data;
    int fd;
    CSinkFDCondition cond;
    CSinkFDCallbackFunc func;
    CSinkPolledFD *next;
};
CSinkPolledFD *_csink_polled_fd_list;

void *
csink_polled_fd_add (int fd, CSinkFDCondition cond,
		     CSinkCallbackFunc func, void *data)
{
    CSinkPolledFD *rec = (CSinkPolledFD *) malloc (sizeof (CSinkPolledFD));

    CDEBUG (("csink", "adding a csink."));

    rec->fd = fd;
    rec->cond = cond;
    rec->data = data;
    rec->func = (CSinkFDCallbackFunc) func;

    rec->next = _csink_polled_fd_list;
    _csink_polled_fd_list = rec;

    {
	CSinkPolledFD *this = _csink_polled_fd_list;
	int i = 0;

	CDEBUG (("csink", "counting fds watched.."));
	for (i = 0, this = _csink_polled_fd_list; this; i++, this = this->next);

	CDEBUG (("csink", "%d fds being watched.", i));
    }

    return rec;
}

void
csink_polled_fd_remove (void *tag)
{
    CSinkPolledFD *this = _csink_polled_fd_list;
    CSinkPolledFD *last = NULL;
    CSinkPolledFD *caught = NULL;

    CDEBUG (("csink", "removing an fd watcher"));

    while (this) {
	if (this == tag) {
	    caught = this;
	    if (last)
		last->next = this->next;
	    else {
		_csink_polled_fd_list = this->next;
		CDEBUG (("csink", "had to change the list head."));
	    }
	    break;
	}
	last = this;
	this = this->next;
	CDEBUG (("csink", "floop"));
    }

    if (caught) {
	CDEBUG (("csink", "found it and NOTfreeing it."));
	// free (caught);    
    } else {
	CDEBUG (("csink", "not found."));
    }

    {
	CSinkPolledFD *this = _csink_polled_fd_list;
	int i = 0;

	CDEBUG (("csink", "counting fds watched.."));
	for (i = 0, this = _csink_polled_fd_list; this; i++, this = this->next);

	CDEBUG (("csink", "%d fds being watched.", i));
    }

}

void
csink_do_poll (void)
{
    CSinkPolledFD *this = _csink_polled_fd_list;

    fd_set in_set,
     out_set,
     err_set;
    struct timeval timeout;

    int res;

    // CDEBUG(("csink", "preparing to poll"));

    FD_ZERO (&in_set);
    FD_ZERO (&out_set);
    FD_ZERO (&err_set);

    while (this) {
	if (this->cond & EIO_READ)
	    FD_SET (this->fd, &in_set);
	if (this->cond & EIO_WRITE)
	    FD_SET (this->fd, &out_set);
	if (this->cond & EIO_ERROR)
	    FD_SET (this->fd, &err_set);
	this = this->next;
    }

    timeout.tv_sec = timeout.tv_usec = 0;

    // CDEBUG(("csink", "calling select()"));

    res = select (FD_SETSIZE, &in_set, &out_set, &err_set, &timeout);

    if (res != 0)
	CDEBUG (("csink", "select() returned %d", res));

    if (res == -1) {
	switch (errno) {
	case EBADF:		/* we've passed in a bad file descriptor -
				 * BUG */
	    CDEBUG (("csink", "bad fd in list of fds to watch - bailing."));
	    return;
	    break;
	case EINVAL:		/* i've specified an incorrect timeout - BUG */
	    CDEBUG (("csink", "jim screwed up the select() call - bailing."));
	    return;
	    break;
	case EINTR:		/* interrupted by a signal */
	    /* we actually shouldn't get this, since the nice GNU macro
	     * 'TEMP_FAILURE_RETRY' checks for it .. */
	    CDEBUG (("csink", "EINTR from select() call - not fatal."));
	    break;
	default:
	    CDEBUG (
		    ("csink",
		     "falling off the end of the world after select()"));
	}
    }

    if (res != 0) {
	CSinkPolledFD *next;
	CDEBUG (("csink", "got activity, so we're gonna check it out.."));

	/* this is done in three loops because a csink might be removed each
	 * loop. */

	CDEBUG (("csink", "checking for activity in the input set"));
	this = _csink_polled_fd_list;
	while (this) {
	    next = this->next;
	    if (FD_ISSET (this->fd, &in_set)) {
		CDEBUG (("csink", "responding to activity in the input set"));
		this->func (this->fd, this->cond, this->data);
		return;
	    }
	    this = next;
	}

	CDEBUG (("csink", "checking for activity in the output set"));
	this = _csink_polled_fd_list;
	while (this) {
	    next = this->next;
	    if (FD_ISSET (this->fd, &out_set)) {
		CDEBUG (("csink", "responding to activity in the output set"));
		this->func (this->fd, this->cond, this->data);
		return;
	    }
	    this = next;
	}

	CDEBUG (("csink", "checking for activity in the exception set"));
	this = _csink_polled_fd_list;
	while (this) {
	    next = this->next;
	    if (FD_ISSET (this->fd, &err_set)) {
		CDEBUG (
			("csink",
			 "responding to activity in the exception set"));
		this->func (this->fd, this->cond, this->data);
		return;
	    }
	    this = next;
	}

	CDEBUG (("csink", "done checking"));
    }
}

void
csink_set_polling_fd_funcs (void)
{
    csink_init_funcs ((CSinkAddFDFunc) csink_polled_fd_add,
		      (CSinkRemoveFDFunc) csink_polled_fd_remove);
    _csink_polled_fd_list = NULL;
}
