#ifndef __CSINK_H__
#define __CSINK_H__

/* csink.h * * Author(s): Mike Harrison, Cory Stone */

/* Headers */
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <unistd.h>
#include <glib.h>
#include <errno.h>

#include "cbuf.h"
#include "csinkerr.h"

#ifdef ENTITY_EMBED
#  include "entity.h"
#  define CDEBUG(x) edebug x
#else				/* not ENTITY_EMBED */
#  include <cdebug.h>
#  define CDEBUG(x) cdebug x
#endif				/* ENTITY_EMBED */


/* Casting macros */
#define CSINK(sink) 		( (CSink*)sink )
typedef struct _CSink CSink;


typedef void (*CSinkFreeFunc) (CSink * sink);
typedef void (*CSinkCloseFunc) (CSink * sink);
typedef CBuf *(*CSinkReadFunc) (CSink * sink);
typedef gint (*CSinkWriteFunc) (CSink * sink, CBuf * data);
typedef void (*CSinkOpenFunc) (CSink * sink);
typedef void (*CSinkCallbackFunc) (CSink * sink);


struct _CSink {
  GPtrArray *inQueue;
  GPtrArray *outQueue;
  
  int err;
  CSink_Error *error_record;
  
  CSinkReadFunc read;
  CSinkWriteFunc write;
  CSinkOpenFunc open;
  CSinkCloseFunc close;
  CSinkFreeFunc free;
  
  CSinkCallbackFunc on_error;	/* error signalled */
  CSinkCallbackFunc on_new_data; /* new data available */
  CSinkCallbackFunc on_close;	/* sink connection closed  */
  CSinkCallbackFunc on_connect;	/* sink connection is active  */
  CSinkCallbackFunc on_empty_send_queue; /* hum */

  gpointer user_data;

  /* child holds the 'contained' sink, parent is the sink that contains the
     sink. used for inheritance and for delegation */
  CSink *child; 
  CSink *parent; 
};


/* get and set the user data */
void *csink_get_user_data (CSink * sink);
void csink_set_user_data (CSink * sink, void *data);

/* modify the callbacks */
void csink_set_new_data_func (CSink * sink, CSinkCallbackFunc func);
void csink_set_empty_send_queue_func (CSink * sink, CSinkCallbackFunc func);
void csink_set_error_func (CSink * sink, CSinkCallbackFunc func);
void csink_set_close_func (CSink * sink, CSinkCallbackFunc func);
void csink_set_connect_func (CSink * sink, CSinkCallbackFunc func);


/* error handling */
gchar *csink_errstr (CSink * sink);
gint csink_error (CSink * sink);
gchar *csink_errormsg (CSink * sink);


/* basic interaction methods */
void csink_open (CSink * sink);	/* tries to open a connection */
CBuf *csink_read (CSink * sink); /* read from the sink, NULL if nothing is avail */
gint csink_write (CSink * sink, CBuf * data); /* write data to the out queue */
void csink_close (CSink * sink); /* close the connection */
void csink_free (CSink * sink);	/* free all data associated with the sink */


/* information about the buffers */
gint csink_send_queue_size (CSink * sink);
gint csink_receive_queue_size (CSink * sink);

/* call the associated callbacks */
void csink_on_connect (CSink * sink);
void csink_on_new_data (CSink * sink);
void csink_on_error (CSink * sink, char *errname); /* sets error before calling */


/* FD Watching interface */
/* CSinks will use whatever functions the user wants to watch activity on
   fds. Also, a simple polling implementaiton is provided. */

typedef EIOCond CSinkFDCondition;
typedef EIOFunc CSinkFDCallbackFunc;

typedef void *(*CSinkAddFDFunc) (int fd, CSinkFDCondition cond,
				 CSinkFDCallbackFunc func, CSink * sink);
typedef void (*CSinkRemoveFDFunc) (void *tag);

void csink_init_funcs (CSinkAddFDFunc fd_add_func,
		       CSinkRemoveFDFunc fd_remove_func);
void *csink_add_fd (int fd, CSinkFDCondition cond, CSinkCallbackFunc func,
		    CSink * sink);
void csink_remove_fd (void *tag);


/* The polling implementation. use csink_set_polling_fd_funcs to set it up, 
   then call csink_do_poll somewhere in your main loop. */

void csink_set_polling_fd_funcs (void);
void csink_do_poll (void);

#endif
