/* csinksocket.c * * Description: * * Author(s): Cory Stone * * Created:
 * 05/25/2000 * * Last Modified: $Id: csinkinet.c,v 1.25 2000/10/10 18:16:02
 * imain Exp $ * */

#include "csinkinet.h"

/* Local functions */
static void csink_inet_free (CSinkInet * sink);
static gint csink_inet_read (CSink * sink);
static void csink_inet_close (CSinkInet * sink);
static gint csink_inet_write (CSinkInet * sink, CBuf * msg);

static void csink_inet_connect_action (CSink * this_sink);
static void csink_inet_accept_action (CSink * this_sink);
static void csink_inet_socket_action (CSink * this_sink);

static void csink_inet_remote_sync (CSinkInet * sink);
static void csink_inet_local_sync (CSinkInet * sink);
static void csink_inet_set_on_dns_lookup_success (CSinkInet *sink, 
						  CSinkCallbackFunc cb);
static CSinkCallbackFunc csink_inet_get_on_dns_lookup_success (CSinkInet *sink);


/**csink_inet_new
 * Creates storage for an inet connection.
 * 
 * $CSinkInet The new sink.
 */
CSinkInet *
csink_inet_new ()
{
    CSinkInet *sink;

    sink = g_new0 (CSinkInet, 1);

    sink->s.inQueue = g_ptr_array_new ();
    sink->s.outQueue = g_ptr_array_new ();

    sink->fd = -1;

    sink->ip = INADDR_NONE;
    sink->port = -1;

    sink->localip = INADDR_ANY;
    sink->localport = -1;

    /* Initialize function pointers. */
    sink->s.free = (CSinkFreeFunc) csink_inet_free;
    sink->s.close = (CSinkCloseFunc) csink_inet_close;
    sink->s.read = (CSinkReadFunc) csink_inet_read;
    sink->s.write = (CSinkWriteFunc) csink_inet_write;

    return sink;
}


void
csink_on_connect (CSink * sink)
{
    if (sink->on_connect)
	sink->on_connect (sink);
}


/**csink_inet_read
 * Read in a message from the socket.
 *
 * !sink Sink to read.
 * $gint Flag -- success or failure, TRUE or FALSE.
 */
static int
csink_inet_read (CSink * sink_)
{
    char buf[256];
    CBuf *newmsg;
    int res;
    CSinkInet *sink = CSINK_INET(sink_);

    CDEBUG (("csink-inet", "in csink_inet_read\n"));

    /* ensure some sanity */
    if (!sink) {
      CDEBUG (("csink-inet", "csink_inet_read called with null sink!\n"));
      return FALSE;
    }

    /* can't write to a sink that isn't connected */
    if (! (sink->status & ISS_CONNECTED)) {
      csink_on_error (sink, "read from unconnected sink.");
      return FALSE;
    }

    
    /* Read a small block. */
    res = read (sink->fd, buf, 255);

    /* end-of-file */
    if (res == 0) {
	CDEBUG (("csink-inet",
		 "Connection closed on other end. closing link.\n"));
	csink_close (CSINK (sink));
	return (FALSE);
    }

    /* error */
    if (res < 0) {
      /* libc info seems to think that these are the only errors that can't 
       * be recovered from just by another read() call .. */
      switch (errno) {
      case EBADF:		/* bad file descriptor or file not open for
				 * reading */
	csink_close (CSINK(sink));
	csink_on_error (CSINK(sink), "READ_EBADF");
	return FALSE;
      case EIO:		/* hardware failure */
	csink_close (CSINK(sink));
	csink_on_error (CSINK(sink), " READ_EIO");
	return FALSE;
      default:
	
      }
      return FALSE;
    }
    
    newmsg = cbuf_new_with_data (buf, res);	/* Create a CBuf from the
						 * buf. */
    buf[res]='\0';
    g_ptr_array_add (CSINK (sink)->inQueue, (gpointer) newmsg);

    CDEBUG (("csink-inet", "leaving csink_inet_read\n"));

    return (TRUE);
}				/* END csink_inet_read */


/**csink_inet_write
 * Add this new message to the sink, writing it out if we can.
 *
 * !sink Sink to enqueue to.
 * !message Message to enqueue.
 * $gint TRUE if enqueue succeeded, FALSE if we got disconnected.
 */
static gint
csink_inet_write (CSinkInet * sink, CBuf * message)
{
    CBuf *newMesg;
    gint res;


    /* a little sanity check */
    if (!sink) {
      CDEBUG (("csink-inet", "csink_inet_write passed null sink\n"));
      return FALSE;
    }


    /* check the fd number to see if that's how things are breaking */
    CDEBUG (("csink-inet", "csink_inet_write with fd=%i, status=%i\n", 
	     sink->fd, sink->status));

    if (!message) {
      csink_on_error(CSINK(sink), "can't write out a null message!");
      return FALSE;
    }      

    /* If we're not connected, leave. */
    if (!(sink->status & ISS_CONNECTED)) {
	csink_on_error (CSINK (sink), "DISCONN_WRITE");
	return FALSE;
    }

    /* Add the message to the out queue. */
    newMesg = cbuf_new_with_cbuf (message);
    g_ptr_array_add (CSINK (sink)->outQueue, (gpointer) newMesg);

    /* Send loop for sending the queue. */
    while ((CSINK (sink)->outQueue->len != 0)
	   && (sink->status & ISS_CONNECTED)) {
	CBuf *cur;
	int written = 0;
	CDEBUG (("csink-inet", "SEND LOOP--CONNECTED (1)\n"));

	cur = (CBuf *) g_ptr_array_index (CSINK (sink)->outQueue, 0);
	g_ptr_array_remove_index (CSINK (sink)->outQueue, 0);

	while (written != cur->len) {
	    res = write (sink->fd, cur->str + written, cur->len - written);

	    if (res < 0) {	/* error */
		switch (errno) {
		case EAGAIN: /* this is not really a problem .. */
		    return TRUE;
		case EFBIG:
		    csink_on_error (CSINK (sink), "SEND_EFBIG");
		    return FALSE;
		case EINTR:
		    csink_on_error (CSINK (sink), "SEND_EINTR");
		    return FALSE;
		case ENOSPC:
		    csink_on_error (CSINK (sink), "SEND_NOSPC");
		    return FALSE;
		case EPIPE:
		    csink_on_error (CSINK (sink), "SEND_PIPE");
		    csink_close (CSINK (sink));
		    return FALSE;
		case EBADF:	/* bad or not open-for-writing fd */
		    csink_close (CSINK (sink));
		    csink_on_error (CSINK (sink), "SEND_EBADF");
		    return FALSE;
		case EIO:	/* hardware failure */
		    csink_close (CSINK (sink));
		    csink_on_error (CSINK (sink), "SEND_EIO");
		    return FALSE;
		default:
		    csink_on_error (CSINK (sink), "Unknown");
		    return FALSE;
		}
	    } else		/* success */
		written += res;
	}
    }

    return TRUE;
}




static void
csink_inet_free (CSinkInet * sink)
{
    int i;

    CDEBUG (("csink-inet", "freeing a csinkinet"));

    /* Free the hostname. */
    g_free (sink->hostname);
    g_free (sink->localinterface);

    /* Free the queues. */
    if (sink->s.inQueue) {
	for (i = 0; i < sink->s.inQueue->len; i++)
	    cbuf_free (g_ptr_array_index (sink->s.inQueue, i));
	g_ptr_array_free (sink->s.inQueue, TRUE);
    }

    if (sink->s.outQueue) {
	for (i = 0; i < sink->s.outQueue->len; i++)
	    cbuf_free (g_ptr_array_index (sink->s.outQueue, i));
	g_ptr_array_free (sink->s.outQueue, TRUE);
    }

    g_free (sink);
}

static void
csink_inet_close (CSinkInet * sink)
{
    CDEBUG (("csink-inet", "closing fd: %d\n", sink->fd));

    /* remove us from the fd watchers */
    if (sink->watch_tag)
	csink_remove_fd (sink->watch_tag);
    /* No we need to say there is no watch tag, we removed it. */
    sink->watch_tag = NULL;

    close (sink->fd);
    sink->fd = -1;

    sink->status = 0;
}



void
csink_inet_connect (CSinkInet * sink)
{
    int len = 0;		/* Size of sockaddr. */
    int res = 0;		/* Success flag.     */

    CDEBUG (("csink-inet", "connect, status = %i\n", sink->status));

    if (sink->status & (ISS_CONNECTED)) {
	/* should this be an error condition? */
	CDEBUG (("csink-inet", "we are connected already.\n"));
	return;
    }


    if (sink->status & ISS_DNS_INPROGRESS) {
      if (sink->status & ISS_CONNECT_INPROGRESS) {
	/* this means we are waiting for the host to respond, and not waiting for
	   DNS to give us our ip address */
	CDEBUG (("csink-inet", "alredy connecting.\n"));
      }
      else {
	/* our intention is to connect to the address found by the lookup, 
	   so we want to chain the connect on to the end of the DNS lookup
	   action.
	*/
	/* magically, csink_inet_connect has pretty much the same signature as any
	   other CSinkCallbackFunc :) 
	*/
	sink->status |= ISS_CONNECT_INPROGRESS;
	csink_inet_set_on_dns_lookup_success(sink, csink_inet_connect);
	CDEBUG (("csink-inet", "queued connect after lookup completes..\n"));
      }
      return;
    }
    

    /* get a socket */
    sink->fd = socket (AF_INET, SOCK_STREAM, 0);

    if (sink->fd < 0) {		/* error */
	csink_on_error (CSINK (sink), "SOCKET_CREATE_FAILED");
	return;
    }

    /* io in csink is ALWAYS nonblocking. */
    fcntl (sink->fd, F_SETFL, O_NONBLOCK);

    /* try to connect to the server */
    len = sizeof (sink->address);
    res = connect (sink->fd, (struct sockaddr *) &sink->address, len);
    if (res < 0) {		/* error */
	CDEBUG (
		("csink-inet", "error calling connect: %s\n",
		 g_strerror (errno)));

	switch (errno) {
	case EINPROGRESS:
	case EALREADY:
	    /* TODO: we have to make the select() stuff realize to check this 
	     */
	  CDEBUG (("csink-inet",
		   "returned in progress, adding to main loop"));
	  sink->status |= ISS_CONNECT_INPROGRESS;
	  sink->watch_tag = csink_add_fd (sink->fd, EIO_READ | EIO_WRITE,
					  csink_inet_connect_action,
					  CSINK (sink));
	  break;
	default:
	    CDEBUG (("csink-inet", "can't connect to %s\n", sink->hostname));

	    sink->status &= ~(ISS_CONNECT_INPROGRESS | ISS_CONNECTED);
	    csink_close (CSINK (sink));
	    csink_on_error (CSINK (sink), "NO_CONNECT");
	}
    } else {
      CDEBUG (("csink-inet", "connected\n"));
      sink->status &= ~ISS_CONNECT_INPROGRESS;
      sink->status = sink->status | ISS_CONNECTED;
      csink_add_fd (sink->fd, EIO_READ, csink_inet_socket_action,
		    CSINK (sink));
      csink_on_connect (CSINK (sink));
    }
}				/* END csink_inet_socket_connect */


static void
csink_inet_set_on_dns_lookup_success (CSinkInet *sink, CSinkCallbackFunc cb)
{
  sink->on_dns_lookup_success = cb;
}

static CSinkCallbackFunc
csink_inet_get_on_dns_lookup_success (CSinkInet *sink)
{
  return sink->on_dns_lookup_success;
}



static void
csink_host_cancel_lookup(CSinkInetDNSLookup *lookup)
{
  if (lookup->active) {
    /* close the pipe, if still open */
    pclose (lookup->fp);
    /* remove the watch tag, if still set */
    csink_remove_fd (lookup->fd_watch_tag);
    lookup->active = FALSE;
  }
}

static void
csink_host_free_lookup(CSinkInetDNSLookup *lookup)
{
  CDEBUG(("csink-inet", "freeing a lookup struct"));

  lookup->active = FALSE;

  /* free the mem */
  /* g_free(lookup);
   */
}


void
csink_dns_result (CSink *sink_cbdata)
{
  /* undo our cheating */
  CSinkInetDNSLookup *lookup = (CSinkInetDNSLookup*)sink_cbdata;
  CSinkInet *sink = lookup->sink;
  char buf[80];
  int count, res, fd;

  CDEBUG(("csink-inet", "csink_dns_result called\n"));

  if (!lookup->active) {
    CDEBUG(("csink-inet", "csink_dns_result called for an inactive lookup\n"));
    return;
  }

  fd = fileno(lookup->fp);
  count = read (fd, buf, 80);
  buf[count] = '\0';		/* this is to make it "safe" */
  res = pclose(lookup->fp);
  csink_remove_fd (lookup->fd_watch_tag);
  lookup->active = FALSE;

  if (res == EXIT_FAILURE) {
    CDEBUG(("csink-inet", "hostname lookup failed"));
    /* signal error */
    return;
  }

  CDEBUG(("csink-inet", "csink_dns_result ip returned was %s\n", buf));

  /* set the IP address */
  *lookup->in_addr_target = inet_addr(buf);  

  if (lookup->on_lookup) {
    lookup->on_lookup(CSINK(sink));
  }

}


/* We are using popen so we have to kill all the special
 * chars in the command. */
void normalize_resolve_cmd (gchar * resolve_cmd)
{
  char * str;

  str = resolve_cmd;

  while (*str != '\0')
    {
    
      if (	'\\' == *str	||
		'\'' == *str	||
		'\"' == *str	||
		'*'  == *str	||
		'&'  == *str	||
		'#'  == *str	||
		'('  == *str	||
		')'  == *str	||
		'|'  == *str	||
		'~'  == *str	||
		'\t' == *str	||
		'$'  == *str	||
		'{'  == *str	||
		'}'  == *str
	  )
        {
          *str = '_';	/* Assume that want _. :) */
        }

      str++;
    }
}

static CSinkInetDNSLookup *
csink_host_lookup_cb (CSinkInet * sink, gchar * hostname, uint32_t *inaddr,
		      CSinkCallbackFunc cb)
{
  FILE *fp;
  int fd;
  char resolve_cmd[256 + PATH_MAX + 1];
  CSinkInetDNSLookup *lookup;


  if(!hostname) {
    CDEBUG(("csink-inet", "asked to lookup null hostname.\n"));
    return NULL;
  }

  snprintf (resolve_cmd, 
	sizeof (resolve_cmd)-1, RESOLVER_BIN" %s", hostname);
  
  /* Need to keep malicious code from being ran when we popen(3). */
  normalize_resolve_cmd (resolve_cmd);

  fp = popen(resolve_cmd, "r");

  if(!fp) {
    CDEBUG(("csink-inet", "error opening resolve command\n"));
    return NULL;
  }


  /* Fileno doesn't fail if fp is good... */
  fd = fileno(fp);


  lookup = g_new0(CSinkInetDNSLookup, 1);
  lookup->active = TRUE;
  lookup->sink = sink;
  lookup->fp = fp;
  lookup->in_addr_target = inaddr;
  lookup->on_lookup = cb;

  lookup->fd_watch_tag = 
    csink_add_fd (fd, EIO_READ | EIO_ERROR, 
		  csink_dns_result, 
		  CSINK(lookup)); /* slight cheating here */

  return lookup;

#if 0
    struct hostent *host;
    host = gethostbyname (hostname);
    if (!host) {
	/* thow an error and leave */
	csink_close (CSINK (sink));
	switch (h_errno) {
	case HOST_NOT_FOUND:
	    csink_on_error (CSINK (sink), "host lookup HOST_NOT_FOUND");
	    break;
	case TRY_AGAIN:
	    /* should this be an error? for now, yes. */
	    csink_on_error (CSINK (sink), "host lookup TRY_AGAIN");
	    break;
	case NO_RECOVERY:
	    csink_on_error (CSINK (sink), "host lookup NO_RECOVERY");
	    break;
	case NO_ADDRESS:
	    csink_on_error (CSINK (sink), "host lookup NO_ADDRESS");
	    break;
	default:
	    csink_on_error (CSINK (sink), "host lookup unknown error.");
	}
	return (NULL);
    }

    return host;
#endif

}


static void
csink_set_remote_addr_action (CSink *sink_)
{
  CSinkInet *sink = CSINK_INET(sink_);

  memset (&sink->address, 0, sizeof (struct sockaddr_in));
  sink->address.sin_family = AF_INET;	/* internet naming format */

  /* note that this is only good for ipv4, 32bit addresses */
  memcpy ((char *) &sink->address.sin_addr, &sink->ip, sizeof(uint32_t));
  sink->address.sin_port = htons (sink->port);

  sink->status &= ~(ISS_DNS_INPROGRESS_REMOTE);
  csink_host_free_lookup(sink->remote_lookup);
  sink->remote_lookup = NULL;

  if (sink->on_dns_lookup_success) {
    sink->on_dns_lookup_success(CSINK(sink));
  }
}

static void
csink_inet_remote_sync (CSinkInet * sink)
{
  sink->ip = INADDR_NONE;

  /* cancel any previous lookup */
  if (sink->remote_lookup) {
    csink_host_cancel_lookup (sink->remote_lookup);
    csink_host_free_lookup (sink->remote_lookup);
    sink->remote_lookup = NULL;
  }
  
  /* this garauntees either an error or to set the address; eventually */
  sink->remote_lookup = 
    csink_host_lookup_cb (sink, sink->hostname,
			  &sink->ip, csink_set_remote_addr_action);

  if (sink->remote_lookup) {
    sink->status |= ISS_DNS_INPROGRESS_REMOTE;
  }
}


static void
csink_set_local_addr_action (CSink *sink_)
{
  CSinkInet *sink = CSINK_INET(sink_);

  memset (&sink->localaddress, 0, sizeof (struct sockaddr_in));
  sink->localaddress.sin_family = AF_INET;	/* internet naming format */

  /* note that this is only good for ipv4, 32bit addresses */
  memcpy ((char *) &sink->localaddress.sin_addr, &sink->localip, 
	  sizeof(uint32_t));
  sink->localaddress.sin_port = htons (sink->localport);

  sink->status &= ~(ISS_DNS_INPROGRESS_LOCAL);

  if (sink->local_lookup) {
    csink_host_free_lookup(sink->local_lookup);
    sink->local_lookup = NULL;
  }

  if (sink->on_dns_lookup_success)
    sink->on_dns_lookup_success(CSINK(sink));
}

static void
csink_inet_local_sync (CSinkInet * sink)
{

  /* cancel any previous lookup */
  if (sink->local_lookup) {
    csink_host_cancel_lookup (sink->local_lookup);
    csink_host_free_lookup (sink->local_lookup);
    sink->local_lookup = NULL;
  }

  sink->status |= ISS_DNS_INPROGRESS_LOCAL;
  if (sink->localinterface) {
    sink->local_lookup = 
      csink_host_lookup_cb (sink, sink->localinterface, &sink->localip, 
			    csink_set_local_addr_action);
  } else {
    sink->localip = htonl (INADDR_ANY);
    csink_set_local_addr_action (CSINK(sink));
  }
}

void
csink_inet_set_remote_host (CSinkInet * sink, const gchar * hostname)
{
    /* save the hostname */
    if (sink->hostname)
	g_free (sink->hostname);
    sink->hostname = g_strdup (hostname);

    CDEBUG (("csink-inet", "setting remote host to '%s'", hostname));

    /* magicly synch remote address structure */
    csink_inet_remote_sync (sink);
}

const gchar *
csink_inet_get_remote_host (CSinkInet * sink)
{
    return sink->hostname;
}

void
csink_inet_set_remote_port (CSinkInet * sink, const gint port)
{
    /* should have a sanity check on port here */
    sink->port = port;

    CDEBUG (("csink-inet", "setting remote port to %d", port));
    csink_inet_remote_sync (sink);
}

gint
csink_inet_get_remote_port (CSinkInet * sink)
{
    return sink->port;
}

void
csink_inet_set_local_port (CSinkInet * sink, const gint port)
{
    /* should have a sanity check on port here */
    sink->localport = port;

    csink_inet_local_sync (sink);
}

gint
csink_inet_get_local_port (CSinkInet * sink)
{
    return sink->localport;
}

void
csink_inet_set_local_interface (CSinkInet * sink, const gchar * hostname)
{
    /* save hostname */
    if (sink->localinterface)
	g_free (sink->localinterface);
    sink->localinterface = g_strdup (hostname);

    csink_inet_local_sync (sink);
}

const gchar *
csink_inet_get_local_interface (CSinkInet * sink)
{
    return sink->localinterface;
}


void
csink_inet_listen (CSinkInet * sink)
{
    int res;
    int true = 1;

    /* TODO: check state here and close down if nescessary. */

    if (sink->status & (ISS_CONNECTED |  ISS_CONNECT_INPROGRESS)) {
      CDEBUG (("csink-inet", 
	       "attempt to listen on an already connected sink."));
      return;
    }

    if (sink->status & ISS_DNS_INPROGRESS) {
      CDEBUG (("csink-inet",
	       "hostname lookup in progress, chaining on it's success."));
      csink_inet_set_on_dns_lookup_success (sink, csink_inet_listen);
      sink->status |= ISS_CONNECT_INPROGRESS;
      return;
    }

    sink->status &= ~(ISS_CONNECT_INPROGRESS);

    /* Do all of the fun stuff necessary to initialize a listening socket. */
    sink->fd = socket (AF_INET, SOCK_STREAM, 0);

    if (sink->fd < 0) {
	csink_on_error (CSINK (sink), "NO_SOCKET");
	return;
    }
    /* Set the SO_REUSEADDR option on the socket; this gets around the nasty
     * stupid fact that sockets aren't closed properly when a process exits. */
    res = setsockopt (sink->fd, SOL_SOCKET, SO_REUSEADDR, &true, sizeof (int));
    if (res < 0) {
	switch (errno) {
	case EBADF:		/* bad socket descriptor */
	    csink_on_error (CSINK (sink), "EBADF");
	    break;
	case ENOTSOCK:		/* fd was not a socket */
	    csink_on_error (CSINK (sink), "ENOTSOCK");
	    break;
	case ENOPROTOOPT:	/* that option doesn't make any sense at that 
				 * level */
	    csink_on_error (CSINK (sink), "ENOPROTOOPT");
	    break;
	default:
	    csink_on_error (CSINK (sink), "Unknown");
	}
	return;
    }


    /* Socket stuff. */
    fcntl (sink->fd, F_SETFL, O_NONBLOCK);

    res = bind (sink->fd, (struct sockaddr *) &sink->localaddress,
		sizeof (sink->localaddress));
    if (res < 0) {
	switch (errno) {
	case EBADF:		/* bad socket descriptor */
	    csink_on_error (CSINK (sink), "EBADF");
	    break;
	case ENOTSOCK:		/* fd was not a socket */
	    csink_on_error (CSINK (sink), "ENOTSOCK");
	    break;
	case EADDRNOTAVAIL:	/* local address/port  not available on this
				 * machine */
	    csink_on_error (CSINK (sink), "EADDRNOTAVAIL");
	    break;
	case EADDRINUSE:	/* that address/port is in use. */
	    csink_on_error (CSINK (sink), "EADDRINUSE");
	    break;
	case EINVAL:		/* socket already has an address */
	    csink_on_error (CSINK (sink), "EINVAL");
	    break;		/* is this fatal?? I don't know. */
	case EACCES:
	    csink_on_error (CSINK (sink), "EACCES");
	    break;
	default:
	    csink_on_error (CSINK (sink), "Unknown.");
	    break;
	}
	return;
    }

    res = listen (sink->fd, 2);
    if (res < 0) {
	switch (errno) {
	case EBADF:		/* bad socket descriptor */
	    csink_on_error (CSINK (sink), "EBADF");
	    break;
	case ENOTSOCK:		/* fd was not a socket */
	    csink_on_error (CSINK (sink), "ENOTSOCK");
	    break;
	case EOPNOTSUPP:	/* this is not a listenable socket */
	    csink_on_error (CSINK (sink), "EOPNOTSUPP");
	    break;
	}
	return;
    }

    sink->status |= ISS_CONNECTED;

    sink->watch_tag = csink_add_fd (sink->fd, EIO_READ | EIO_READ | EIO_ERROR,
				    csink_inet_accept_action, CSINK (sink));
}

/* a new accept()ed connection */
static void
csink_inet_accept_action (CSink * this_sink)
{
    CSinkInet *sink = CSINK_INET (this_sink);
    CSink *newsink;
    int connfd;
    struct sockaddr_in addr;
    socklen_t addrlen;


    CDEBUG (("csink-inet", "inet_accept on fd: %d\n", CSINK_INET (sink)->fd));

    /* accept the new socket connection */
    addrlen = sizeof (addr);
    connfd = accept (sink->fd, (struct sockaddr *) &addr, &addrlen);

    /* check for sanity */
    if (connfd == -1) {
	csink_on_error (CSINK (sink), "NO_SOCKET");
	return;
    }

    CDEBUG (("csink-inet", "connection from %s, port %d\n",
		inet_ntoa (*(struct in_addr*)&addr.sin_addr.s_addr), 
		ntohs (addr.sin_port)));    

    /* ensure that the new socket is non-blocking; whether or not it inherits
       this from it's associated listen() socket seems to be different from unix
       to unix. */
    fcntl (connfd, F_SETFL, O_NONBLOCK);


    newsink = CSINK (csink_inet_new ());


    csink_inet_set_local_port (CSINK_INET (newsink), sink->localport);
    csink_inet_set_remote_host (CSINK_INET(newsink), inet_ntoa(addr.sin_addr));
    csink_inet_set_remote_port (CSINK_INET(newsink), addr.sin_port);
    CSINK_INET (newsink)->fd = connfd;

    csink_set_new_data_func (newsink, CSINK (sink)->on_new_data);
    csink_set_user_data (newsink, CSINK (sink)->user_data);
    csink_set_close_func (newsink, CSINK (sink)->on_close);
    csink_set_connect_func (newsink, CSINK (sink)->on_connect);

    CSINK_INET (newsink)->status = ISS_CONNECTED;

    /* Save the ip address from the newly accepted connection. */
    CSINK_INET (newsink)->ip = (uint32_t) addr.sin_addr.s_addr;

    csink_on_connect (newsink);


    CSINK_INET (newsink)->watch_tag = csink_add_fd (connfd, EIO_READ,
						    csink_inet_socket_action,
						    newsink);
}


/* Connection poll function. */
static void
csink_inet_connect_action (CSink * this_sink)
{
    CSinkInet *sink = CSINK_INET (this_sink);

    CSINK_INET (sink)->status = ISS_CONNECTED;

    CDEBUG (("csink-inet", "connect callback.."));

    /* remove our watch, make a new one watching for incoming data */
    csink_remove_fd (sink->watch_tag);
    sink->watch_tag =
	csink_add_fd (sink->fd, EIO_READ, csink_inet_socket_action,
		      CSINK (this_sink));

    csink_on_connect (CSINK (sink));
}

/* there is data to be read  */
static void
csink_inet_socket_action (CSink * this_sink)
{
    CSinkInet *sink = CSINK_INET (this_sink);

    csink_on_new_data (CSINK (sink));
}
