#include <gtk/gtk.h>
#include <stdlib.h>
#include "entity.h"
#include "gtk-common.h"
#include "gtk-widget-attr.h"

/* 
 * See t/clist.e
 */

/* 
 * Because of the nature of of gtk clists we can't
 * grow the size of them, so we will create the clist with cols+3 columns
 * and only show the ones that need to be showed.  Or, if you set in the 
 * clist the max cols we will create the clist with that many instead.
 * 
 * Clists are pretty lightweight but, we don't want to bog down the system
 * with 256 columns or some ridiculus size when we only need 4 or 5
 * usually.
 */

/* This is the overall max cols there can be because we want to make a *
 * static version of the text[][] that we start with when we create * a new
 * row. */

#define MAX_CLIST_COLS 256

static void rendgtk_clist_title_child_attr (ENode * node,
					    EBuf * attr, EBuf * value,
					    gpointer user_data);
static void rendgtk_clist_row_child_attr (ENode * node, EBuf * attr,
					  EBuf * value, gpointer user_data);


static void
rendgtk_clist_click_column (GtkCList * clist, gint n, gpointer user_data)
{
    EDEBUG (("clist-renderer", "setting sort column to %d", n));
    gtk_clist_set_sort_column (GTK_CLIST (clist), n);
    gtk_clist_sort (GTK_CLIST (clist));
    /* gtk_clist_columns_autosize (GTK_CLIST (clist)); */
}

static void
rendgtk_clist_select_row (GtkWidget * widget, gint row, gint column,
			  GdkEventButton * event, gpointer user_data)
{
    ENode *clist_node = user_data;
    ENode *clrow = NULL;
    GtkWidget *clist;
    gchar *onselect;
    ENode *last_selected_row;

    clist = enode_get_kv (clist_node, "bottom-widget");
    if (!clist)
	return;

    clrow = gtk_clist_get_row_data (GTK_CLIST (clist), row);
    if (!clrow)
	return;

    last_selected_row = enode_get_kv (clist_node, "last_selected");
    if (last_selected_row) {
	enode_attrib_quiet (last_selected_row,
			    "selected", ebuf_new_with_false ());
    }
    enode_set_kv (clist_node, "last_selected", clrow);
    enode_attrib_quiet (clrow, "selected", ebuf_new_with_true ());

    onselect = enode_attrib_str (clrow, "onselect", NULL);
    if (onselect) {
	enode_call_ignore_return (clrow, onselect, "");
	return;
    }

    onselect = enode_attrib_str (clist_node, "onselect", NULL);
    enode_call_ignore_return (clrow, onselect, "");
}

static void
rendgtk_clist_unselect_row (GtkWidget * widget, gint row, gint column,
			    GdkEventButton * event, gpointer user_data)
{
    ENode *clist_node = user_data;
    ENode *clrow = NULL;
    GtkWidget *clist;
    gchar *ondeselect;

    clist = enode_get_kv (clist_node, "bottom-widget");
    if (!clist)
	return;

    clrow = gtk_clist_get_row_data (GTK_CLIST (clist), row);
    if (!clrow)
	return;

    ondeselect = enode_attrib_str (clrow, "ondeselect", NULL);
    if (ondeselect) {
	enode_call_ignore_return (clrow, ondeselect, "");
	return;
    }

    ondeselect = enode_attrib_str (clist_node, "ondeselect", NULL);
    enode_call_ignore_return (clrow, ondeselect, "");
}

static gint
rendgtk_clist_frozen_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *clist;

    clist = enode_get_kv (node, "bottom-widget");
    if (!clist)
	return TRUE;

    if (erend_value_is_true (value))
	gtk_clist_freeze (GTK_CLIST (clist));
    else
	gtk_clist_thaw (GTK_CLIST (clist));

    return TRUE;
}


#if 0
/* Borked */
static gint
rendgtk_clist_clheader_visible_attr_set (ENode * node, EBuf * attr,
					 EBuf * value)
{
    ENode *clist_node;
    GtkWidget *clist;
    gint col;

    clist_node = enode_parent (node, "clist");
    if (!clist_node)
	return (TRUE);

    col = GPOINTER_TO_INT (enode_get_kv (node, "rendgtk-clist-cl-header-col"));

    /* Show this col. */
    gtk_clist_set_column_visibility (GTK_CLIST (clist), col,
				     erend_value_is_true (value));

    return TRUE;
}

static gint
rendgtk_clist_sort_col_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *clist;
    int col;

    col = erend_get_integer (value);
    clist = enode_get_kv (node, "top-widget");
    gtk_clist_set_sort_column (GTK_CLIST (clist), col);
    gtk_clist_sort (GTK_CLIST (clist));
    return (TRUE);
}
#endif


static void
rendgtk_clist_new_child (ENode * new_node, gpointer user_data)
{
    ENode *clist_node = user_data;
    gint row;
    GtkWidget *clist;

    EDEBUG (("clist-renderer", "New child type %s", new_node->element->str));

    clist = enode_get_kv (clist_node, "bottom-widget");
    if (!clist) {
	EDEBUG (
		("clist-renderer",
		 "UUhh, clist widget not attached to node ?"));
	return;
    }

    if (ebuf_equal_str (new_node->element, "cl-row")) {
	int cols;
	static gchar *text[MAX_CLIST_COLS] = { "", /* ... */  };

	cols = GPOINTER_TO_INT (enode_get_kv (enode_parent (new_node, NULL),
					      "rendgtk-clist-createdcols"));

	/* Because the size is dynamic we need to make sure that we have *
	 * enough space in the text[][]. */

	/* @%*^% clist... */
	row = gtk_clist_append (GTK_CLIST (clist), text);
	EDEBUG (("clist-renderer", "Created new clist row, row #%d", row));
	gtk_clist_set_row_data (GTK_CLIST (clist), row, new_node);
	gtk_clist_columns_autosize (GTK_CLIST (clist));
	return;
    } else if (ebuf_equal_str (new_node->element, "string")) {
	GSList *attribs;
	GSList *tmp;
	ENode *parent;
	EBuf *attr;

	parent = enode_parent (new_node, NULL);
	if (!parent)
	    return;

	if (ebuf_equal_str (parent->element, "cl-row")) {
	    enode_event_watch_attrib (new_node,
				      rendgtk_clist_row_child_attr, clist_node);
	    attribs = enode_list_set_attribs (new_node);

	    tmp = attribs;
	    while (tmp) {
		attr = tmp->data;
		rendgtk_clist_row_child_attr (new_node, attr,
					      enode_attrib (new_node, attr->str,
							    NULL), clist_node);
		tmp = tmp->next;
	    }
	    g_slist_free (attribs);
	} else if (ebuf_equal_str (parent->element, "cl-title")) {
	    enode_event_watch_attrib (new_node,
				      rendgtk_clist_title_child_attr,
				      clist_node);
	    attribs = enode_list_set_attribs (new_node);

	    tmp = attribs;
	    while (tmp) {
		attr = tmp->data;
		rendgtk_clist_title_child_attr (new_node, attr,
						enode_attrib (new_node,
							      attr->str, NULL),
						clist_node);
		tmp = tmp->next;
	    }
	    g_slist_free (attribs);

	}
    }
}


static void
rendgtk_clist_row_child_attr (ENode * node, EBuf * attr,
			      EBuf * value, gpointer user_data)
{
    ENode *clist_node = user_data;
    ENode *row_node;
    gint row;
    gint column;
    GtkWidget *clist;

    EDEBUG (("clist-renderer", "in rendgtk_clist_row_child_attr"));

    if (!ebuf_equal_str (clist_node->element, "clist"))
	return;

    clist = enode_get_kv (clist_node, "bottom-widget");
    if (!clist)
	return;

    row_node = enode_parent (node, NULL);
    if (!ebuf_equal_str (row_node->element, "cl-row"))
	return;

    column = g_slist_index (row_node->children, node);
    row = gtk_clist_find_row_from_data (GTK_CLIST (clist), row_node);

    if (ebuf_equal_str (attr, "text")) {
	gtk_clist_set_text (GTK_CLIST (clist), row, column, value->str);

	EDEBUG (("clist-renderer",
		 "cl-row, adding text '%s' to (%d,%d)",
		 value->str, row, column));
    }
}


static void
rendgtk_clist_title_child_attr (ENode * node, EBuf * attr,
				EBuf * value, gpointer user_data)
{
    ENode *clist_node = user_data;
    ENode *title_node;
    gint column;
    GtkWidget *clist;

    EDEBUG (("clist-renderer", "in rendgtk_clist_title_child_attr"));

    if (!ebuf_equal_str (clist_node->element, "clist"))
	return;

    clist = enode_get_kv (clist_node, "bottom-widget");
    if (!clist)
	return;

    title_node = enode_parent (node, NULL);
    if (!ebuf_equal_str (title_node->element, "cl-title"))
	return;

    column = g_slist_index (title_node->children, node);

    if (ebuf_equal_str (attr, "text")) {
	EDEBUG (("clist-renderer", "Setting column %d to %s",
		 column, value->str));
	gtk_clist_set_column_title (GTK_CLIST (clist), column, value->str);

	if (ebuf_empty (value))
	    gtk_clist_set_column_visibility (GTK_CLIST (clist), column, FALSE);
	else {
	    gtk_widget_realize (clist);
	    gtk_clist_set_column_visibility (GTK_CLIST (clist), column, TRUE);
	}

    } else if (ebuf_equal_str (attr, "width")) {
	int width;
	EBuf *visible;

	width = erend_get_integer (value);
	EDEBUG (("clist-renderer", "Setting width of %i to %i", column, width));

	/* FIXME, doesn't seem to be working... */
	gtk_clist_set_column_width (GTK_CLIST (clist), column, width);

	if (width) {
	    visible = enode_attrib (node, "visible", NULL);
	    if (ebuf_empty (visible) || erend_value_is_true (visible)) {
		/* Make sure we set it visible if it should be showing. */
		gtk_clist_set_column_visibility (GTK_CLIST (clist), column,
						 TRUE);
	    }
	} else {		/* If 0 we'll just set it invisible.  Nice
				 * trick for the user. */

	    gtk_clist_set_column_visibility (GTK_CLIST (clist), column, FALSE);
	}
    }
}


static void
rendgtk_clist_delete_child (ENode * del_node, gpointer user_data)
{
    ENode *clist_node = user_data;
    gint row;
    GtkWidget *clist;

    EDEBUG (
	    ("clist-renderer", "Deleting child type %s",
	     del_node->element->str));

    clist = enode_get_kv (clist_node, "bottom-widget");

    ECHECK_RET (clist != NULL);

    if (ebuf_equal_str (del_node->element, "cl-row")) {
	row = gtk_clist_find_row_from_data (GTK_CLIST (clist), del_node);
	EDEBUG (("clist-renderer", "Deleting row %d from list", row));
	gtk_clist_remove (GTK_CLIST (clist), row);
    }
}


static void
rendgtk_clist_render (ENode * node)
{
    GtkWidget *clist;
    GtkWidget *sw;
    int i;
    int cols;
    int maxcols = 0;
    int createdcols = 0;

    cols = erend_get_integer (enode_attrib (node, "cols", NULL));
    maxcols = erend_get_integer (enode_attrib (node, "maxcols", NULL));


    /* We just can't go over this value or there will be major problems. */
    if (maxcols > MAX_CLIST_COLS)
	maxcols = MAX_CLIST_COLS;

    /* We need to make sure we can grow the clists if there is need. * The
     * next few lines tries to set up the default values to use * for the
     * user. */

    /* If maxcols can't be right(less than created.) then, use cols +3 */
    if (cols > maxcols)
	createdcols = cols + 3;
    else
	createdcols = maxcols;

    /* If the number of cols to create is 0 then make extra so we can grow. */
    if (createdcols == 0)
	createdcols = 20;

    /* Show only the first col if they don't set cols="". */
    if (cols == 0)
	cols = 1;

    createdcols += 3;		/* Add 3 for good measure. :) */


    /* Will need to know this in the parenting process. */
    enode_set_kv (node,
		  "rendgtk-clist-createdcols", GINT_TO_POINTER (createdcols));

    EDEBUG (("clist-renderer",
	     "Creating clist with %d columns, %d visible", createdcols, cols));
    clist = gtk_clist_new (createdcols);

    gtk_signal_connect (GTK_OBJECT (clist), "click_column",
			GTK_SIGNAL_FUNC (rendgtk_clist_click_column), NULL);
    gtk_signal_connect (GTK_OBJECT (clist), "select_row",
			GTK_SIGNAL_FUNC (rendgtk_clist_select_row), node);
    gtk_signal_connect (GTK_OBJECT (clist), "unselect_row",
			GTK_SIGNAL_FUNC (rendgtk_clist_unselect_row), node);

    sw = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_container_add (GTK_CONTAINER (sw), clist);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_widget_show (sw);

    enode_set_kv (node, "top-widget", sw);
    enode_set_kv (node, "bottom-widget", clist);


    for (i = 0; i < cols; i++) {
	EDEBUG (("clist-renderer", "Showing column %i", i));
	gtk_clist_set_column_visibility (GTK_CLIST (clist), i, TRUE);
    }

    /* Don't show the extra cols we create. */
    for (i = cols; i < createdcols; i++) {
	EDEBUG (("clist-renderer", "Hiding column %i", i));
	gtk_clist_set_column_visibility (GTK_CLIST (clist), i, FALSE);
    }

    gtk_clist_column_titles_show (GTK_CLIST (clist));

    enode_attribs_sync (node);

    enode_event_watch_child_new (node, rendgtk_clist_new_child, node);
    enode_event_watch_child_delete (node, rendgtk_clist_delete_child, node);

    rendgtk_show_cond (node, clist);
}

static gint
rendgtk_clrow_selected_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *clist;
    gint row;

    clist = enode_get_kv (enode_parent (node, "clist"), "bottom-widget");
    row = gtk_clist_find_row_from_data (GTK_CLIST (clist), node);

    if (NULL == clist || -1 == row) {
	return TRUE;
    }

    if (erend_value_is_true (value)) {
	gtk_clist_select_row (GTK_CLIST (clist), row, 0);
	/* Emits the selected signal. */
    } else {
	gtk_clist_unselect_row (GTK_CLIST (clist), row, 0);
	/* Emits the unselected signal. */
    }
    return TRUE;
}

static gint
rendgtk_clrow_style_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkStyle *style;
    GtkWidget *clist;
    gint row;
    gint cols;
    gint i;

    EDEBUG (("clist-renderer", "in rendgtk_clrow_style_attr_set\n"));

    if (!node->parent)
	return TRUE;

    clist = enode_get_kv (enode_parent (node, "clist"), "bottom-widget");
    if (!clist)
	return TRUE;

    row = gtk_clist_find_row_from_data (GTK_CLIST (clist), node);
    if (row == -1)
	return TRUE;

    cols = GPOINTER_TO_INT (enode_get_kv (enode_parent (node, NULL),
					  "rendgtk-clist-createdcols"));

    for (i = 0; i < cols; i++) {
	/* Build a new style. */
	style = rendgtk_style_parser (value, rendgtk_rc_get_style (clist));

	gtk_clist_set_cell_style (GTK_CLIST (clist), row, i, style);
    }

    return TRUE;
}


void
clist_renderer_register (void)
{
    Element *element;
    ElementAttr *e_attr;

  /**** <clist> ****/

    element = g_new0 (Element, 1);
    element->render_func = rendgtk_clist_render;
    element->destroy_func = rendgtk_element_destroy;
    element->tag = "clist";
    element->description = "Create a columned list widget";
    element_register (element);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onselect";
    e_attr->description = "Function to call when a cl-row is selected";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(selected_node)";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "ondeselect";
    e_attr->description = "Function to call when a cl-row is deselected";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(deselected_node)";
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "frozen";
    e_attr->description = "When set true the clist is frozen so dynamic "
	"updates don't show until it is set false.";
    e_attr->value_desc = "boolean";
    e_attr->possible_values = "true,false";
    e_attr->set_attr_func = rendgtk_clist_frozen_attr_set;
    element_register_attrib (element, e_attr);

    rendgtk_widget_attr_register (element, GTK_TYPE_CLIST);


  /**** <cl-row> ****/

    element = g_new0 (Element, 1);
    element->tag = "cl-row";
    element->description =
	"Create a row within a clist.  Use the 'string' tag for individual entries.";
    element_register (element);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onselect";
    e_attr->description = "Function to call when a cl-row is selected";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(selected_node)";
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "ondeselect";
    e_attr->description = "Function to call when a cl-row is deselected";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(deselected_node)";
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "selected";
    e_attr->description = "The selected state of this row.";
    e_attr->value_desc = "boolean";
    e_attr->possible_values = "true,false";
    e_attr->set_attr_func = rendgtk_clrow_selected_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "style";
    e_attr->description = "The look and color of this row.";
    e_attr->value_desc = "string";
    e_attr->possible_values = "Comma seperated style rules.";
    e_attr->set_attr_func = rendgtk_clrow_style_attr_set;
    element_register_attrib (element, e_attr);

}
