#include <gtk/gtk.h>
#include <stdlib.h>
#include "entity.h"
#include "gtk-common.h"
#include "gtk-widget-attr.h"

/* 
 * See t/ctree.e for example usage.  To make an sub tree just put a new
 * <ctree-row> inside the parent <ctree-row>.
 */

/* 
 * Because of the nature of of gtk ctrees we can't grow the size
 * of them, so we will create the ctree with cols+3 columns
 * and only show the ones that need to be showed.  Or, if you set in the 
 * clist the max cols we will create the clist with that many instead.
 * 
 * Ctrees are pretty lightweight but, we don't want to bog down the system
 * with 256 columns or some ridiculus size when we only need 4 or 5
 * usually.
 */

/* This is the overall max cols there can be because we want to make a *
 * static version of the text[][] that we start with when we create * a new
 * row. */

#define MAX_CTREE_COLS 256


/* Pixmap Cache */
typedef struct {
    GdkPixmap *pixmap;
    GdkBitmap *mask;
} PixmapCacheEntry;


/**********************************************************************
 * CTree Cell Renderer
 *********************************************************************/
static void
rendgtk_ctree_cell_load_xpm_image (ENode * node, gchar * filename,
				   GdkPixmap ** ret_pixmap,
				   GdkBitmap ** ret_mask)
{
    GdkPixmap *pixmap;
    GdkBitmap *mask;
    gchar *realfile;
    static GtkWidget *window = NULL;
    static GHashTable *cache;
    PixmapCacheEntry *pce;

    *ret_pixmap = NULL;
    *ret_mask = NULL;

    if (!window) {
	window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_widget_realize (window);
	cache = g_hash_table_new (g_str_hash, g_str_equal);
    }

    realfile = eutils_file_search (node, filename);

    if (!realfile) {
	/* FIXME: Should use onerror handler */
	g_warning ("Unable to locate file '%s'", filename);
	return;
    }

    pce = g_hash_table_lookup (cache, realfile);
    if (pce) {
	EDEBUG (("ctree-image-renderer", "Pulling file %s from cache",
		 realfile));
	*ret_pixmap = pce->pixmap;
	*ret_mask = pce->mask;
	gdk_pixmap_ref (pce->pixmap);
	gdk_pixmap_ref (pce->mask);
    } else {
	EDEBUG (("ctree-image-renderer",
		 "attempting to load real imagefile '%s'", realfile));

	pixmap = gdk_pixmap_create_from_xpm (window->window, &mask,
					     &window->style->
					     bg[GTK_STATE_NORMAL], realfile);
	pce = g_malloc (sizeof (PixmapCacheEntry));
	pce->pixmap = pixmap;
	pce->mask = mask;
	gdk_pixmap_ref (pixmap);
	gdk_pixmap_ref (mask);

	EDEBUG (("ctree-image-renderer",
		 "inserting filename '%s' into cache", realfile));

	g_hash_table_insert (cache, g_strdup (realfile), pce);

	*ret_pixmap = pixmap;
	*ret_mask = mask;
    }

    g_free (realfile);
}


static gint
rendgtk_ctree_cell_text_image_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *ctree;
    ENode *ctree_node;
    ENode *parent;
    GtkCTreeNode *parent_cnode = NULL;
    gint column;
    EBuf *text_value;

    column = GPOINTER_TO_INT (enode_get_kv (node, "ctree-cell-column-number"));
    parent = enode_get_kv (node, "ctree-cell-parent-row");
    if (!parent)
	return (TRUE);

    parent_cnode = enode_get_kv (parent, "ctree-row-node");
    if (!parent_cnode)
	return (TRUE);

    ctree_node = enode_parent (node, "ctree");
    if (!ctree_node)
	return (TRUE);

    ctree = enode_get_kv (ctree_node, "bottom-widget");
    if (!ctree)
	return (TRUE);

    EDEBUG (("ctree-renderer",
	     "Setting cell text for column %d, on ctree node %p, ctree %p to %s",
	     column, ctree_node, ctree, value->str));


    text_value = enode_attrib (node, "text", NULL);

    if (column == 0) {
	EBuf *collapsed_image;
	EBuf *expanded_image;
	EBuf *image;
	GdkPixmap *pixmap = NULL;
	GdkBitmap *mask = NULL;
	GdkPixmap *pixmap_expanded = NULL;
	GdkBitmap *mask_expanded = NULL;
	GdkPixmap *pixmap_collapsed = NULL;
	GdkBitmap *mask_collapsed = NULL;

	/* Check for old pixmaps first */
	pixmap = enode_get_kv (node, "gtk-ctree-cell-pixmap");
	if (pixmap) {
	    gdk_pixmap_unref (pixmap);
	    enode_set_kv (node, "gtk-ctree-cell-pixmap", NULL);
	}
	mask = enode_get_kv (node, "gtk-ctree-cell-mask");
	if (mask) {
	    gdk_pixmap_unref (mask);
	    enode_set_kv (node, "gtk-ctree-cell-mask", NULL);
	}
	pixmap = enode_get_kv (node, "gtk-ctree-cell-pixmap-expanded");
	if (pixmap) {
	    gdk_pixmap_unref (pixmap);
	    enode_set_kv (node, "gtk-ctree-cell-pixmap-expanded", NULL);
	}
	mask = enode_get_kv (node, "gtk-ctree-cell-mask-expanded");
	if (mask) {
	    gdk_pixmap_unref (mask);
	    enode_set_kv (node, "gtk-ctree-cell-mask-expanded", NULL);
	}
	pixmap = enode_get_kv (node, "gtk-ctree-cell-pixmap-collapsed");
	if (pixmap) {
	    gdk_pixmap_unref (pixmap);
	    enode_set_kv (node, "gtk-ctree-cell-pixmap-collapsed", NULL);
	}
	mask = enode_get_kv (node, "gtk-ctree-cell-mask-collapsed");
	if (mask) {
	    gdk_pixmap_unref (mask);
	    enode_set_kv (node, "gtk-ctree-cell-mask-collapsed", NULL);
	}

	collapsed_image = enode_attrib (node, "collapsed-image", NULL);
	expanded_image = enode_attrib (node, "expanded-image", NULL);

	/* We only do it if they are both set */
	if (ebuf_not_empty (expanded_image) && ebuf_not_empty (collapsed_image)) {

	    EDEBUG (("ctree-renderer", "Loading dual expanded/collapsed"
		     " images for expander column."));

	    rendgtk_ctree_cell_load_xpm_image (node, collapsed_image->str,
					       &pixmap_collapsed,
					       &mask_collapsed);

	    enode_set_kv (node, "gtk-ctree-cell-pixmap-collapsed",
			  pixmap_collapsed);
	    enode_set_kv (node, "gtk-ctree-cell-mask-collapsed",
			  mask_collapsed);

	    rendgtk_ctree_cell_load_xpm_image (node, expanded_image->str,
					       &pixmap_expanded,
					       &mask_expanded);

	    enode_set_kv (node, "gtk-ctree-cell-pixmap-expanded",
			  pixmap_expanded);

	    enode_set_kv (node, "gtk-ctree-cell-mask-expanded", mask_expanded);
	}

	/* If those aren't set, we have to check for just the image attribute 
	 */
	if (!pixmap_expanded || !pixmap_collapsed) {
	    EDEBUG (("ctree-renderer",
		     "doing single image load for expanded/collapsed"));
	    image = enode_attrib (node, "image", NULL);
	    EDEBUG (("ctree-renderer", "image value is '%s'", image->str));
	    if (ebuf_not_empty (image)) {
		EDEBUG (("ctree-renderer",
			 "actually going through with single load!"));

		rendgtk_ctree_cell_load_xpm_image (node, image->str,
						   &pixmap_collapsed,
						   &mask_collapsed);

		enode_set_kv (node, "gtk-ctree-cell-pixmap", pixmap_collapsed);
		enode_set_kv (node, "gtk-ctree-cell-mask", mask_collapsed);
		pixmap_expanded = pixmap_collapsed;
		mask_expanded = mask_collapsed;
	    }
	}

	if (pixmap_expanded && pixmap_collapsed) {
	    EDEBUG (("ctree-renderer",
		     "doing full gtk_ctree_set_node_info for column 0"));

	    gtk_ctree_set_node_info (GTK_CTREE (ctree), parent_cnode,
				     text_value->str, 3,
				     pixmap_collapsed, mask_collapsed,
				     pixmap_expanded, mask_expanded,
				     FALSE, FALSE);
	} else {
	    EDEBUG (("ctree-renderer", "loading text only for column 0"));

	    gtk_ctree_node_set_text (GTK_CTREE (ctree), parent_cnode,
				     column, value->str);
	}

    } else {
	EBuf *image;
	GdkPixmap *pixmap;
	GdkBitmap *mask;

	/* Check for old pixmaps first */
	pixmap = enode_get_kv (node, "gtk-ctree-cell-pixmap");
	if (pixmap) {
	    gdk_pixmap_unref (pixmap);
	    enode_set_kv (node, "gtk-ctree-cell-pixmap", NULL);
	}
	mask = enode_get_kv (node, "gtk-ctree-cell-mask");
	if (mask) {
	    gdk_pixmap_unref (mask);
	    enode_set_kv (node, "gtk-ctree-cell-mask", NULL);
	}

	image = enode_attrib (node, "image", NULL);

	if (ebuf_not_empty (image)) {

	    rendgtk_ctree_cell_load_xpm_image (node, image->str,
					       &pixmap, &mask);
	    enode_set_kv (node, "gtk-ctree-cell-pixmap", pixmap);
	    enode_set_kv (node, "gtk-ctree-cell-mask", mask);
	}
	if (pixmap)
	    gtk_ctree_node_set_pixtext (GTK_CTREE (ctree),
					parent_cnode, column, text_value->str,
					3, pixmap, mask);
	else
	    gtk_ctree_node_set_text (GTK_CTREE (ctree), parent_cnode,
				     column, value->str);
    }
    return (TRUE);
}

/* Render a cell.. the only thing we really do is get it's row number, so *
 * that it can easily set the contents with the attrib_sync */

static void
rendgtk_ctree_cell_render (ENode * node)
{
    GSList *tmp;
    ENode *parent;
    gint column = 0;

    parent = enode_parent (node, NULL);
    if (!ebuf_equal_str (enode_type (parent), "ctree-row")) {
	g_warning ("<ctree-cell> may only be placed directly"
		   " below a <ctree-row>");
	return;
    }

    tmp = parent->children;

    while (tmp) {
	ENode *tmpnode = tmp->data;

	if (tmpnode == node)
	    break;

	column++;
	tmp = tmp->next;
    }

    enode_set_kv (node, "ctree-cell-column-number", GINT_TO_POINTER (column));
    enode_set_kv (node, "ctree-cell-parent-row", parent);

    enode_attribs_sync (node);
}

static void
rendgtk_ctree_cell_destroy (ENode * node)
{
    GdkPixmap *pixmap;
    GdkBitmap *mask;

    pixmap = enode_get_kv (node, "gtk-ctree-cell-pixmap");
    mask = enode_get_kv (node, "gtk-ctree-cell-mask");

    enode_attrib (node, "image", ebuf_new_with_str (""));
    enode_attrib (node, "text", ebuf_new_with_str (""));
}


/**********************************************************************
 * CTree Row Renderer
 *********************************************************************/

/* from gtkctree.c */
/* gives the left pixel of the given column in context of
 * the clist's hoffset */
#define COLUMN_LEFT_XPIXEL(clist, colnum)  ((clist)->column[(colnum)].area.x + \
					    (clist)->hoffset)


static void
rendgtk_ctree_button_press_event_callback (GtkWidget * widget,
					   GdkEventButton * event,
					   gpointer user_data)
{
    ENode *ctree_node = user_data;
    GSList *tmp,
    *children;
    ENode *cell_node = NULL;
    ENode *row_node;
    GtkWidget *ctree;
    gint row;
    gint column;
    gint click_in_ctree;
    gfloat offset;
    gchar *function;
    gchar *callback;
    
    ctree = enode_get_kv (ctree_node, "bottom-widget");
    if (!ctree)
	return;

    click_in_ctree = gtk_clist_get_selection_info (GTK_CLIST (ctree),
						   event->x, event->y, &row,
						   &column);

    if (!click_in_ctree)
	return;

    offset = event->x - COLUMN_LEFT_XPIXEL(GTK_CLIST (ctree), column);
    
    EDEBUG (("ctree-renderer", 
	"Button press event in column %d, row %d.  x %f, y %f, offset %f\n", 
	column, row, event->x, event->y, offset));
   

    row_node = gtk_clist_get_row_data (GTK_CLIST (ctree), row);

    if (!row_node)
	return;

    /* Find the child cell */
    children = enode_children (row_node, "ctree-cell");
    tmp = children;
    while (tmp) {
	ENode *child = tmp->data;
	gint tmp_column;

	tmp_column = GPOINTER_TO_INT (enode_get_kv
				      (child, "ctree-cell-column-number"));
	if (tmp_column == column)
	    cell_node = child;

	tmp = tmp->next;
    }
    g_slist_free (children);

    /* 
     * g_print ("click in ctree, row %d, column %d, node %s.%s\n", row,
     * column, row_node->element->str, enode_attrib_str (row_node, "name",
     * NULL)); */

    if (event->type == GDK_2BUTTON_PRESS) {
	callback = "ondoubleclick";
    } else {
	callback = "onbuttonpress";
    }

    function = enode_attrib_str (row_node, callback, NULL);
    if (!function)
	function = enode_attrib_str (ctree_node, callback, NULL);

    if (function)
	enode_call_ignore_return (ctree_node, function, "nniii",
				  row_node, cell_node, event->button, column, (gint) offset);
}

static void
rendgtk_ctree_row_selected_callback (GtkWidget * ctree_widget,
				     GtkCTreeNode * cnode, gint column,
				     gpointer user_data)
{
    ENode *node;
    ENode *ctree = user_data;
    gchar *function;

    node = gtk_ctree_node_get_row_data (GTK_CTREE (ctree_widget), cnode);

    EDEBUG (("ctree-renderer",
	     "Calling onselect callback on column %d, node %s.%s",
	     column, node->element->str,
	     enode_attrib_str (node, "name", NULL)));

    enode_attrib_quiet (node, "selected", ebuf_new_with_true ());

    /* Check row node first */
    function = enode_attrib_str (node, "onselect", NULL);
    if (!function)
	function = enode_attrib_str (ctree, "onselect", NULL);

    if (function)
	enode_call_ignore_return (ctree, function, "ni", node, column);
}

static void
rendgtk_ctree_row_unselected_callback (GtkWidget * ctree_widget,
				       GtkCTreeNode * cnode, gint column,
				       gpointer user_data)
{
    ENode *node;
    ENode *ctree = user_data;
    gchar *function;

    node = gtk_ctree_node_get_row_data (GTK_CTREE (ctree_widget), cnode);

    EDEBUG (("ctree-renderer",
	     "Calling ondeselect callback on column %d, node %s.%s",
	     column, node->element->str,
	     enode_attrib_str (node, "name", NULL)));

    enode_attrib_quiet (node, "selected", ebuf_new_with_str (""));

    /* Check row node first */
    function = enode_attrib_str (node, "ondeselect", NULL);
    if (!function)
	function = enode_attrib_str (ctree, "ondeselect", NULL);

    if (function)
	enode_call_ignore_return (ctree, function, "ni", node, column);
}

static void
rendgtk_ctree_row_expand_callback (GtkWidget * ctree_widget,
				   GtkCTreeNode * cnode, gpointer user_data)
{
    ENode *node;
    ENode *ctree = user_data;
    gchar *function;

    node = gtk_ctree_node_get_row_data (GTK_CTREE (ctree_widget), cnode);

    EDEBUG (("ctree-renderer", "Calling expanded callback for node %s.%s",
	     node->element->str, enode_attrib_str (node, "name", NULL)));

    enode_attrib_quiet (node, "expanded", ebuf_new_with_true ());

    /* Check row node first */
    function = enode_attrib_str (node, "onexpand", NULL);
    if (!function)
	function = enode_attrib_str (ctree, "onexpand", NULL);

    if (function)
	enode_call_ignore_return (ctree, function, "n", node);
}

static void
rendgtk_ctree_row_collapse_callback (GtkWidget * ctree_widget,
				     GtkCTreeNode * cnode, gpointer user_data)
{
    ENode *node;
    ENode *ctree = user_data;
    gchar *function;

    node = gtk_ctree_node_get_row_data (GTK_CTREE (ctree_widget), cnode);

    EDEBUG (("ctree-renderer", "Calling collapse callback for node %s.%s",
	     node->element->str, enode_attrib_str (node, "name", NULL)));

    enode_attrib_quiet (node, "expanded", ebuf_new_with_str (""));

    /* Check row node first */
    function = enode_attrib_str (node, "oncollapse", NULL);
    if (!function)
	function = enode_attrib_str (ctree, "oncollapse", NULL);

    if (function)
	enode_call_ignore_return (ctree, function, "n", node);
}


static gint
rendgtk_ctree_row_expanded_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    ENode *ctree_node;
    GtkWidget *ctree;
    GtkCTreeNode *cnode;

    ctree_node = enode_parent (node, "ctree");
    if (!ctree_node)
	return (TRUE);

    ctree = enode_get_kv (ctree_node, "bottom-widget");
    if (!ctree)
	return (TRUE);

    cnode = enode_get_kv (node, "ctree-row-node");

    if (!cnode)
	return (TRUE);

    EDEBUG (("ctree-renderer", "Setting expanded attribute to %s", value->str));

    if (erend_value_is_true (value)) {
	gtk_ctree_expand (GTK_CTREE (ctree), cnode);
    } else {
	gtk_ctree_collapse (GTK_CTREE (ctree), cnode);
    }

    return (TRUE);
}

static gint
rendgtk_ctree_row_selected_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    ENode *ctree_node;
    GtkWidget *ctree;
    GtkCTreeNode *cnode;

    ctree_node = enode_parent (node, "ctree");
    if (!ctree_node)
	return (TRUE);

    ctree = enode_get_kv (ctree_node, "bottom-widget");
    if (!ctree)
	return (TRUE);

    cnode = enode_get_kv (node, "ctree-row-node");

    if (!cnode)
	return (TRUE);

    EDEBUG (("ctree-renderer", "Setting selected attribute to %s", value->str));

    if (erend_value_is_true (value))
	gtk_ctree_select (GTK_CTREE (ctree), cnode);
    else
	gtk_ctree_unselect (GTK_CTREE (ctree), cnode);

    return (TRUE);
}

static gint
rendgtk_ctree_row_style_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    ENode *ctree_node;
    GtkStyle *style;
    GtkWidget *ctree;
    GtkCTreeNode *cnode;
    gint cols;
    gint i;

    EDEBUG (("ctree-renderer", "ctree row style attr set"));

    ctree_node = enode_parent (node, "ctree");
    if (!ctree_node)
	return (TRUE);

    ctree = enode_get_kv (ctree_node, "bottom-widget");
    if (!ctree)
	return (TRUE);

    cnode = enode_get_kv (node, "ctree-row-node");
    if (!cnode)
	return (TRUE);

    cols = GPOINTER_TO_INT (enode_get_kv (ctree_node,
					  "rendgtk-ctree-createdcols"));

    for (i = 0; i < cols; i++) {
	style = gtk_ctree_node_get_cell_style (GTK_CTREE (ctree), cnode, i);

	if (!style)
	    style = gtk_style_copy (GTK_WIDGET (ctree)->style);

	/* Build a new style. */
	style = rendgtk_style_parser (value, rendgtk_rc_get_style (ctree));

	gtk_ctree_node_set_cell_style (GTK_CTREE (ctree), cnode, i, style);
	gtk_style_unref (style);
    }

    return (TRUE);
}


static void
rendgtk_ctree_row_render (ENode * node)
{
    ENode *ctree_node;
    ENode *parent;
    GtkCTreeNode *parent_cnode = NULL;
    GtkCTreeNode *sibling_cnode;
    GtkWidget *ctree;

    EDEBUG (("ctree-renderer", "Creating ctree row"));

    ctree_node = enode_parent (node, "ctree");
    if (!ctree_node) {
	return;
    }

    ctree = enode_get_kv (ctree_node, "bottom-widget");
    if (!ctree)
	return;

    parent = enode_parent (node, "ctree-row");
    if (parent)
	parent_cnode = enode_get_kv (parent, "ctree-row-node");

    EDEBUG (("ctree-renderer",
	     "Creating ctree row, using tree %p, parent ctree node %p",
	     ctree, parent_cnode));

    sibling_cnode = gtk_ctree_insert_node (GTK_CTREE (ctree),
					   GTK_CTREE_NODE (parent_cnode), NULL,
					   NULL, 5, NULL, NULL, NULL, NULL,
					   FALSE, FALSE);

    EDEBUG (("ctree-renderer", "Created new ctree sibling %p", sibling_cnode));

    enode_set_kv (node, "ctree-row-node", sibling_cnode);
    gtk_ctree_node_set_row_data (GTK_CTREE (ctree), sibling_cnode, node);
}

static void
rendgtk_ctree_row_destroy (ENode * node)
{
    GtkCTreeNode *sibling_cnode;
    GtkWidget *ctree;
    ENode *ctree_node;

    sibling_cnode = enode_get_kv (node, "ctree-row-node");

    ctree_node = enode_parent (node, "ctree");
    if (!ctree_node) {
	return;
    }

    ctree = enode_get_kv (ctree_node, "bottom-widget");
    if (!ctree)
	return;

    gtk_ctree_remove_node (GTK_CTREE (ctree), sibling_cnode);
}


/**********************************************************************
 * CTree Column Renderer
 *********************************************************************/

static gint
rendgtk_ctree_column_active_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    ENode *ctree_node;
    GtkWidget *ctree;
    gint column;

    ctree_node = enode_parent (node, "ctree");
    if (!ctree_node)
	return (TRUE);

    ctree = enode_get_kv (ctree_node, "bottom-widget");
    if (!ctree)
	return TRUE;

    column = GPOINTER_TO_INT (enode_get_kv (node, "ctree-column-number"));

    if (erend_value_is_true (value))
	gtk_clist_column_title_active (GTK_CLIST (ctree), column);
    else
	gtk_clist_column_title_passive (GTK_CLIST (ctree), column);

    return TRUE;
}

static gint
rendgtk_ctree_column_title_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *ctree;
    ENode *ctree_node;
    gint column;

    ctree_node = enode_parent (node, "ctree");
    if (!ctree_node)
	return (TRUE);

    ctree = enode_get_kv (ctree_node, "bottom-widget");
    if (!ctree)
	return TRUE;

    column = GPOINTER_TO_INT (enode_get_kv (node, "ctree-column-number"));

    gtk_clist_set_column_title (GTK_CLIST (ctree), column, value->str);

    return TRUE;
}

static gint
rendgtk_ctree_column_auto_resize_attr_set (ENode * node, EBuf * attr,
					   EBuf * value)
{
    GtkWidget *ctree;
    ENode *ctree_node;
    gint column;
    gint val;

    ctree_node = enode_parent (node, "ctree");
    if (!ctree_node)
	return (TRUE);

    ctree = enode_get_kv (ctree_node, "bottom-widget");
    if (!ctree)
	return TRUE;

    column = GPOINTER_TO_INT (enode_get_kv (node, "ctree-column-number"));

    val = erend_value_is_true (value);
    gtk_clist_set_column_auto_resize (GTK_CLIST (ctree), column, val);

    return TRUE;
}


static gint
rendgtk_ctree_column_resizeable_attr_set (ENode * node, EBuf * attr,
					  EBuf * value)
{
    GtkWidget *ctree;
    ENode *ctree_node;
    gint column;
    gint val;

    ctree_node = enode_parent (node, "ctree");
    if (!ctree_node)
	return (TRUE);

    ctree = enode_get_kv (ctree_node, "bottom-widget");
    if (!ctree)
	return TRUE;

    column = GPOINTER_TO_INT (enode_get_kv (node, "ctree-column-number"));

    val = erend_value_is_true (value);
    gtk_clist_set_column_resizeable (GTK_CLIST (ctree), column, val);

    return TRUE;
}


static gint
rendgtk_ctree_column_width_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *ctree;
    ENode *ctree_node;
    gint column;
    gint val;

    ctree_node = enode_parent (node, "ctree");
    if (!ctree_node)
	return (TRUE);

    ctree = enode_get_kv (ctree_node, "bottom-widget");
    if (!ctree)
	return TRUE;

    column = GPOINTER_TO_INT (enode_get_kv (node, "ctree-column-number"));

    val = erend_get_integer (value);
    gtk_clist_set_column_width (GTK_CLIST (ctree), column, val);

    return TRUE;
}

static void
rendgtk_ctree_columns_sync (ENode * ctree_node)
{
    GtkWidget *ctree;
    GSList *children;
    GSList *tmp;
    int column = 0;

    ctree = enode_get_kv (ctree_node, "bottom-widget");
    if (!ctree)
	return;

    children = enode_children (ctree_node, NULL);

    tmp = children;
    while (tmp) {
	ENode *node = tmp->data;

	if (ebuf_equal_str (enode_type (node), "ctree-column")) {
	    gchar *title = enode_attrib_str (node, "title", NULL);
	    gtk_clist_set_column_title (GTK_CLIST (ctree), column, title);
	    gtk_clist_set_column_visibility (GTK_CLIST (ctree), column, TRUE);
	    enode_set_kv (node, "ctree-column-number",
			  GINT_TO_POINTER (column));
	    column++;
	}

	tmp = tmp->next;
    }

    g_slist_free (children);
}


static void
rendgtk_ctree_column_render (ENode * node)
{
    ENode *ctree_node;

    ctree_node = enode_parent (node, "ctree");
    if (!ctree_node) {
	return;
    } else {
	rendgtk_ctree_columns_sync (ctree_node);
    }

    enode_attribs_sync (node);
}

static void
rendgtk_ctree_column_destroy (ENode * node)
{
    GtkWidget *ctree;
    ENode *ctree_node;
    int column;

    ctree_node = enode_parent (node, "ctree");
    if (!ctree_node)
	return;


    ctree = enode_get_kv (ctree_node, "bottom-widget");
    if (!ctree)
	return;

    column = GPOINTER_TO_INT (enode_get_kv (node, "ctree-column-number"));

    gtk_clist_set_column_visibility (GTK_CLIST (ctree), column, FALSE);
    rendgtk_ctree_columns_sync (ctree_node);
}


/**********************************************************************
 * CTree Renderer
 *********************************************************************/

static gint
rendgtk_ctree_frozen_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *ctree;

    ctree = enode_get_kv (node, "bottom-widget");
    if (!ctree)
	return TRUE;

    if (erend_value_is_true (value))
	gtk_clist_freeze (GTK_CLIST (ctree));
    else
	gtk_clist_thaw (GTK_CLIST (ctree));

    return TRUE;
}

static gint
rendgtk_ctree_show_titles_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *ctree;

    ctree = enode_get_kv (node, "bottom-widget");
    if (!ctree)
	return TRUE;

    if (erend_value_is_true (value))
	gtk_clist_column_titles_show (GTK_CLIST (ctree));
    else
	gtk_clist_column_titles_hide (GTK_CLIST (ctree));

    return TRUE;
}

static gint
rendgtk_ctree_line_style_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *ctree;

    ctree = enode_get_kv (node, "bottom-widget");
    if (!ctree)
	return TRUE;

    if (ebuf_equal_str (value, "solid")) {
	gtk_ctree_set_line_style (GTK_CTREE (ctree), GTK_CTREE_LINES_SOLID);
    } else if (ebuf_equal_str (value, "none")) {
	gtk_ctree_set_line_style (GTK_CTREE (ctree), GTK_CTREE_LINES_NONE);
    } else {
	gtk_ctree_set_line_style (GTK_CTREE (ctree), GTK_CTREE_LINES_DOTTED);
    }

    return TRUE;
}

static gint
rendgtk_ctree_expander_style_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *ctree;

    ctree = enode_get_kv (node, "bottom-widget");
    if (!ctree)
	return TRUE;

    if (ebuf_equal_str (value, "none")) {
	gtk_ctree_set_expander_style (GTK_CTREE (ctree),
				      GTK_CTREE_EXPANDER_NONE);
    } else if (ebuf_equal_str (value, "square")) {
	gtk_ctree_set_expander_style (GTK_CTREE (ctree),
				      GTK_CTREE_EXPANDER_SQUARE);
    } else {
	gtk_ctree_set_expander_style (GTK_CTREE (ctree),
				      GTK_CTREE_EXPANDER_TRIANGLE);
    }

    return TRUE;
}



static gint
rendgtk_ctree_spacing_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *ctree;
    gint spacing;

    ctree = enode_get_kv (node, "bottom-widget");
    if (!ctree)
	return (TRUE);

    spacing = erend_get_integer (value);
    gtk_ctree_set_spacing (GTK_CTREE (ctree), spacing);
    return (TRUE);
}

static gint
rendgtk_ctree_row_height_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *ctree;
    gint height;

    ctree = enode_get_kv (node, "bottom-widget");
    if (!ctree)
	return (TRUE);

    height = erend_get_integer (value);
    gtk_clist_set_row_height (GTK_CLIST (ctree), height);
    return (TRUE);
}

static gint
rendgtk_ctree_indent_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *ctree;
    gint indent;

    ctree = enode_get_kv (node, "bottom-widget");
    if (!ctree)
	return (TRUE);

    indent = erend_get_integer (value);
    gtk_ctree_set_indent (GTK_CTREE (ctree), indent);
    return (TRUE);
}

static gint
rendgtk_ctree_selection_type_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *ctree;

    ctree = enode_get_kv (node, "bottom-widget");
    if (!ctree)
	return (TRUE);

    if (ebuf_equal_str (value, "multiple")) {
	gtk_clist_set_selection_mode (GTK_CLIST (ctree),
				      GTK_SELECTION_MULTIPLE);
    } else if (ebuf_equal_str (value, "browse")) {
	gtk_clist_set_selection_mode (GTK_CLIST (ctree), GTK_SELECTION_BROWSE);
    } else if (ebuf_equal_str (value, "extended")) {
	gtk_clist_set_selection_mode (GTK_CLIST (ctree),
				      GTK_SELECTION_EXTENDED);
    } else {
	gtk_clist_set_selection_mode (GTK_CLIST (ctree), GTK_SELECTION_SINGLE);
    }

    return (TRUE);
}

static void
rendgtk_ctree_render (ENode * node)
{
    GtkWidget *ctree;
    int i;
    int cols;
    int maxcols = 0;
    int createdcols = 0;

    cols = erend_get_integer (enode_attrib (node, "columns", NULL));
    maxcols = erend_get_integer (enode_attrib (node, "maxcolumns", NULL));

    /* We just can't go over this value or there will be major problems. */
    if (maxcols > MAX_CTREE_COLS)
	maxcols = MAX_CTREE_COLS;

    /* We need to make sure we can grow the ctrees if there is need. * The
     * next few lines tries to set up the default values to use * for the
     * user. */

    /* If maxcols can't be right(less than created.) then, use cols +3 */
    if (cols > maxcols)
	createdcols = cols + 3;
    else
	createdcols = maxcols;

    /* If the number of cols to create is 0 then make extra so we can grow. */
    if (createdcols == 0)
	createdcols = 20;

    /* Show only the first col if they don't set cols="". */
    if (cols == 0)
	cols = 1;

    /* Add 3 for good measure. :) */
    createdcols += 3;


    /* Will need to know this in the parenting process. */
    enode_set_kv (node, "rendgtk-ctree-createdcols",
		  GINT_TO_POINTER (createdcols));

    EDEBUG (("ctree-renderer", "Creating ctree with %d columns, %d visible",
	     createdcols, cols));

    ctree = gtk_ctree_new (createdcols, 0);

    gtk_ctree_set_expander_style (GTK_CTREE (ctree),
				  GTK_CTREE_EXPANDER_TRIANGLE);

    /* default */
    gtk_ctree_set_line_style (GTK_CTREE (ctree), GTK_CTREE_LINES_DOTTED);

    gtk_clist_column_titles_show (GTK_CLIST (ctree));
    gtk_clist_column_titles_active (GTK_CLIST (ctree));
    gtk_ctree_set_indent (GTK_CTREE (ctree), 10);
    gtk_ctree_set_spacing (GTK_CTREE (ctree), 5);


    gtk_signal_connect (GTK_OBJECT (ctree), "tree_select_row",
			GTK_SIGNAL_FUNC (rendgtk_ctree_row_selected_callback),
			node);
    gtk_signal_connect (GTK_OBJECT (ctree), "tree_unselect_row",
			GTK_SIGNAL_FUNC (rendgtk_ctree_row_unselected_callback),
			node);
    gtk_signal_connect (GTK_OBJECT (ctree), "tree_expand",
			GTK_SIGNAL_FUNC (rendgtk_ctree_row_expand_callback),
			node);
    gtk_signal_connect_after (GTK_OBJECT (ctree), "tree_collapse",
			      GTK_SIGNAL_FUNC
			      (rendgtk_ctree_row_collapse_callback), node);
    gtk_signal_connect (GTK_OBJECT (ctree), "button_press_event",
			GTK_SIGNAL_FUNC
			(rendgtk_ctree_button_press_event_callback), node);

    enode_set_kv (node, "top-widget", ctree);
    enode_set_kv (node, "bottom-widget", ctree);


    for (i = 0; i < cols; i++) {
	EDEBUG (("ctree-renderer", "Showing column %i", i));
	gtk_clist_set_column_visibility (GTK_CLIST (ctree), i, TRUE);
    }

    /* Don't show the extra cols we create. */
    for (i = cols; i < createdcols; i++) {
	EDEBUG (("ctree-renderer", "Hiding column %i", i));
	gtk_clist_set_column_visibility (GTK_CLIST (ctree), i, FALSE);
    }

    enode_attribs_sync (node);
    rendgtk_show_cond (node, ctree);
}



void
ctree_renderer_register (void)
{
    Element *element;
    ElementAttr *e_attr;

    /**** <ctree> ****/

    element = g_new0 (Element, 1);
    element->render_func = rendgtk_ctree_render;
    element->destroy_func = rendgtk_element_destroy;
    element->tag = "ctree";
    element->description = "Create a columned tree widget";
    element_register (element);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onexpand";
    e_attr->description = "Function to call when a child row is expanded";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(ctree_node, expanded_node)";
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "oncollapse";
    e_attr->description = "Function to call when a child row is collapsed.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(ctree_node, collapsed_node)";
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onselect";
    e_attr->description = "Function to call when a child row is selected";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(ctree_node, selected_node, selected_column)";
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "ondeselect";
    e_attr->description = "Function to call when a child row is deselected";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(ctree_node, deselected_node, selected_column)";
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "frozen";
    e_attr->description = "When set true the ctree is frozen so dynamic "
	"updates don't show until it is set false.";
    e_attr->value_desc = "boolean";
    e_attr->possible_values = "true,false";
    e_attr->set_attr_func = rendgtk_ctree_frozen_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "expander-style";
    e_attr->description = "Set the style of the expander used to draw trees.";
    e_attr->value_desc = "choice";
    e_attr->possible_values = "triangle,square,none";
    e_attr->set_attr_func = rendgtk_ctree_expander_style_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "line-style";
    e_attr->description = "Set line style used for drawing trees.";
    e_attr->value_desc = "choice";
    e_attr->possible_values = "dotted,solid,none";
    e_attr->set_attr_func = rendgtk_ctree_line_style_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "show-titles";
    e_attr->description =
	"Enable or disable the showing of the column titles in the ctree.";
    e_attr->value_desc = "boolean";
    e_attr->set_attr_func = rendgtk_ctree_show_titles_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "row-height";
    e_attr->description = "Set the height of a single row.";
    e_attr->value_desc = "integer";
    e_attr->possible_values = "0,*";
    e_attr->set_attr_func = rendgtk_ctree_row_height_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "spacing";
    e_attr->description = "Set spacing between items.";
    e_attr->value_desc = "integer";
    e_attr->possible_values = "0,*";
    e_attr->set_attr_func = rendgtk_ctree_spacing_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "indent";
    e_attr->description = "Set indent for child node.";
    e_attr->value_desc = "integer";
    e_attr->possible_values = "0,*";
    e_attr->set_attr_func = rendgtk_ctree_indent_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "selection-type";
    e_attr->description = "Set selection type";
    e_attr->value_desc = "choice";
    e_attr->possible_values = "single,multiple,browse,extended";
    e_attr->set_attr_func = rendgtk_ctree_selection_type_attr_set;
    element_register_attrib (element, e_attr);

    rendgtk_widget_attr_register (element, GTK_TYPE_CTREE);

    /* This one has to go after the widget_attr_register, as we are
     * overriding * the default version for widgets. */
    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onbuttonpress";
    e_attr->description = "Function to call whenever a mouse button is pressed in a row";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(ctree_node, row_node, cell_node, button, row_column, x_position)";
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "ondoubleclick";
    e_attr->description = "Function to call when a row is double clicked.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(ctree_node, row_node, cell_node, button, row_column, x_position)";
    element_register_attrib (element, e_attr);


    /**** <ctree-column> ****/

    element = g_new0 (Element, 1);
    element->tag = "ctree-column";
    element->description = "Create columns for a clist";
    element->render_func = rendgtk_ctree_column_render;
    element->destroy_func = rendgtk_ctree_column_destroy;
    element_register (element);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onclick";
    e_attr->description = "Function to call when a ctree-title is clicked";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(selected_node)";
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "active";
    e_attr->description = "Specify if title should allow keypress events"
	" etc., or just be a passive title.";
    e_attr->value_desc = "boolean";
    e_attr->set_attr_func = rendgtk_ctree_column_active_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "title";
    e_attr->description = "Set title for column.";
    e_attr->value_desc = "string";
    e_attr->set_attr_func = rendgtk_ctree_column_title_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "width";
    e_attr->description = "Set width of column.";
    e_attr->value_desc = "integer";
    e_attr->possible_values = "0,*";
    e_attr->set_attr_func = rendgtk_ctree_column_width_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "resizeable";
    e_attr->description = "Specify if the column is user resizeable.";
    e_attr->value_desc = "boolean";
    e_attr->set_attr_func = rendgtk_ctree_column_resizeable_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "auto-resize";
    e_attr->description =
	"Specify if column should automatically resize to fit its contents.";
    e_attr->value_desc = "boolean";
    e_attr->set_attr_func = rendgtk_ctree_column_auto_resize_attr_set;
    element_register_attrib (element, e_attr);

    /**** <ctree-row> ****/

    element = g_new0 (Element, 1);
    element->tag = "ctree-row";
    element->description = "Create a row within a clist.  Use the"
	" 'ctree-cell' tag for individual entries.";
    element->render_func = rendgtk_ctree_row_render;
    element->destroy_func = rendgtk_ctree_row_destroy;
    element_register (element);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "expanded";
    e_attr->description =
	"Determine or set if this row is expanded to show its children.";
    e_attr->value_desc = "boolean";
    e_attr->set_attr_func = rendgtk_ctree_row_expanded_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onexpand";
    e_attr->description = "Function to call when a row is expanded";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(ctree_node, expanded_node)";
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "oncollapse";
    e_attr->description = "Function to call when a row is collapsed.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(ctree_node, collapsed_node)";
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "selected";
    e_attr->description = "Determine or set if this row is selected.";
    e_attr->value_desc = "boolean";
    e_attr->set_attr_func = rendgtk_ctree_row_selected_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onselect";
    e_attr->description = "Function to call when a row is selected";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(ctree_node, selected_node, selected_column)";
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "ondeselect";
    e_attr->description = "Function to call when a row is deselected";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(ctree_node, deselected_node, selected_column)";
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "ondoubleclick";
    e_attr->description = "Function to call when a row is double clicked.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(ctree_node, row_node, cell_node, button, row_column, x_position)";
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onbuttonpress";
    e_attr->description =
	"Function to call whenever a mouse button is pressed in a row";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(ctree_node, row_node, cell_node, button, row_column, x_position)";
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "style";
    e_attr->description = "The look and color of this row.";
    e_attr->value_desc = "string";
    e_attr->possible_values = "Comma seperated style rules.";
    e_attr->set_attr_func = rendgtk_ctree_row_style_attr_set;
    element_register_attrib (element, e_attr);

    /**** <ctree-cell> ****/

    element = g_new0 (Element, 1);
    element->tag = "ctree-cell";
    element->description = "Create a column within a row.";
    element->render_func = rendgtk_ctree_cell_render;
    element->destroy_func = rendgtk_ctree_cell_destroy;
    element_register (element);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "text";
    e_attr->description = "Text to be displayed in a cell.";
    e_attr->value_desc = "string";
    e_attr->set_attr_func = rendgtk_ctree_cell_text_image_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "image";
    e_attr->description = "Image to use within a cell.";
    e_attr->value_desc = "file";
    e_attr->set_attr_func = rendgtk_ctree_cell_text_image_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "expanded-image";
    e_attr->description = "If this cell is in the expander column,"
	" set the image to be shown when expanded.";
    e_attr->value_desc = "file";
    e_attr->set_attr_func = rendgtk_ctree_cell_text_image_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "collapsed-image";
    e_attr->description = "If this cell is in the expander column, set"
	" the image to be shown when collapsed.";
    e_attr->value_desc = "file";
    e_attr->set_attr_func = rendgtk_ctree_cell_text_image_attr_set;
    element_register_attrib (element, e_attr);
}


