#include <gtk/gtk.h>
#include "entity.h"
#include "gtk-common.h"
#include "gtk-widget-attr.h"

static gint
rendgtk_window_idle_show (gpointer user_data)
{
    ENode *node = user_data;
    GtkWidget *window;

    window = enode_get_kv (node, "top-widget");
    if (window)
	rendgtk_show_cond (node, window);
    return (FALSE);
}

static gint
rendgtk_window_realize_position (gpointer user_data)
{
    ENode *node = user_data;

    gint xipos;
    gint yipos;
    EBuf *val;

    GtkWidget *window;

    window = enode_get_kv (node, "top-widget");

    val = enode_attrib (node, "y-position", NULL);
    if (ebuf_not_empty (val))
	yipos = erend_get_integer (val);
    else
	yipos = -1;

    val = enode_attrib (node, "x-position", NULL);
    if (ebuf_not_empty (val))
	xipos = erend_get_integer (val);
    else
	xipos = -1;

#ifndef WIN32
    gtk_window_reposition (GTK_WINDOW (window), xipos, yipos);
#endif				/* WIN32 */
    return (FALSE);
}

static gint
rendgtk_window_realize_icon (gpointer user_data)
{
    ENode *node = user_data;

    GtkWidget *window;
    GdkPixmap *icon;

    gchar *filename;
    gchar *fpath;

    window = enode_get_kv (node, "top-widget");
    if (!window)
	return (FALSE);

    if (!window->window)
	gtk_widget_realize (window);

    filename = enode_attrib_str (node, "icon", NULL);
    fpath = eutils_file_search (node, filename);

    icon = gdk_pixmap_create_from_xpm (window->window, NULL, NULL, fpath);
    gdk_window_set_icon (window->window, NULL, icon, NULL);
    enode_set_kv (node, "icon-pixmap", icon);
    return (FALSE);
}

static gint
rendgtk_window_ondelete_callback (GtkWidget * widget,
				  GdkEvent * event, gpointer user_data)
{
    ENode *node = user_data;
    gchar *function = NULL;

    function = enode_attrib_str (node, "ondelete", NULL);

    enode_call_ignore_return (node, function, "");

    return (TRUE);
}

static void
rendgtk_window_resize_callback (GtkWidget * widget,
				GdkEventConfigure * event, gpointer user_data)
{
    ENode *node = user_data;
    gchar value[100];
    EBuf *val;

    /* to remember window width and height.  May also want to check an *
     * attribute in case they don't want this to be saved */

    /* We only save the new value, if the old value was set. */
    val = enode_attrib (node, "width", NULL);
    if (ebuf_not_empty (val)) {
	g_snprintf (value, 100, "%d", event->width);
	enode_attrib_quiet (node, "width", ebuf_new_with_str (value));
    }

    val = enode_attrib (node, "height", NULL);
    if (ebuf_not_empty (val)) {
	g_snprintf (value, 100, "%d", event->height);
	enode_attrib_quiet (node, "height", ebuf_new_with_str (value));
    }
}

static gint
rendgtk_window_key_repeat_attr (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *window;

    window = enode_get_kv (node, "top-widget");
    if (!window)
	return FALSE;

    if (enode_attrib_is_true (node, "key-auto-repeat"))
	gdk_key_repeat_restore ();
    else
	gdk_key_repeat_disable ();

    return (TRUE);
}


static gint
rendgtk_window_set_title_attr (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *window;

    window = enode_get_kv (node, "top-widget");
    if (!window)
	return FALSE;

    gtk_window_set_title (GTK_WINDOW (window), value->str);

    /* 2nd arg is wmclass_name, 3rd arg is wmclass_class */
    /* May want to move this into a seperate attr */
    if (!GTK_WIDGET_REALIZED (window))
	gtk_window_set_wmclass (GTK_WINDOW (window), value->str, value->str);

    return (TRUE);
}

static gint
rendgtk_window_set_xyposition_attr (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *window;

    window = enode_get_kv (node, "top-widget");
    if (!window)
	return (TRUE);

    gtk_idle_add (rendgtk_window_realize_position, node);

    return (TRUE);
}

static gint
rendgtk_window_set_position_attr (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *window;
    gint intval;

    window = enode_get_kv (node, "top-widget");
    if (!window)
	return FALSE;

    intval = GTK_WIN_POS_NONE;

    if (ebuf_equal_str (value, "center"))
	intval = GTK_WIN_POS_CENTER;

    if (ebuf_equal_str (value, "mouse"))
	intval = GTK_WIN_POS_MOUSE;

    gtk_window_set_position (GTK_WINDOW (window), intval);

    return (TRUE);
}

static gint
rendgtk_window_set_icon (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *window;

    window = enode_get_kv (node, "top-widget");
    if (!window)
	return (FALSE);

    gtk_idle_add (rendgtk_window_realize_icon, node);

    return (TRUE);
}

static void
rendgtk_window_pack (ENode * parent_node, ENode * child_node)
{
    GtkWidget *window;

    /* Only show window once something gets packed into it, should make it
     * pop up better */
    window = enode_get_kv (parent_node, "top-widget");

    rendgtk_box_pack (parent_node, child_node);
}

static void
rendgtk_window_destroy (ENode * node)
{
    GdkPixmap *icon;

    icon = enode_get_kv (node, "icon-pixmap");
    if (icon)
	gdk_pixmap_unref (icon);

    gdk_key_repeat_restore ();

    rendgtk_element_destroy (node);
}

static void
rendgtk_window_render (ENode * node)
{
    GtkWidget *window;
    GtkWidget *vbox;

    if (enode_attrib_is_true (node, "dialog")) {
	window = gtk_window_new (GTK_WINDOW_DIALOG);
    } else {
	window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    }

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (window), vbox);

    enode_set_kv (node, "top-widget", window);
    enode_set_kv (node, "bottom-widget", vbox);

    gtk_signal_connect (GTK_OBJECT (window), "delete_event",
			GTK_SIGNAL_FUNC (rendgtk_window_ondelete_callback),
			(gpointer) node);

    gtk_signal_connect (GTK_OBJECT (window), "configure_event",
			GTK_SIGNAL_FUNC (rendgtk_window_resize_callback),
			(gpointer) node);

    gtk_window_set_policy (GTK_WINDOW (window), TRUE, TRUE, FALSE);

    enode_attribs_sync (node);

    gtk_widget_show (vbox);

    gtk_idle_add (rendgtk_window_idle_show, node);
    /* rendgtk_show_cond (node, window); */
}


void
window_renderer_register (void)
{
    Element *element;
    ElementAttr *e_attr;

    element = g_new0 (Element, 1);
    element->render_func = rendgtk_window_render;
    element->destroy_func = rendgtk_window_destroy;
    element->parent_func = rendgtk_window_pack;
    element->tag = "window";
    element->description = "Create a new top-level window.";
    element_register (element);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "ondelete";
    e_attr->description = "Specify function to call when window is 'closed'.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(window_node)";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "title";
    e_attr->description = "Set title and wmclass of window.";
    e_attr->value_desc = "string";
    e_attr->set_attr_func = rendgtk_window_set_title_attr;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "position";
    e_attr->description = "Set position of window.";
    e_attr->value_desc = "choice";
    e_attr->possible_values = "any,center,mouse";
    e_attr->set_attr_func = rendgtk_window_set_position_attr;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "x-position";
    e_attr->description = "Set absolute x position of window.";
    e_attr->value_desc = "integer";
    e_attr->possible_values = "-1,*";
    e_attr->set_attr_func = rendgtk_window_set_xyposition_attr;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "y-position";
    e_attr->description = "Set absolute y position of window.";
    e_attr->value_desc = "integer";
    e_attr->possible_values = "-1,*";
    e_attr->set_attr_func = rendgtk_window_set_xyposition_attr;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "icon";
    e_attr->description = "The xpm file to use as the icon";
    e_attr->value_desc = "string";
    e_attr->set_attr_func = rendgtk_window_set_icon;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "dialog";
    e_attr->description =
	"Specify if this window should be a dialog window, or regular toplevel window.";
    e_attr->value_desc = "boolean";
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "key-auto-repeat";
    e_attr->description = "Enable or disable auto repeat on the keyboard.";
    e_attr->value_desc = "boolean";
    e_attr->set_attr_func = rendgtk_window_key_repeat_attr;
    element_register_attrib (element, e_attr);

    rendgtk_widget_attr_register (element, GTK_TYPE_WINDOW);
    rendgtk_containerbox_attr_register (element);
}
