#ifndef lint
static char *RCSid = "$Header: functions.c,v 1.1 87/01/18 13:26:06 halazar Exp $";
#endif

/*
 * functions.c - functions for the webster server
 *
 * David A. Curry
 * Purdue University
 * Engineering Computer Network
 * April, 1986
 *
 * $Log:	functions.c,v $
 * Revision 1.1  87/01/18  13:26:06  halazar
 * Initial revision
 * 
 * Revision 1.2  86/12/26  22:02:17  davy
 * Changed to work with DBM files.
 * 
 */
#include <sys/types.h>
#include <ctype.h>
#include <stdio.h>

#include "../h/index.h"
#include "../h/webster.h"

extern FILE *output;			/* output file pointer		*/
extern char *wordbuf;			/* word storage buffer		*/
extern char *websterhelp;		/* path to help file		*/

extern int nanswers;			/* number of answers in answers	*/
extern int answers_valid;		/* non-zero if numbers work	*/
extern struct answer answers[MAXANSWERS]; /* answers found in search	*/

int comp();

/*
 * define - find and print the definition of a word.
 */
define(ac, av)
int ac;
char **av;
{
	register char *s;
	register int i, j;
	char word[BUFSIZ];

	s = word;
	*s = NULL;

	for (i=1; i < ac; i++) {
		strcat(word, av[i]);

		if ((ac - i) > 1)
			strcat(word, " ");
	}

	/*
	 * Make the word all lower case.
	 */
	while (*s) {
		if (isupper(*s))
			*s = tolower(*s);
		s++;
	}

	/* 
	 * If the answers are valid and the user sent a number,
	 * define the word in answers[number].
	 */
	if (answers_valid && isnumber(word)) {
		i = atoi(word);

		if (i > nanswers) {
			error(0, "%d: out of range.", i);
			return(-1);
		}

		av[1] = answers[i-1].a_word;
		answers_valid = 0;

		return(define(ac, av));
	}

	/*
	 * If the word has wildcards in it, look it up
	 * and send back a WILD response.
	 */
	if (haswild(word)) {
		wildsearch(word, 0);

		if (nanswers == 0) {
			fprintf(output, "WILD 0\r\n");
			return(1);
		}

		/*
		 * Sort the answers.
		 */
		qsort(answers, nanswers, sizeof(struct answer), comp);

		fprintf(output, "WILD\r\n");

		/*
		 * Output words and word numbers.
		 */
		j = 1;
		for (i=0; i < nanswers; i++) {
			if ((i > 0) && (strcmp(answers[i].a_word, answers[i-1].a_word) == 0))
				continue;
			fprintf(output, "%d %s\r\n", j++, answers[i].a_word);
		}

		/*
		 * End output with an EOF marker.
		 */
		putc(EOFCH, output);
		answers_valid = 1;

		return(1);
	}

	/*
	 * Look the word up the easy way.
	 */
	easysearch(word, 0);

	/*
	 * If we found it, print the definition.
	 */
	if (nanswers == 1) {
		printdef(&answers[0]);
		answers_valid = 0;

		return(1);
	}

	/*
	 * Do What I Mean.
	 */
	return(dwim(word));
}

/*
 * complete - attempt to do Tenex-style word completion.
 */
complete(ac, av)
int ac;
char **av;
{
	register int i;
	register char *s;
	char word[BUFSIZ];

	s = word;
	*s = NULL;

	for (i=1; i < ac; i++) {
		strcat(word, av[i]);

		if ((ac - i) > 1)
			strcat(word, " ");
	}

	/*
	 * Convert to lower case.
	 */
	while (*s) {
		if (isupper(*s))
			*s = tolower(*s);
		s++;
	}

	/*
	 * Add a "match 0 or more" wildcard.
	 */
	*s++ = MANYCH;
	*s = NULL;

	/*
	 * Search for the word.
	 */
	wildsearch(word, 0);
	answers_valid = 0;

	/*
	 * It's either ambiguous, meaning we didn't find "just one",
	 * or we complete it.
	 */
	if (nanswers != 1) {
		fprintf(output, "AMBIGUOUS %d\r\n", nanswers);
		return(0);
	}

	fprintf(output, "COMPLETION %s\r\n", answers[0].a_word);
	return(1);
}

/*
 * endings - do Tenex-style endings stuff.
 */
endings(ac, av)
int ac;
char **av;
{
	register char *s;
	register int i, j;
	char word[BUFSIZ];

	s = word;
	*s = NULL;

	for (i=1; i < ac; i++) {
		strcat(word, av[i]);

		if ((ac - i) > 1)
			strcat(word, " ");
	}

	/*
	 * Convert to lower case.
	 */
	while (*s) {
		if (isupper(*s))
			*s = tolower(*s);
		s++;
	}

	/*
	 * Add a "match zero or more" wildcard.
	 */
	*s++ = MANYCH;
	*s = NULL;

	/*
	 * Search for the word.
	 */
	wildsearch(word, 0);

	/*
	 * Not found.
	 */
	if (nanswers == 0) {
		fprintf(output, "MATCHS 0\r\n");
		return(1);
	}

	/*
	 * Print the matches, along with numbers.
	 */
	fprintf(output, "MATCHS\r\n");

	qsort(answers, nanswers, sizeof(struct answer), comp);

	j = 1;
	for (i=0; i < nanswers; i++) {
		if ((i > 0) && (strcmp(answers[i].a_word, answers[i-1].a_word) == 0))
			continue;
		fprintf(output, "%d %s\r\n", j++, answers[i].a_word);
	}

	putc(EOFCH, output);
	answers_valid = 1;

	return(1);
}

/*
 * spell - check the spelling of a word.
 */
spell(ac, av)
int ac;
char **av;
{
	register int i;
	register char *s;
	char word[BUFSIZ];

	s = word;
	*s = NULL;

	for (i=1; i < ac; i++) {
		strcat(word, av[i]);

		if ((ac - i) > 1)
			strcat(word, " ");
	}

	/*
	 * Convert to lower case.
	 */
	while (*s) {
		if (isupper(*s))
			*s = tolower(*s);
		s++;
	}

	/*
	 * No wildcards.
	 */
	if (haswild(word)) {
		error(0, "Wildcards not allowed in SPELL requests.", 0);
		return(-1);
	}

	/*
	 * Look it up the easy way.
	 */
	easysearch(word, 0);

	/*
	 * Spelled correctly.
	 */
	if (nanswers == 1) {
		fprintf(output, "SPELLING 1\r\n", nanswers);
		answers_valid = 0;

		return(nanswers);
	}

	/*
	 * Look for possible alternate spellings.
	 */
	return(dwim(word));
}

/*
 * help - print the protocol document.
 */
help(ac, av)
int ac;
char **av;
{
	FILE *fp;
	register int c;

	if ((fp = fopen(websterhelp, "r")) == NULL) {
		error(0, "Cannot open \"%s\".", websterhelp);
		return(-1);
	}

	while ((c = getc(fp)) != EOF) {
		if (c == '\n')
			putc('\r', output);

		putc(c, output);
	}

	putc(EOFCH, output);
	fclose(fp);

	return(1);
}

/*
 * dwim - Do What I Mean.  We try to find words that are "almost"
 *	  what the user wanted.  This code is a crock, but it's neato.
 */
dwim(word)
char *word;
{
	int i, j;
	char buf[BUFSIZ];
	register char *p, *q, *s, *t;

	nanswers = 0;

	/*
	 * Try searching as if they transposed two letters.
	 */
	s = word;
	while (*(s+1)) {
		p = buf;
		t = word;

		while (*t) {
			if (t == s) {
				*p++ = *(t+1);
				*p++ = *t;
				t += 2;
			}
			else {
				*p++ = *t++;
			}
		}

		*p = NULL;
		easysearch(buf, nanswers);

		s++;
	}

	/*
	 * Try searching as if they didn't give us one
	 * letter.
	 */
	s = word + 1;
	while (*s) {
		p = buf;
		t = word;

		while (t < s)
			*p++ = *t++;
		*p++ = ONECH;

		while (*t)
			*p++ = *t++;

		*p = NULL;
		wildsearch(buf, nanswers);

		s++;
	}

	strcat(strcpy(buf, word), "%");
	wildsearch(buf, nanswers);

	/*
	 * Try searching as if they had given us an extra letter.
	 */
	s = word;
	while (*s) {
		p = buf;
		t = word;

		while (*t) {
			if (t != s)
				*p++ = *t;
			t++;
		}

		*p = NULL;
		easysearch(buf, nanswers);

		s++;
	}

	/*
	 * Try searching as if they gave us one letter wrong.
	 * Assume the first letter is right though.
	 */
	s = word + 1;
	while (*s) {
		p = buf;
		t = word;

		while (*t) {
			if (t == s)
				*p++ = ONECH;
			else
				*p++ = *t;
			t++;
		}

		*p = NULL;
		wildsearch(buf, nanswers);

		s++;
	}

	/*
	 * All that work, and we didn't find anything.
	 */
	if (nanswers == 0) {
		fprintf(output, "SPELLING 0\r\n");
		answers_valid = 0;

		return(0);
	}

	/*
	 * Show them what we found.
	 */
	fprintf(output, "SPELLING\r\n");

	qsort(answers, nanswers, sizeof(struct answer), comp);

	j = 1;
	for (i=0; i < nanswers; i++) {
		if ((i > 0) && (strcmp(answers[i].a_word, answers[i-1].a_word) == 0))
			continue;
		fprintf(output, "%d %s\r\n", j++, answers[i].a_word);
	}

	putc(EOFCH, output);
	answers_valid = 1;

	return(1);
}

comp(a, b)
struct answer *a, *b;
{
	return(strcmp(a->a_word, b->a_word));
}
