/*
 *
 * h a s h  . c			-- Hash Tables 
 *
 * Copyright (C) 1993, 1994 Erick Gallesio - I3S - CNRS / UNSA <eg@unice.fr>
 * 
 *
 * Permission to use, copy, and/or distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that both the above copyright notice and this permission notice appear in
 * all copies and derived works.  Fees for distribution or use of this
 * software or derived works may only be charged with express written
 * permission of the copyright holder.  
 * This software is provided ``as is'' without express or implied warranty.
 *
 * This software is a derivative work of other copyrighted softwares; the
 * copyright notices of these softwares are placed in the file COPYRIGHTS
 *
 *
 *           Author: Erick Gallesio [eg@kaolin.unice.fr]
 *    Creation date: 17-Jan-1994 17:49
 * Last file update: 25-May-1994 19:45
 */

#ifdef USE_STKLOS 	/* must be changed to USE_HASH */

#include <stk.h>

/**** Definitions for new type tc_hash ****/

static void free_hash_table(SCM ht);
static void mark_hash_table(SCM ht);

static int tc_hash;
static extended_scheme_type hash_type = {
  "hash-table",		/* name */
  0,			/* is_procp */
  mark_hash_table,	/* gc_mark_fct */
  free_hash_table,	/* gc_sweep_fct */
  NULL,			/* apply_fct */
  NULL			/* display_fct */
};


#define HASHTABLE(x)  ((struct Tcl_HashTable *)(x->storage_as.extension.data))
#define HASHTABLEP(x) (TYPEP(x, tc_hash))


static PRIMITIVE make_hash_table(void)
{
  SCM z;

  NEWCELL(z, tc_hash);
  HASHTABLE(z) = (Tcl_HashTable *) must_malloc(sizeof(Tcl_HashTable));
  Tcl_InitHashTable(HASHTABLE(z),  TCL_ONE_WORD_KEYS);
  return z;
}


static PRIMITIVE hash_table_p(SCM obj)
{
  return HASHTABLEP(obj) ? truth: ntruth;
}


static PRIMITIVE hash_table_put(SCM ht, SCM key, SCM val)
{
  Tcl_HashEntry *entry;
  int new;

  if (!HASHTABLEP(ht)) err("hash-table-put!: bad hash table", ht);
  
  entry = Tcl_CreateHashEntry(HASHTABLE(ht), (char *) key, &new);
  Tcl_SetHashValue(entry, val);
  return UNDEFINED;
}


static PRIMITIVE hash_table_get(SCM ht, SCM key, SCM default_value)
{
  Tcl_HashEntry *entry;
  
  if (!HASHTABLEP(ht)) err("hash-table-get: bad hash table", ht);
  
  if (entry = Tcl_FindHashEntry(HASHTABLE(ht), (char *) key)) {
    /* Key alrady in hash table */
    return Tcl_GetHashValue(entry);
  }
  else {
    /* Key not present in table */
    if (default_value == UNBOUND)
      err("hash-table-get: entry not defined", key);
    return default_value;
  }
}

static PRIMITIVE hash_table_remove(SCM ht, SCM key)
{
  Tcl_HashEntry *entry;

  if (!HASHTABLEP(ht)) err("hash-table-remove!: bad hash table", ht);
  
  if (entry = Tcl_FindHashEntry(HASHTABLE(ht), (char *) key)) {
    /* Key alrady in hash table */
    Tcl_DeleteHashEntry(entry);
  }
  return UNDEFINED;
}

static PRIMITIVE hash_table_for_each(SCM ht, SCM proc)
{
  Tcl_HashEntry *entry;
  Tcl_HashSearch search;

  if (!HASHTABLEP(ht)) err("hash-table-for-each: bad hash table", ht);
  if (procedurep(proc)==ntruth) err("hash-table-for-each: bad procedure", proc);
  
  for (entry = Tcl_FirstHashEntry(HASHTABLE(ht), &search);
       entry;
       entry = Tcl_NextHashEntry(&search)) {

    apply(proc, LIST2((SCM) Tcl_GetHashKey(HASHTABLE(ht), entry),
		      (SCM) Tcl_GetHashValue(entry)));
    
  }
  return UNDEFINED;
}

static PRIMITIVE hash_table_map(SCM ht, SCM proc)
{
  Tcl_HashEntry *entry;
  Tcl_HashSearch search;
  SCM result = NIL;

  if (!HASHTABLEP(ht)) err("hash-table-map: bad hash table", ht);
  if (procedurep(proc)==ntruth) err("hash-table-map: bad procedure", proc);
  
  for (entry = Tcl_FirstHashEntry(HASHTABLE(ht), &search);
       entry;
       entry = Tcl_NextHashEntry(&search)) {

    result = cons(apply(proc, LIST2((SCM) Tcl_GetHashKey(HASHTABLE(ht), entry),
				    (SCM) Tcl_GetHashValue(entry))),
		  result);
  }
  return result;
}

static PRIMITIVE hash_table_stats(SCM ht)
{
  Tcl_HashEntry *entry;
  char *s;

  if (!HASHTABLEP(ht)) err("hash-table-stats: bad hash table", ht);

  s = Tcl_HashStats(HASHTABLE(ht));
  fprintf(stderr, "%s\n", s);
  free(s);

  return UNDEFINED;
}


static void free_hash_table(SCM ht)
{
  Tcl_DeleteHashTable(HASHTABLE(ht));
  free(HASHTABLE(ht));
}


static void mark_hash_table(SCM ht)
{
  Tcl_HashEntry *entry;
  Tcl_HashSearch search;

  for (entry = Tcl_FirstHashEntry(HASHTABLE(ht), &search);
       entry;
       entry = Tcl_NextHashEntry(&search)) {

    /* mark key and value */
    gc_mark((SCM) Tcl_GetHashKey(HASHTABLE(ht), entry));
    gc_mark((SCM) Tcl_GetHashValue(entry));
  }
}


/******************************************************************************/

PRIMITIVE init_hash(void)
{
  tc_hash = add_new_type(&hash_type);

  add_new_primitive("make-hash-table",	  tc_subr_0,	  make_hash_table);
  add_new_primitive("hash-table?",	  tc_subr_1, 	  hash_table_p);
  add_new_primitive("hash-table-put!",	  tc_subr_3,	  hash_table_put);
  add_new_primitive("hash-table-get", 	  tc_subr_2_or_3, hash_table_get);
  add_new_primitive("hash-table-remove!", tc_subr_2,	  hash_table_remove);
  add_new_primitive("hash-table-for-each",tc_subr_2,	  hash_table_for_each);
  add_new_primitive("hash-table-map",	  tc_subr_2,	  hash_table_map);
  add_new_primitive("hash-table-stats",	  tc_subr_1,	  hash_table_stats);
  return UNDEFINED;
}

#endif /* defined USE_STKLOS */
