/*
 * s l i b . c				-- Misc functions
 *
 * Copyright (C) 1993, 1994 Erick Gallesio - I3S - CNRS / UNSA <eg@unice.fr>
 * 
 *
 * Permission to use, copy, and/or distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that both the above copyright notice and this permission notice appear in
 * all copies and derived works.  Fees for distribution or use of this
 * software or derived works may only be charged with express written
 * permission of the copyright holder.  
 * This software is provided ``as is'' without express or implied warranty.
 *
 * This software is a derivative work of other copyrighted softwares; the
 * copyright notices of these softwares are placed in the file COPYRIGHTS
 *
 *           Author: Erick Gallesio [eg@unice.fr]
 *    Creation date: ??-Oct-1993 ??:?? 
 * Last file update:  3-Jun-1994 22:26
 *
 */

#include "stk.h"
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/times.h>



#ifndef _DEBUG_MALLOC_INC
void *must_malloc(unsigned long size)
{
  void *tmp;

  tmp = malloc(size);
  if (tmp == NULL)
    err("failed to allocate storage from system", NIL);
 return(tmp);
}

void *must_realloc(void *ptr, unsigned long size)
{
  void *tmp;

  tmp = (char *) realloc(ptr, size);
  if (tmp == NULL)
    err("failed to re-allocate storage from system",NIL);
 return(tmp);
}
#endif

int strcmpi(register char *p1, register char *p2)
{
  for( ; tolower(*p1) == tolower(*p2); p1++, p2++) 
    if (!*p1) return 0;

  return tolower(*p1) - tolower(*p2);
}



SCM internal_eval_string(char *s, long context, SCM env)
{
  jmp_buf jb, *prev_jb = top_jmp_buf;
  long prev_context     = error_context;
  SCM result, port;
  int k;
  
  /* Create a string port to read the command and evaluate it in a new context */
  port = internal_open_input_string(s);

  /* save normal error jmpbuf  so that eval error don't lead to toplevel */
  /* If in a "catch", keep the ERR_IGNORED bit set */
  if ((k = setjmp(jb)) == 0) {
    top_jmp_buf   = &jb;
    error_context = (error_context & ERR_IGNORED) | ERR_READ_FROM_STRING;
    result = leval(lreadf(port->storage_as.port.f, FALSE), env);
  }
  top_jmp_buf   = prev_jb;;
  error_context = prev_context;
 
  if (k == 0) return result;
  /* if we are here, an error has occured during the string reading 
   * Two cases:
   *    - we are in a catch. Do a longjump to the catch to signal it a fail
   *    - otherwise error has already signaled, just return EVAL_ERROR
   */
  if (error_context & ERR_IGNORED) longjmp(*top_jmp_buf, k);
  return EVAL_ERROR;
}

SCM internal_read_from_string(SCM port, int *eof, int case_significant)
{
  jmp_buf jb, *prev_jb = top_jmp_buf;
  long prev_context     = error_context;
  SCM result;
  int k;

  /* save normal error jmpbuf  so that read error don't lead to toplevel */
  /* If in a "catch", keep the ERR_IGNORED bit set */
  if ((k = setjmp(jb)) == 0) {
    top_jmp_buf   = &jb;
    error_context = (error_context & ERR_IGNORED) | ERR_READ_FROM_STRING;
    result 	  = lreadf(port->storage_as.port.f, case_significant);
    *eof   	  = Eof(port->storage_as.port.f);
  }
  top_jmp_buf   = prev_jb;;
  error_context = prev_context;
  
  if (k == 0) return result;
  
  /* if we are here, an error has occured during the string reading 
   * Two cases:
   *    - we are in a catch. Do a longjump to the catch to signal it a fail
   *    - otherwise error has already signaled, just return EVAL_ERROR
   */
  if (error_context & ERR_IGNORED) longjmp(*top_jmp_buf, k);
  return EVAL_ERROR;
}

PRIMITIVE lcatch(SCM expr, SCM env)
{
  jmp_buf jb, *prev_jb = top_jmp_buf;
  long prev_context     = error_context;
  SCM l;
  int k;
  
  if (llength(expr) == -1) err("catch: bad list of expressions", expr);
  /* save normal error jmpbuf  so that eval error don't lead to toplevel */
  if ((k = setjmp(jb)) == 0) {
    top_jmp_buf   = &jb;
    error_context |= ERR_IGNORED;

    /* Evaluate the list of expressions */
    for (l = expr; NNULLP(l); l = CDR(l)) 
      leval(CAR(l), env);
  }
  top_jmp_buf   = prev_jb;
  error_context = prev_context; /* Don't use a mask to allow nested cal to catch */

  return (k == 0)? ntruth: truth;
}

PRIMITIVE quit_interpreter(SCM retcode)
{
  long ret = 0;

  if (retcode != UNBOUND) {
    if ((ret=integer_value(retcode)) == LONG_MIN)
      err("quit: bad return code", retcode);
  }
  lunwind_all();
  exit(ret);
}

PRIMITIVE lversion(void)
{
  return makestrg(strlen(STK_VERSION), STK_VERSION);
}

PRIMITIVE lrandom(SCM n)
{
  if (NINTEGERP(n)) err("random: bad number", n);
  return lmodulo(makeinteger(rand()), n);
}

PRIMITIVE set_random_seed(SCM n)
{
  if (NINTEGERP(n)) err("set-random-seed!: bad number", n);
  srand(INTEGER(n));
  return UNDEFINED;
}


static double _time()
{
  struct tms time_buffer;
  times(&time_buffer);
  return 1000 * (time_buffer.tms_utime + time_buffer.tms_stime) / 60.0;
}


PRIMITIVE ltime(SCM expr, SCM env)
{
  double rt;
  SCM res;

  if (llength(expr) != 1) err("stats: bad expression to stat" , expr);
  
  alloccells = 0;
  rt = _time();
  res = EVALCAR(expr);
  fprintf(stderr, ";;  Time: %.2fms\n;; Cells: %g\n", _time() - rt, alloccells);
  return res;
}


/* When STk evaluates an expression, it recode it in a manner which permits it
   to be more efficient for further evaluations. The uncode functions permits to 
   do the reverse job: it takes an exppression and returns a form similar to the 
   original one. 
   Warning: when a macro has been expanded, there is no mean to "revert" it to 
   its original form 
*/


static SCM associate(SCM l1, SCM l2)
{
  SCM z;

  if (NULLP(l1)) return NIL;
  
  for(z= NIL; NNULLP(l1); l1=CDR(l1), l2=CDR(l2))
    z = cons(LIST2(CAR(l1), luncode(CAR(l2))), z);
  
  return reverse(z);
}

static SCM uncode_let(char *type, SCM expr)
{
  return cons(intern(type),
	      LIST2(associate(CAR(expr), CAR(CDR(expr))),
		    luncode(CAR(CDR(CDR(expr))))));
}
  
PRIMITIVE luncode(SCM expr)
{
  switch (TYPE(expr)) {
    case tc_cons: switch (TYPE(CAR(expr))) {
    		    case tc_let:     return uncode_let("let",    CDR(expr)); 
      		    case tc_letstar: return uncode_let("let*",   CDR(expr)); 
		    case tc_letrec:  return uncode_let("letrec", CDR(expr)); 
		    case tc_if: 
		         expr = CDR(expr);
		         if (EQ(CAR(CDR(CDR(expr))), UNDEFINED)) 
			   return cons(intern("if"),
				       LIST2(luncode(CAR(expr)),
					     luncode(CAR(CDR(expr)))));
			 else
			   return cons(intern("if"),
				       LIST3(luncode(CAR(expr)),
					     luncode(CAR(CDR(expr))),
					     luncode(CAR(CDR(CDR(expr))))));
		    default: return cons(luncode(CAR(expr)), luncode(CDR(expr)));
		  }
    case tc_quote:	return intern("quote");
    case tc_lambda:	return intern("lambda");
    case tc_if:		return intern("if");
    case tc_setq:	return intern("set!");
    case tc_cond:	return intern("cond");
    case tc_and:	return intern("and");
    case tc_or:		return intern("or");
    case tc_let:	return intern("let");
    case tc_letstar:	return intern("letstar");
    case tc_letrec: 	return intern("letrec");
    case tc_begin:	return intern("begin");
    case tc_globalvar:  return VCELL(expr);
    case tc_localvar:   return expr->storage_as.localvar.symbol;
    default:		return expr;
  }
}

#ifdef USE_TK
/******************************************************************************
 *
 * Trace var
 *
 ******************************************************************************/

static char *TraceVarFct(ClientData clientData, Tcl_Interp *interp, 
			 char *name1, char *name2,
			 int flags)
{
  /* 
   * ClientData is the only field which insterest us here. It contains a 
   * string to evaluate 
   */
  Tcl_GlobalEval(interp, (char*) clientData);
  return NULL; /* to make the compiler happy */
}

PRIMITIVE ltrace_var(SCM var, SCM code)
{
  char *s1, *s2;
  
  if (NSYMBOLP(var)) err("trace-var: bad variable name", var);

  s1 = convert_for_tk(code);
  /* 
   * Build a copy of s1 and set it as the ClientData for Tcl_TraceVar
   * Duplication of s1 permits to be sure that the hash table has its 
   * own copy of the code to evaluate since original code can be garbaged.
   * TCL_TRACE_READS flags is used here to indicate to untrace to free
   * the code string.
   */
  s2 = (char *) must_malloc(strlen(s1) + 1); 
  strcpy(s2, s1);

  Tcl_TraceVar(main_interp, PNAME(var), TCL_TRACE_READS|TCL_TRACE_WRITES, 
	       TraceVarFct, (ClientData) s2);
  return UNDEFINED;
}

PRIMITIVE luntrace_var(SCM var)
{
  if (NSYMBOLP(var)) err("untrace-var: bad variable name", var);
  Tcl_CompleteUntraceVar(main_interp, PNAME(var));
  return UNDEFINED;
}
#endif
