/*
 *
 * t c l - l i b . c 		- A library remplacement for simulating 
 *				  a Tcl interpreter in Scheme
 *
 * Copyright (C) 1993, 1994 Erick Gallesio - I3S - CNRS / UNSA <eg@unice.fr>
 * 
 *
 * Permission to use, copy, and/or distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that both the above copyright notice and this permission notice appear in
 * all copies and derived works.  Fees for distribution or use of this
 * software or derived works may only be charged with express written
 * permission of the copyright holder.  
 * This software is provided ``as is'' without express or implied warranty.
 *
 * This software is a derivative work of other copyrighted softwares; the
 * copyright notices of these softwares are placed in the file COPYRIGHTS
 *
 *
 *            Author: Erick Gallesio [eg@unice.fr]
 *    Creation date: 19-Feb-1993 22:15
 * Last file update:  1-Jun-1994 11:24
 *
 */

#ifdef USE_TK
#include <varargs.h>
#include "stk.h"


/******************************************************************************
 *
 *				Eval functions
 *
 ******************************************************************************/

int Tcl_GlobalEval(interp, s)
     Tcl_Interp *interp;
     char *s;
{
  char *buffer = "";
  SCM result;

  if (*s == '\0') return TCL_OK;

  /* In some situations Tk append some data (numbers) to the callback. This
   * arise for scrollbars and scales. This parameters are normally used to
   * reflect slider position. When such a situation arises, we have to 
   * specify the callback as a string and add a pair of parenthesis around
   * this string to form a valid sexpr. To recognize such cases, we look
   * at first character: if it is not an open parenthesis, we add a pair of ()
   * around the callback string 
   * 
   */

  if (*s != '(') {
    /* Build the command to evaluate by adding a pair of parenthesis */
    buffer = must_malloc(strlen(s)+3);
    sprintf(buffer, "(%s)", s);
    s = buffer;
  }
  result = internal_eval_string(s, ERR_TK_BACKGROUND, NIL);
  Tcl_ResetResult(interp);
    
  if (*buffer) free(buffer);

  if (result != EVAL_ERROR) {
    Tcl_SetResult(interp, STk_Stringify(convert_for_tk(result), 0), TCL_DYNAMIC);
    return TCL_OK;
  }
  
  return TCL_ERROR;
}


#ifndef lint
int Tcl_VarEval(va_alist)
#else
int Tcl_VarEval(iPtr, p, va_alist)
    Tcl_Interp *iPtr;		/* Interpreter in which to execute command. */
    char *p;			/* One or more strings to concatenate,
				 * terminated with a NULL string. */
#endif
    va_dcl
{
    va_list argList;
#define FIXED_SIZE 200
    char fixedSpace[FIXED_SIZE+1];
    int spaceAvl, spaceUsed, length;
    char *string, *cmd;
    Tcl_Interp *interp;
    int result;

    /*
     * Copy the strings one after the other into a single larger
     * string.  Use stack-allocated space for small commands, but if
     * the commands gets too large than call ckalloc to create the
     * space.
     */

    va_start(argList);
    interp = va_arg(argList, Tcl_Interp *);
    spaceAvl = FIXED_SIZE;
    spaceUsed = 0;
    cmd = fixedSpace;
    while (1) {
	string = va_arg(argList, char *);
	if (string == NULL) {
	    break;
	}
	length = strlen(string);
	if ((spaceUsed + length) > spaceAvl) {
	    char *new;

	    spaceAvl = spaceUsed + length;
	    spaceAvl += spaceAvl/2;
	    new = ckalloc((unsigned) spaceAvl);
	    memcpy((VOID *) new, (VOID *) cmd, spaceUsed);
	    if (cmd != fixedSpace) {
		ckfree(cmd);
	    }
	    cmd = new;
	}
	strcpy(cmd + spaceUsed, string);
	spaceUsed += length;
    }
    va_end(argList);
    cmd[spaceUsed] = '\0';

    result = Tcl_GlobalEval(interp, cmd);
    if (cmd != fixedSpace) {
	ckfree(cmd);
    }
    return result;
}


/******************************************************************************
 *
 *	      Variable accesses (GetVar, GetVar2, SetVar, SetVar2)
 *
 ******************************************************************************/

char *Tcl_GetVar(interp, var, flags)
     Tcl_Interp *interp;	/* not used */
     char *var;
     int flags;
{
  SCM V = VCELL(intern(var));
  return (V == UNBOUND) ? NULL : convert_for_tk(V);
}

char *Tcl_GetVar2(interp, name1, name2, flags)
     Tcl_Interp *interp;	/* not used */
     char *name1, *name2;
     int flags;
{
  if (name2 && *name2) {
    char *res, *s = must_malloc(strlen(name1) + strlen(name2) + 3);

    sprintf(s, "%s(%s)", name1, name2);
    res = Tcl_GetVar(interp, s, flags);
    free(s);
    return res;
  }
  return Tcl_GetVar(interp, name1, flags);
}

char *Tcl_SetVar(interp, var, val, flags)
     Tcl_Interp *interp;
     char *var, *val;
     int flags;
{
  /* Eval the following expression: (set! var val) */
  VCELL(intern(var)) = makestrg(strlen(val), val);
  Tcl_ChangeValue(var); 
  return val;
}

char *Tcl_SetVar2(interp, name1, name2, val, flags)
     Tcl_Interp *interp;
     char *name1, *name2, *val;
     int flags;
{ 
  if (name2 && *name2) {
    char *res, *s = must_malloc(strlen(name1) + strlen(name2) + 3);

    sprintf(s, "%s(%s)", name1, name2);
    res = Tcl_SetVar(interp, s, val, flags);
    free(s);
    return res;
  }
  return Tcl_SetVar(interp, name1, val, flags);
}

/******************************************************************************
 *
 *			    Tcl command management
 *
 ******************************************************************************/


int Tcl_DeleteCommand(interp, cmdName)
     Tcl_Interp *interp;
     char *cmdName;
{
  SCM V   = intern(cmdName);

  if (symbol_boundp(V, globenv) == ntruth) return -1;
  /* Undefine "cmdName" by doing a (set! cmdname #<unbound>) */
  VCELL(V) = UNBOUND;
  return 0;
}


void Tcl_CreateCommand(interp, cmdName, proc, clientData, deleteProc)
     Tcl_Interp *interp;		
     char *cmdName;
     Tcl_CmdProc *proc;
     ClientData clientData;
     Tcl_CmdDeleteProc *deleteProc;
{
  SCM z, var;
  struct Tk_command *W;

  /* Build a new cell. Initialize it later */
  NEWCELL(z, tc_tkcommand);

  /* Define a variable whose name is the command name */
  var = intern(cmdName);
  VCELL(var) = z;

  W = (struct Tk_command *) must_malloc(sizeof(struct Tk_command)+strlen(cmdName));
  W->ptr 	   = clientData;
  W->fct 	   = proc;
  W->string_result = 0; /* by default */
  strcpy(W->Id, cmdName);

  /* 
   * Save the current environment in the object. 
   * This permit to have callback which can reference things in the
   * environment creation. Furthermore, this environment being known by the
   * Scheme interpreter, it will not be garbaged.
   *
   */
  W->environment = top_env_stack();
  
  z->storage_as.tk.data   = W;
  z->storage_as.tk.l_data = ntruth;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_GetCommandInfo --
 *
 *	Returns various information about a Tcl command.
 *
 * Results:
 *	If cmdName exists in interp, then *infoPtr is modified to
 *	hold information about cmdName and 1 is returned.  If the
 *	command doesn't exist then 0 is returned and *infoPtr isn't
 *	modified.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tcl_GetCommandInfo(interp, cmdName, infoPtr)
    Tcl_Interp *interp;			/* Interpreter in which to look
					 * for command. */
    char *cmdName;			/* Name of desired command. */
    Tcl_CmdInfo *infoPtr;		/* Where to store information about
					 * command. */
{
    Tcl_HashEntry *hPtr;
    Command *cmdPtr;

    SCM v = intern(cmdName);
    
    if (NTKCOMMP(VCELL(v))) return 0;

    infoPtr->proc       = (VCELL(v)->storage_as.tk.data)->fct;
    infoPtr->clientData = (VCELL(v)->storage_as.tk.data)->ptr;
    infoPtr->deleteProc = NULL;
    infoPtr->deleteData = NULL;
    return 1;
}


/******************************************************************************
 *
 *			  Tcl interpreter management
 *
 ******************************************************************************/

Tcl_Interp *Tcl_CreateInterp()
{
  register Interp *iPtr = (Interp *) ckalloc(sizeof(Interp));
  
  iPtr->result	 = iPtr->resultSpace;
  iPtr->freeProc	 = 0;
  iPtr->errorLine	 = 0;
  iPtr->resultSpace[0] = 0;
  
  iPtr->appendResult	 = NULL;
  iPtr->appendAvl	 = 0;
  iPtr->appendUsed	 = 0;
  
  return (Tcl_Interp *) iPtr;
}

void Tcl_DeleteInterp(interp)
    Tcl_Interp *interp;
{
  Interp *iPtr = (Interp *) interp;
  
  if (iPtr->appendResult != NULL) {
    ckfree(iPtr->appendResult);
  }
  ckfree((char *) iPtr);
}

void Tcl_AddErrorInfo(interp, message)
     Tcl_Interp *interp;
     char *message;	
{
  register Interp *iPtr = (Interp *) interp;
  char *p;
  
  /*
   * If an error is already being logged, then the new errorInfo
   * is the concatenation of the old info and the new message.
   * If this is the first piece of info for the error, then the
   * new errorInfo is the concatenation of the message in
   * interp->result and the new message.
   */
  
  if (!(iPtr->flags & ERR_IN_PROGRESS)) {
    char *p = STk_Stringify(interp->result, 0);

    p[strlen(p)] = 0;
    Tcl_SetVar2(interp, "errorInfo", (char *) NULL, p+1,TCL_GLOBAL_ONLY);
    iPtr->flags |= ERR_IN_PROGRESS;
    
    /*
     * If the errorCode variable wasn't set by the code that generated
     * the error, set it to "NONE".
     */
    
    if (!(iPtr->flags & ERROR_CODE_SET)) {
      (void) Tcl_SetVar2(interp, "errorCode", (char *) NULL, "NONE",
			 TCL_GLOBAL_ONLY);
    }
  }
  free(p);

  p = STk_Stringify(message, 0);
  Tcl_SetVar2(interp,"errorInfo",(char *)NULL,p+1,TCL_GLOBAL_ONLY|TCL_APPEND_VALUE);
  free(p);
}
#endif /* USE_TK */
