/*
 *
 * t o p l e v e l . c				-- The REP loop
 *
 * Copyright (C) 1993, 1994 Erick Gallesio - I3S - CNRS / UNSA <eg@unice.fr>
 * 
 *
 * Permission to use, copy, and/or distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that both the above copyright notice and this permission notice appear in
 * all copies and derived works.  Fees for distribution or use of this
 * software or derived works may only be charged with express written
 * permission of the copyright holder.  
 * This software is provided ``as is'' without express or implied warranty.
 *
 * This software is a derivative work of other copyrighted softwares; the
 * copyright notices of these softwares are placed in the file COPYRIGHTS
 *
 *
 *           Author: Erick Gallesio [eg@kaolin.unice.fr]
 *    Creation date:  6-Apr-1994 14:46
 * Last file update:  6-Apr-1994 15:06
 */

#include "stk.h"

extern char **environ;

/* Define constants used everywhere */
static struct obj VNIL     = {0, tc_nil};
static struct obj VUNDEF   = {0, tc_undefined}; 
static struct obj VUNBOUND = {0, tc_unbound};
static struct obj Vtruth   = {0, tc_boolean};
static struct obj Vntruth  = {0, tc_boolean};


long nointerrupt = 1;
long interrupt_differed = 0;

/******************************************************************************
 *
 * Handlers
 *
 ******************************************************************************/

static void err_ctrl_c(void)
{
  err("Control-C interrupt",NIL);
}

static void handle_sigfpe(int sig)
{
  signal(SIGFPE, handle_sigfpe);
  err("Floating point exception",NIL);
}

static void handle_sigint(int sig)
{
  signal(SIGINT,handle_sigint);
  if (nointerrupt)
    interrupt_differed = 1;
  else
    err_ctrl_c();
}

long no_interrupt(long n)
{
  long x;
  x = nointerrupt;
  nointerrupt = n;
  if ((nointerrupt == 0) && (interrupt_differed == 1)){
    interrupt_differed = 0;
    err_ctrl_c();
  }
  return(x);
}

/******************************************************************************
 *
 * Unix environment saving
 *
 * 	The two following function permit to save/restore the argc/argv/envp
 * in a file. They are used upon image restauration.
 *
 ******************************************************************************/
static void save_unix_environment(int argc, char **argv)
{
  FILE *f;
  char **env;
  char filename[50];
  
  /* Open a file in which we will save argc/argv/envp */
  sprintf(filename, "/usr/tmp/STktmp%d", getpid());
  if ((f = fopen(filename, "w")) == NULL) {
    fprintf(stderr, "Cannot save environment in %s.\n ABORT.\n", filename);
    exit(1);
  }

  /* print argc */
  fprintf(f, "%d\n", argc);

  /* print argv */
  for ( ; argc; argc-=1, argv+=1) {
    fprintf(f, "%d %s\n", strlen(*argv), *argv);
  }

  /* print environment */
  for (env=environ; *env; env++) {};
  fprintf(f, "%d\n", env-environ);

  for (env=environ; *env; env++) {
    fprintf(f, "%d %s\n", strlen(*env), *env);
  }

  /* Close file */
  fclose(f);
}

static void restore_unix_environment(int *argc, char ***argv)
{
  FILE *f;
  int i, l, Argc, env_len;
  char **Argv, **Env;
  char filename[50];
  
  /* Open a file in which we have saved argc/argv/envp */
  sprintf(filename, "/usr/tmp/STktmp%d", getpid());
  if ((f = fopen(filename, "r")) == NULL) {
    fprintf(stderr, "Cannot re-open environment in %s.\n ABORT.\n", filename);
    exit(1);
  }

  /* Read argc */
  fscanf(f, "%d", &Argc); getc(f);

  /* Read argv */
  Argv = malloc((Argc+1) * sizeof(char *));

  for (i=0; i<Argc; i++) {
    fscanf(f, "%d", &l); getc(f);
    Argv[i] = malloc(l+1);
    fread(Argv[i], 1, l, f);
    Argv[i][l] = '\0';
  }
  Argv[Argc] = NULL;

  /* Read environment */
  fscanf(f, "%d", &env_len); getc(f);
  Env = malloc((env_len+1) * sizeof(char *));
  for (i=0; i<env_len; i++) {
    fscanf(f, "%d", &l); getc(f);
    Env[i] = malloc(l+1);
    fread(Env[i], 1, l, f);
    Env[i][l] = '\0';
  }

  /* Save read values in global variables */
  *argc   = Argc;
  *argv   = Argv;
  environ = Env;
  
  /* close & delete temporary file */
  fclose(f);
  unlink(filename);
}

static void load_init_file(void)
{
  /* Try to load init.stk in ".", $STK_LIBRARY or a default directory */
  static char init_file[] = "init.stk";
  char file[256];
  char *tklib= getenv("STK_LIBRARY");

  sprintf(file, "./%s", init_file);
  if (loadfile(file, 0) == truth) return;

  if (tklib) {
    sprintf(file, "%s/%s",tklib, init_file);
    if (loadfile(file, 0) == truth) return;
  }

  sprintf(file, "%s/%s", STK_LIBRARY, init_file);
  loadfile(file, 1);
}

static void init_interpreter(void)
{
  long j;

  /* Global variables to initialize */
  NIL           = &VNIL;
  UNDEFINED     = &VUNDEF;
  UNBOUND	= &VUNBOUND;
  tkbuffer 	= (char *) must_malloc(TKBUFFERN+1);

  /* Initialize GC */
  init_gc();

  /* Initialize symbol table */
  for(j=0; j<OBARRAY_SIZE; j++) obarray[j] = NIL;

  gc_protect(truth  = &Vtruth);
  gc_protect(ntruth = &Vntruth);

  gc_protect(sym_progn  	  = intern("begin"));
  gc_protect(sym_lambda 	  = intern("lambda"));
  gc_protect(sym_quote  	  = intern("quote"));
  gc_protect(sym_imply  	  = intern("=>"));
  gc_protect(sym_dot    	  = intern(".")); 
  gc_protect(sym_debug  	  = intern(DEBUG_MODE));
  gc_protect(sym_else   	  = intern("else"));  
  gc_protect(sym_define 	  = intern("define"));
  gc_protect(sym_letrec 	  = intern("letrec"));
  gc_protect(sym_quasiquote 	  = intern("quasiquote"));
  gc_protect(sym_unquote	  = intern("unquote"));
  gc_protect(sym_unquote_splicing = intern("unquote-splicing")); 

  gc_protect(globenv    = makeenv(NIL));
  gc_protect(wind_stack = NIL);

  /* Initialize standard ports */
  init_standard_ports();

  /* Initialize Scheme primitives */
  init_primitives();
}

static void print_banner(void)
{
  fprintf(stderr, "Welcome to the STk interpreter version %s\n", STK_VERSION);
  fprintf(stderr, "Copyright (C) 1993, 1994 Erick Gallesio - ");
  fprintf(stderr, "I3S - CNRS / UNSA <eg@unice.fr>\n");
}

static void repl_loop(void)
{
  /* The print/eval/read loop */
  for( ; ; ) {
    SCM x;

    if (interactivep) fprintf(stderr, "STk> ");
    if (EQ(x=lreadf(stdin, FALSE), eof_object)) break;
    lprint(leval(x, NIL), stdout, WRT_MODE);
    if (dumped_core) {
      dumped_core = 0;
      longjmp(*top_jmp_buf, JMP_RESTORE);
    }      
    Putc('\n', stdout);
  }
}

static void repl_driver(int argc, char **argv)
{
  int k;
  char buffer[100];

  if (argc >= 3  && strcmp(argv[1], "-image") == 0) {
    /* We have an image to restore */
    strcpy(buffer, argv[2]);
    save_unix_environment(argc, argv);
    internal_restore(buffer);
  }
  else {
    /* Normal initialisation */
    reset_eval_stack();
  }

  /* The ultimate point */
  k = setjmp(*top_jmp_buf);
  
  signal(SIGINT, handle_sigint);
  signal(SIGFPE, handle_sigfpe);
  
  interrupt_differed = 0;
  nointerrupt 	     = 0;
  error_context      = ERR_OK;  
  interactivep 	     = isatty(fileno(stdin));

  if (k == 0 || k == JMP_RESTORE) { 
    /* It is not an error */
    switch (k) {
      case 0: 		Argc = argc; Argv = argv;
		        whence(*Argv, Argv0);
			init_interpreter();
			load_init_file();
#ifdef USE_TK
	      	        Tk_main(Argc, Argv);
#endif
			if (interactivep) print_banner();
			/* Execute application specific initialisations */
			STk_user_init();
		        break;
      case JMP_RESTORE: restore_unix_environment(&Argc, &Argv);
			whence(*Argv, Argv0);
#ifdef USE_TK
		        Tk_main(Argc, Argv);
#endif
			if (interactivep) print_banner();
			break;
    }
  }

  repl_loop();
  if (interactivep) fprintf(stderr, "Bye.\n");
  quit_interpreter(UNBOUND);
}

/******************************************************************************
 *
 * Toplevel
 * 
 ******************************************************************************/

void toplevel(int argc, char **argv)
{
  char stack_start; /* Unused variable. Its the first stack allocated varaiable */

  stack_start_ptr = &stack_start;
  repl_driver(argc, argv);
}
