/*****************************************************************************
 *
 * Microchip DeviceNet Stack (Identity Object Header)
 *
 *****************************************************************************
 * FileName:        ident.h
 * Dependencies:    
 * Processor:       PIC18F with CAN
 * Compiler:       	C18 02.10.02 or higher
 * Linker:          MPLINK 03.20.01 or higher
 * Company:         Microchip Technology Incorporated
 *
 * Software License Agreement
 *
 * The software supplied herewith by Microchip Technology Incorporated
 * (the "Company") is intended and supplied to you, the Company's
 * customer, for use solely and exclusively with products manufactured
 * by the Company. 
 *
 * The software is owned by the Company and/or its supplier, and is 
 * protected under applicable copyright laws. All rights are reserved. 
 * Any use in violation of the foregoing restrictions may subject the 
 * user to criminal sanctions under applicable laws, as well as to 
 * civil liability for the breach of the terms and conditions of this 
 * license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN "AS IS" CONDITION. NO WARRANTIES, 
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED 
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT, 
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR 
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 *
 * Refer to Chapter 6 of Volume 2 for information about this object.
 *
 * Author               Date        Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Ross Fosler			04/03/03	...	
 * 
 *****************************************************************************/



/*********************************************************************
 * Identity object data structure
 ********************************************************************/
typedef struct _DNET_ID
{
	union _VENDOR_ID				// Vendor ID
	{
		UINT W;
		struct
		{
			USINT	L;
			USINT	H;
		}bytes;
	}vendorID;
	
	union _DEV_TYPE					// Device type
	{
		UINT W;
		struct
		{
			USINT	L;
			USINT	H;
		}bytes;
	}deviceType;

	union _PROD_CODE				// Product code
	{
		UINT W;
		struct
		{
			USINT	L;
			USINT	H;
		}bytes;
	}productCode;

	struct _REV						// Revision		
	{
		USINT	major;
		USINT 	minor;
	}revision;

	WORD status;					// Status

	union _SERIAL_NUM				// Serial number
	{
		UDINT	W;
		struct
		{
			USINT	L;
			USINT	H;
			USINT	UL;
			USINT	UH;
		}bytes;
	}serial;

	struct _DEV_NAME				// Device name
	{
		unsigned char len;
		ROM unsigned char 	*p;		
	}name;

} DNET_ID;



/*********************************************************************
 * Identity object data structure declared global
 ********************************************************************/
extern DNET_ID uIdentity;




		   
/*********************************************************************
 * Function:        unsigned char _IdentityExplMsgHandler(void)
 *
 * PreCondition:    
 *
 * Input:           
 *					
 *                  
 * Output:         
 *				
 *				
 *			
 *
 * Side Effects:    
 *
 * Overview:        Handler for explicit messaging 
 *
 * Note:            None
 ********************************************************************/
unsigned char _IdentityExplMsgHandler(void);





/*********************************************************************
 * Function:        unsigned char IdentityInst0GetAttrib(void)
 
 *					unsigned char DNetGetAttrib(Instance, Attribute, *length)
 *
 * PreCondition:    
 *
 * Input:           gInstance - contains the requested instance.
 *					gAttribute - contains the attribute number.
 *                  
 * Output:          gResponse - contains the response or error code
 *					and any bytes following gResponse may also contain
 *					data.
 *					Returns 1 for success or 0 for error.
 *
 * Side Effects:    Overwrites gInstance, gAttribute, gResponse, and more
 *
 * Overview:        Gets an attribute within this object (DeviceNet Object) 
 *
 * Note:            None
 ********************************************************************/
unsigned char _IdentityInst0GetAttrib(void);



/*********************************************************************
 * Function:        unsigned char IdentityInst1GetAttrib(void)
 
 *					unsigned char DNetGetAttrib(Instance, Attribute, *length)
 *
 * PreCondition:    
 *
 * Input:           gInstance - contains the requested instance.
 *					gAttribute - contains the attribute number.
 *                  
 * Output:          gResponse - contains the response or error code
 *					and any bytes following gResponse may also contain
 *					data.
 *					Returns 1 for success or 0 for error.
 *
 * Side Effects:    Overwrites gInstance, gAttribute, gResponse, and more
 *
 * Overview:        Gets an attribute within this object (DeviceNet Object) 
 *
 * Note:            None
 ********************************************************************/
unsigned char _IdentityInst1GetAttrib(void);



/*********************************************************************
 * Function:        unsigned char IdentityReset(void)
 *
 * PreCondition:    
 *
 * Input:           
 *					
 *                  
 * Output:         
 *				
 *				
 *			
 *
 * Side Effects:    
 *
 * Overview:        Handler for explicit messaging 
 *
 * Note:            None
 ********************************************************************/
unsigned char IdentityReset(void);



/*********************************************************************
 * Function:        void IdentityInit(vendorID, deviceType, productCode, 
 *										majorRev, minorRev, status, 
 *										serial, nameLen, pName)
 *
 * PreCondition:    
 *
 * Input:           
 *					
 *                  
 * Output:         
 *				
 *				
 *			
 *
 * Side Effects:    
 *
 * Overview:        Init 
 *
 * Note:            None
 ********************************************************************/
#define mIdentityInit(vendor_id, device_type, product_code, major_rev, 		\
					 minor_rev, status_bits, serial_no, name_len, p_name) {	\
							uIdentity.name.p = p_name;						\
							uIdentity.name.len = name_len;					\
							uIdentity.serial.W = serial_no;					\
							uIdentity.status.W = status_bits;				\
							uIdentity.revision.minor = minor_rev;			\
							uIdentity.revision.major = major_rev;			\
							uIdentity.productCode.W.word = product_code;	\
						 	uIdentity.deviceType.W.word = device_type;		\
						 	uIdentity.vendorID.W.word = vendor_id;}				   					
					  	  		  


/*********************************************************************
 * Function:        void IdentitySetVendorID(UINT vendorID)
 *					void IdentitySetVendorIDL(USINT vendorID_L)
 *					void IdentitySetVendorIDH(USINT vendorID_H)
 *
 * PreCondition:    
 *
 * Input:    		vendorID       
 *                  
 * Output:     		none    
 *				
 * Side Effects:  	  
 *
 * Overview:        Writes the Identity class variable, vendorID. 
 *
 * Note:            None
 ********************************************************************/
#define	mIdentitySetVendorID(vendor_id)			(uIdentity.vendorID.W.word = vendor_id)
#define mIdentitySetVendorIDL(vendor_id_L)		(uIdentity.vendorID.bytes.L = vendor_id_L)
#define mIdentitySetVendorIDH(vendor_id_H)		(uIdentity.vendorID.bytes.H = vendor_id_H)



/*********************************************************************
 * Function:        void IdentitySetDeviceType(UINT deviceType)
 *					void IdentitySetDeviceTypeL(USINT deviceType_L)
 *					void IdentitySetDeviceTypeH(USINT deviceType_H)
 *
 * PreCondition:    
 *
 * Input:    		deviceType       
 *                  
 * Output:     		none    
 *				
 * Side Effects:  	  
 *
 * Overview:        Writes the Identity class variable, device type. 
 *
 * Note:            None
 ********************************************************************/
#define mIdentitySetDeviceType(device_type)		(uIdentity.deviceType.W.word = device_type)
#define mIdentitySetDeviceTypeL(device_type_L)	(uIdentity.deviceType.bytes.L = device_type_L)
#define mIdentitySetDeviceTypeH(device_type_H)	(uIdentity.deviceType.bytes.H = device_type_H)




/*********************************************************************
 * Function:        void IdentitySetProductCode(UINT productCode)
 *					void IdentitySetProductCodeL(USINT productCode_L)
 *					void IdentitySetProductCodeH(USINT productCode_H)
 *
 * PreCondition:    
 *
 * Input:    		productCode       
 *                  
 * Output:     		none    
 *				
 * Side Effects:  	  
 *
 * Overview:        Writes the Identity class variable, product code. 
 *
 * Note:            None
 ********************************************************************/
#define mIdentitySetProductCode(product_code)		(uIdentity.productCode.W.word = product_code)
#define mIdentitySetProductCodeL(product_code_L)	(uIdentity.productCode.bytes.L = product_code_L)
#define mIdentitySetProductCodeH(product_code_H)	(uIdentity.productCode.bytes.H = product_code_H)



/*********************************************************************
 * Function:        void IdentitySetMajorRevision(UINT majorRev)
 *					void mIdentitySetMinorRevision(USINT minorRev)
 *
 * PreCondition:    
 *
 * Input:    		revision       
 *                  
 * Output:     		none    
 *				
 * Side Effects:  	  
 *
 * Overview:        Writes the Identity class variable, revision. 
 *
 * Note:            None
 ********************************************************************/
#define	mIdentitySetMajorRevision(major_rev)		(uIdentity.revision.major = major_rev) 
#define	mIdentitySetMinorRevision(minor_rev)		(uIdentity.revision.minor = minor_rev)


/*********************************************************************
 * Function:        void IdentitySetSerial(UDINT serial)
 *					void IdentitySetSerialL(USINT serial_L)
 *					void IdentitySetSerialH(USINT serial_H)
 *					void IdentitySetSerialUL(USINT serial_UH)
 *					void IdentitySetSerialUH(USINT serial_UH)
 *
 * PreCondition:    
 *
 * Input:    		serial       
 *                  
 * Output:     		none    
 *				
 * Side Effects:  	  
 *
 * Overview:        Writes the Identity class variable, serial number. 
 *
 * Note:            None
 ********************************************************************/
#define mIdentitySetSerial(serial_num)				(uIdentity.serial.W = serial_num)
#define mIdentitySetSerialL(serial_num_L)			(uIdentity.serial.bytes.L = serial_num_L)
#define mIdentitySetSerialH(serial_num_H)			(uIdentity.serial.bytes.H = serial_num_H)
#define mIdentitySetSerialUL(serial_num_UL)			(uIdentity.serial.bytes.UL = serial_num_UL)
#define mIdentitySetSerialUH(serial_num_UH)			(uIdentity.serial.bytes.UH = serial_num_UH)



/*********************************************************************
 * Function:        void IdentitySetStatus(WORD status)
 *					void IdentitySetStatusL(USINT status_L)
 *					void IdentitySetStatusH(USINT status_H)
 *
 * PreCondition:    
 *
 * Input:    		status       
 *                  
 * Output:     		none    
 *				
 * Side Effects:  	  
 *
 * Overview:        Writes the Identity class variable, status. 
 *
 * Note:            None
 ********************************************************************/
#define	mIdentitySetStatus(status_bits)				(uIdentity.status.W = status_bits)
#define	mIdentitySetStatusL(status_bits_L) 			(uIdentity.status.bits.lbyte.L = status_bits_L)
#define	mIdentitySetStatusH(status_bits_H) 			(uIdentity.status.bits.lbyte.H = status_bits_H)



/*********************************************************************
 * Function:        void IdentitySetName(unsigned char *pName)
 *
 * PreCondition:    
 *
 * Input:    		*pName       
 *                  
 * Output:     		none    
 *				
 * Side Effects:  	  
 *
 * Overview:        Writes the Identity class variable pointer to the
 *					device name. 
 *
 * Note:            None
 ********************************************************************/
#define	mIdentitySetNameP(p_name)					(uIdentity.name.p = p_name)
#define mIdentitySetNameLen(name_len)				(uIdentity.name.len = name_len)


/*********************************************************************
 * Function:        UINT IdentityGetVendorID(void)
 *					USINT IdentityGetVendorIDL(void)
 *					USINT IdentityGetVendorIDH(void)
 *
 * PreCondition:    
 *
 * Input:    		none       
 *                  
 * Output:     		VendorID    
 *				
 * Side Effects:  	  
 *
 * Overview:        Reads the vendor ID. 
 *
 * Note:            None
 ********************************************************************/
#define	mIdentityGetVendorID()					(uIdentity.vendorID.W)
#define	mIdentityGetVendorIDL()					(uIdentity.vendorID.bytes.L)
#define	mIdentityGetVendorIDH()					(uIdentity.vendorID.bytes.H)


/*********************************************************************
 * Function:        UINT IdentityGetDeviceType(void)
 *					USINT IdentityGetDeviceTypeL(void)
 *					USINT IdentityGetDeviceTypeH(void)
 *
 * PreCondition:    
 *
 * Input:    		none       
 *                  
 * Output:     		Device Type    
 *				
 * Side Effects:  	  
 *
 * Overview:        Reads the Device Type. 
 *
 * Note:            None
 ********************************************************************/
#define mIdentityGetDeviceType()				(uIdentity.deviceType.W)
#define mIdentityGetDeviceTypeL()				(uIdentity.deviceType.bytes.L)
#define mIdentityGetDeviceTypeH()				(uIdentity.deviceType.bytes.H)



/*********************************************************************
 * Function:        UINT IdentityGetProductCode(void)
 *					USINT IdentityGetProductCodeL(void)
 *					USINT IdentityGetProductCodeH(void)
 *
 * PreCondition:    
 *
 * Input:    		none       
 *                  
 * Output:     		Product Code    
 *				
 * Side Effects:  	  
 *
 * Overview:        Reads the Product Code. 
 *
 * Note:            None
 ********************************************************************/
#define	mIdentityGetProductCode()				(uIdentity.productCode.W)
#define	mIdentityGetProductCodeL()				(uIdentity.productCode.bytes.L)
#define	mIdentityGetProductCodeH()				(uIdentity.productCode.bytes.H)



/*********************************************************************
 * Function:        USINT IdentityGetMajorRevision(void)
 *					USINT IdentityGetMinorRevision(void)
 *
 * PreCondition:    
 *
 * Input:    		none       
 *                  
 * Output:     		Revision    
 *				
 * Side Effects:  	  
 *
 * Overview:        Reads the revision. 
 *
 * Note:            None
 ********************************************************************/
#define mIdentityGetMajorRevision()				(uIdentity.revision.major)
#define mIdentityGetMinorRevision()				(uIdentity.revision.minor)


/*********************************************************************
 * Function:        UDINT IdentityGetSerial(void)
 *					USINT IdentityGetSerialL(void)
 *					USINT IdentityGetSerialH(void)
 *					USINT IdentityGetSerialUL(void)
 *					USINT IdentityGetSerialUH(void)
 *
 * PreCondition:    
 *
 * Input:    		none       
 *                  
 * Output:     		Serial number    
 *				
 * Side Effects:  	  
 *
 * Overview:        Reads the serial number. 
 *
 * Note:            None
 ********************************************************************/
#define mIdentityGetSerial()					(uIdentity.serial.W)
#define mIdentityGetSerialL()					(uIdentity.serial.bytes.L)
#define mIdentityGetSerialH()					(uIdentity.serial.bytes.H)
#define mIdentityGetSerialUL()					(uIdentity.serial.bytes.UL)
#define mIdentityGetSerialUH()					(uIdentity.serial.bytes.UH)


/*********************************************************************
 * Function:        WORD IdentityGetStatus(void)
 *					USINT IdentityGetStatusL(void)
 *					USINT IdentityGetStatusH(void)
 *
 * PreCondition:    
 *
 * Input:    		none       
 *                  
 * Output:     		Status    
 *				
 * Side Effects:  	  
 *
 * Overview:        Reads the status. 
 *
 * Note:            None
 ********************************************************************/
#define mIdentityGetStatus()					(uIdentity.status.W)
#define mIdentityGetStatusL()					(uIdentity.status.bits.lbyte.L)
#define mIdentityGetStatusH()					(uIdentity.status.bits.lbyte.H)
