package antichess;

import java.util.*;
import antichess.ai.GameAI;

/**
 * <p>A <code>AIPlayer</code> is a <code>GamePlayer</code> that uses a
 * <code>GameAI</code> to make moves. </p>
 */
public class AIPlayer<M extends Move, B extends Board<M>> implements GamePlayer, Runnable {

	private B board;
	private Player player;
	private Move move;
	private GameController controller;
	private GameAI<M, B> ai;

	/**
	 * Rep invariant:
	 * * At most one of move and controller is non-null
	 * * board != null
	 * * ai != null
	 */

	/**
	 * Creates a new AIPlayer object
	 * 
	 * @param board
	 * @param player
	 */
	public AIPlayer(B board, Player player, GameAI<M, B> ai) {
		this.board = board;
		this.player = player;
		this.ai = ai;
		move = null;
		controller = null;
		new Thread(this).start();
	}

	/**
	 * <p>
	 * The thread in the run() method is responsible for actually
	 * asking the <code>GameAI</code> to make moves.
	 * </p>
	 *
	 * <p> The thread spends most of its time sleeping. When
	 * <code>getMove</code> is called, it sets controller and calls
	 * <code>notify()</code> to wake the thread. Once the thread wakes
	 * and observes <code>controller</code> to be non-null, it
	 * computes a move using the <code>GameAI</code>. When the move
	 * returns, it sets the internal <code>move</code> variable, calls
	 * <code>notify()</code> on the Game controller, and then sets the
	 * <code>controller</code> back to null. </p>
	 */
	public void run() {
		while(true) {
			while(controller == null) {
				try {
					this.wait();
				} catch(Exception e) {}
			}
			if(player != board.getPlayer())
				throw new RuntimeException("The AI was asked to make a move for the other player!");
			long timeLeft;
			long opponentTimeLeft;
			GameClock clock = controller.getClock();
			if(clock.hasPlayer(player))
				timeLeft = clock.getTime(player);
			else
				timeLeft = 0;
			if(clock.hasPlayer(player.otherPlayer()))
				opponentTimeLeft = clock.getTime(player.otherPlayer());
			else
				opponentTimeLeft = 0;
			
			Move m = ai.findMove(board, timeLeft, opponentTimeLeft);
			synchronized(this) {
				move = m;
				controller.notifyControllerHasMove();
				controller = null;
			}
		}
	}
	
	/* 
	 * @see antichess.GamePlayer#getMove()
	 */
	public synchronized Move getMove(GameController controller) {
		synchronized(this) {
			if(move != null) {
				Move m = move;
				move = null;
				return m;
			}
		}
		this.controller = controller;
		this.notify();
		return null;
	}

	
	public Player getPlayerType(){
		return this.player;
	}
	
}
