// -*- mode: java; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*-

package antichess;

import java.util.ArrayList;
import java.util.List;

/**
 * A ChessBoard represents the state of a Chess Game at some time.
 * The ChessBoard keeps track of the location of all of the pieces
 * both on the board and that have been captured, as well as a
 * list of optionally timestamped moves that have been made thus
 * far in the game.
 * <p> 
 * ChessBoards are mutable and cloneable.
 * <p>
 * ChessBoards also include some static variables that describe various
 * end-game conditions, including ChessBoard.CHECKMATE (value = 2) and
 * ChessBoard.STALEMATE (value = 3). Checks of the end-game information
 * should compare the stored GameOverReason() against these values.
 * <p>
 * 
 * @specfield capturedPieces : Set<Pieces>	// The Pieces in the captured pile
 * @specfield moveStack	 // A stack of all moves previously performed on this board, optionally associated with timestamps
 **/
public class ChessBoard extends Board<ChessMove> implements Cloneable
{
	// Flag to control debugging
	private static final boolean DEBUG = false;
	
	// End conditions for Chess
	/**
	 * End-game condition variable to represent checkmate.
	 */
	public static final int CHECKMATE = 2;
	/**
	 * End-game condition variable to represent stalemate.
	 */
	public static final int STALEMATE = 3;
	
	// Abstraction Function
	// For all ChessBoards cb:
	// 	pieces(cb) => Set(non-null entries in cb.board)
	// 	capturedPieces(cb) => cb.capturedPieces;
	// 	moveStack(cb) => cb.moveStack
	
	// Representation Invariant
	// For all ChessBoards cb:
	// 	Let i be a row number (0 <= i <= 7);
	// 	Let j be a column number (0 <= j <= 7);
	// 	For all Piece entires p at cb.board[i][j]:
	// 		p == null OR (p.getRow() == i AND p.getColumn() == j);
	// 	For all Pieces p in cb.capturedPieces:
	// 		p.getRow() == -1 AND p.getColumn() == -1
	
	// The dimensions of the board
	private static final int ROWS = 8;
	private static final int COLUMNS = 8; 
	
	// FIELDS
	// pieces are located at board[row][column]
	private Piece[][] board;
	private List<Piece> capturedPieces;
	private MoveHistory<ChessMove> moveStack;
	
	/**
	 * Private method to check that the representation invariant
	 * for ChessBoards holds.
	 */
	protected void checkRep() {
		for (int i = 0; i < ROWS; i++)
			for (int j = 0; j < COLUMNS; j++)
				if (!(board[i][j] == null || (board[i][j].getRow() == i && board[i][j].getColumn() == j)))
					throw new RuntimeException("Piece found in unexpected location");
		
		for (Piece p : capturedPieces)
			if(!(p.getRow() == -1 && p.getColumn() == -1))
				throw new RuntimeException("Captured piece had invalid location flags");
	}
	
	/**
	 * Creates a new ChessBoard object, with an unfilled board,
	 * an unfilled capturedPieces pile, and an empty moveStack.
	 */
	public ChessBoard() {
		super();
		board = new Piece[ROWS][COLUMNS];
		player = Player.WHITE;
		capturedPieces = new ArrayList<Piece>(64);
		moveStack = null;
	}
	
	@Override
	public ChessBoard clone() {
		ChessBoard cb = new ChessBoard();
		copyThis(cb);
		return cb;
	}
	
	/**
	 * Inheritable helper method for copying various
	 * data members from this Board to an empty Board.
	 * In particular, this method takes an empty
	 * ChessBoard and fills it with the this
	 * ChessBoard's data.
	 * @param b the empty board into which this board
	 * should be copied.
	 */
	protected void copyThis(ChessBoard cb) {
		super.copyThis(cb);
		// Copy the player
		cb.player = player;
		// Copy the board
		for (int i = 0; i < ROWS; i++)
			for (int j = 0; j < COLUMNS; j++)
				cb.board[i][j] = this.board[i][j];
		// Copy the captured pile
		for (Piece capturedPiece : this.capturedPieces)
			cb.capturedPieces.add(capturedPiece);
		//Copy the move history
		cb.moveStack = this.moveStack;
		
		if (DEBUG)
			cb.checkRep();
	}
	
	/**
	 * Populates the board and captured piles with the list of pieces
	 * @param pieces the Pieces to add to the board and capturedPieces piles 
	 * 
	 * @modifies this.board and this.capturedPieces
	 * @effects adds Pieces in <code>pieces</code> to this.board and this.capturedPieces 
	 */
	private void populate(List<Piece> pieces) {
		// Initialize board with a bunch of nulls
		for (int i = 0; i < ROWS; i++)
			for (int j = 0; j < COLUMNS; j++)
				board[i][j] = null;
		// Clear the list of captured Piecs
		capturedPieces.clear();
		
		// Repopulate the board and capturedPieces piles
		for (Piece piece : pieces)
			if (piece.getRow() == -1) // Piece is captured
				capturedPieces.add(piece);
			else // Piece is on the board
				board[piece.getRow()][piece.getColumn()] = piece;
	}
	
	/**
	 * Initializes the board for a new game
	 * @modifies this
	 * @effects populates the board and initializes its parameters for a new game.
	 */
	public void newGame() {
		populate(createInitialPieces());
		player = Player.WHITE;
		moveStack = null;
		endGame(NOTOVER, Player.NONE);
		
		if (DEBUG)
			checkRep();
	}
	
	/**
	 * Loads the board parameters from the given state
	 * @modifies this
	 * @effects populates the board and initializes its paramteres from
	 * the given parameters
	 */
	public void loadGame(List<Piece> pieces, Player player, MoveHistory<ChessMove> moveStack) {
		populate(pieces);
		this.player = player;
		this.moveStack = moveStack;
		
		checkForEndCondition();
		
		if (DEBUG)
			checkRep();
	}
	
	/**
	 * Private method for creating the initial antichess pieces with
	 * their initial locations on the board.
	 * @effects initializes initialPieces as a new List of Pieces containing the normal
	 * chess pieces in their default starting locations.
	 */
	private static List<Piece> createInitialPieces() {
		List<Piece> initialPieces = new ArrayList<Piece>();
		
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getRookType(), 0, 0));
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getKnightType(), 0, 1));
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getBishopType(), 0, 2));
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getQueenType(), 0, 3));
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getKingType(), 0, 4));
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getBishopType(), 0, 5));
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getKnightType(), 0, 6));
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getRookType(), 0, 7));
		
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getRookType(), 7, 0));
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getKnightType(), 7, 1));
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getBishopType(), 7, 2));
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getQueenType(), 7, 3));
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getKingType(), 7, 4));
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getBishopType(), 7, 5));
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getKnightType(), 7, 6));
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getRookType(), 7, 7));
		
		for (int i = 0; i < COLUMNS; i++) {
			initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getPawnType(), 1, i));
			initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getPawnType(), 6, i));
		}
		
		return initialPieces;
	}
	
	/**
	 * @modifies this
	 * @effects sets this.player to the other player
	 */
	protected void switchPlayer() {
		player = player.otherPlayer();
		
		checkForEndCondition();
	}
	
	/**
	 * @modifies this
	 * @effects if necessary, sets the end-condition variables
	 * for this game to terminate the game
	 */
	protected void checkForEndCondition() {
		// Check for stalemate if only kings are left
		if (getPieces(player).size() == 1 && getPieces(player.otherPlayer()).size() == 1) {
			endGame(STALEMATE, Player.NONE);
		} else if (allLegalMoves().isEmpty()) {
			if (isPlayerInCheck(player))
				endGame(CHECKMATE, player.otherPlayer());
			else
				endGame(STALEMATE, Player.NONE);
		} else
			endGame(NOTOVER, Player.NONE);
	}
	
	/**
	 * @return true iff the specified player's king is in check
	 **/
	public boolean isPlayerInCheck(Player p) {
		// Find Player p's king
		Piece pKing = null;
		
		List<Piece> pieces = getPieces(p);
		for (Piece piece : pieces) {
			if (piece.getType().equals(PieceTypeFactory.getKingType())) {
				pKing = piece;
				break;
			}
		}
		
		if (pKing == null)
			throw new RuntimeException("The king is gone!");
		
		// Check to see if the other player has pieces that can attack the king
		pieces = getPieces(p.otherPlayer());
		
		for (Piece piece : pieces) {
			if (piece.getMovementType().isMoveLegal(new ChessMove(piece, pKing.getRow(), pKing.getColumn(), pKing), this))
				return true;
		}
		return false;
	}
	
	public int getRows() {
		return ROWS;
	}
	
	public int getColumns() {
		return COLUMNS;
	}
	
	public Piece getPieceAt(int row, int column) {
		return board[row][column];
	}
	
	public void doMove(ChessMove m, String timestamp) throws IllegalMoveException {
		if (!doMoveInternal(m, timestamp))
			throw new IllegalMoveException(m);
		switchPlayer();
		notifyObservers();
	}

	public void undoLastMove() {
		popMove();
		switchPlayer();
		notifyObservers();
	}
	
	public List<Piece> getPieces() {
		List<Piece> pieces = new ArrayList<Piece>();
		for (Piece[] pieceRow : board)
			for (Piece piece : pieceRow)
				if (piece != null)
					pieces.add(piece);
		return pieces;
	}
	
	public List<Piece> getPieces(Player player) {
		List<Piece> pieces = new ArrayList<Piece>();
		for (Piece[] pieceRow : board)
			for (Piece piece : pieceRow)
				if (piece != null && piece.getPlayer().equals(player))
					pieces.add(piece);
		return pieces;
	}

	/**
	 * @return a list of Pieces in this.capturedPieces in an unspecified order
	 **/
	public List<Piece> getCapturedPieces() {
		return capturedPieces;
	}

	/**
	 * @return a list of Pieces in {p in this.capturedPieces | p.player ==
	 * player} in an unspecified order
	 **/
	public List<Piece> getCapturedPieces(Player player) {
		List<Piece> capturedPiecesForPlayer = new ArrayList<Piece>();
		
		for (Piece piece : capturedPieces)
			if (piece.getPlayer().equals(player))
				capturedPiecesForPlayer.add(piece);
		
		return capturedPiecesForPlayer;
	}
	
	public List<ChessMove> allLegalMoves() {
		List<ChessMove> allLegalMoves = new ArrayList<ChessMove>();
		List<Piece> pieces = getPieces(player);
		
		for (Piece p : pieces)
			allLegalMoves.addAll(legalMoves(p));
		
		return allLegalMoves;
	}
	
	public List<ChessMove> legalMoves(Piece p) {
		assert p.getPlayer().equals(player) : p + " is the wrong player";
		List<ChessMove> legalMoves = new ArrayList<ChessMove>();
		List<ChessMove> possibleMoves = p.getMovementType().getMoves(p, this);
		
		for (ChessMove possibleMove : possibleMoves)
			if (isMoveLegal(possibleMove))
				legalMoves.add(possibleMove);
		
		return legalMoves;
	}
    
	/**
	 * @return true iff the given move is an allowed move for the
	 * current player
	 **/
	public boolean isMoveLegal(ChessMove m) {
		boolean isLegal = doMoveInternal(m, "");
		if(isLegal) {
			popMove();
		}
		return isLegal;
	}
	
    /**
     * @return the move history for this ChessBoard
     */
    public MoveHistory<ChessMove> getMoveHistory() {
    	return moveStack;
    }
    
    /**
     * Attempt to perform a move, returning true if it's legal
     * @return true if the move was successfully performed
     * @modifies this
     * @effects Performs the specified move
     **/
	protected boolean doMoveInternal(ChessMove m, String timestamp) {
		Piece piece = m.getPiece();
        
		// Check that this move applies to the current player
		if(m.getPlayer() != getPlayer())
			return false;
		if(!piece.equals(getPieceAt(piece.getRow(),
									piece.getColumn())))
			return false;
        // Check that this move is legal
        if(!piece.getMovementType().isMoveLegal(m, this))
            return false;
        
        // Take care of captures
        Piece capture = m.getCapturedPiece();
        if(capture != null) {
            removePiece(capture);
        }
        
        // Move the piece
        movePiece(piece, m.getRow(), m.getColumn());
        // If a pawn hits the edge, promote it to a Queen
        if (piece.getType().equals(PieceTypeFactory.getPawnType()) &&
			((piece.getPlayer().equals(Player.BLACK) && m.getRow() == 0) ||
			 (piece.getPlayer().equals(Player.WHITE) && m.getRow() == 7)))
        	board[m.getRow()][m.getColumn()] = new Piece(piece.getPlayer(), PieceTypeFactory.getQueenType(), m.getRow(), m.getColumn());
        // Add move to the move history
        pushMove(m, timestamp);
        
        // Check if the the player who just moved is in check 
        if(isPlayerInCheck(m.getPlayer())) {
            popMove();
            return false;
        }
        
        return true;
    }

	/**
	 * Pushes the given move onto the internal piece stack
	 * @requires m != null
	 * @modifies this
	 * @effects adds the given move to the this.moveStack
	 **/
	protected void pushMove(ChessMove m, String timestamp) {
		if (moveStack == null)
			moveStack = new MoveHistory<ChessMove>(m, timestamp);
		else
			moveStack = moveStack.addMove(m, timestamp);
	}

	/**
	 * Pop a move off the internal move stack, undoing its effects
	 * @modifies this
	 * @effects undoes the last move and removes it from this.moveStack
	 **/
	protected void popMove() {
		ChessMove cm = moveStack.getLastMove();
		Piece oldPiece = cm.getPiece();
		Piece capturedPiece = cm.getCapturedPiece();
		
		board[oldPiece.getRow()][oldPiece.getColumn()] = oldPiece;
		board[cm.getRow()][cm.getColumn()] = null;
		
		if (capturedPiece != null) {
			board[capturedPiece.getRow()][capturedPiece.getColumn()] = capturedPiece;
			capturedPieces.remove(capturedPiece.moveTo(-1, -1));
		}
		
		moveStack = moveStack.removeLastMove();
		
		if (DEBUG)
			checkRep();
	}
	
    /**
     * Remove a piece from its square and move it to a destination
     * square
     * @requires this.getPieceAt(row, column) == null && this.pieces contains piece
     * @modifies this
     * @effects moves the given piece on the board. Does no legality checking
     **/
    private void movePiece(Piece piece, int row, int column) {
    	board[row][column] = piece.moveTo(row, column);
    	board[piece.getRow()][piece.getColumn()] = null;
    	
    	if (DEBUG)
    		checkRep();
    }

	/**
	 * Removes a piece from the board
	 * @requires this.pieces contains p
	 * @modifies this
	 * @effects this.pieces = this.pieces - p
	 **/
	private void removePiece(Piece p) {
		board[p.getRow()][p.getColumn()] = null;
    	
    	capturedPieces.add(p.moveTo(-1, -1));
    	
    	if (DEBUG)
    		checkRep();
    }
}
