// -*- mode: java; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*-

package antichess;

/**
 * A <tt>ChessMove</tt> represents a move in a Chess game.
 * <tt>ChessMove</tt>s are immutable.
 *
 * @specfield row : int			   // The row the piece is moving to
 * @specfield column : int		   // The column the piece is moving to 
 * @specfield capturedPiece : Piece	// The piece that is captured by this Move. May be null.
 **/
public class ChessMove extends Move
{
	private int row;
	private int column;
	// Optional piece that is caputred. Stored in its state before capture.
	private Piece capturedPiece;
	
	public ChessMove(Piece piece, int row, int column, Piece capturedPiece) {
		super(piece);
		this.row = row;
		this.column = column;
		this.capturedPiece = capturedPiece;
	}
	
	public ChessMove(Piece piece, int row, int column) {
		this(piece, row, column, null);
	}

	/**
	 * @return this.row
	 */
	public int getRow() {
		return row;
	}

	/**
	 * @return this.column
	 */
	public int getColumn() {
		return column;
	}
	
	/**
	 * @return this.capturedPiece
	 */
	public Piece getCapturedPiece() {
		return capturedPiece;
	}

	/**
	 * @return a textual description of this move
	 **/
	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append("ChessMove[");
		sb.append(piece);
		sb.append(" -> ");
		sb.append(row);
		sb.append(",");
		sb.append(column);
		if (capturedPiece != null) {
			sb.append(" takes ");
			sb.append(capturedPiece);
		}
		sb.append("]");
		return sb.toString();
	}
	
	/**
	 * @param row
	 * @return the standard chess representation of a row
	 */
	private String chessRow(int row) {
		return Integer.toString(row+1);
	}
	
	/**
	 * @param column
	 * @return the standard chess representation of a column
	 */
	private String chessColumn(int column) {
		return Integer.toString(column+10, 18);
	}
	
	/**
	 * @return a textual representation of the move
	 * that makes sense to a user
	 */
	// And by "makes sense to a user" I mean "conforms
	// to their retarded specs."
	public String humanReadableString() {
		StringBuilder sb = new StringBuilder();
		int startRow = piece.getRow(), startCol = piece.getColumn();
		sb.append(chessColumn(startCol));
		sb.append(chessRow(startRow));
		sb.append("-");
		sb.append(chessColumn(column));
		sb.append(chessRow(row));
		return sb.toString();
	}
	
	/**
	 * @param o the Object against which to test equality
	 * @return true if the Object o is a ChessMove equal to this; false otherwise
	 */
	@Override
	public boolean equals(Object o) {
		if (!(o instanceof ChessMove))
			return false;
		ChessMove cm = (ChessMove)o;
		if ((this.row == cm.row) && (this.column == cm.column)) {
			if ((this.capturedPiece == null && this.capturedPiece == cm.capturedPiece) ||
				(this.capturedPiece != null && this.capturedPiece.equals(cm.capturedPiece)))
				return true;
			else
				return false;
		}
		return false;
	}
}
