// -*- mode: java; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*-

package antichess.ai;

import antichess.*;
import java.io.File;
import java.util.List;

/**
 * <code>MinimaxAI</code> is a <code>GameAI</code> that implements the
 * minimax game search algorithm
 * 
 **/
public abstract class MinimaxAI<M extends Move, B extends Board<M>>
	extends GameAI<M, B>
{
	private static final long SEARCH_TIME = 4 * 1000;

	/**
	 * AIMove is a helper class to allow the minimax search routine to
	 * return a <move, value> tuple.
	 */
	private class AIMove {
		public AIMove(M m, int v) {
			this.move = m;
			this.value = v;
		}

		/**
		 * The game move represented by this AIMove
		 */
		public M move;

		/**
		 * The value of that move to the player making it
		 */
		public int value;

		/**
		 * <code>line</code> is used to store a linked list of the
		 * principal variation discovered by minimax
		 */
		public AIMove line;
	}

	/**
	 * @see GameAI.findMove
	 */
	public M findMove(B board, long timeLeft, long opponentTimeLeft) {
		AIMove move = null;
		int depth;
		long startTime = System.currentTimeMillis();
		
		List<M> allMoves = board.allLegalMoves();
		if(allMoves.size() == 0) return null;
		if(allMoves.size() == 1) return allMoves.get(0);
		
		for(depth = 1; (System.currentTimeMillis() - startTime) < SEARCH_TIME; depth++) {
			move =  minimaxMove((B)board.clone(),
								Integer.MIN_VALUE,
								Integer.MAX_VALUE,
								depth, move);
			System.err.println("ply " + depth + ": " + (System.currentTimeMillis() - startTime) + "ms");
		}
		
		AIMove m = move;
		System.err.println("Principal variation (depth=" + depth + ",val=" + m.value + "):");
		while(m != null) {
			System.err.println(m.move);
			m = m.line;
		}
		System.err.println("----");

		return move.move;
	}

	/**
	 * <p>Return an <code>AIMove</code> representing the information discovered by an
	 * alpha-beta pruned minimax search of the specified depth on the
	 * given board.</p>
	 *
	 * <p>The returned <code>AIMove</code> contains fields indicating
	 * the actual move on the board that was uncovered, the heuristic
	 * value of that move, and a linked chain of <code>AIMove</code>
	 * elements representing the principal variation discovered by the
	 * search</p>
	 *
	 * @param board : The Board to search over. The method will modify
	 * this board while searching, but will undo any moves that it
	 * makes.
	 * @param alpha : The alpha value for alpha-beta pruning
	 * @param beta : The beta value for alpha-beta pruning
	 * @param depth : The depth to which the tree should be searched
	 * @param tryMove : A line of moves that the caller has reason to
	 * believe represents a good move from the given position. This
	 * may be given as <code>null</code>. Specifying a move here does
	 * not change the semantics of the method; However, specifying a
	 * good move may potentially speed up the search. It is not
	 * required that <code>tryMove.move</code> be a legal move from
	 * the given board.
	 */
	private AIMove minimaxMove(B board, int alpha, int beta, int depth,
							   AIMove tryMove) {
		boolean white = board.getPlayer() == Player.WHITE;
		if(depth == 0 || board.isGameOver())
			return new AIMove(null, (white ? 1 : -1 ) * boardValue(board));
		AIMove bestMove = new AIMove(null, alpha);
		AIMove pv;
		int value;
		List<M> moves = board.allLegalMoves();
		if(tryMove != null && tryMove.move != null && board.isMoveLegal(tryMove.move)) {
			bestMove.line = tryMove.line;
			moves.add(0, tryMove.move);
		}
		for(M m : moves) {
			try {
				board.doMove(m);
			} catch(IllegalMoveException e) {
				System.err.println("Illegal move: " + m);
				System.err.println("Player: " + board.getPlayer());
				System.err.println("Board saved to debug.xml");
				try {
					new GameWriter((ChessBoard)board).writeGame(new File("debug.xml"));
				} catch(Exception e2){}
				throw new RuntimeException("allLegalMoves returned an illegal move: " + m);
			}
			pv = minimaxMove(board, -beta, -bestMove.value, depth-1, bestMove.line);
			pv.value = -pv.value;
			board.undoLastMove();
			//System.err.println("(" + depth + ")" + m + ": " + pv.value);
			if(pv.value > bestMove.value) {
				bestMove.move = m;
				bestMove.value = pv.value;
				bestMove.line = pv;
				if(bestMove.value >= beta) {
					return bestMove;
				}
			}
		}
		return bestMove;
	}

	/**
	 * @return a heuristic value for the given board. Larger values
	 * should indicate positions more favorable for the White player.
	 */
	public abstract int boardValue(B board);
}
