package antichess.test;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import antichess.*;
import junit.framework.TestCase;

/**
 * Tests for the functionality of AntichessBoardTests. Focuses
 * on that functionality AntichessBoards have over ChessBoards.
 */
public class AntichessBoardTests extends TestCase {

	/**
	 * A variable to hold the pieces on a new ChessBoard.
	 */
	private List<Piece> initialPieces;
	/**
	 * The maximum move depth into which the self consistency should
	 * be checked.
	 */
	private int MAX_RECURSE = 3;
	
	public AntichessBoardTests(String name) {
		super(name);
		
		// Create a list of the pieces on a new-game chess board, sorted by player color
		// First 16 pieces are White, while the second 16 are Black
		initialPieces = new ArrayList<Piece>();
		
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getRookType(), 0, 0));
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getKnightType(), 0, 1));
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getBishopType(), 0, 2));
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getQueenType(), 0, 3));
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getKingType(), 0, 4));
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getBishopType(), 0, 5));
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getKnightType(), 0, 6));
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getRookType(), 0, 7));
		
		for (int i = 0; i < 8; i++)
			initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getPawnType(), 1, i));
		
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getRookType(), 7, 0));
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getKnightType(), 7, 1));
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getBishopType(), 7, 2));
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getQueenType(), 7, 3));
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getKingType(), 7, 4));
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getBishopType(), 7, 5));
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getKnightType(), 7, 6));
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getRookType(), 7, 7));
		
		for (int i = 0; i < 8; i++)
			initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getPawnType(), 6, i));
	}
	
	// Code in the following tests is blocked together by individual moves
	/**
	 * Tests that moves applied to an AntichessBoard have the expected legality
	 * and have the expected effects on the Board.
	 */
	public void testMove() {
		List<Piece> pieces = initialPieces;
		AntichessBoard cb = new AntichessBoard();
		MoveHistory<ChessMove> expected = null;
		ChessMove nextMove;
		cb.newGame();
		
		nextMove = new ChessMove(cb.getPieceAt(1, 4), 3, 4);
		checkMoveLegality(cb, nextMove, true);
		checkGetMoveHistory(cb, expected);
		pieces = doMove(cb, nextMove, pieces);
		expected = new MoveHistory<ChessMove>(nextMove);
		checkPieces(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, false);
		checkGameOver(cb, false);
		checkEndGameConditions(cb, AntichessBoard.NOTOVER, Player.NONE);
		
		nextMove = new ChessMove(cb.getPieceAt(6, 4), 4, 4);
		checkMoveLegality(cb, nextMove, true);
		pieces = doMove(cb, nextMove, pieces);
		expected = expected.addMove(nextMove);
		checkPieces(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, false);
		checkGameOver(cb, false);
		checkEndGameConditions(cb, AntichessBoard.NOTOVER, Player.NONE);
		
		nextMove = new ChessMove(cb.getPieceAt(0, 3), 4, 7);
		checkMoveLegality(cb, nextMove, true);
		pieces = doMove(cb, nextMove, pieces);
		expected = expected.addMove(nextMove);
		checkPieces(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, false);
		checkGameOver(cb, false);
		checkEndGameConditions(cb, AntichessBoard.NOTOVER, Player.NONE);
		
		nextMove = new ChessMove(cb.getPieceAt(7, 1), 5, 2);
		checkMoveLegality(cb, nextMove, true);
		pieces = doMove(cb, nextMove, pieces);
		expected = expected.addMove(nextMove);
		checkPieces(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, false);
		checkGameOver(cb, false);
		checkEndGameConditions(cb, AntichessBoard.NOTOVER, Player.NONE);
		
		nextMove = new ChessMove(cb.getPieceAt(0, 5), 3, 2);
		checkMoveLegality(cb, nextMove, false);
		nextMove = new ChessMove(cb.getPieceAt(4, 7), 6, 5, cb.getPieceAt(6, 5));
		checkMoveLegality(cb, nextMove, true);
		pieces = doMove(cb, nextMove, pieces);
		expected = expected.addMove(nextMove);
		checkPieces(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, true);
		checkGameOver(cb, false);
		checkEndGameConditions(cb, AntichessBoard.NOTOVER, Player.NONE);
		
		nextMove = new ChessMove(cb.getPieceAt(7, 4), 6, 5, cb.getPieceAt(6, 5));
		checkMoveLegality(cb, nextMove, true);
		pieces = doMove(cb, nextMove, pieces);
		expected = expected.addMove(nextMove);
		checkPieces(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, false);
		checkGameOver(cb, false);
		checkEndGameConditions(cb, AntichessBoard.NOTOVER, Player.NONE);
	}
	
	/**
	 * Tests that moves applied to an AntichessBoard have the expected legality
	 * and have the expected effects on the Board. Uses a different series of
	 * moves from testMove() that ultimately result in checkmate.
	 */
	public void testMove2() {
		List<Piece> pieces = initialPieces;
		AntichessBoard cb = new AntichessBoard();
		MoveHistory<ChessMove> expected = null;
		ChessMove nextMove;
		cb.newGame();
		
		nextMove = new ChessMove(cb.getPieceAt(1, 5), 2, 5);
		checkMoveLegality(cb, nextMove, true);
		checkGetMoveHistory(cb, expected);
		pieces = doMove(cb, nextMove, pieces);
		expected = new MoveHistory<ChessMove>(nextMove);
		checkPieces(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, false);
		checkGameOver(cb, false);
		checkEndGameConditions(cb, AntichessBoard.NOTOVER, Player.NONE);
		
		nextMove = new ChessMove(cb.getPieceAt(6, 4), 4, 4);
		checkMoveLegality(cb, nextMove, true);
		pieces = doMove(cb, nextMove, pieces);
		expected = expected.addMove(nextMove);
		checkPieces(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, false);
		checkGameOver(cb, false);
		checkEndGameConditions(cb, AntichessBoard.NOTOVER, Player.NONE);
		
		nextMove = new ChessMove(cb.getPieceAt(1, 6), 3, 6);
		checkMoveLegality(cb, nextMove, true);
		checkGetMoveHistory(cb, expected);
		pieces = doMove(cb, nextMove, pieces);
		expected = expected.addMove(nextMove);
		checkPieces(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, false);
		checkGameOver(cb, false);
		checkEndGameConditions(cb, AntichessBoard.NOTOVER, Player.NONE);
		
		nextMove = new ChessMove(cb.getPieceAt(7, 3), 3, 7);
		checkMoveLegality(cb, nextMove, true);
		pieces = doMove(cb, nextMove, pieces);
		expected = expected.addMove(nextMove);
		checkPieces(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, true);
		checkCheckStatus(cb, Player.BLACK, false);
		checkGameOver(cb, true);
		checkEndGameConditions(cb, AntichessBoard.CHECKMATE, Player.BLACK);
	}
	
	/**
	 * Loads an arbitrary Board configuration onto the Board and verifies
	 * that this has the expected effect on the Board. Then executes a series
	 * of legal moves on this Board, verifying that these moves have the
	 * expected results and legalities for the Board.
	 */
	public void testLoadAndMove() {
		List<Piece> pieces = new ArrayList<Piece>();
		pieces.add(new Piece(Player.BLACK, PieceTypeFactory.getRookType(), 7, 0));
		pieces.add(new Piece(Player.BLACK, PieceTypeFactory.getKingType(), 7, 7));
		pieces.add(new Piece(Player.BLACK, PieceTypeFactory.getPawnType(), 6, 6));
		pieces.add(new Piece(Player.BLACK, PieceTypeFactory.getPawnType(), 6, 7));
		pieces.add(new Piece(Player.WHITE, PieceTypeFactory.getKnightType(), 4, 4));
		pieces.add(new Piece(Player.WHITE, PieceTypeFactory.getQueenType(), 2, 1));
		pieces.add(new Piece(Player.WHITE, PieceTypeFactory.getKingType(), 0, 6));
		
		AntichessBoard cb = new AntichessBoard();
		MoveHistory<ChessMove> expected = null;
		ChessMove nextMove;
		cb.loadGame(pieces, Player.WHITE, expected);
		checkPieceLocations(cb, pieces);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, false);
		checkGameOver(cb, false);
		checkEndGameConditions(cb, AntichessBoard.NOTOVER, Player.NONE);
		
		nextMove = new ChessMove(cb.getPieceAt(4, 4), 6, 5);
		checkMoveLegality(cb, nextMove, true);
		checkGetMoveHistory(cb, expected);
		pieces = doMove(cb, nextMove, pieces);
		expected = new MoveHistory<ChessMove>(nextMove);
		checkPieceLocations(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, true);
		checkGameOver(cb, false);
		checkEndGameConditions(cb, AntichessBoard.NOTOVER, Player.NONE);
		
		nextMove = new ChessMove(cb.getPieceAt(7, 7), 7, 6);
		checkMoveLegality(cb, nextMove, true);
		pieces = doMove(cb, nextMove, pieces);
		expected = expected.addMove(nextMove);
		checkPieceLocations(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, false);
		checkGameOver(cb, false);
		checkEndGameConditions(cb, AntichessBoard.NOTOVER, Player.NONE);
		
		nextMove = new ChessMove(cb.getPieceAt(6, 5), 5, 3);
		checkMoveLegality(cb, nextMove, true);
		pieces = doMove(cb, nextMove, pieces);
		expected = expected.addMove(nextMove);
		checkPieceLocations(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, true);
		checkGameOver(cb, false);
		checkEndGameConditions(cb, AntichessBoard.NOTOVER, Player.NONE);
		
		nextMove = new ChessMove(cb.getPieceAt(7, 6), 7, 7);
		checkMoveLegality(cb, nextMove, true);
		pieces = doMove(cb, nextMove, pieces);
		expected = expected.addMove(nextMove);
		checkPieceLocations(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, false);
		checkGameOver(cb, false);
		checkEndGameConditions(cb, AntichessBoard.NOTOVER, Player.NONE);
		
		nextMove = new ChessMove(cb.getPieceAt(2, 1), 7, 6);
		checkMoveLegality(cb, nextMove, true);
		pieces = doMove(cb, nextMove, pieces);
		expected = expected.addMove(nextMove);
		checkPieceLocations(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, true);
		checkGameOver(cb, false);
		checkEndGameConditions(cb, AntichessBoard.NOTOVER, Player.NONE);
		
		nextMove = new ChessMove(cb.getPieceAt(7, 0), 7, 6, cb.getPieceAt(7, 6));
		checkMoveLegality(cb, nextMove, true);
		pieces = doMove(cb, nextMove, pieces);
		expected = expected.addMove(nextMove);
		checkPieceLocations(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, false);
		checkGameOver(cb, false);
		checkEndGameConditions(cb, AntichessBoard.NOTOVER, Player.NONE);
		
		nextMove = new ChessMove(cb.getPieceAt(5, 3), 6, 5);
		checkMoveLegality(cb, nextMove, true);
		pieces = doMove(cb, nextMove, pieces);
		expected = expected.addMove(nextMove);
		checkPieceLocations(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, true);
		checkGameOver(cb, true);
		checkEndGameConditions(cb, AntichessBoard.CHECKMATE, Player.WHITE);
	}
	
	/**
	 * Tests the functionality of the Board to recognize the OUTOFPIECES
	 * end-game condition.
	 */
	public void testOutOfPiecesEnd() {
		AntichessBoard cb = new AntichessBoard();
		List<Piece> pieces = initialPieces.subList(0, 16);
		pieces.add(new Piece(Player.BLACK, PieceTypeFactory.getKingType(), 7, 4));
		
		cb.loadGame(pieces, Player.BLACK, null);
		checkGameOver(cb, true);
		checkEndGameConditions(cb, AntichessBoard.OUTOFPIECES, Player.BLACK);
	}
	
	/**
	 * Checks that legal moves returned from the board can be successfully
	 * executed. This is done to MAX_RECURSE levels of recursion from a
	 * new-game board.
	 */
	public void testSelfConsistency() {
		List<Piece> pieces = initialPieces;
		AntichessBoard cb = new AntichessBoard();
		cb.newGame();
		checkSelfConsistency(cb, pieces, 1);
	}
	
	/**
	 * Private method to verify that the chessboard has pieces in the expected
	 * locations, both on the board and in the captured piles. This tests
	 * the getPieceAt() method and getCapturePieces() methods.
	 * @param cb the chessboard
	 * @param pieces the list of pieces, with their expected locations on the board
	 * and in the captured pile.
	 */
	private void checkPieceLocations(AntichessBoard cb, List<Piece> pieces) {
		Piece test;
		boolean found;
		// Test that each of the pieces in the given list is in the correct position
		for (Piece p : pieces) {
			if (p.getRow() != -1 && p.getColumn() != -1) {
				test = cb.getPieceAt(p.getRow(), p.getColumn());
				assertEquals("Incorrect piece found at " + p.getRow() + ", " + p.getColumn(),
						p, test);
			} else if (p.getRow() == -1 && p.getColumn() == -1) {
				found = false;
				List<Piece> capturedPile = cb.getCapturedPieces();
				for (Piece cp : capturedPile)
					if (p.equals(cp))
						found = true;
				assertTrue("Expected captured piece not found in captured pile", found);
			} else {
				fail("Expected piece had an invalid location");
				return;
			}
		}
	}
	
	/**
	 * Checks that the pieces for a given player (or both players if player == null)
	 * have the expected locations. This allows for verification of the getPieces()
	 * and getPieces(player) methods.
	 * @param cb the chessboard
	 * @param pieces the list of expected pieces for the given player
	 * @param player the player in question.
	 */
	private void checkGetPieces(AntichessBoard cb, List<Piece> pieces, Player player) {
		List<Piece> piecesCopy = new ArrayList<Piece>(pieces);
		List<Piece> currentPieceList;
		
		if (player == null)
			currentPieceList = cb.getPieces();
		else
			currentPieceList = cb.getPieces(player);
		
		for (Piece nextPiece : currentPieceList) {
			assertTrue("Board contains an unexpected piece: " + nextPiece.toString(),
					piecesCopy.contains(nextPiece));
			piecesCopy.remove(nextPiece);
		}
		
		if (player == null)
			currentPieceList = cb.getCapturedPieces();
		else
			currentPieceList = cb.getCapturedPieces(player);
		
		for (Piece nextPiece : currentPieceList) {
			assertTrue("Board's captured pile contains an unexpected piece: " + nextPiece.toString(),
					piecesCopy.contains(nextPiece));
			piecesCopy.remove(nextPiece);
		}
		
		assertTrue("Board did not contain all of the given pieces on the board or in the captured pile",
				piecesCopy.size() == 0);
	}
	
	/**
	 * Tests that the MoveHistory from the chessboard matches the expected MoveHistory.
	 * @param cb the chessboard
	 * @param expected the expected movehistory
	 */
	private void checkGetMoveHistory(AntichessBoard cb, MoveHistory<ChessMove> expected) {
		if (expected == null)
			assertTrue("Non-null move history found in AntichessBoard where null move history expected",
					cb.getMoveHistory() == null);
		else {
			List<ChessMove> expectedHistory = expected.getHistory();
			List<String> expectedTimes = expected.getMoveTimes();
			List<ChessMove> actualHistory = cb.getMoveHistory().getHistory();
			List<String> actualTimes = cb.getMoveHistory().getMoveTimes();
			
			assertEquals("Expected move history does not match actual move history.",
					expectedHistory, actualHistory);
			assertEquals("Expected move times do not match actual move times.",
					expectedTimes, actualTimes);
		}
	}
	
	/**
	 * Helper method to quickly check all of the methods for getting
	 * the pieces on the board or in the capture pile. This assumes
	 * that pieces includes all of the standard chess pieces.
	 * @param cb the chessboard
	 * @param pieces the full list of chess pieces in a standard game
	 * of chess.
	 */
	private void checkPieces(AntichessBoard cb, List<Piece> pieces) {
		checkPieceLocations(cb, pieces);
		checkGetPieces(cb, pieces, null);
		checkGetPieces(cb, pieces.subList(0, 16), Player.WHITE);
		checkGetPieces(cb, pieces.subList(16, 32), Player.BLACK);
	}
	
	/**
	 * Verifies that the legality of a given move matches the given expected
	 * value. Tests isMoveLegal.
	 * @param cb the chessboard
	 * @param cm the ChessMove whose legality is in question.
	 * @param legal the expected legality
	 * @return a new list of pieces with the piece locations properly
	 * updated.
	 */
	private void checkMoveLegality(AntichessBoard cb, ChessMove cm, boolean legal) {
		if (legal)
			assertTrue("AntichessBoard declared a legal move illegal: " + cm,
					cb.isMoveLegal(cm));
		else
			assertFalse("AntichessBoard declared an illegal move legal: " + cm,
					cb.isMoveLegal(cm));
	}
	
	/**
	 * Helper method to execute a ChessMove on this board and update
	 * the list of pieces correctly. ChessMove <code>cm</code> must
	 * be a legal move for this chessboard.
	 * @param cb the chessboard
	 * @param cm the ChessMove to execute
	 * @param start the original list of pieces on the board to update
	 * @return a new list of pieces with the piece locations properly
	 * updated.
	 */
	private List<Piece> doMove(AntichessBoard cb, ChessMove cm, List<Piece> start) {
		List<Piece> end = new ArrayList<Piece>(start);
		int pieceToMoveIndex = start.indexOf(cm.getPiece());
		int capturedPieceIndex;
		
		try {
			cb.doMove(cm);
		} catch (Exception e) {
			fail("Board disallowed a legal move: " + e.getMessage());
			return end;
		}
		
		// Pawn promotions and timestamped moves have already been tested
		// in ChessBoardTests, and those methods are unchanged in
		// AntichessBoardTests in ways that would affect the functionality
		// of those features.
		end.set(pieceToMoveIndex, end.get(pieceToMoveIndex).moveTo(cm.getRow(), cm.getColumn()));
		if (cm.getCapturedPiece() != null) {
			capturedPieceIndex = start.indexOf(cm.getCapturedPiece());
			end.set(capturedPieceIndex, end.get(capturedPieceIndex).moveTo(-1, -1));
		}
		return end;
	}
	
	/**
	 * Helper method to undo the last ChessMove executed on a given
	 * chessboard and update the list of pieces properly. At least
	 * one move must have been executed on the given chessboard and
	 * recorded in the board's move history.
	 * @param cb the chessboard
	 * @param start the list of pieces with their current locations
	 * on the board.
	 * @return a new list of pieces with the piece locations properly
	 * updated.
	 */
	private List<Piece> undoMove(AntichessBoard cb, List<Piece> start) {
		List<Piece> end = new ArrayList<Piece>(start);
		// Grab the last move from the history
		List<ChessMove> history = cb.getMoveHistory().getHistory();
		ChessMove last = history.get(history.size()-1);
		
		int pieceMovedIndex = start.indexOf(last.getPiece().moveTo(last.getRow(), last.getColumn()));
		int capturedPieceIndex;
		
		try {
			cb.undoLastMove();
		} catch (Exception e) {
			fail("Board disallowed undoing move: " + e.getMessage());
			return end;
		}
		
		// Pawn promotions and timestamped moves have already been tested
		// in ChessBoardTests, and those methods are unchanged in
		// AntichessBoardTests in ways that would affect the functionality
		// of those features.
		end.set(pieceMovedIndex, last.getPiece());
		if (last.getCapturedPiece() != null) {
			capturedPieceIndex = start.indexOf(last.getCapturedPiece().moveTo(-1, -1));
			end.set(capturedPieceIndex, last.getCapturedPiece());
		}
		return end;
	}
	
	/**
	 * Verifies that the "in check" status of the given player is the
	 * given expected value.
	 * @param cb the chessboard
	 * @param player the Player in question
	 * @param expected the expected value of isPlayerInCheck(player).
	 */
	private void checkCheckStatus(AntichessBoard cb, Player player, boolean expected) {
		if (expected)
			assertTrue("Board failed to recognize that " + player.toString() + " is in check.",
					cb.isPlayerInCheck(player));
		else
			assertFalse("Board mistakenly recognized that " + player.toString() + " is in check.",
					cb.isPlayerInCheck(player));
	}
	
	/**
	 * Verifies that the game-over characteristic of the given
	 * chessboard is as expected. Test isGameOver().
	 * @param cb the chessboard
	 * @param expected
	 */
	private void checkGameOver(AntichessBoard cb, boolean expected) {
		if (expected) {
			assertTrue("Board failed to recognize the end of the game",
					cb.isGameOver());
		} else {
			assertFalse("Board mistook to recognize the end of the game",
					cb.isGameOver());
		}
	}
	
	/**
	 * Tests the setting of the end-game conditions to verify that they
	 * have the given expected values. This method tests
	 * getGameOverReason() and getWinner().
	 * @param cb the chessboard
	 * @param expectedGameOverReason the expected GameOverReason
	 * @param expectedWinner the expected winner
	 */
	private void checkEndGameConditions(AntichessBoard cb, int expectedGameOverReason, Player expectedWinner) {
		assertEquals("Board has an unexpected GameOverReason",
				expectedGameOverReason, cb.getGameOverReason());
		assertEquals("Board has an unexpected winner",
				expectedWinner, cb.getWinner());
	}
	
	 /**
	  * Helper method to recursively check the self-consistency of
	  * the board, verifying that moves returned by allLegalMoves()
	  * can be made with the expected consequences.
	  * @param cb the chessboard
	  * @param pieces the list holding the arrangement of pieces
	  * on the given chessboard
	  * @param i the level of recursion at which this method is
	  * currently running. Will be less than MAX_RECURSE.
	  */
	private void checkSelfConsistency(AntichessBoard cb, List<Piece> pieces, int i) {
		AntichessBoard localcb = (AntichessBoard)cb.clone();
		List<ChessMove> allLegal = localcb.allLegalMoves();
		
		for (ChessMove cm : allLegal) {
			checkMoveLegality(localcb, cm, true);
			pieces = doMove(localcb, cm, pieces);
			checkPieces(localcb, pieces);
			
			if (i < MAX_RECURSE)
				checkSelfConsistency(localcb, new ArrayList<Piece>(pieces), i+1);
			
			pieces = undoMove(localcb, pieces);
			checkPieces(localcb, pieces);
		}
	}
}
