package antichess.test;

import java.util.*;

import junit.framework.TestCase;
import antichess.*;

/**
 * Tests for the functionality of ChessBoards and their methods. In
 * conjunction with integration testing this provides a thurough
 * testing system of the Board.
 */
public class ChessBoardTests extends TestCase {

	/**
	 * A variable to hold the pieces on a new ChessBoard.
	 */
	private List<Piece> initialPieces;
	/**
	 * The maximum move depth into which the self consistency should
	 * be checked.
	 */
	private int MAX_RECURSE = 3;
	
	public ChessBoardTests(String name) {
		super(name);
		
		initialPieces = new ArrayList<Piece>();
		
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getRookType(), 0, 0));
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getKnightType(), 0, 1));
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getBishopType(), 0, 2));
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getQueenType(), 0, 3));
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getKingType(), 0, 4));
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getBishopType(), 0, 5));
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getKnightType(), 0, 6));
		initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getRookType(), 0, 7));
		
		for (int i = 0; i < 8; i++)
			initialPieces.add(new Piece(Player.WHITE, PieceTypeFactory.getPawnType(), 1, i));
		
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getRookType(), 7, 0));
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getKnightType(), 7, 1));
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getBishopType(), 7, 2));
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getQueenType(), 7, 3));
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getKingType(), 7, 4));
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getBishopType(), 7, 5));
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getKnightType(), 7, 6));
		initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getRookType(), 7, 7));
		
		for (int i = 0; i < 8; i++)
			initialPieces.add(new Piece(Player.BLACK, PieceTypeFactory.getPawnType(), 6, i));
	}
	
	/**
	 * Verifies that chessboards may be constructed through their
	 * constructor and that doing so has the desired effect.
	 *
	 */
	public void testConstructor() {
		ChessBoard cb = new ChessBoard();
		checkBoardDimensions(cb);
	}
	
	/**
	 * Tests creating new games on a chessboard.
	 */
	public void testNewGame() {
		ChessBoard cb = new ChessBoard();
		cb.newGame();
		
		checkBoardDimensions(cb);
		checkPieces(cb, initialPieces);
	}
	
	// Code in the following tests is blocked together by individual moves
	/**
	 * Tests that pawn promotion has the expected result, and can be done
	 * and undone.
	 */
	public void testPawnPromote() {
		ChessBoard cb = new ChessBoard();
		List<Piece> pieces = new ArrayList<Piece>();
		pieces.add(new Piece(Player.WHITE, PieceTypeFactory.getPawnType(), 6, 0));
		pieces.add(new Piece(Player.BLACK, PieceTypeFactory.getPawnType(), 1, 1));
		pieces.add(new Piece(Player.WHITE, PieceTypeFactory.getKingType(), 5, 7));
		pieces.add(new Piece(Player.BLACK, PieceTypeFactory.getKingType(), 3, 7));
		MoveHistory<ChessMove> expected = null;
		cb.loadGame(pieces, Player.WHITE, expected);
		ChessMove nextMove;
		
		nextMove = new ChessMove(cb.getPieceAt(6, 0), 7, 0);
		checkMoveLegality(cb, nextMove, true);
		pieces = doMove(cb, nextMove, pieces);
		expected = new MoveHistory<ChessMove>(nextMove);
		checkGetMoveHistory(cb, expected);
		checkPieceLocations(cb, pieces);
		
		nextMove = new ChessMove(cb.getPieceAt(1, 1), 0, 1);
		checkMoveLegality(cb, nextMove, true);
		pieces = doMove(cb, nextMove, pieces);
		expected = expected.addMove(nextMove);
		checkGetMoveHistory(cb, expected);
		checkPieceLocations(cb, pieces);
		
		pieces = undoMove(cb, pieces);
		expected = expected.removeLastMove();
		checkGetMoveHistory(cb, expected);
		checkPieceLocations(cb, pieces);
		
		pieces = undoMove(cb, pieces);
		expected = expected.removeLastMove();
		checkGetMoveHistory(cb, expected);
		checkPieceLocations(cb, pieces);
	}
	
	/**
	 * Tests that moves applied to a ChessBoard have the expected legality
	 * and have the expected effects on the Board. Also verifies that the
	 * board can take timestamps with moves with the expected results.
	 */
	public void testMove() {
		List<Piece> pieces = initialPieces;
		ChessBoard cb = new ChessBoard();
		MoveHistory<ChessMove> expected = null;
		ChessMove nextMove;
		cb.newGame();
		
		nextMove = new ChessMove(cb.getPieceAt(1, 4), 3, 4);
		checkMoveLegality(cb, nextMove, true);
		checkGetMoveHistory(cb, expected);
		pieces = doMove(cb, nextMove, pieces);
		expected = new MoveHistory<ChessMove>(nextMove);
		checkPieces(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, false);
		checkEndGameConditions(cb, ChessBoard.NOTOVER, Player.NONE);
		
		nextMove = new ChessMove(cb.getPieceAt(6, 4), 4, 4);
		checkMoveLegality(cb, nextMove, true);
		pieces = doMove(cb, nextMove, pieces, "");
		expected = expected.addMove(nextMove, "");
		checkPieces(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, false);
		checkEndGameConditions(cb, ChessBoard.NOTOVER, Player.NONE);
		
		nextMove = new ChessMove(cb.getPieceAt(0, 3), 4, 7);
		checkMoveLegality(cb, nextMove, true);
		pieces = doMove(cb, nextMove, pieces, "12.144141414");
		expected = expected.addMove(nextMove, "12.144141414");
		checkPieces(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, false);
		checkEndGameConditions(cb, ChessBoard.NOTOVER, Player.NONE);
		
		nextMove = new ChessMove(cb.getPieceAt(7, 1), 5, 2);
		checkMoveLegality(cb, nextMove, true);
		pieces = doMove(cb, nextMove, pieces, "1234");
		expected = expected.addMove(nextMove, "1234");
		checkPieces(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, false);
		checkEndGameConditions(cb, ChessBoard.NOTOVER, Player.NONE);
		
		nextMove = new ChessMove(cb.getPieceAt(0, 5), 3, 2);
		checkMoveLegality(cb, nextMove, true);
		pieces = doMove(cb, nextMove, pieces);
		expected = expected.addMove(nextMove);
		checkPieces(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, false);
		checkEndGameConditions(cb, ChessBoard.NOTOVER, Player.NONE);
		
		nextMove = new ChessMove(cb.getPieceAt(7, 6), 5, 5);
		checkMoveLegality(cb, nextMove, true);
		pieces = doMove(cb, nextMove, pieces, "22222222222222222222222222");
		expected = expected.addMove(nextMove, "22222222222222222222222222");
		checkPieces(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, false);
		checkEndGameConditions(cb, ChessBoard.NOTOVER, Player.NONE);
		
		nextMove = new ChessMove(cb.getPieceAt(4, 7), 6, 5, cb.getPieceAt(6, 5));
		checkMoveLegality(cb, nextMove, true);
		pieces = doMove(cb, nextMove, pieces);
		expected = expected.addMove(nextMove);
		checkPieces(cb, pieces);
		checkGetMoveHistory(cb, expected);
		checkCheckStatus(cb, Player.WHITE, false);
		checkCheckStatus(cb, Player.BLACK, true);
		checkEndGameConditions(cb, ChessBoard.CHECKMATE, Player.WHITE);
	}
	
	/**
	 * Checks that legal moves returned from the board can be successfully
	 * executed. This is done to MAX_RECURSE levels of recursion from a
	 * new-game board.
	 */
	public void testSelfConsistency() {
		List<Piece> pieces = initialPieces;
		ChessBoard cb = new ChessBoard();
		cb.newGame();
		checkSelfConsistency(cb, pieces, 1);
	}
	
	/**
	 * Verifies that cloning a board works and has the desired effect.
	 */
	public void testClone() {
		ChessBoard cb2;
		ChessBoard cb = new ChessBoard();
		cb.newGame();
		
		try {
			cb2 = (ChessBoard)cb.clone();
		} catch (Exception e) {
			fail("Threw an exception rather than cloning the ChessBoard into another ChessBoard.");
			return;
		}
		
		checkBoardDimensions(cb2);
		checkPieces(cb2, initialPieces);
	}
	
	/**
	 * Really simple test for the initial move history. The main tests
	 * of the MoveHistory in the Board rests in testMove().
	 */
	public void testMoveHistory() {
		ChessBoard cb = new ChessBoard();
		MoveHistory<ChessMove> expected = null;
		checkGetMoveHistory(cb, expected);
		
		cb.newGame();
		checkGetMoveHistory(cb, expected);
	}
	
	/**
	 * Test loading games and the stalemate end-game conditions
	 */
	public void testLoadAndStalemate() {
		ChessBoard cb = new ChessBoard();
		// Test the Stalemate condition 
		List<Piece> pieces = new ArrayList<Piece>();
		pieces.add(new Piece(Player.BLACK, PieceTypeFactory.getKingType(), 7, 7));
		pieces.add(new Piece(Player.WHITE, PieceTypeFactory.getKingType(), 0, 6));
		
		cb.loadGame(pieces, Player.BLACK, null);	
		checkEndGameConditions(cb, ChessBoard.STALEMATE, Player.NONE);
		
		pieces.clear();
		pieces.add(new Piece(Player.BLACK, PieceTypeFactory.getKingType(), 7, 7));
		pieces.add(new Piece(Player.WHITE, PieceTypeFactory.getKingType(), 5, 7));
		pieces.add(new Piece(Player.WHITE, PieceTypeFactory.getRookType(), 0, 6));
		pieces.add(new Piece(Player.WHITE, PieceTypeFactory.getRookType(), 0, 0));
		
		cb.loadGame(pieces, Player.WHITE, new MoveHistory<ChessMove>(
				new ChessMove(
						new Piece(Player.BLACK, PieceTypeFactory.getKingType(), 6, 6),
						7, 7)));
		checkEndGameConditions(cb, ChessBoard.NOTOVER, Player.NONE);
		doMove(cb,
				new ChessMove(new Piece(Player.WHITE, PieceTypeFactory.getRookType(), 0, 0), 6, 0),
				pieces);
		checkEndGameConditions(cb, ChessBoard.STALEMATE, Player.NONE);
	}
	
	/**
	 * Tests the OUTOFTIME end-game condition, mainly, that
	 * setting this condition has the expected results.
	 */
	public void testOutOfTimeEnd() {
		ChessBoard cb = new ChessBoard();
		cb.newGame();
		
		cb.endGame(ChessBoard.OUTOFTIME, Player.WHITE);
		checkGameOver(cb, true);
		checkEndGameConditions(cb, ChessBoard.OUTOFTIME, Player.WHITE);
	}
	
	/**
	 * Verifies that the dimensions of the chessboard are as expected.
	 * @param cb the chessboard
	 */
	private void checkBoardDimensions(ChessBoard cb) {
		assertTrue("Unexpected number of rows for ChessBoard " + cb.getRows(),
				cb.getRows() == 8);
		assertTrue("Unexpected number of columns for ChessBoard " + cb.getColumns(),
				cb.getColumns() == 8);
	}
	
	/**
	 * Private method to verify that the chessboard has pieces in the expected
	 * locations, both on the board and in the captured piles. This tests
	 * the getPieceAt() method and getCapturePieces() methods.
	 * @param cb the chessboard
	 * @param pieces the list of pieces, with their expected locations on the board
	 * and in the captured pile.
	 */
	private void checkPieceLocations(ChessBoard cb, List<Piece> pieces) {
		Piece test;
		boolean found;
		// Test that each of the pieces in the given list is in the correct position
		for (Piece p : pieces) {
			if (p.getRow() != -1 && p.getColumn() != -1) {
				test = cb.getPieceAt(p.getRow(), p.getColumn());
				assertEquals("Incorrect piece found at " + p.getRow() + ", " + p.getColumn(),
						p, test);
			} else if (p.getRow() == -1 && p.getColumn() == -1) {
				found = false;
				List<Piece> capturedPile = cb.getCapturedPieces();
				for (Piece cp : capturedPile)
					if (p.equals(cp))
						found = true;
				assertTrue("Expected captured piece not found in captured pile", found);
			} else {
				fail("Expected piece had an invalid location");
				return;
			}
		}
	}
	
	/**
	 * Checks that the pieces for a given player (or both players if player == null)
	 * have the expected locations. This allows for verification of the getPieces()
	 * and getPieces(player) methods.
	 * @param cb the chessboard
	 * @param pieces the list of expected pieces for the given player
	 * @param player the player in question.
	 */
	private void checkGetPieces(ChessBoard cb, List<Piece> pieces, Player player) {
		List<Piece> piecesCopy = new ArrayList<Piece>(pieces);
		List<Piece> currentPieceList;
		
		if (player == null)
			currentPieceList = cb.getPieces();
		else
			currentPieceList = cb.getPieces(player);
		
		for (Piece nextPiece : currentPieceList) {
			assertTrue("Board contains an unexpected piece: " + nextPiece.toString(),
					piecesCopy.contains(nextPiece));
			piecesCopy.remove(nextPiece);
		}
		
		if (player == null)
			currentPieceList = cb.getCapturedPieces();
		else
			currentPieceList = cb.getCapturedPieces(player);
		
		for (Piece nextPiece : currentPieceList) {
			assertTrue("Board's captured pile contains an unexpected piece: " + nextPiece.toString(),
					piecesCopy.contains(nextPiece));
			piecesCopy.remove(nextPiece);
		}
		
		assertTrue("Board did not contain all of the given pieces on the board or in the captured pile",
				piecesCopy.size() == 0);
	}
	
	/**
	 * Tests that the MoveHistory from the chessboard matches the expected MoveHistory.
	 * @param cb the chessboard
	 * @param expected the expected movehistory
	 */
	private void checkGetMoveHistory(ChessBoard cb, MoveHistory<ChessMove> expected) {
		if (expected == null)
			assertTrue("Non-null move history found in ChessBoard where null move history expected",
					cb.getMoveHistory() == null);
		else {
			List<ChessMove> expectedHistory = expected.getHistory();
			List<String> expectedTimes = expected.getMoveTimes();
			List<ChessMove> actualHistory = cb.getMoveHistory().getHistory();
			List<String> actualTimes = cb.getMoveHistory().getMoveTimes();
			
			assertEquals("Expected move history does not match actual move history.",
					expectedHistory, actualHistory);
			assertEquals("Expected move times do not match actual move times.",
					expectedTimes, actualTimes);
		}
	}
	
	/**
	 * Helper method to quickly check all of the methods for getting
	 * the pieces on the board or in the capture pile. This assumes
	 * that pieces includes all of the standard chess pieces.
	 * @param cb the chessboard
	 * @param pieces the full list of chess pieces in a standard game
	 * of chess.
	 */
	private void checkPieces(ChessBoard cb, List<Piece> pieces) {
		checkPieceLocations(cb, pieces);
		checkGetPieces(cb, pieces, null);
		checkGetPieces(cb, pieces.subList(0, 16), Player.WHITE);
		checkGetPieces(cb, pieces.subList(16, 32), Player.BLACK);
	}
	
	/**
	 * Verifies that the legality of a given move matches the given expected
	 * value. Tests isMoveLegal.
	 * @param cb the chessboard
	 * @param cm the ChessMove whose legality is in question.
	 * @param legal the expected legality
	 * @return a new list of pieces with the piece locations properly
	 * updated.
	 */
	private void checkMoveLegality(ChessBoard cb, ChessMove cm, boolean legal) {
		if (legal)
			assertTrue("ChessBoard declared a legal move illegal: " + cm,
					cb.isMoveLegal(cm));
		else
			assertFalse("ChessBoard declared an illegal move legal: " + cm,
					cb.isMoveLegal(cm));
	}
	
	/**
	 * Executes doMove with an empty timestamp.
	 */
	private List<Piece> doMove(ChessBoard cb, ChessMove cm, List<Piece> start) {
		return doMove(cb, cm, start, "");
	}
	
	/**
	 * Helper method to execute a ChessMove on this board and update
	 * the list of pieces correctly. ChessMove <code>cm</code> must
	 * be a legal move for this chessboard.
	 * @param cb the chessboard
	 * @param cm the ChessMove to execute
	 * @param start the original list of pieces on the board to update
	 * @param timestamp the timestamp to associate with <code>cm</code>
	 * @return a new list of pieces with the piece locations properly
	 * updated.
	 */
	private List<Piece> doMove(ChessBoard cb, ChessMove cm, List<Piece> start, String timestamp) {
		List<Piece> end = new ArrayList<Piece>(start);
		int pieceToMoveIndex = start.indexOf(cm.getPiece());
		int capturedPieceIndex;
		Piece p;
		
		try {
			if (timestamp.equals(""))
				cb.doMove(cm);
			else
				cb.doMove(cm, timestamp);
		} catch (Exception e) {
			fail("Board disallowed a legal move: " + e.getMessage());
			return end;
		}
		
		end.set(pieceToMoveIndex, end.get(pieceToMoveIndex).moveTo(cm.getRow(), cm.getColumn()));
		p = end.get(pieceToMoveIndex);
		// Take care of pawn promotions
		if (p.getType().equals(PieceTypeFactory.getPawnType()) && 
				((p.getPlayer().equals(Player.WHITE) && p.getRow() == 7) ||
				 (p.getPlayer().equals(Player.BLACK) && p.getRow() == 0)))
			end.set(pieceToMoveIndex, new Piece(p.getPlayer(), PieceTypeFactory.getQueenType(), p.getRow(), p.getColumn()));
		
		if (cm.getCapturedPiece() != null) {
			capturedPieceIndex = start.indexOf(cm.getCapturedPiece());
			end.set(capturedPieceIndex, end.get(capturedPieceIndex).moveTo(-1, -1));
		}
		return end;
	}
	
	/**
	 * Helper method to undo the last ChessMove executed on a given
	 * chessboard and update the list of pieces properly. At least
	 * one move must have been executed on the given chessboard and
	 * recorded in the board's move history.
	 * @param cb the chessboard
	 * @param start the list of pieces with their current locations
	 * on the board.
	 * @return a new list of pieces with the piece locations properly
	 * updated.
	 */
	private List<Piece> undoMove(ChessBoard cb, List<Piece> start) {
		List<Piece> end = new ArrayList<Piece>(start);
		// Grab the last move from the history
		List<ChessMove> history = cb.getMoveHistory().getHistory();
		ChessMove last = history.get(history.size()-1);
		
		int pieceMovedIndex = start.indexOf(last.getPiece().moveTo(last.getRow(), last.getColumn()));
		if (pieceMovedIndex == -1) // Pawn Promotion
			pieceMovedIndex = start.indexOf(new Piece(last.getPiece().getPlayer(), PieceTypeFactory.getQueenType(), last.getRow(), last.getColumn()));
		int capturedPieceIndex;
		
		try {
			cb.undoLastMove();
		} catch (Exception e) {
			fail("Board disallowed undoing move: " + e.getMessage());
			return end;
		}
		
		end.set(pieceMovedIndex, last.getPiece());
		if (last.getCapturedPiece() != null) {
			capturedPieceIndex = start.indexOf(last.getCapturedPiece().moveTo(-1, -1));
			end.set(capturedPieceIndex, last.getCapturedPiece());
		}
		return end;
	}
	
	/**
	 * Verifies that the "in check" status of the given player is the
	 * given expected value.
	 * @param cb the chessboard
	 * @param player the Player in question
	 * @param expected the expected value of isPlayerInCheck(player).
	 */
	private void checkCheckStatus(ChessBoard cb, Player player, boolean expected) {
		if (expected)
			assertTrue("Board failed to recognize that " + player.toString() + " is in check.",
					cb.isPlayerInCheck(player));
		else
			assertFalse("Board mistakenly recognized that " + player.toString() + " is in check.",
					cb.isPlayerInCheck(player));
	}
	
	/**
	 * Verifies that the game-over characteristic of the given
	 * chessboard is as expected. Test isGameOver().
	 * @param cb the chessboard
	 * @param expected
	 */
	private void checkGameOver(ChessBoard cb, boolean expected) {
		if (expected) {
			assertTrue("Board failed to recognize the end of the game",
					cb.isGameOver());
		} else {
			assertFalse("Board mistook to recognize the end of the game",
					cb.isGameOver());
		}
	}
	
	/**
	 * Tests the setting of the end-game conditions to verify that they
	 * have the given expected values. This method tests
	 * getGameOverReason() and getWinner().
	 * @param cb the chessboard
	 * @param expectedGameOverReason the expected GameOverReason
	 * @param expectedWinner the expected winner
	 */
	private void checkEndGameConditions(ChessBoard cb, int expectedGameOverReason, Player expectedWinner) {
		assertEquals("Board has an unexpected GameOverReason",
				expectedGameOverReason, cb.getGameOverReason());
		assertEquals("Board has an unexpected winner",
				expectedWinner, cb.getWinner());
	}
	
	 /**
	  * Helper method to recursively check the self-consistency of
	  * the board, verifying that moves returned by allLegalMoves()
	  * can be made with the expected consequences.
	  * @param cb the chessboard
	  * @param pieces the list holding the arrangement of pieces
	  * on the given chessboard
	  * @param i the level of recursion at which this method is
	  * currently running. Will be less than MAX_RECURSE.
	  */
	private void checkSelfConsistency(ChessBoard cb, List<Piece> pieces, int i) {
		ChessBoard localcb = (ChessBoard)cb.clone();
		List<ChessMove> allLegal = localcb.allLegalMoves();
		
		for (ChessMove cm : allLegal) {
			checkMoveLegality(localcb, cm, true);
			pieces = doMove(localcb, cm, pieces);
			checkPieces(localcb, pieces);
			
			if (i < MAX_RECURSE)
				checkSelfConsistency(localcb, new ArrayList<Piece>(pieces), i+1);
			
			pieces = undoMove(localcb, pieces);
			checkPieces(localcb, pieces);
		}
	}
}