package antichess.test;

import junit.framework.TestCase;
import antichess.*;

/**
 * Automated tests for ChessMoves.
 */
public class ChessMoveTests extends TestCase
{
	private Piece whitePiece, blackPiece;
	
	public ChessMoveTests(String name) {
		super(name);
		
		PieceType pieceType = new NullPieceType();
		whitePiece = new Piece(Player.WHITE, pieceType, 0, 0);
		blackPiece = new Piece(Player.BLACK, pieceType, 7, 5);
	}
	
	/**
	 * Tests that the Constructor can create ChessMoves with the expected values
	 * when given legal inputs, withough unexpected consequences.
	 */
	public void testConstructor() {
		ChessMove cm;
		try {
			// Test for a basic ChessMove
			cm = new ChessMove(whitePiece, 1, 1);
		} catch (Exception e) {
			fail("ChessMove constructor threw Exception rather than accepting valid input.");
			return;
		}
		checkChessMove(cm, whitePiece, 1, 1, null);
		
		try {
			// Test a ChessMove with an explicitly "null" captured piece
			cm = new ChessMove(blackPiece, 6, 3, null);
		} catch (Exception e) {
			fail("ChessMove constructor threw Exception rather than accepting valid input.");
			return;
		}
		checkChessMove(cm, blackPiece, 6, 3, null);
		
		try {
			// Test a ChessMove with a non-null captured piece
			cm = new ChessMove(whitePiece, 6, 3, blackPiece.moveTo(6, 3));
		} catch (Exception e) {
			fail("ChessMove constructor threw Exception rather than accepting valid input.");
			return;
		}
		checkChessMove(cm,whitePiece, 6, 3, blackPiece.moveTo(6, 3));
	}
	
	/**
	 * Tests the equality between ChessMoves, verifying that ChessMoves that
	 * should be equal are equal, unequal ChessMoves are unequal, and
	 * ChessMoves aren't equal to non-ChessMoves.
	 */
	public void testEquals() {
		// Standard ChessMove
		ChessMove cm = new ChessMove(whitePiece, 1, 1);
		assertTrue("ChessMove does not equal itself",
				cm.equals(cm));
		assertTrue("ChessMove does not equal equivalent ChessMove",
				cm.equals(new ChessMove(whitePiece, 1, 1)));
		assertFalse("Valid ChessMove equivalent to non-valid ChessMove",
				cm == null);
		assertFalse("Valid ChessMove equivalent to non-valid ChessMove",
				cm.equals(new Object()));
		
		// Explicitly null captured piece
		ChessMove cm2 = new ChessMove(blackPiece, 6, 3, null);
		assertTrue("ChessMove does not equal itself",
				cm2.equals(cm2));
		assertTrue("ChessMove does not equal equivalent ChessMove",
				cm2.equals(new ChessMove(blackPiece, 6, 3, null)));
		assertFalse("Valid ChessMove equivalent to non-valid ChessMove",
				cm2 == null);
		assertFalse("Valid ChessMove equivalent to non-valid ChessMove",
				cm2.equals(new Object()));
		assertFalse("ChessMove equivalent to non-equivalent ChessMove",
				cm2.equals(cm));
		
		// Non-null captured piece
		ChessMove cm3 = new ChessMove(whitePiece, 6, 3, blackPiece.moveTo(6, 3));
		assertTrue("ChessMove does not equal itself",
				cm3.equals(cm3));
		assertTrue("ChessMove does not equal equivalent ChessMove",
				cm3.equals(new ChessMove(whitePiece, 6, 3, blackPiece.moveTo(6, 3))));
		assertFalse("Valid ChessMove equivalent to non-valid ChessMove",
				cm3 == null);
		assertFalse("Valid ChessMove equivalent to non-valid ChessMove",
				cm3.equals(new Object()));
		assertFalse("ChessMove equivalent to non-equivalent ChessMove",
				cm3.equals(cm));
		assertFalse("ChessMove equivalent to non-equivalent ChessMove",
				cm3.equals(cm2));
	}
	
	/**
	 * Helper method to verify that a given ChessMove has the expected visible
	 * data.
	 * @param cm the ChessMove
	 * @param p the Piece in the ChessMove
	 * @param row the new row of the Piece of the ChessMove
	 * @param col the new column of the Piece of the ChessMove
	 * @param captured the (potentially null) captured piece in this ChessMove
	 */
	private void checkChessMove(ChessMove cm, Piece p, int row, int col, Piece captured) {
		assertEquals("Unexpected Piece in ChessMove",
				p, cm.getPiece());
		assertEquals("Unexpected row in ChessMove",
				row, cm.getRow());
		assertEquals("Unexpected column in ChessMove",
				col, cm.getColumn());
		assertEquals("Unexpected captured piece in ChessMove",
				captured, cm.getCapturedPiece());
	}
}
