// -*- mode: java; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*-

package antichess.test;

import antichess.*;
import java.io.*;
import java.util.*;
import junit.framework.*;
import javax.xml.validation.*;
import javax.xml.transform.stream.StreamSource;
import javax.xml.*;
import org.xml.sax.SAXException;

/**
 * Test the GameWriter class
 **/
public class GameWriterTests extends TestCase
{
    Validator validator;
	public GameWriterTests() {
		this("");
	}

	public GameWriterTests(String s) {
		super(s);
		Schema schema;
		try {
			schema = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI).
				newSchema(GameReader.class.getResource("antichess.xsd"));
		} catch(SAXException e) {
			throw new RuntimeException("Unable to read antichess.xsd: " + e);
		}
		validator = schema.newValidator();
	}

	private void validate(String xml) {
			try {
				validator.validate(new StreamSource(new StringReader(xml)));
			} catch(SAXException e) {
				fail("Validation error: " + e);
			} catch(IOException e) {
				fail("IO Exception reading from a string: " +  e);
			}
	}

	public void testDefaultBoard() {
		ChessBoard board = new ChessBoard();
		board.newGame();
		try {
			GameWriter writer = new GameWriter(board);
			String xml = writer.writeGame();
			validate(xml);
		} catch(Exception e){
			e.printStackTrace();
			fail("Exception: " + e.toString() + "\n");
		}
	}

	public void testTimedGame() {
		GameTimer whiteTimer = new GameTimer(300 * 1000, 500);
		GameTimer blackTimer = new GameTimer(300 * 1000, 500);
		whiteTimer.start();
		try {
			Thread.sleep(500);
		} catch(Exception e) {}
		ChessBoard board = new ChessBoard();
		board.newGame();
		try {
			GameWriter writer = new GameWriter(board, whiteTimer, blackTimer);
			String xml = writer.writeGame();
			validate(xml);
		} catch(Exception e){
			e.printStackTrace();
			fail("Exception: " + e.toString() + "\n");
		}
	}

	private void doMove(AntichessBoard board, String move) {
		ChessMove m = GameReader.parseMove(move, board);
		try {
			board.doMove(m);
		} catch(Exception e) {
			e.printStackTrace();
			fail(move + ": " + e.toString());
		}
	}

	/**
	 * Check that both boards are the ``same'' -- same pieces, same
	 * move history, same player, and same win state
	 */
	private void checkBoards(ChessBoard b1, ChessBoard b2) {
		assertEquals(b1.getPlayer(), b2.getPlayer());
		assertEquals(b1.getWinner(), b2.getWinner());
		assertEquals(b1.getGameOverReason(), b2.getGameOverReason());

		for(int i=0;i<b1.getRows();i++) {
			for(int j=0;j<b2.getColumns();j++) {
				assertEquals("Piece at " + i + "," + j,
							 b1.getPieceAt(i,j),
							 b2.getPieceAt(i,j));
			}
		}

		List<ChessMove> moves1 = b1.getMoveHistory().getHistory();
		List<ChessMove> moves2 = b2.getMoveHistory().getHistory();
		assertEquals("Moves are equal", moves1, moves2);
	}

	/**
	 * Test that we can save and restore a game and get the same
	 * thing.
	 */
	public void testRoundTrip() {
		AntichessBoard board = new AntichessBoard();
		board.newGame();
		// Play the opening of some game
		doMove(board, "e2-e4");
		doMove(board, "e7-e5");
		doMove(board, "g1-f3");
		doMove(board, "d7-d5");
		doMove(board, "e4-d5");
		doMove(board, "d8-d5");
		doMove(board, "f3-e5");
		doMove(board, "d5-d2");
		doMove(board, "e1-d2");
		doMove(board, "c8-h3");

		testRoundTrip(board);
	}

	/**
	 * Save the given board to an XML string, make sure it validates,
	 * and then load it and make sure the result is the same.
	 */
	private void testRoundTrip(ChessBoard board) {
		String xml;
		try {
			GameWriter writer = new GameWriter(board, null, null);
			xml = writer.writeGame();
			validate(xml);
		} catch(Exception e){
			e.printStackTrace();
			fail("Exception: " + e.toString() + "\n");
			return;
		}

		GameReader reader;
		try {
			reader = new GameReader(xml);
		} catch(Exception e) {
			fail("Failure loading initial_game.xml: " + e);
			return;
		}

		ChessBoard load = reader.getBoard();

		checkBoards(board, load);
	}

	/**p
	 * Test round-tripping with a game that's over.
	 */
	public void testGameOver() {
		AntichessBoard board = new AntichessBoard();
		board.newGame();
		doMove(board, "f2-f3");
		doMove(board, "e7-e6");
		doMove(board, "g2-g4");
		doMove(board, "d8-h4");
		testRoundTrip(board);
		
	}

}
