package antichess.test;

import java.util.ArrayList;
import java.util.List;

import junit.framework.TestCase;
import antichess.ChessBoard;
import antichess.ChessMove;
import antichess.Piece;
import antichess.PieceTypeFactory;
import antichess.Player;
import antichess.viewgui.MoveHistoryView;

/**
 * Tests to verify the output of the MoveHistoryView, namely, that
 * moves are being displayed correctly in the Move History section
 * of the GUI.
 */
public class MoveHistoryViewTests extends TestCase {

	public MoveHistoryViewTests(String name) {
		super(name);
	}
	
	/**
	 * Test to verify that the column and row spaces of each of the new moves
	 * submitted to the move history are valid, and that moves are formatted
	 * properly in this Text Box.
	 *
	 */
	// Code blocked off roughly based on moves
	public void testColumnAndRowSpace() {
		ChessBoard cb = new ChessBoard();
		List<Piece> pieces = new ArrayList<Piece>();
		pieces.add(new Piece(Player.WHITE, PieceTypeFactory.getKingType(), 1, 0));
		pieces.add(new Piece(Player.BLACK, PieceTypeFactory.getKingType(), 0, 7));
		pieces.add(new Piece(Player.WHITE, PieceTypeFactory.getPawnType(), 3, 3));
		pieces.add(new Piece(Player.BLACK, PieceTypeFactory.getPawnType(), 3, 4));
		
		cb.loadGame(pieces, Player.WHITE, null);
		
		MoveHistoryView mhv = new MoveHistoryView(cb, 12, 12);
		
		ChessMove cm = new ChessMove(new Piece(Player.WHITE, PieceTypeFactory.getKingType(), 1, 0), 0, 0);
		doMove(cb, cm);
		mhv.postRefresh();
		checkColumnAndRow(mhv.getText());
		
		cm = new ChessMove(new Piece(Player.BLACK, PieceTypeFactory.getKingType(), 0, 7), 0, 6);
		doMove(cb, cm);
		mhv.postRefresh();
		checkColumnAndRow(mhv.getText());
		
		cm = new ChessMove(new Piece(Player.WHITE, PieceTypeFactory.getKingType(), 0, 0), 1, 0);
		doMove(cb, cm);
		mhv.postRefresh();
		checkColumnAndRow(mhv.getText());
		
		cm = new ChessMove(new Piece(Player.BLACK, PieceTypeFactory.getKingType(), 0, 6), 0, 5);
		doMove(cb, cm);
		mhv.postRefresh();
		checkColumnAndRow(mhv.getText());
		
		cm = new ChessMove(new Piece(Player.WHITE, PieceTypeFactory.getKingType(), 1, 0), 2, 0);
		doMove(cb, cm);
		mhv.postRefresh();
		checkColumnAndRow(mhv.getText());

		cm = new ChessMove(new Piece(Player.BLACK, PieceTypeFactory.getKingType(), 0, 5), 0, 4);
		doMove(cb, cm);
		mhv.postRefresh();
		checkColumnAndRow(mhv.getText());
		
		cm = new ChessMove(new Piece(Player.WHITE, PieceTypeFactory.getKingType(), 2, 0), 3, 0);
		doMove(cb, cm);
		mhv.postRefresh();
		checkColumnAndRow(mhv.getText());

		cm = new ChessMove(new Piece(Player.BLACK, PieceTypeFactory.getKingType(), 0, 4), 0, 3);
		doMove(cb, cm);
		mhv.postRefresh();
		checkColumnAndRow(mhv.getText());
		
		cm = new ChessMove(new Piece(Player.WHITE, PieceTypeFactory.getKingType(), 3, 0), 4, 0);
		doMove(cb, cm);
		mhv.postRefresh();
		checkColumnAndRow(mhv.getText());

		cm = new ChessMove(new Piece(Player.BLACK, PieceTypeFactory.getKingType(), 0, 3), 0, 2);
		doMove(cb, cm);
		mhv.postRefresh();
		checkColumnAndRow(mhv.getText());
		
		cm = new ChessMove(new Piece(Player.WHITE, PieceTypeFactory.getKingType(), 4, 0), 5, 0);
		doMove(cb, cm);
		mhv.postRefresh();
		checkColumnAndRow(mhv.getText());

		cm = new ChessMove(new Piece(Player.BLACK, PieceTypeFactory.getKingType(), 0, 2), 0, 1);
		doMove(cb, cm);
		mhv.postRefresh();
		checkColumnAndRow(mhv.getText());
		
		cm = new ChessMove(new Piece(Player.WHITE, PieceTypeFactory.getKingType(), 5, 0), 6, 0);
		doMove(cb, cm);
		mhv.postRefresh();
		checkColumnAndRow(mhv.getText());

		cm = new ChessMove(new Piece(Player.BLACK, PieceTypeFactory.getKingType(), 0, 1), 0, 0);
		doMove(cb, cm);
		mhv.postRefresh();
		checkColumnAndRow(mhv.getText());
		
		cm = new ChessMove(new Piece(Player.WHITE, PieceTypeFactory.getKingType(), 6, 0), 7, 0);
		doMove(cb, cm);
		mhv.postRefresh();
		checkColumnAndRow(mhv.getText());
	}
	
	/**
	 * Private method to execute the given (must be) legal move
	 * on the chessboard
	 * @param cb the chessboard
	 * @param cm the ChessMove to make
	 */
	private void doMove(ChessBoard cb, ChessMove cm) {
		try {
			cb.doMove(cm);
		} catch (Exception e) {
			fail("ChessBoard failed to make the legal move: " + cm.toString());
		}
	}
	
	/**
	 * Private method to verify that the format of the moves
	 * printed in the Move History box are correct.
	 * @param mhvText the text in the Move History JTextArea
	 */
	private void checkColumnAndRow(String mhvText) {
		String[] lines = mhvText.split("\\n");
		String lastLine = lines[lines.length-1];
		String moveText = lastLine.split(" ")[1].trim();
		
		assertTrue("ChessMove display text is not in the expected format",
				moveText.matches("[a-h][1-8]-[a-h][1-8]"));
	}
}
