// -*- mode: java; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*-
package antichess.viewgui;

import java.awt.AlphaComposite;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.geom.Rectangle2D;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;

import antichess.Board;
import antichess.BoardObserver;
import antichess.ChessBoard;
import antichess.ChessMove;
import antichess.HumanPlayer;
import antichess.Piece;
import antichess.Player;
/**
 * ChessBoardView is the graphic representation of a Board for use in the
 * view.  ChessBoardView extends the abstract class BoardView. 
 *
 * The ChessBoardView draws the viewedBoard and allows the user to input 
 * moves when requested.
 * 
 * When setAcceptingInput is called the Board processes the users mouse clicks
 * and reports a click from a square containing a piece for the HumanPlayer specified
 * in the parameters to setAcceptingInput to a destination square.  Determining if this
 * is a valid move is the responsibility of that HumanPlayer.  
 * setNotAcceptingInput cancels out a setAcceptingInput. 
 * 
 * 
 * 
 * @author nlharr
 *
 * @specfield acceptingInput 	//boolean that determines which state we are in
 * @specfield startLocation		//Stores the value of the start location on the board
 * 								//for the user inputed move
 * @specfield endLocation 		//Stores the value of the end location on the board
 * 								//for the user inputed move
 * @specfield boardPlayer		//stores the human player the BoardView is updates information to
 * @specfield viewedBoard		//the ChessBoard that this ChessGUI is displaying
 */


public class ChessBoardView extends BoardView implements MouseListener, BoardObserver{
	
	//just to make 
	static final long serialVersionUID = 1000;
	
	//For determing regions outside of the board
	private static final int NULLCOOR = -32;
	
	//stores whether the ChessBoardView is accepting Input
	private boolean acceptingInput;
	
	private boolean hasFirstSquare;
	
	
	private ImageMap imageMap;
	
	private ChessBoard viewedBoard;
	private HumanPlayer currentPlayer;
	
	private int[] startRow;
	private int[] startColumn;
	private int[] endRow;
	private int[] endColumn;

	
	//variables that help when getting input and determing the
	//painting squares
	private int boardXOffset;
	private int boardYOffset;
	private int boardWidth;
	private int boardHeight;
	private int squareWidth;
	private int squareHeight;
	
	
	//Abstraction Function
	//acceptingInput = acceptingInput
	//startLocation  = (startRow[0], startColumn[0])
	//endLocation    = (endRow[0], endColumn[0])
	//boardPlayer    = currentPlayer
	//viewedBoard    = viewedBoard
	
	//Representation Invariants
	//viewedBoard != null
	//if acceptingInput == true then currentPlayer != null
	//startRow.length = 1
	//endRow.length = 1
	//startColumn.length = 1
	//endColumn.length = 1
	//startRow[0] = NULLCOOR or 0<=startRow[0]<=7
	//endRow[0] = NULLCOOR or 0<=endRow[0]<=7
	//startColumn[0] = NULLCOOR or 0<=startColumn[0]<=7
	//endColumn[0] = NULLCOOR or 0<=endColumn[0]<=7
	//imageMap != null
	//imageMap contains images mapped to by the names:
	// 		Rantiking
	// 		RBishop
	// 		RKing
	// 		RKnight
	// 		RPawn 
	// 		RQueen
	// 		RRook
	// 		WAntiking
	// 		WBishop
	// 		WKing
	// 		WKnight
	// 		WPawn
	// 		WQueen
	// 		WRook
	// 		board
	//boardXOffset >= 0
	//boardYOffset >= 0
	//boardWidth > 0
	//boardHeight > 0
	//squareWidth > 0
	//squareHeight > 0
	
	
	private void testArray(int[] array){
		if (array.length!=1)
			throw new RuntimeException("position array is improper length");
		if (!((array[0]==NULLCOOR) || ((array[0]<=7) && (array[0]>=0)))){
			throw new RuntimeException("position value is out of the allowed range");
		}
	}
	
	private void checkRep(){
		if (viewedBoard == null) throw new RuntimeException("viewedBoard is null");
		if (acceptingInput == true && currentPlayer == null) 
			throw new RuntimeException("currentPlayer is null while accepting input");
		testArray(startRow);
		testArray(endRow);
		testArray(startColumn);
		testArray(endColumn);
		if (imageMap == null)throw new RuntimeException("imageMap is null");
		try{
			imageMap.getImage("RAntiking");
			imageMap.getImage("RBishop");
			imageMap.getImage("RKing");
			imageMap.getImage("RKnight");
			imageMap.getImage("RPawn");
			imageMap.getImage("RQueen");
			imageMap.getImage("RRook");
			imageMap.getImage("WAntiking");
			imageMap.getImage("WBishop");
			imageMap.getImage("WKing");
			imageMap.getImage("WKnight");
			imageMap.getImage("WPawn");
			imageMap.getImage("WQueen");
			imageMap.getImage("WRook");
			imageMap.getImage("board");
		}catch (Exception ex){
			throw new RuntimeException("imageMap does not contain an image it should");
		}
		if (boardXOffset < 0) throw new RuntimeException("boardXOffest is < 0");
		if (boardYOffset < 0) throw new RuntimeException("boardYOffset is < 0");
		if (boardWidth <= 0) throw new RuntimeException("boardWidth is <= 0");
		if (boardHeight <= 0) throw new RuntimeException("boardHeight is <= 0");
		if (squareWidth <= 0) throw new RuntimeException("squareWidth is <= 0");
		if (squareHeight <= 0) throw new RuntimeException("squareHeight is <= 0");
		
	}
	
	
	
	/**
	 * Constructor that creates a new ChessBoardView
	 * 
	 * @requires board != null, imageMap != null
	 *  boardXOffset >= 0, boardYOffset >= 0, boardWidth > 0, boardHeight>0
	 *  imageMap maps these strings to non null values:
	 *		Rantiking
	 * 		RBishop
	 * 		RKing
	 * 		RKnight
	 * 		RPawn 
	 * 		RQueen
	 * 		RRook
	 * 		WAntiking
	 * 		WBishop
	 * 		WKing
	 * 		WKnight
	 * 		WPawn
	 * 		WQueen
	 * 		WRook
	 * 		board
	 */
	public ChessBoardView(ChessBoard board, ImageMap imageMap, 
			int boardXOffset, int boardYOffset, int boardWidth, int boardHeight){
		addMouseListener(this);
		board.addObserver(this);
		this.imageMap = imageMap;
		this.viewedBoard = board;
		startRow = new int[1];
		startColumn = new int[1];
		endRow = new int[1];
		endColumn = new int[1];
		startRow[0]=NULLCOOR;
		startColumn[0]=NULLCOOR;
		endRow[0]=NULLCOOR;
		endColumn[0]=NULLCOOR;
		
		
		acceptingInput = false;
		hasFirstSquare = false;
		
		//stores the board information
		this.boardXOffset = boardXOffset;
		this.boardYOffset = boardYOffset;
		this.boardWidth = boardWidth;
		this.boardHeight = boardHeight;
		
		//calculates the squareWidth and squareHeight from given values
		this.squareWidth = Math.round(boardWidth/(viewedBoard.getColumns()));
		this.squareHeight = Math.round(boardHeight/(viewedBoard.getRows()));
		checkRep();
	}
	
	/**
	 * @modifies viewedBoard
	 * @effects viewedBoard = newViewedBoard
	 */
	public void switchViewedBoard(ChessBoard newViewedBoard){
		viewedBoard.removeObserver(this);
		startRow[0]=NULLCOOR;
		startColumn[0]=NULLCOOR;
		endRow[0]=NULLCOOR;
		endColumn[0]=NULLCOOR;
		acceptingInput = false;
		hasFirstSquare = false;	
		viewedBoard = newViewedBoard;
		viewedBoard.addObserver(this);
		this.postRefresh();
		checkRep();
	}
	
	
	/**
	 * Re-renders the Board taking any updated information
	 * about the board into account. 
	 *
	 */
	public void postRefresh(){
		this.repaint();
	}

	//comparator for pieces so the pieces are returned from bottom the top for drawing
	private class PieceComparer implements Comparator<Piece>{
		//@returns
		//		-1 if a.getColumn()<b.getColumn()
		//		0 if a.getColumn()==b.getColumn()
		//		-1 if a.getColumn()>b.getColumn()
		
		public int compare(Piece a, Piece b){
				if (a.getRow() > b.getRow()){
					return -1;
				}else if (a.getRow()==b.getRow()){
					return 0;
				}else {
					return 1;
				}		
		}
	}
	
	
	//
	 // Draws a square over the specified square in the graphics g
	 // @requires 0<=row<#rows in viewedBoard 
	 // 			 0<=column<#columns in viewedBoard 
	 // @effects Draws a square over the square identified by row and column 
	 // 			of the specified color
	 //
	
	private void paintSquare(int row, int column, Color color, Graphics g){
		Graphics2D g2 = (Graphics2D)g;
		int leftPixel = squareWidth*column+boardXOffset;
		int topPixel = (viewedBoard.getRows()-1-row)*squareWidth+boardYOffset;
		g2.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER));
		g2.setPaint(color);
		g2.fill(new Rectangle2D.Double(leftPixel, topPixel,
                squareWidth,
                squareHeight));
		
	}
	
	//Gets the piece at row, column.  If the piece has available moves, it
	//draws a square over the destinations for all the possible moves. 
	private void paintPossibleMoves(int row, int column, Color color, Graphics g){
		Piece pieceAt = viewedBoard.getPieceAt(row, column);
		List<ChessMove> availableMoves = viewedBoard.legalMoves(pieceAt);
		for(ChessMove tempMove : availableMoves){
			paintSquare(tempMove.getRow(), tempMove.getColumn(), color, g);
		}
		
	}
	
	//Draws all the pieces in their proper locations
	private void paintPieces(Graphics g){
        List<Piece> pieceSet = viewedBoard.getPieces();
              
        //sorts the array
        Piece[] sortedPieceArray =  pieceSet.toArray(new Piece[pieceSet.size()]);
        Arrays.sort(sortedPieceArray, new PieceComparer());
        
        Image currentImage;
        for (Piece currentPiece : sortedPieceArray){
        	
	        //gets the image appropriate for the Player
	        if (currentPiece.getPlayer().equals(Player.WHITE) ){
	        	currentImage = imageMap.getImage(  "W" + currentPiece.getType().getName());
	        }else{
	        	currentImage = imageMap.getImage(  "R" + currentPiece.getType().getName());
	        }
	        //draws the image
	        g.drawImage(currentImage, 
	        		getPiecePixelX(currentPiece.getColumn(), currentImage), 
	        		getRowPixel(currentPiece.getRow(), currentImage), null
	        		);
        }
	}
	
	
	
	
	/**
	 * Draws the viewedBoard onto g. 
	 */
	public void paint(Graphics g) {
		Dimension size = getSize();
		Image boardImage = imageMap.getImage("board");
        g.drawImage(boardImage, 
                    0, 0, size.width, size.height,
                    0, 0, boardImage.getWidth(null),boardImage.getHeight(null),
                    null);  
     
        //if there is a piece selected, draw the square the piece is on blue, and
        //possible moves for the piece green
        if (hasFirstSquare){
        	if (!((startRow[0]== NULLCOOR)||(startColumn[0]==NULLCOOR))){
        		
        		paintSquare(startRow[0],startColumn[0],new Color(0, 0, 255, 150),g);
        		paintPossibleMoves(startRow[0],startColumn[0],new Color(0, 255, 0, 150),g);
        	}
        	
        }
        //Draw the pieces
        paintPieces(g);
        
        //If the viewedBoard says the game is over draw this
        if (viewedBoard.isGameOver()){
        	Font font = new Font("Serif", Font.BOLD, 36);
            g.setFont(font); 
            Color color = new Color(255, 0, 255);
            g.setColor(color);
            if (viewedBoard.getWinner().equals(Player.NONE)){
            	g.drawString("Game Over: Stalemate", boardXOffset, boardYOffset-3);
                
            }else{
            	g.drawString("Game Over: "+viewedBoard.getWinner().toString().substring(0, 1)+
            			viewedBoard.getWinner().toString().toLowerCase().substring(1)+" Wins", boardXOffset, boardYOffset-3);
            }
         }
	}

    
    /**
     * Sets the preferred size of the boardView to be the size of the board image
     */
    public Dimension getPreferredSize() {
    	Image boardImage = imageMap.getImage("board");
        return new Dimension(boardImage.getWidth(null), boardImage.getHeight(null));
    }
	
	
	
	
	/**
	 * @modifies acceptingInput, boardPlayer
	 * @effects acceptingInput = true, boardPlayer = humanPlayer
	 */
	public void setAcceptingInput(HumanPlayer humanPlayer){
		if (!acceptingInput){
			this.currentPlayer = humanPlayer;
			acceptingInput = true;
			System.out.println("Accepting input from: "+humanPlayer.getPlayerType().toString());
		}
	}
	
	/**
	 * @modifies acceptingInput, startLocation, endLocation
	 * @effects acceptingInput = false
	 */
	public void setNotAcceptingInput(){
		acceptingInput = false;
	}
	
	
	
	/**
	 * Needed for inheritance does not do anything
	 */
	public void mouseClicked(MouseEvent e) {
	    }
	/**
	 * Needed for inheritance does not do anything
	 */
	public void mouseReleased(MouseEvent e) {
	    }
	/**
	 * Needed for inheritance does not do anything
	 */
	public void mouseEntered(MouseEvent e) {
	    }
	/**
	 * Needed for inheritance does not do anything
	 */
	public void mouseExited(MouseEvent e) {
	    }

	
	/**
	 * @modifies startLocation, endLocation
	 * @effects
	 * 		If acceptingInput == true{
	 * 			if the location clicked on contains a piece of 
	 * 			else
	 * 				sets endLocation to the square that was clicked on
	 * 				calls postMove with the move generated from startLocation,
	 * 				and endLocation 
	 * 				resets startLocation, resets endLocation
	 * 
	 * 
	 */
	public void mousePressed(MouseEvent e) {
			System.out.println("X: "+e.getX());
			System.out.println("Column: "+xToColumn(e.getX()));
			System.out.println("Y: "+e.getY());
			System.out.println("Row: "+ yToRow(e.getY()));
			
			int row =  yToRow(e.getY());
			int column = xToColumn(e.getX());
			if (!(( row == NULLCOOR) || (column ==NULLCOOR))){
				if (acceptingInput){
					
					//determines if there is not already a piece
					//if the space clicked on has a piece 
					//and if that piece belongs to the current player
					if( (viewedBoard.getPieceAt(row, column)!=null)
						&&(viewedBoard.getPieceAt(row, column).getPlayer().equals(currentPlayer.getPlayerType())))
						{
						System.out.println("Has First Square");
						startRow[0] = yToRow(e.getY());
						startColumn[0] = xToColumn(e.getX());
						hasFirstSquare = true;
						this.postRefresh();
					}else if (startRow[0]!= NULLCOOR &&
							  startColumn[0] != NULLCOOR && hasFirstSquare){
						endRow[0]=yToRow(e.getY());
						endColumn[0] = xToColumn(e.getX());
						if( currentPlayer.setMove(startRow, startColumn, 
								endRow, endColumn)){
							this.setNotAcceptingInput();
							currentPlayer.notifyHasMove();
						}
						hasFirstSquare=false;
						this.postRefresh();
					}
				}
			}
			
			
	
	}
	
	//returns the column a pixel with an x value would be in
	//returns NULLCOOR if none
	private int xToColumn(int x){
		if ((x<boardXOffset)||(x>boardXOffset+boardWidth)){
			return NULLCOOR;
		}
		
		return (int) Math.floor((x-1-boardXOffset)/squareWidth);
		
	}
	//returns the row a pixel with an y value would be in
	//returns NULLCOOR if none
	private int yToRow(int y){
		if ((y<boardYOffset)||(y>boardYOffset+boardHeight)){
			return NULLCOOR;
		}
		
		return  viewedBoard.getRows()-1 - (int)Math.floor((y-1-boardYOffset)/squareHeight);
		
	}
	

	//0 indexed
	//returns the x pixel that the piece needs to be rendered in
	private int getPiecePixelX(int column, Image pieceImage){
		//gets the left most pixel for the column
		int leftPixel = squareWidth*column+boardXOffset;
		//adds the appropriate offset
		return leftPixel+squareWidth/2-pieceImage.getWidth(null)/2;
	}
	
	//returns the y pixel that the piece needs to be rendered in
	private int getRowPixel(int row, Image pieceImage){
		//gets the bottom pixel of the square that contains this piece
		int topPixel = (viewedBoard.getRows()-row)*squareWidth+boardYOffset;
		//adds the appropriate offset
		return topPixel-pieceImage.getHeight(null)-squareHeight/10;
	}
	
	
}