/*
 * tkZephyr.c --
 *
 *      Provides zephyr library calls for tk programs.  Maybe this'll
 * be the next fad?
 *
 * by Chris Shabsin <shabby@mit.edu>
 *
 * begun 11/19/93
 *
 */

#include <stdlib.h>
#include <string.h>
#include "tkz.h"

static int tkzInit = 0;
extern Tcl_Interp *interp;

int 
Zep_Init (Tcl_Interp * interp)
{
  Tcl_CreateCommand (interp, "zinit", tkzZInitialize, (ClientData) NULL,
		     (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand (interp, "zopen", tkzZOpenPort, (ClientData) NULL,
		     (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand (interp, "zclose", tkzZClosePort, (ClientData) NULL,
		     (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand (interp, "zsub", tkzZSubscribeCmd, (ClientData) NULL,
		     (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand (interp, "zunsub", tkzZUnsubscribeCmd, (ClientData) NULL,
		     (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand (interp, "znotice", tkzZNotice_tCmd, (ClientData) NULL,
		     (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand (interp, "zpending", tkzZPendingCmd, (ClientData) NULL,
		     (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand (interp, "zreceive", tkzZReceiveCmd, (ClientData) NULL,
		     (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand (interp, "zset_handler", tkzZSetHandlerCmd,
		     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
  return TCL_OK;
}

int 
tkzZInitialize (ClientData clientData, Tcl_Interp * interp, int argc,
		char *argv[])
{
  int zi;

  if (tkzInit)
    {
      Tcl_SetResult (interp, "ZInitialize: attempted second init", TCL_STATIC);
      return TCL_ERROR;
    }

  if (argc != 1)
    {
      Tcl_SetResult (interp, "ZInitialize: takes no args", TCL_STATIC);
      return TCL_ERROR;
    }
  if ((zi = ZInitialize ()) == ZERR_NONE)
    {
      tkzInit = 1;
      Tcl_SetResult (interp, "", TCL_STATIC);
      return TCL_OK;
    }
  else
    {
      Tcl_SetResult (interp, "ZInitialize: ", TCL_STATIC);
      Tcl_AppendResult (interp, error_message (zi), (char *) NULL);
      return TCL_ERROR;
    }
}

int 
tkzZOpenPort (ClientData clientData, Tcl_Interp * interp, int argc,
	      char *argv[])
{
  unsigned short p = 0;
  int zi;
  char res[10];

  if (!tkzInit)
    {
      Tcl_SetResult (interp, "ZOpenPort: not ZInitialized", TCL_STATIC);
      return TCL_ERROR;
    }

  if (argc != 1)
    {
      Tcl_SetResult (interp, "ZOpenPort: takes no args", TCL_STATIC);
      return TCL_ERROR;
    }
  if ((zi = ZOpenPort (&p)) == ZERR_NONE)
    {
      sprintf (res, "%u", (unsigned int) p);
      Tcl_SetResult (interp, res, TCL_VOLATILE);
      return TCL_OK;
    }
  else
    {
      Tcl_SetResult (interp, "ZOpenPort: ", TCL_STATIC);
      Tcl_AppendResult (interp, error_message (zi), (char *) NULL);
      return TCL_ERROR;
    }
}

int 
tkzZClosePort (ClientData clientData, Tcl_Interp * interp, int argc,
	       char *argv[])
{
  int zi;

  if (!tkzInit)
    {
      Tcl_SetResult (interp, "ZClosePort: not ZInitialized", TCL_STATIC);
      return TCL_ERROR;
    }

  if (argc != 1)
    {
      Tcl_SetResult (interp, "ZClosePort: takes no args", TCL_STATIC);
      return TCL_ERROR;
    }
  if ((zi = ZClosePort ()) == ZERR_NONE)
    {
      Tcl_SetResult (interp, "", TCL_STATIC);
      return TCL_OK;
    }
  else
    {
      Tcl_SetResult (interp, "ZClosePort: ", TCL_STATIC);
      Tcl_AppendResult (interp, error_message (zi), (char *) NULL);
      return TCL_ERROR;
    }
}

int 
tkzZNotice_tCmd (ClientData clientData, Tcl_Interp * interp, int argc,
		 char *argv[])
{
  tkzZNotice_t *notice;

  if (argc != 2)
    {
      Tcl_SetResult (interp, "ZNotice_t: takes one arg", TCL_STATIC);
      return TCL_ERROR;
    }

  notice = (tkzZNotice_t *) calloc (1, sizeof (tkzZNotice_t));

  Tcl_CreateCommand (interp, argv[1], tkzZNoticeObjCmd, (ClientData) notice,
		     (Tcl_CmdDeleteProc *) tkzZNoticeObjFree);

  return TCL_OK;
}

#define ZN ((ZNotice_t *)clientData)
#define TZN ((tkzZNotice_t *)clientData)

int 
tkzZNoticeObjCmd (ClientData clientData, Tcl_Interp * interp, int argc,
		  char *argv[])
{
  if (argc < 2)
    {
      Tcl_SetResult (interp, "ZNotice_t: set or get", TCL_STATIC);
      return TCL_ERROR;
    }

  if (!strcasecmp (argv[1], "set"))
    {
      if (argc != 4)
	{
	  Tcl_SetResult (interp, "ZNotice_t: set requires one argument", TCL_STATIC);
	  return TCL_ERROR;
	}
      if (!strcasecmp (argv[2], "kind"))
	{
	  if (!strcasecmp (argv[3], "unsafe"))
	    ZN->z_kind = UNSAFE;
	  else if (!strcasecmp (argv[3], "unacked"))
	    ZN->z_kind = UNACKED;
	  else if (!strcasecmp (argv[3], "acked"))
	    ZN->z_kind = ACKED;
	  else if (!strcasecmp (argv[3], "hmack"))
	    ZN->z_kind = HMACK;
	  else if (!strcasecmp (argv[3], "hmctl"))
	    ZN->z_kind = HMCTL;
	  else if (!strcasecmp (argv[3], "servack"))
	    ZN->z_kind = SERVACK;
	  else if (!strcasecmp (argv[3], "servnak"))
	    ZN->z_kind = SERVNAK;
	  else if (!strcasecmp (argv[3], "clientack"))
	    ZN->z_kind = CLIENTACK;
	  else if (!strcasecmp (argv[3], "stat"))
	    ZN->z_kind = CLIENTACK;
	  else
	    {
	      Tcl_SetResult (interp, "ZNotice_t: invalid kind", TCL_STATIC);
	      return TCL_ERROR;
	    }
	}
      else if (!strcasecmp (argv[2], "port"))
	{
	  ZN->z_port = atoi (argv[3]);
	}
      else if (!strcasecmp (argv[2], "class"))
	{
	  if (ZN->z_class)
	    free (ZN->z_class);

	  ZN->z_class = (char *) malloc (strlen (argv[3]) + 1);
	  strcpy (ZN->z_class, argv[3]);
	}
      else if (!strcasecmp (argv[2], "class_inst"))
	{
	  if (ZN->z_class_inst)
	    free (ZN->z_class_inst);

	  ZN->z_class_inst = (char *) malloc (strlen (argv[3]) + 1);
	  strcpy (ZN->z_class_inst, argv[3]);
	}
      else if (!strcasecmp (argv[2], "opcode"))
	{
	  if (ZN->z_opcode)
	    free (ZN->z_opcode);

	  ZN->z_opcode = (char *) malloc (strlen (argv[3]) + 1);
	  strcpy (ZN->z_opcode, argv[3]);
	}
      else if (!strcasecmp (argv[2], "sender"))
	{
	  if (ZN->z_sender)
	    free (ZN->z_sender);

	  ZN->z_sender = (char *) malloc (strlen (argv[3]) + 1);
	  strcpy (ZN->z_sender, argv[3]);
	}
      else if (!strcasecmp (argv[2], "recipient"))
	{
	  if (ZN->z_recipient)
	    free (ZN->z_recipient);

	  ZN->z_recipient = (char *) malloc (strlen (argv[3]) + 1);
	  strcpy (ZN->z_recipient, argv[3]);
	}
      else if (!strcasecmp (argv[2], "default_format"))
	{
	  if (ZN->z_default_format)
	    free (ZN->z_default_format);

	  ZN->z_default_format = (char *) malloc (strlen (argv[3]) + 1);
	  strcpy (ZN->z_default_format, argv[3]);
	}
      else if (!strcasecmp (argv[2], "num_fields"))
	{
	  int of = TZN->nfields;
	  char **oldfields = TZN->fields;
	  int nf = atoi (argv[3]);
	  char **newfields = (char **) calloc (nf, sizeof (char *));
	  int i;

	  for (i = 0; i < (of < nf) ? of : nf; i++)
	    {
	      if (oldfields[i])
		{
		  newfields[i] = (char *) malloc (strlen (oldfields[i]) + 1);
		  strcpy (newfields[i], oldfields[i]);
		  free (oldfields[i]);
		}
	    }
	  if (oldfields)
	    free (oldfields);
	  TZN->fields = newfields;
	  TZN->nfields = atoi (argv[3]);
	}
      else if (!strncasecmp (argv[2], "field", 5))
	{
	  int fn = atoi (argv[2] + 5);
	  char **field;

	  if (fn >= TZN->nfields)
	    {
	      Tcl_SetResult (interp, "ZNotice_t: attempt to set field out of bounds",
			     TCL_STATIC);
	      return TCL_ERROR;
	    }

	  field = TZN->fields + fn;

	  if (*field)
	    free (*field);

	  *field = (char *) malloc (strlen (argv[3]) + 1);
	  strcpy (*field, argv[3]);
	}
      else
	{
	  Tcl_SetResult (interp, "ZNotice_t: invalid set", TCL_STATIC);
	  return TCL_ERROR;
	}
      Tcl_SetResult (interp, "", TCL_STATIC);
      return TCL_OK;
    }
  else if (!strcasecmp (argv[1], "get"))
    {
      if (argc != 3)
	{
	  Tcl_SetResult (interp, "ZNotice_t: get requires one argument", 
			 TCL_STATIC);
	  return TCL_ERROR;
	}
      if (!strcasecmp (argv[2], "kind"))
	{
	  switch (ZN->z_kind)
	    {
	    case UNSAFE:
	      Tcl_SetResult (interp, "unsafe", TCL_STATIC);
	      break;
	    case UNACKED:
	      Tcl_SetResult (interp, "unacked", TCL_STATIC);
	      break;
	    case ACKED:
	      Tcl_SetResult (interp, "acked", TCL_STATIC);
	      break;
	    case HMACK:
	      Tcl_SetResult (interp, "hmack", TCL_STATIC);
	      break;
	    case HMCTL:
	      Tcl_SetResult (interp, "hmctl", TCL_STATIC);
	      break;
	    case SERVACK:
	      Tcl_SetResult (interp, "servack", TCL_STATIC);
	      break;
	    case SERVNAK:
	      Tcl_SetResult (interp, "servnak", TCL_STATIC);
	      break;
	    case CLIENTACK:
	      Tcl_SetResult (interp, "clientack", TCL_STATIC);
	      break;
	    case STAT:
	      Tcl_SetResult (interp, "stat", TCL_STATIC);
	      break;
	    default:
	      Tcl_SetResult (interp, "ZNotice_t: kind hasn't been set", TCL_STATIC);
	      return TCL_ERROR;
	    }
	}
      else if (!strcasecmp (argv[2], "port"))
	{
	  char pt[10];
	  sprintf (pt, "%u", (unsigned int) ZN->z_port);
	  Tcl_SetResult (interp, pt, TCL_VOLATILE);
	}
      else if (!strcasecmp (argv[2], "class"))
	{
	  Tcl_SetResult (interp, ZN->z_class, TCL_VOLATILE);
	}
      else if (!strcasecmp (argv[2], "class_inst"))
	{
	  Tcl_SetResult (interp, ZN->z_class_inst, TCL_VOLATILE);
	}
      else if (!strcasecmp (argv[2], "opcode"))
	{
	  Tcl_SetResult (interp, ZN->z_opcode, TCL_VOLATILE);
	}
      else if (!strcasecmp (argv[2], "sender"))
	{
	  Tcl_SetResult (interp, ZN->z_sender, TCL_VOLATILE);
	}
      else if (!strcasecmp (argv[2], "recipient"))
	{
	  Tcl_SetResult (interp, ZN->z_recipient, TCL_VOLATILE);
	}
      else if (!strcasecmp (argv[2], "default_format"))
	{
	  Tcl_SetResult (interp, ZN->z_default_format, TCL_VOLATILE);
	}
      else if (!strcasecmp (argv[2], "num_fields"))
	{
	  char pt[10];
	  sprintf (pt, "%u", (unsigned int) TZN->nfields);
	  Tcl_SetResult (interp, pt, TCL_VOLATILE);
	}
      else if (!strcasecmp (argv[2], "time"))
	{
	  Tcl_SetResult (interp, ctime (&(ZN->z_time.tv_sec)), TCL_STATIC);
	}
      else if (!strncasecmp (argv[2], "field", 5))
	{
	  int fn = atoi (argv[2] + 5);
	  char **field;

	  if (fn >= TZN->nfields)
	    {
	      Tcl_SetResult (interp, 
			     "ZNotice_t: attempt to get field out of bounds",
			     TCL_STATIC);
	      return TCL_ERROR;
	    }

	  field = TZN->fields + fn;

	  if (*field)
	    Tcl_SetResult (interp, *field, TCL_VOLATILE);
	  else
	    Tcl_SetResult (interp, "", TCL_STATIC);
	}
      else
	{
	  Tcl_SetResult (interp, "ZNotice_t: invalid get", TCL_STATIC);
	  return TCL_ERROR;
	}
      return TCL_OK;
    }
  return TCL_ERROR;
}

void 
tkzZNoticeObjFree (char *clientData)
{
  ZNotice_t *n = (ZNotice_t *) clientData;
  tkzZNotice_t *tn = (tkzZNotice_t *) clientData;
  int i;

  if (n->z_class)
    free (n->z_class);
  if (n->z_class_inst)
    free (n->z_class_inst);
  if (n->z_opcode)
    free (n->z_opcode);
  if (n->z_sender)
    free (n->z_sender);
  if (n->z_recipient)
    free (n->z_recipient);
  if (n->z_default_format)
    free (n->z_default_format);

  if (tn->nfields)
    {
      for (i = 0; i < tn->nfields; i++)
	{
	  if (tn->fields[i])
	    free (tn->fields[i]);
	}
      free (tn->fields);
    }
  free (clientData);
}


int
tkzZSubscribeCmd(ClientData clientData, Tcl_Interp *interp,
		 int argc, char *argv[])
{
  static ZSubscription_t zsubt;
  int code;

  if (argc != 4)
    {
      Tcl_SetResult(interp,
		    "ZSubscribe: needs 3 args, class instance recipient",
		    TCL_STATIC);
      return TCL_ERROR;
    }
  zsubt.zsub_class = argv[1];
  zsubt.zsub_classinst = argv[2];
  zsubt.zsub_recipient = argv[3];
  code = ZSubscribeTo(&zsubt, 1, 0);
  if (code)
    {
      Tcl_SetResult (interp, "ZSubscribe: ", TCL_STATIC);
      Tcl_AppendResult (interp, error_message (code));
      return TCL_ERROR;
    }
  return TCL_OK;
}

int
tkzZUnsubscribeCmd(ClientData clientData, Tcl_Interp *interp,
		   int argc, char *argv[])
{
  static ZSubscription_t zsubt;
  int code;
  
  if (argc != 4)
    {
      Tcl_AppendResult (interp,
			"ZUnsubscribe: needs 3 args, class instance recipient",
			TCL_STATIC);
      return TCL_ERROR;
    }
  zsubt.zsub_class = argv[1];
  zsubt.zsub_classinst = argv[2];
  zsubt.zsub_recipient = argv[3];
  code = ZUnsubscribeTo(&zsubt, 1, 0);
  if (code)
    {
      Tcl_SetResult (interp, "ZUnsubscribe: ", TCL_STATIC);
      Tcl_AppendResult (interp, error_message (code));
      return TCL_ERROR;
    }
  return TCL_OK;
}


int
tkzZPendingCmd(ClientData clientData, Tcl_Interp *interp,
	       int argc, char *argv[])
{
  int code;

  if (argc != 1)
    {
      Tcl_SetResult (interp, "zpending: takes one arg", TCL_STATIC);
      return TCL_ERROR;
    }

  code = ZPending ();
  if (code == -1)		/* error condition */
    {
      Tcl_SetResult (interp, "zpending: ", TCL_STATIC);
      Tcl_AppendResult (interp, error_message(errno));
      return TCL_ERROR;
    }
  else
    {
      sprintf(interp->result, "%d", code);
      return TCL_OK;
    }
}


int
tkzZReceiveCmd(ClientData clientData, Tcl_Interp * interp,
	       int argc, char *argv[])
{
  tkzZNotice_t *notice;
  static char buf[64];
  static int n;
  int code;

  if (argc != 1)
    {
      Tcl_SetResult (interp, "zreceive: takes no args", TCL_STATIC);
      return TCL_ERROR;
    }
  
  notice = (tkzZNotice_t *) calloc (1, sizeof (tkzZNotice_t));
  if (notice == NULL)
    {
      Tcl_SetResult (interp, "zreceive: malloc failed", TCL_STATIC);
      return TCL_ERROR;
    }

  code = ZReceiveNotice (&(notice->notice), &(notice->from));
  if (code)
    {
      Tcl_SetResult (interp, "zreceive: ", TCL_STATIC);
      Tcl_AppendResult (interp, error_message(code));
      return TCL_ERROR;
    }
  else
    {
      register int i;
      register int c = 0;
      char *msg = notice->notice.z_message;
      int msglen = notice->notice.z_message_len;
      int fieldsize = 20;

      if (msglen != 0)
	{
	  notice->fields = (char **) malloc (sizeof (char *) * fieldsize);
	  notice->fields[c++] = msg;
	  for (i = 0; i < msglen; i++)
	    {
	      if (msg[i] == '\0')
		{
		  if (c >= fieldsize) /* need more space */
		    {
		      fieldsize *= 2;
		      notice->fields
			= (char **) realloc ((void *) notice->fields,
					     fieldsize * sizeof (char *));
		    }
		  notice->fields[c++] = &(msg[i+1]);
		}
	    }
	  /* Supposedly, z_message_len is supposed to include the final
	     NULL, but it doesn't look like it does. */
	  msg[msglen] = '\0';
	}
      notice->nfields = c;
      sprintf(buf, "notice<%d>", n++);
      Tcl_CreateCommand (interp, buf, tkzZNoticeObjCmd, (ClientData) notice,
			 (Tcl_CmdDeleteProc *) tkzZNoticeObjFree);
      Tcl_SetResult (interp, buf, TCL_STATIC);
      return TCL_OK;
    }
}

struct _command {
  char str[128];
  Tcl_Interp *interp;
};

void
tkzFileProc(ClientData clientData, int mask)
{
  int code = ZPending ();
  struct _command *com = (struct _command *) clientData;

  if (mask & TK_EXCEPTION || code == -1)
    {
      perror ("tkzFileProc");
      exit (1);
    }

  for ( ; code > 0; code--)
    {
      Tcl_Eval (com->interp, com->str);
      Tcl_Eval (com->interp, "update");
    }
}


int
tkzZSetHandlerCmd(ClientData clientData, Tcl_Interp * interp,
		  int argc, char *argv[])
{
  int fd = ZGetFD ();
  struct _command *com;

  if (argc != 2)
    {
      Tcl_SetResult (interp, "zset_handler: takes one argument", TCL_STATIC);
      return TCL_ERROR;
    }
  if (fd == -1)
    {
      Tcl_SetResult (interp, "zset_handler: no open connection", TCL_STATIC);
      return TCL_ERROR;
    }
  com = (struct _command *) malloc (sizeof (struct _command));
  com->interp = interp;
  strcpy(com->str, argv[1]);
  Tk_CreateFileHandler (fd, TK_READABLE | TK_EXCEPTION, tkzFileProc, com);
  return TCL_OK;
}
