/*****************************************************************************
 *
 * Microchip DeviceNet Stack (Router Object Source)
 *
 *****************************************************************************
 * FileName:        route.c
 * Dependencies:    
 * Processor:       PIC18F with CAN
 * Compiler:       	C18 02.20.00 or higher
 * Linker:          MPLINK 03.40.00 or higher
 * Company:         Microchip Technology Incorporated
 *
 * Software License Agreement
 *
 * The software supplied herewith by Microchip Technology Incorporated
 * (the "Company") is intended and supplied to you, the Company's
 * customer, for use solely and exclusively with products manufactured
 * by the Company. 
 *
 * The software is owned by the Company and/or its supplier, and is 
 * protected under applicable copyright laws. All rights are reserved. 
 * Any use in violation of the foregoing restrictions may subject the 
 * user to criminal sanctions under applicable laws, as well as to 
 * civil liability for the breach of the terms and conditions of this 
 * license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN "AS IS" CONDITION. NO WARRANTIES, 
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED 
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT, 
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR 
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 *
 * This file contains the Router object described in section 6-3 of 
 * volume 2 of the DeviceNet specification.
 *
 * The router provides several parameters to Explicit Message Handlers.
 * All of the parameters are globals defined in an access section for 
 * fast access. 
 *	
 * Receive Buffer									Transmit Buffer
 * ---------------									---------------	
 * |   Header    | 									|   Header    |
 * |   Service   |									|   Service   |
 * |   ClassID   |									|    .....    | <- pOutBuf
 * | InstanceID  |									|    .....    |
 * | AttributeID | <- pInBuf, Ptr to rx data		|    .....    |
 * |    .....    |    aInBufLen, len of buf			|    .....    |
 * |    .....    |	  aInBufDataLen, len of data	|    .....    |
 * |    .....    |									|    .....    |
 * |    .....    | 									|    .....    |
 * |    .....    |									|    .....    |
 * ---------------									---------------		
 *
 * Author               Date        Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Ross Fosler			04/03/03	...	
 * 
 *****************************************************************************/


#include	"dnet.def"				// Global definitions file
#include 	"typedefs.h"

#include	"route.h"				// Internal prototypes

#include	"services.h"			// Service codes
#include	"errors.h"				// Error codes
#include	"class.h"				// Class codes


#include	"ident.h"
#include	"dnet.h"
#include 	"conn.h"



#if CLASS_USER_DEFINED_1 > 0
unsigned char CLASS_USER_DEFINED_1_NAME (void);
#endif

#if CLASS_USER_DEFINED_2 > 0
unsigned char CLASS_USER_DEFINED_2_NAME (void);
#endif

#if CLASS_USER_DEFINED_3 > 0
unsigned char CLASS_USER_DEFINED_3_NAME (void);
#endif

#if CLASS_USER_DEFINED_4 > 0
unsigned char CLASS_USER_DEFINED_4_NAME (void);
#endif

#if CLASS_USER_DEFINED_5 > 0
unsigned char CLASS_USER_DEFINED_5_NAME (void);
#endif

#if CLASS_USER_DEFINED_6 > 0
unsigned char CLASS_USER_DEFINED_6_NAME (void);
#endif

#if CLASS_USER_DEFINED_7 > 0
unsigned char CLASS_USER_DEFINED_7_NAME (void);
#endif

#if CLASS_USER_DEFINED_8 > 0
unsigned char CLASS_USER_DEFINED_8_NAME (void);
#endif




/*********************************************************************
 * The following section of global variables are provided specifically
 * to enhance performance in the router and explicit message handling.
 * They are public to all message handlers.
 ********************************************************************/
#if USE_ACCESS == TRUE
#pragma	udata access	A_ROUTE_REGISTERS
#endif

NEAR ROUTE route;



/*********************************************************************
 * Set the default data section type.
 ********************************************************************/
#pragma	udata




/*********************************************************************
 * Function:        unsigned char _RouteExplEvent(void)
 *
 * PreCondition:    
 *
 * Input:           
 *					
 *                  
 * Output:         
 *				
 *				
 *			
 *
 * Side Effects:    
 *
 * Overview:        Handler for explicit messaging 
 *
 * Note:            None
 ********************************************************************/
unsigned char _RouteExplMsgHandler(void)
{
	switch (mRouteGetInstanceID())
	{
		case 0:
			switch(mRouteGetServiceID())
			{
				default:
					mRoutePutError(ERR_SERVICE_NOT_SUPPORTED);
					break;
			}
			break;

		case 1:
			switch(mRouteGetServiceID())
			{
				default:
					mRoutePutError(ERR_SERVICE_NOT_SUPPORTED);
					break;
			}
			break;

		default:
			mRoutePutError(ERR_OBJECT_DOES_NOT_EXIST);
			break;
	}

 	return (1);
}


						 
/*********************************************************************
 * Function:        unsigned char RouteMessage(void)
 *
 * PreCondition:    
 *
 * Input:   		aHeader, aService, aClassID, aInstanceID         
 *					aAttributeID, pInBuf, pOutBuf, aInBufLen
 *					aInBufDataLen, aOutBufLen, aOutBufDataLen 
 *                  
 * Output:       	aHeader, aService, pInBuf, pOutBuf, aOutBufDataLen	
 *
 * Side Effects:    
 *
 * Overview:        This is the start of message routing. This function
 * 					decodes the class then calls the appropriate
 *					object to decode the instance, service, and if 
 *					necessary the attribute. If the class is not 
 *					defined in this object then an error is returned 
 *					in aService (= _ERROR_RESPONSE).
 *
 *					pOutBuf is loaded with the responce data and the
 *					length aOutBufDataLen is adjusted.
 *
 * Note:            Edit the case statement below to add more defined
 *					classes of objects to the list.
 ********************************************************************/
unsigned char RouteMessage(void)
{
	switch (mRouteGetClassID())
	 {
	 	case CLASS_IDENTITY:
	 		return (_IdentityExplMsgHandler());
	 		
	 	case CLASS_MESSAGE_ROUTER:
	 		return (_RouteExplMsgHandler());
	 		
	 	case CLASS_DEVICENET:
	 		return (_DNetExplMsgHandler());
	 			 			
	 	case CLASS_CONNECTION:
	 		return (_ConnExplMsgHandler());
	

		#if CLASS_USER_DEFINED_1 > 0
		case CLASS_USER_DEFINED_1:
			return (CLASS_USER_DEFINED_1_NAME ());
			break;
		#endif

		#if CLASS_USER_DEFINED_2 > 0
		case CLASS_USER_DEFINED_2:
			return (CLASS_USER_DEFINED_2_NAME ());
			break;
		#endif

		#if CLASS_USER_DEFINED_3 > 0
		case CLASS_USER_DEFINED_3:
			return (CLASS_USER_DEFINED_3_NAME ());
			break;
		#endif

		#if CLASS_USER_DEFINED_4 > 0
		case CLASS_USER_DEFINED_4:
			return (CLASS_USER_DEFINED_4_NAME ());
			break;
		#endif

		#if CLASS_USER_DEFINED_5 > 0
		case CLASS_USER_DEFINED_5:
			return (CLASS_USER_DEFINED_5_NAME ());
			break;
		#endif
		
		#if CLASS_USER_DEFINED_6 > 0
		case CLASS_USER_DEFINED_6:
			return (CLASS_USER_DEFINED_6_NAME ());
			break;
		#endif
		
		#if CLASS_USER_DEFINED_7 > 0
		case CLASS_USER_DEFINED_7:
			return (CLASS_USER_DEFINED_7_NAME ());
			break;
		#endif
		
		#if CLASS_USER_DEFINED_8 > 0
		case CLASS_USER_DEFINED_8:
			return (CLASS_USER_DEFINED_8_NAME ());
			break;
		#endif

	 	
	 	// Any other classes addressed that are not supported
	 	default:
	 		mRoutePutError(ERR_OBJECT_DOES_NOT_EXIST);
	 		break;
	 }

	return (1);
}



/*********************************************************************
 * Function:        USINT RouteGetByte(void)
 *
 * PreCondition:    
 *
 * Input:           None.
 *       
 * Output:       	USINT - byte from the buffer.		
 *
 * Side Effects:    
 *
 * Overview:      	This function returns a byte from the router's 
 *					input buffer. Plus it auto adjusts to point the 
 *					the next byte in the buffer.   
 *
 * Note:            
 ********************************************************************/
USINT RouteGetByte(void)
{
	unsigned char outData;

	outData = 0;

	if (route.inBufDataLen > 0)
	{
		outData = *route.pInBuf;
		route.inBufDataLen--;
		route.pInBuf++;
	}

	return (outData);
}




/*********************************************************************
 * Function:        void RoutePutByte(USINT dataByte)
 *
 * PreCondition:    
 *
 * Input:           USINT - a byte to put into the output buffer
 *       
 * Output:       		
 *
 * Side Effects:    
 *
 * Overview:      	This function puts a byte in the output buffer.
 *					Plus it automatically adjusts the buffer count and
 *					pointer into the buffer. 
 *
 * Note:          	 
 ********************************************************************/
void RoutePutByte(USINT dataByte)
{
	if (route.outBufDataLen <= route.outBufLen)
	{
		*route.pOutBuf = dataByte;
		route.pOutBuf++;
		route.outBufDataLen++;
	}
}



/*********************************************************************
 * Function:        unsigned char RouteTestValidInputDataLen(unsigned char expectedSize)
 *
 * PreCondition:    
 *
 * Input:           
 *       
 * Output:       		
 *
 * Side Effects:    
 *
 * Overview:      	This function tests the input buffer against 
 *					the expected size and automatically generates
 *					the necessary error packet if failed. 
 *
 * Note:          	 
 ********************************************************************/
unsigned char RouteTestValidInputDataLen(unsigned char expectedSize)
{
	if (mRouteGetInBufferDataLength() == expectedSize)
	{
		return(1);
	}
	else if (mRouteGetInBufferDataLength() < expectedSize)
	{
		mRoutePutError(ERR_NOT_ENOUGH_DATA);
	}
	else
	{
		mRoutePutError(ERR_TOO_MUCH_DATA);
	}
	
	return (0);
}




/*********************************************************************
 * Function:        void RoutePutError(unsigned char errorCode)
 *
 * PreCondition:    
 *
 * Input:           
 *       
 * Output:       		
 *
 * Side Effects:    
 *
 * Overview:      	This function writes the requested error to the 
 *					buffer. 
 *
 * Note:          	 
 ********************************************************************/
void RoutePutError(unsigned char errorCode)
{
	mRoutePutServiceID(SRVS_ERROR_RESPONSE);
	mRoutePutByte(errorCode);
}


