// -*- mode: java; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*-

package antichess.movegen;

import antichess.*;
import java.util.*;

/**
 * A LinearMoveGenerator represents movement along a line an unbounded
 * distance, stopping at other pieces or the edge of the board.
 *
 * @specfield diagonal : boolean    // Whether diagonal movement is allowed
 * @specfield rowColumn : boolean   // Whether movement along a single row or column is allowed
 **/
public class LinearMoveGenerator extends ChessMoveGenerator
{
    boolean rowColumn;
    boolean diagonal;

    /**
     * @effects Create a new LinearMoveGenerator with the specified
     * parameters
     **/
    public LinearMoveGenerator(boolean diag, boolean rc) {
        rowColumn = rc;
        diagonal = diag;
    }

    public List<ChessMove> getMoves(Piece piece, Board board) {
        List<ChessMove> moves = new ArrayList<ChessMove>();
        int dr, dc;
        int row, col;
        Piece otherPiece;
        // Iterate over every direction
        for(dr=-1;dr<=1;dr++) {
            for(dc=-1;dc<=1;dc++) {
                if(dr == 0 && dc == 0) continue;
                if(!diagonal && dr != 0 && dc != 0) continue;
                if(!rowColumn && (dr == 0 || dc == 0)) continue;
                row = piece.getRow();
                col = piece.getColumn();
                while(true) {
                    row += dr;
                    col += dc;
                    if(row < 0 || row == board.getRows()) break;
                    if(col < 0 || col == board.getColumns()) break;
                    otherPiece = board.getPieceAt(row, col);
                    if(otherPiece != null) {
                        if(otherPiece.getPlayer() != piece.getPlayer()) {
                            moves.add(new ChessMove(piece, row, col, otherPiece));
                        }
                        break;
                    } else {
                        moves.add(new ChessMove(piece, row, col, null));
                    }
                }
            }
        }
        return moves;
    }

	public boolean isChessMoveLegal(ChessMove cm, Board b) {
		int dr, dc;
		int row, col;

		dr = cm.getRow() - cm.getPiece().getRow();
		dc = cm.getColumn() - cm.getPiece().getColumn();

		if(dr == 0 && dc == 0)
			return false;

		if(!((rowColumn && (dr == 0 || dc == 0))
			 || (diagonal && Math.abs(dr) == Math.abs(dc)))) {
			return false;
		}

		dr = (int)Math.signum(dr);
		dc = (int)Math.signum(dc);

		// Walk from the start to the end location, making sure
		// there's nothing in the way
		row = cm.getPiece().getRow();
		col = cm.getPiece().getColumn();
		while(row != cm.getRow() || col != cm.getColumn()) {
			row += dr;
			col += dc;
			if(b.getPieceAt(row, col) != null) break;
		}

		// If we didn't make it, there was a piece in the way
		if(row != cm.getRow() || col != cm.getColumn()) {
			return false;
		}

		return true;
	}
}
